/*
 * plprintf.c
 *   A tiny printf and vprintf function for pl...-applications
 *
 * Copyright (C) 1990 V.Siebert
 */

#include "prologdef.h"		/* System dependent information */
#include "plstd.h"		/* Standard definitions */
#include "plprintf.h"		/* includes also either stdarg.h or varargs.h */

#ifndef prolog_putchar
extern int prolog_putchar PROTO((int ch));
#endif

static int prolog_putint PROTO((long value,int radix));
#if BUILTIN_FLOATS
static int prolog_putfloat PROTO((double d));
#endif
static int prolog_putsquoted PROTO((char *str,int quote));

static int
prolog_putint(value,radix)
    long value;
    int radix;
{
    char *numbers = "0123456789ABCDEF";
    char buffer[32];
    char *bufput = buffer;
    int back,sign = 0;

    if (radix == 10 && value < 0) {
	sign = 1;
	value = -value;
    }
    do {
	*bufput++ = numbers[(int) (value % (long)radix)];
	value /= (long) radix;
    } while (value);
    if (sign)
	*bufput++ = '-';
    back = bufput - buffer;
    while (bufput > buffer)
	prolog_putchar(*--bufput);
    return back;
}

#if BUILTIN_FLOATS
static int
prolog_putfloat(double d)
{
    int done = 0,expo = 0;
    double dint,dfrac;
    long lfrac;

    if (d == 0.0) {
	prolog_putchar('0');
	prolog_putchar('.');
	prolog_putchar('0');
	return 3;
    }
    if (d < 0) {
	prolog_putchar('-');
	done++;
	d = -d;
    }
    if (d < 0.0001) {
    	while (d < 1.0)
    	    expo--, d *= 10.0;
    } else if (d > 10000.0) {
    	while (d > 10.0)
    	    expo++, d /= 10.0;
    }
    dint = (double)(long) d;
    dfrac = d - dint;
    if (expo == 0) {
    	done += prolog_putint((long) dint, 10);
	prolog_putchar('.');
	lfrac = (long) (dfrac * 100000.0);
	while (lfrac != 0L && lfrac % 10L == 0L)
	    lfrac /= 10L;
	done += prolog_putint(lfrac, 10) + 1;
    } else {
    	done += prolog_putint((long) dint, 10);
	lfrac = (long) (dfrac * 1e8);
	while (lfrac != 0L && lfrac % 10L == 0L)
	    lfrac /= 10L;
	if (lfrac != 0L) {
	    prolog_putchar('.');
	    done += prolog_putint(lfrac, 10) + 1;
	}
    	prolog_putchar('e');
    	done += prolog_putint((long) expo, 10) + 1;
    }
    return done;
}
#endif

static int
prolog_putsquoted(str,quote)
    char *str;
    int quote;
{
    int done;

    for (done = 0; *str; str++, done++) {
	if (*str == quote) {
	    prolog_putchar(quote);
	    done++;
	}
	prolog_putchar(*str);
    }
    return done;
}

int
prolog_vprintf (format,argv)
    char *format;
    va_list argv;
{
    int done = 0;
    char *scan,*str;
    int quoted,islong;
    long ival;
    unsigned long uval;
#if BUILTIN_FLOATS
    double dval;
#endif

    for (scan = format; *scan; scan++) {
	if (*scan == '%') {
	    islong = 0;
	    quoted = 0;
 fetch_again:
	    switch (*++scan) {
		case 'l':
		case 'L':
		    islong = 1;
		    goto fetch_again;
		case '\'':
		case '\"':
		    quoted = scan[-1];
		    goto fetch_again;
		case 'c':
		    ival = va_arg(argv,int);
		    prolog_putchar(ival);
		    done++;
		    break;
		case 'd':
		    ival = islong ? va_arg(argv,long) : (long) va_arg(argv,int);
		    done += prolog_putint(ival,10);
		    break;
		case 'u':
		    uval = islong ? va_arg(argv,unsigned long) : (unsigned long) va_arg(argv,unsigned);
		    done += prolog_putint(uval,10);
		    break;
		case 'x':
		    uval = islong ? va_arg(argv,unsigned long) : (unsigned long) va_arg(argv,unsigned);
		    done += prolog_putint(uval,16);
		    break;
		case 'o':
		    uval = islong ? va_arg(argv,unsigned long) : (unsigned long) va_arg(argv,unsigned);
		    done += prolog_putint(uval,8);
		    break;
		case 's':
		    str = va_arg(argv,char *);
		    done += prolog_putsquoted(str,quoted);
		    break;
#if BUILTIN_FLOATS
		case 'e':
		case 'f':
		case 'g':
		    dval = va_arg(argv,double);
		    done += prolog_putfloat(dval);
		    break;
#endif
		default:
		    prolog_putchar(*scan);
		    done++;
		    break;
	    }
	} else {
	    prolog_putchar(*scan);
	    done++;
	}
    }
    return done;
}

#if !defined(__STDC__)
int
prolog_printf (format,va_alist)
    char *format;
    va_dcl
#else
int
prolog_printf (char *format,...)
#endif
{
    va_list argv;
    int done;

#ifndef __STDC__
    va_start(argv);
#else
    va_start(argv,format);
#endif
    done = prolog_vprintf(format,argv);
    va_end(argv);
    return done;
}
