
/*
 *
 *	sowam_c
 *		Bytecode-Interpreter fuer die SOWAM
 *
 *
 *	FILE
 *		debug.c
 *
 *	PURPOSE
 *		Debugging-Funktionen
 *
 *
 *
 *	AUTHORS
 *		Andreas Schwab, Mike Wilhelm, Frank Lindert, Volker Siebert
 *
 */

#include <signal.h>
#include <ctype.h>
#include "sowam.h"
#include "debug.h"
#include "af-area.h"

struct breakpoint {
  struct breakpoint *next;
  code_addr pc;
  int opcode;
};

static struct breakpoint *breakpoints = NULL;
static int debug_depth = 10;

static void set_breakpoint PROTO((code_addr));

static int debug_handler_installed = FALSE;
static void remove_debug_handler PROTO((void));


struct af_entry *
find_name(addr)
  code_addr addr;
{
  struct af_entry *p, *best_match;
  code_addr best_addr = 0;

  for (p = b_af_area; p < b_af_area + initial_symbol_count; p++)
    {
      if (p->narrow_addr == addr || p->rewrite_addr == addr
	  || p->pred_addr == addr)
	return p;
      if (p->narrow_addr > best_addr && p->narrow_addr < addr)
	{
	  best_match = p;
	  best_addr = p->narrow_addr;
	}
      if (p->rewrite_addr > best_addr && p->rewrite_addr < addr)
	{
	  best_match = p;
	  best_addr = p->rewrite_addr;
	}
      if (p->pred_addr > best_addr && p->pred_addr < addr)
	{
	  best_match = p;
	  best_addr = p->pred_addr;
	}
    }
  if (best_addr == 0)
    internal_error("code address not found");

  return best_match;
}

static term *
make_term(addr)
  code_addr addr;
{
  int arity;
  struct af_entry *p;

  p = find_name(addr);
  arity = p->arity;
  CHECK_HEAP(arity+1);
  SET_TAG(*s_regs.h, T_AF);
  SET_VAL(*s_regs.h, (p - b_af_area) | (arity << 16));
  bcopy(&X_REG(1), s_regs.h+1, arity * sizeof(term));
  return s_regs.h;
}

void
write_occ_stack(addr)
  code_addr addr;
{
  term **p, old_ao;

  if (s_regs.ao != UNDEF)
    {
      old_ao = *s_regs.ao;
      if (TAG(old_ao) == T_UNDEF)
	{
	  SET_TAG(*s_regs.ao, T_STRUCT);
	  SET_VAL(*s_regs.ao, INDEX(make_term(addr)));
	}
      putchar('\n');
      for (p = s_regs.op->occ_os; p < s_regs.om; p++)
	{
	  write_internal(stdout, *p, 1200, TRUE, debug_depth);
	  putchar('\n');
	}
      write_internal(stdout, s_regs.ao, 1200, TRUE, debug_depth);
      putchar('\n');
      *s_regs.ao = old_ao;
    }
}

void
write_term_start(addr)
  code_addr addr;
{
  term old_ao;

  old_ao = *s_regs.ao;
  if (TAG(old_ao) == T_UNDEF)
    {
      SET_TAG(*s_regs.ao, T_STRUCT);
      SET_VAL(*s_regs.ao, INDEX(make_term(addr)));
    }
  putchar('\n');
  write_internal(stdout, s_regs.ts, 1200, TRUE, debug_depth);
  putchar('\n');
  *s_regs.ao = old_ao;
}

/* Setzt Breakpoints wo bei einem Skip der Debugger wieder
   gestartet werden soll */
static void
skip(art, default_cp)
     enum debug_type art;
     code_addr default_cp;
{
  term **func = s_regs.om, *t;

  if (art >= D_NARROW)
    {
      /* naechste Funktion auf Occurence-Stack suchen */
      for (;;)
	if (func == s_regs.op->occ_os)
	  {
	    /* war Toplevel-Funktion */
	    set_breakpoint(default_cp);
	    break;
	  }
	else
	  {
	    t = *--func;
	    DEREF_TERMP(t);
	    if (LOC(*t) == SKEL && IS_FUNCTION(*t))
	      {
		/* Funktion gefunden */
		if (art >= D_REWRITE)
		  set_breakpoint(AF_REWRITE(VAL(*t)));
		else
		  set_breakpoint(AF_NARROW(VAL(*t)));
		break;
	      }
	  }
    }
  else
    set_breakpoint(default_cp);

  if (s_regs.tfp != FAIL)
    set_breakpoint(s_regs.tfp);
  else if (s_regs.rfp != FAIL)
    set_breakpoint(s_regs.rfp);
  else if (s_regs.b != NULL)
    set_breakpoint(s_regs.b->bt_regs.p);

  s_glflags.debug = FALSE;
  install_debug_handler();
}

void
debug(art, cp, p)
     enum debug_type art;
     code_addr cp, p;
{
  int c, i;
  term *debug_term;
  int print_depth = debug_depth;
  char line_buffer[80];

  for (;;)
    {
      putchar('\n');
      switch (art)
	{
	case D_CALL:
#ifdef DEBUG_BOX
	  printf("(%d) ", s_regs.p_box);
#endif
	  printf("CALL: ");
	  debug_term = make_term(p);
	  break;

	case D_REDO:
#ifdef DEBUG_BOX
	  printf("(%d) ", s_regs.p_box);
#endif
	  printf("REDO: ");
	  debug_term = make_term(p);
	  break;

	case D_FAIL:
#ifdef DEBUG_BOX
	  printf("(%d) ", s_regs.p_box);
#endif
	  printf("FAIL: ");
	  debug_term = make_term(p);
	  break;

	case D_NARROW:
#ifdef DEBUG_BOX
	  printf("(%d) ", s_regs.p_box);
#endif
	  printf("NARROW: ");
	  debug_term = s_regs.ao;
	  break;

	case D_REWRITE:
#ifdef DEBUG_BOX
	  printf("(%d) ", s_regs.p_box);
#endif
	  printf("REWRITE: ");
	  debug_term = s_regs.ao;
	  break;

	case D_REWRITE_A:
#ifdef DEBUG_BOX
	  printf("(%d) ", s_regs.p_box);
#endif
	  printf("REWRITE: ");
	  debug_term = make_term(p);
	  break;

	default:
	  internal_error("debug");
	  break;
	}
      debug_write_term(debug_term, print_depth);
      print_depth = debug_depth;
      printf(" ? ");
      fflush(stdout);
      do
	{
	  if (fgets(line_buffer, 80, stdin) == NULL)
	    c = EOF;
	  else
	    {
	      if (line_buffer[strlen(line_buffer)-1] != '\n' && !feof(stdin))
		while ((c = getchar()) != '\n' && c != EOF)
		  ;
	      for (i = 0; c = line_buffer[i]; i++)
		if (c == '\n' || !isspace(c))
		  break;
	    }
	}
      while (c == 0);

      switch (c)
	{
	case EOF:
	  putchar('\n');
	  return;

	case 'c':
	case 'C':
	case '\n':
	  return;

	case 'a':
	case 'A':
	  stop(ABORT);
	  return;

	case 'n':
	case 'N':
	  s_glflags.debug = FALSE;
	  install_debug_handler();
	  return;

	case 's':
	case 'S':
	  skip(art, cp);
	  return;

	case 'o':
	case 'O':
	  if (art >= D_NARROW)
	    write_occ_stack(p);
	  break;

	case 't':
	case 'T':
	  if (art >= D_NARROW)
	    write_term_start(p);
	  break;

	case 'd':
	case 'D':
	  s_glflags.trace = !s_glflags.trace;
	  printf("\n[ Trace mode is %s ]\n", s_glflags.trace ? "on" : "off");
	  break;

	case 'p':
	case 'P':
	  break;

	case 'w':
	case 'W':
	  print_depth = 0x7fffffff;
	  break;

	case '<':
	  if (sscanf(&line_buffer[i+1], "%d", &print_depth) == 1)
	    {
	      printf("\n[ Debugger print depth is set to ");
	      if (print_depth == 0)
		{
		  print_depth = 0x7fffffff;
		  printf("unlimited ]\n");
		}
	      else
		{
		  if (print_depth < 0)
		    print_depth = -print_depth;
		  printf("%d ]\n", print_depth);
		}
	      debug_depth = print_depth;
	      break;
	    }
	  /* else fall through */
	      
	default:
	  printf("\n\
<cr>  creep       p   print             a      abort\n\
c     creep       w   write             d      toggle trace mode\n\
s     skip        o   occurrence stack  < <n>  set depth\n\
n     nodebug     t   term start\n\
");
	  break;
	}
    }
}

void
debug_return(pc)
     code_addr pc;
{
  struct breakpoint *bp, *nbp;

  for (bp = breakpoints; bp; bp = nbp)
    {
      *bp->pc = bp->opcode;
      nbp = bp->next;
      free(bp);
    }
  breakpoints = NULL;
  remove_debug_handler();
  s_glflags.debug = TRUE;
}

static void
set_breakpoint(pc)
     code_addr pc;
{
  struct breakpoint *bp;

  bp = (struct breakpoint *)xmalloc(sizeof(struct breakpoint));
  bp->next = breakpoints;
  breakpoints = bp;
  bp->pc = pc;
  bp->opcode = *pc;
  *pc = DEBUG_OPCODE;
}

static void
set_debug_mode(sig)
     int sig;
{
  remove_debug_handler();
  s_glflags.debug = 1;
}

void
install_debug_handler()
{
  if (!isatty(0)) return;
  if (signal(SIGINT, SIG_IGN) != SIG_IGN)
    {
      debug_handler_installed = TRUE;
      signal(SIGINT, set_debug_mode);
    }
}

static void
remove_debug_handler()
{
  if (debug_handler_installed)
    {
      signal(SIGINT, SIG_DFL);
      debug_handler_installed = FALSE;
    }
}
