
%% Prolog-Praedikate zur Uebersetzung von AbSy(SOWAM) in Bytecode
%%   --> hier : Die Bytecode-Definitionen fuer alle SOWAM-Instruktionen
%%  <c> Andreas Schwab, Berthold Josephs
 
%% Standarddef.:

%%  --------------------- SOWAM-Befehl < Name...> --------------------------
%% b_bytecode(SOWAM-Befehl,
%%	      SymbolList, LSymTab, GSymTab, CSymTab, Bytecode, Length) :-
%%
%% b_bytecode/8:
%%
%% Aufgabe:
%%	Diese Praedikate ermitteln die zu einem SOWAM-Befehl
%%	gehoerenden Byte-Codes.
%%
%% Eingabe:
%%	SOWAM-Befehl: waehlt die passende Regel aus
%%	SymbolList:    Die vom Compiler uebergebene Liste von Symbolen:
%%	 [entry(Symbol,Stelligkeit,{cons|pred(Label)|rewr(Label)|func(NLabel,RLabel)},
%%		Operator,Praezedenz),...]
%%	LSymTab: Liste der lokalen Symbole( [label(Name,Adresse),...]
%%      GSymTab: Liste der globalen Symbole ( " )
%%
%% Ausgabe:
%%	Bytecode: Uebersetzter Bytecode
%%	Length: Laenge diese Codes
%% ========================================================================


%% ========================================================================
%% 					       2-Byte Befehle ohne Argument
%% ========================================================================

%% --------------------- SOWAM-Befehl < allocate > --------------------------
b_bytecode(allocate, _, _, _, _,
	   bc0(1), 1).


%% --------------------- SOWAM-Befehl < deallocate > --------------------------
b_bytecode(deallocate, _, _, _, _,
	   bc0(2), 1).


%% --------------------- SOWAM-Befehl < allocate_occ > ------------------------
b_bytecode(allocate_occ, _, _, _, _,
	   bc0(3), 1).


%% --------------------- SOWAM-Befehl < deallocate_occ > ----------------------
b_bytecode(deallocate_occ, _, _, _, _,
	   bc0(4), 1).


%% --------------------- SOWAM-Befehl < proceed > --------------------------
b_bytecode(proceed, _, _, _, _,
	   bc0(5), 1).


%% --------------------- SOWAM-Befehl < unify_nil > --------------------------
b_bytecode(unify_nil, _, _, _, _,
	   bc0(6), 1).


%% --------------------- SOWAM-Befehl < fail > --------------------------
b_bytecode(fail, _, _, _, _,
	   bc0(7), 1).


%% --------------------- SOWAM-Befehl < stop > --------------------------
b_bytecode(stop, _, _, _, _,
	   bc0(8), 1).


%% --------------------- SOWAM-Befehl < pop_occ > --------------------------
b_bytecode(pop_occ, _, _, _, _,
	   bc0(9), 1).


%% --------------------- SOWAM-Befehl < execute_narrowing > -------------------
b_bytecode(execute_narrowing(ao), _, _, _, _,
	   bc0(10), 1).


%% --------------------- SOWAM-Befehl < execute_rewriting > -------------------
b_bytecode(execute_rewriting(ao), _, _, _, _,
	   bc0(11), 1).


%% --------------------- SOWAM-Befehl < inner_reflection > --------------------
b_bytecode(inner_reflection, _, _, _, _,
	   bc0(12), 1).


%% --------------------- SOWAM-Befehl < put_nil_occ > -------------------------
b_bytecode(put_nil_occ, _, _, _, _,
	   bc0(13), 1).


%% --------------------- SOWAM-Befehl < put_list_occ > ------------------------
b_bytecode(put_list_occ, _, _, _, _,
	   bc0(14), 1).


%% --------------------- SOWAM-Befehl < trust_me_else_fail > ------------------
b_bytecode(trust_me_else_fail, _, _, _, _,
	   bc0(15), 1).


%% --------------------- SOWAM-Befehl < r_trust_me_else_fail > ----------------
b_bytecode(r_trust_me_else_fail, _, _, _, _,
	   bc0(16), 1).


%% --------------------- SOWAM-Befehl < rebuild_occ_stack > -------------------
b_bytecode(rebuild_occ_stack, _, _, _, _,
	   bc0(17), 1).


%% --------------------- SOWAM-Befehl < write_nil > --------------------------
b_bytecode(write_nil, _, _, _, _,
	   bc0(18), 1).


%% --------------------- SOWAM-Befehl < read_nil > --------------------------
b_bytecode(read_nil, _, _, _, _,
	   bc0(19), 1).


%% --------------------- SOWAM-Befehl < invalid_os > --------------------------
b_bytecode(invalid_os, _, _, _, _,
	   bc0(20), 1).


%% --------------------- SOWAM-Befehl < push_act_occ > ------------------------
b_bytecode(push_act_occ, _, _, _, _,
	   bc0(21), 1).


%% --------------------- SOWAM-Befehl < l_trust_me_else_fail > ----------------
b_bytecode(l_trust_me_else_fail, _, _, _, _,
	   bc0(22), 1).


%% --------------------- SOWAM-Befehl < call_rewriting > ----------------------
b_bytecode(call_rewriting(ao), _, _, _, _,
	   bc0(23), 1).


%% --------------------- SOWAM-Befehl < reject > --------------------------
b_bytecode(reject, _, _, _, _,
	   bc0(24), 1).


%% --------------------- SOWAM-Befehl < copy_pop_occ > ------------------------
b_bytecode(copy_pop_occ, _, _, _, _,
	   bc0(25), 1).


%% --------------------- SOWAM-Befehl < reflection > --------------------------
b_bytecode(reflection, _, _, _, _,
	   bc0(26), 1).


%% ========================================================================
%%				2-Byte Befehle mit einem Argument im Opcode
%% ========================================================================

%% --------------------- SOWAM-Befehl < write_void > --------------------------
b_bytecode(write_void(N), _, _, _, _,
	   bc1(32, N), 1).


%% --------------------- SOWAM-Befehl < put_nil > --------------------------
b_bytecode(put_nil(x(N)), _, _, _, _,
	   bc1(33, N), 1).


%% --------------------- SOWAM-Befehl < get_nil > --------------------------
b_bytecode(get_nil(x(N)), _, _, _, _,
	   bc1(34, N), 1).


%% --------------------- SOWAM-Befehl < put_list > --------------------------
b_bytecode(put_list(x(N)), _, _, _, _,
	   bc1(35, N), 1).


%% --------------------- SOWAM-Befehl < get_list > --------------------------
b_bytecode(get_list(x(N)), _, _, _, _,
	   bc1(36, N), 1).


%% --------------------- SOWAM-Befehl < unify_void > --------------------------
b_bytecode(unify_void(N), _, _, _, _,
	   bc1(37, N), 1).


%% --------------------- SOWAM-Befehl < put_unsafe_value_occ_y > --------------
b_bytecode(put_unsafe_value_occ(y(N)), _, _, _, _,
	   bc1(38, N), 1).


%% --------------------- SOWAM-Befehl < unify_variable_x > --------------------
b_bytecode(unify_variable(x(N)), _, _, _, _,
	   bc1(39, N), 1).


%% --------------------- SOWAM-Befehl < write_variable_y > --------------------
b_bytecode(write_variable(x(N)), _, _, _, _,
	   bc1(40, N), 1).


%% --------------------- SOWAM-Befehl < unify_variable_y > --------------------
b_bytecode(unify_variable(y(N)), _, _, _, _,
	   bc1(41, N), 1).


%% --------------------- SOWAM-Befehl < unify_value_x > -----------------------
b_bytecode(unify_value(x(N)), _, _, _, _,
	   bc1(42, N), 1).


%% --------------------- SOWAM-Befehl < write_value_x > -----------------------
b_bytecode(write_value(x(N)), _, _, _, _,
	   bc1(43, N), 1).


%% --------------------- SOWAM-Befehl < unify_value_y > -----------------------
b_bytecode(unify_value(y(N)), _, _, _, _,
	   bc1(44, N), 1).


%% --------------------- SOWAM-Befehl < unify_local_value_x > -----------------
b_bytecode(unify_local_value(x(N)), _, _, _, _,
	   bc1(45, N), 1).


%% --------------------- SOWAM-Befehl < write_local_value_x > -----------------
b_bytecode(write_local_value(x(N)), _, _, _, _,
	   bc1(46, N), 1).


%% --------------------- SOWAM-Befehl < unify_local_value_y > -----------------
b_bytecode(unify_local_value(y(N)), _, _, _, _,
	   bc1(47, N), 1).


%% --------------------- SOWAM-Befehl < put_var_occ_x > -----------------------
b_bytecode(put_var_occ(x(N)), _, _, _, _,
	   bc1(48, N), 1).


%% --------------------- SOWAM-Befehl < put_var_occ_y > -----------------------
b_bytecode(put_var_occ(y(N)), _, _, _, _,
	   bc1(49, N), 1).


%% --------------------- SOWAM-Befehl < put_value_occ_x > ---------------------
b_bytecode(put_value_occ(x(N)), _, _, _, _,
	   bc1(50, N), 1).


%% --------------------- SOWAM-Befehl < put_value_occ_y > ---------------------
b_bytecode(put_value_occ(y(N)), _, _, _, _,
	   bc1(51, N), 1).


%% --------------------- SOWAM-Befehl < load_occ_x > --------------------------
b_bytecode(load_occ(x(N)), _, _, _, _,
	   bc1(52, N), 1).


%% --------------------- SOWAM-Befehl < set_begin_of_term_x > -----------------
b_bytecode(set_begin_of_term(x(N)), _, _, _, _,
	   bc1(53, N), 1).


%% --------------------- SOWAM-Befehl < push_occ_x > --------------------------
b_bytecode(push_occ(x(N)), _, _, _, _,
	   bc1(54, N), 1).


%% --------------------- SOWAM-Befehl < write_variable_y > --------------------
b_bytecode(write_variable(y(N)), _, _, _, _,
	   bc1(55, N), 1).


%% --------------------- SOWAM-Befehl < write_value_y > -----------------------
b_bytecode(write_value(y(N)), _, _, _, _,
	   bc1(56, N), 1).


%% --------------------- SOWAM-Befehl < write_local_value_y > -----------------
b_bytecode(write_local_value(y(N)), _, _, _, _,
	   bc1(57, N), 1).


%% --------------------- SOWAM-Befehl < match_list > --------------------------
b_bytecode(match_list(x(N)), _, _, _, _,
	   bc1(59, N), 1).


%% --------------------- SOWAM-Befehl < load_occ_y > --------------------------
b_bytecode(load_occ(y(N)), _, _, _, _,
	   bc1(60, N), 1).


%% --------------------- SOWAM-Befehl < push_occ_y > --------------------------
b_bytecode(push_occ(y(N)), _, _, _, _,
	   bc1(61, N), 1).


%% --------------------- SOWAM-Befehl < write_and_ask > -----------------------
b_bytecode(write_and_ask(x(N)), _, _, _, _,
	   bc1(62, N), 1).


%% --------------------- SOWAM-Befehl < built_in > --------------------------
b_bytecode(built_in(Name/Stell), _, _, _, _,
	   bc1(63, N), 1) :-
    sowam_builtin(Name, Stell, N).


%% --------------------- SOWAM-Befehl < set_begin_of_term_y > -----------------
b_bytecode(set_begin_of_term(y(N)), _, _, _, _,
	   bc1(64, N), 1).


%% --------------------- SOWAM-Befehl < read_void > --------------------------
b_bytecode(read_void(N), _, _, _, _,
	   bc1(65, N), 1).


%% --------------------- SOWAM-Befehl < read_variable_x > ---------------------
b_bytecode(read_variable(x(N)), _, _, _, _,
	   bc1(66, N), 1).


%% --------------------- SOWAM-Befehl < read_variable_y > ---------------------
b_bytecode(read_variable(y(N)), _, _, _, _,
	   bc1(67, N), 1).


%% --------------------- SOWAM-Befehl < read_value_x > ------------------------
b_bytecode(read_value(x(N)), _, _, _, _,
	   bc1(68, N), 1).


%% --------------------- SOWAM-Befehl < read_value_y > ------------------------
b_bytecode(read_value(y(N)), _, _, _, _,
	   bc1(69, N), 1).


%% --------------------- SOWAM-Befehl < match_nil > --------------------------
b_bytecode(match_nil(x(N)), _, _, _, _,
	   bc1(70, N), 1).


%% --------------------- SOWAM-Befehl < put_list > --------------------------
b_bytecode(put_list(y(N)), _, _, _, _,
	   bc1(71, N), 1).


%% ========================================================================
%% 					  6-Byte Befehle mit einem Argument
%% ========================================================================


%% --------------------- SOWAM-Befehl < execute > --------------------------
b_bytecode(execute(Name/Stell), SymbolList, _, GSymTab, _,
	   bc1adr(80, Labeladr), 3) :-
    b_pred_label(Name, Stell, SymbolList, Label),
    b_lookup(GSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < retry_me_else > -----------------------
b_bytecode(retry_me_else(Label), _, LSymTab, _, _,
	   bc1adr(81, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < r_retry_me_else > ---------------------
b_bytecode(r_retry_me_else(Label), _, LSymTab, _, _,
	   bc1adr(82, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < retry > --------------------------
b_bytecode(retry(Label), _, LSymTab, _, _,
	   bc1adr(83, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < r_retry > --------------------------
b_bytecode(r_retry(Label), _, LSymTab, _, _,
	   bc1adr(84, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < trust > --------------------------
b_bytecode(trust(Label), _, LSymTab, _, _,
	   bc1adr(85, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < r_trust > --------------------------
b_bytecode(r_trust(Label), _, LSymTab, _, _,
	   bc1adr(86, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < unify_constant > ----------------------
b_bytecode(unify_constant(C), _, _, _, CSymTab,
	   bc1const(87, Index, 0), 3) :-
    b_lookup(CSymTab, const(C, 0), Index).


%% --------------------- SOWAM-Befehl < put_struct_occ > ----------------------
b_bytecode(put_struct_occ(Name/Stell), _, _, _, CSymTab,
	   bc1const(88, Index, Stell), 3) :-
    b_lookup(CSymTab, const(Name, Stell), Index).


%% --------------------- SOWAM-Befehl < put_const_occ > -----------------------
b_bytecode(put_const_occ(C), _, _, _, CSymTab,
	   bc1const(89, Index, 0), 3) :-
    b_lookup(CSymTab, const(C, 0), Index).


%% --------------------- SOWAM-Befehl < execute_rewriting > -------------------
b_bytecode(execute_rewriting(Name/Stell), SymbolList, _, GSymTab, _,
	   bc1adr(90, Labeladr), 3) :-
    b_rewrite_label(Name, Stell, SymbolList, Label),
    b_lookup(GSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < put_function_occ > --------------------
b_bytecode(put_function_occ(Name/Stell), _, _, _, CSymTab,
	   bc1const(91, Index, Stell), 3) :-
    b_lookup(CSymTab, const(Name, Stell), Index).


%% --------------------- SOWAM-Befehl < call_rewriting > ----------------------
b_bytecode(call_rewriting(Name/Stell), SymbolList, _, GSymTab, _,
	   bc1adr(92, Labeladr), 3) :-
    b_rewrite_label(Name, Stell, SymbolList, Label),
    b_lookup(GSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < write_constant > ----------------------
b_bytecode(write_constant(C), _, _, _, CSymTab,
	   bc1const(93, Index, 0), 3) :-
    b_lookup(CSymTab, const(C, 0), Index).


%% --------------------- SOWAM-Befehl < r_try_me_else > -----------------------
b_bytecode(r_try_me_else(Label), _, LSymTab, _, _,
	   bc1adr(94, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < r_try > --------------------------
b_bytecode(r_try(Label), _, LSymTab, _, _,
	   bc1adr(95, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < read_constant > -----------------------
b_bytecode(read_constant(C), _, _, _, CSymTab,
	   bc1const(96, Index, 0), 3) :-
    b_lookup(CSymTab, const(C, 0), Index).


%% ========================================================================
%% 			  6-Byte Befehle mit zwei Argumenten (1. im Opcode)
%% ========================================================================

%% --------------------- SOWAM-Befehl < call > --------------------------
b_bytecode(call(Name/Stell, N), SymbolList, _, GSymTab, _,
	   bc2adr(112, N, Labeladr), 3) :-
    b_pred_label(Name, Stell, SymbolList, Label),
    b_lookup(GSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < call_narrowing > ----------------------
b_bytecode(call_narrowing(ao,N), _, _, _, _,
	   bc2adr(113, N, 0), 3).


%% --------------------- SOWAM-Befehl < call_rewriting > ----------------------
b_bytecode(call_rewriting(ao,N), _, _, _, _,
	   bc2adr(114, N, 0), 3).


%% --------------------- SOWAM-Befehl < try_me_else > -------------------------
b_bytecode(try_me_else(Label,N), _, LSymTab, _, _,
	   bc2adr(115, N, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < get_structure > -----------------------
b_bytecode(match_structure(Name/Stell, x(I)), _, _, _, CSymTab,
	   bc2const(116, I, Index, Stell), 3) :-
    b_lookup(CSymTab, const(Name, Stell), Index).


%% --------------------- SOWAM-Befehl < try > --------------------------
b_bytecode(try(Label, N), _, LSymTab, _, _,
	   bc2adr(117, N, Labeladr), 3) :-
    b_lookup(LSymTab, Label, Labeladr).
	

%% --------------------- SOWAM-Befehl < match_constant > ----------------------
b_bytecode(match_constant(C, x(I)), _, _, _, CSymTab,
	   bc2const(118, I, Index, 0), 3) :-
    b_lookup(CSymTab, const(C, 0), Index).


%% --------------------- SOWAM-Befehl < call_rewriting > ----------------------
b_bytecode(call_rewriting(Name/Stell, N), SymbolList, _, GSymTab, _,
	   bc2adr(119, N, Labeladr), 3) :-
    b_rewrite_label(Name, Stell, SymbolList, Label),
    b_lookup(GSymTab, Label, Labeladr).


%% --------------------- SOWAM-Befehl < put_structure > -----------------------
b_bytecode(put_structure(Name/Stell, x(I)), _, _, _, CSymTab,
	   bc2const(121, I, Index, Stell), 3) :-
    b_lookup(CSymTab, const(Name, Stell), Index).


%% --------------------- SOWAM-Befehl < put_structure > -----------------------
b_bytecode(put_structure(Name/Stell, y(I)), _, _, _, CSymTab,
	   bc2const(122, I, Index, Stell), 3) :-
    b_lookup(CSymTab, const(Name, Stell), Index).


%% --------------------- SOWAM-Befehl < get_structure > -----------------------
b_bytecode(get_structure(Name/Stell, x(I)), _, _, _, CSymTab,
	   bc2const(123, I, Index, Stell), 3) :-
    b_lookup(CSymTab, const(Name, Stell), Index).


%% --------------------- SOWAM-Befehl < put_constant > ------------------------
b_bytecode(put_constant(C, x(I)), _, _, _, CSymTab,
	   bc2const(125, I, Index, 0), 3) :-
    b_lookup(CSymTab, const(C, 0), Index).


%% --------------------- SOWAM-Befehl < get_constant > ------------------------
b_bytecode(get_constant(C, x(I)), _, _, _, CSymTab,
	   bc2const(126, I, Index, 0), 3) :-
    b_lookup(CSymTab, const(C, 0), Index).


%% ========================================================================
%% 					 4-Byte Befehle mit zwei Argumenten
%% ========================================================================

%% --------------------- SOWAM-Befehl < match_value_x > -----------------------
b_bytecode(match_value(x(N), x(I)), _, _, _, _,
	   bc2(129, N, I), 2).


%% --------------------- SOWAM-Befehl < match_value_y > -----------------------
b_bytecode(match_value(y(N), x(I)), _, _, _, _,
	   bc2(130, N, I), 2).


%% --------------------- SOWAM-Befehl < put_variable_y > ----------------------
b_bytecode(put_variable(y(N), x(I)), _, _, _, _,
	   bc2(131, N, I), 2).


%% --------------------- SOWAM-Befehl < put_value_x > -------------------------
b_bytecode(put_value(x(N), x(I)), _, _, _, _,
	   bc2(132, N, I), 2).


%% --------------------- SOWAM-Befehl < put_value_y > -------------------------
b_bytecode(put_value(y(N), x(I)), _, _, _, _,
	   bc2(133, N, I), 2).


%% --------------------- SOWAM-Befehl < get_variable_x > ----------------------
b_bytecode(get_variable(x(N), x(I)), _, _, _, _,
	   bc2(134, N, I), 2).


%% --------------------- SOWAM-Befehl < get_variable_y > ----------------------
b_bytecode(get_variable(y(N), x(I)), _, _, _, _,
	   bc2(135, N, I), 2).


%% --------------------- SOWAM-Befehl < get_value_x > -------------------------
b_bytecode(get_value(x(N), x(I)), _, _, _, _,
	   bc2(136, N, I), 2).


%% --------------------- SOWAM-Befehl < get_value_y > -------------------------
b_bytecode(get_value(y(N), x(I)), _, _, _, _,
	   bc2(137, N, I), 2).


%% --------------------- SOWAM-Befehl < put_unsafe_value_y > ------------------
b_bytecode(put_unsafe_value(y(N), x(I)), _, _, _, _,
	   bc2(138, N, I), 2).


%% --------------------- SOWAM-Befehl < put_variable_x > ----------------------
b_bytecode(put_variable(x(N), x(I)), _, _, _, _,
	   bc2(139, N, I), 2).


%% --------------------- SOWAM-Befehl < put_unsafe_value_x > ------------------
b_bytecode(put_unsafe_value(x(N), x(I)), _, _, _, _,
	   bc2(140, N, I), 2).


%% ========================================================================
%% 					4-Byte Befehle mit Tabelle als Arg.
%% ========================================================================

%% --------------------- SOWAM-Befehl < switch_on_structure > -----------------
b_bytecode(switch_on_structure(N, Table), SymbolList, LSymTab, _, CSymTab,
	   bc2table(160, N, SwitchTable), Length) :-
    b_struct_switch_table(Table, SymbolList, LSymTab, CSymTab, SwitchTable),
    Length is 4 * N + 2.


%% --------------------- SOWAM-Befehl < switch_on_constant > ------------------
b_bytecode(switch_on_constant(N, Table), SymbolList, LSymTab, _, CSymTab,
	   bc2table(161, N, SwitchTable), Length) :-
    b_const_switch_table(Table, SymbolList, LSymTab, CSymTab, SwitchTable),
    Length is 4 * N + 2.


%% ========================================================================
%% 					   18-Byte Befehle mit 4 Argumenten
%% ========================================================================

%% --------------------- SOWAM-Befehl < switch_on_term > ----------------------
b_bytecode(switch_on_term(Label_v, Label_c, Label_l, Label_s),
	   _, LSymTab, _, _,
	   bc4adr(192,Label_v_adr,Label_c_adr,Label_l_adr,Label_s_adr),
	   9) :-
    b_lookup(LSymTab, Label_v, Label_v_adr),
    b_lookup(LSymTab, Label_c, Label_c_adr),
    b_lookup(LSymTab, Label_l, Label_l_adr),
    b_lookup(LSymTab, Label_s, Label_s_adr).


%% ========================================================================
%% 					    8-Byte Befehle mit 3 Argumenten
%% ========================================================================

%% --------------------- SOWAM-Befehl < l_try_me_else > -----------------------
b_bytecode(l_try_me_else(Label, N, M), _, LSymTab, _, _,
	   bc3adr(224, Labeladr, N, M), 4) :-
    b_lookup(LSymTab, Label, Labeladr).


% ==========++++++++++++========================+++++++++++
% fuer Testzwecke

b_bytecode(fehler(_), _, _, _, _, bc0(0), 1).

b_bytecode(IRGENDWAS, _, _, _, _, bc0(0), 1) :-
    write('Unknown opcode: '), writeq(IRGENDWAS), nl.
