%%% Uebersetzung AbsySOWAM -> Bytecode
%%% hier: Ausgabe des Bytecodes in eine Datei
%%% Berthold Josephs, Andreas Schwab

%%% b_bc_out/8
%%
%% Aufgabe:
%%   Ausgabe des erzeugten Bytecodes incl. Header und Tabellen
%%
%% b_bc_out(BCList, PCMax, AFTable, AFMax, TextTable, TextLength,
%%	    RelocTable, Entry)
%%
%% Eingabe:
%%   BCList:	 der Bytecode
%%   PCMax:	 der Wert des Programmcounters nach Uebersetzung
%%   AFTable:	 Tabelle der Atome/Funktoren
%%   AFMax:	 Anzahl der Eintraege in AFTable
%%   TextTable:	 textuelle Darstellung der Atome/Funktoren als Liste von Bytes
%%   TextLength: Laenge der TextTable
%%   RelocTable: Relocation-Tabelle
%%   Entry:	 Code-Adresse des auszufuehrenden Goals
%%

b_bc_out(BCList, PCMax, AFTable, AFMax, TextTable, TextLength,
	 RelocTable, Entry) :-
    bytecode_version(Version),
    length(RelocTable, RLength),
    b_write_header(Version, PCMax, AFMax, TextLength, RLength, Entry),
%   b_debug_write('BCList', BCList),
    b_write_code(BCList),
    b_write_af_table(AFTable),
    b_write_text_table(TextTable),
    b_write_relocation(RelocTable).

%%% b_write_header/6
%%

b_write_header(Version, PC, AFMax, TextLength, RelocEntries, Entry) :-
    b_put_list("FALF"),
    b_put_four(Version),
    CodeLength is PC * 2,
    b_put_four(CodeLength),
    b_debug_write('CodeLength', CodeLength),
    AFLength is AFMax * 20,
    b_put_four(AFLength),
    b_debug_write('AFLength', AFLength),
    b_put_four(TextLength),
    b_debug_write('TextLength', TextLength),
    RelocLength is RelocEntries * 4,
    b_put_four(RelocLength),
    b_debug_write('RelocLength', RelocLength),
    b_put_four(Entry).


%%% b_write_code/1
%%

b_write_code([BCList | BCRest]) :-
    b_write_code1(BCList),
    b_write_code(BCRest).
b_write_code([]) :- !.

b_write_code1([BC | BCRest]) :-
    b_write_bc(BC),
    b_write_code1(BCRest).
b_write_code1([]).

%%% b_write_bc/1
%%

b_write_bc(bc0(N)) :-
    put(0),
    put(N).
b_write_bc(bc1(X, Arg)) :-
    put(Arg),
    put(X).
b_write_bc(bc1adr(X, Adr)) :-
    put(0),
    put(X),
    b_put_four(Adr).
b_write_bc(bc1const(X, I, S)) :-
    put(0),
    put(X),
    b_put_two(S),
    b_put_two(I).
b_write_bc(bc2adr(X, Arg, Adr)) :-
    put(Arg),
    put(X),
    b_put_four(Adr).
b_write_bc(bc2const(X, Arg, I, S)) :-
    put(Arg),
    put(X),
    b_put_two(S),
    b_put_two(I).
b_write_bc(bc2(X, Arg1, Arg2)) :-
    put(0),
    put(X),
    put(Arg1),
    put(Arg2).
b_write_bc(bc2table(X, Len, Table)) :-
    put(0),
    put(X),
    b_put_two(Len),
    b_write_switch_table(Table).
b_write_bc(bc4adr(X, Adr1, Adr2, Adr3, Adr4)) :-
    put(0),
    put(X),
    b_put_four(Adr1),
    b_put_four(Adr2),
    b_put_four(Adr3),
    b_put_four(Adr4).
b_write_bc(bc3adr(X, Adr, Arg1, Arg2)) :-
    put(0),
    put(X),
    b_put_four(Adr),
    put(Arg1),
    put(Arg2).


%%% b_write_switch_table/1
%%
%% Aufgabe:
%%   Schreibt eine Switch-Tabelle
%%
%% b_write_switch_table(STable)
%%
%% Eingabe:
%%   STable: die Tabelle
%%

b_write_switch_table([entry(AFIndex, Stell, CodeAdr) | TableRest]) :-
    b_put_two(Stell),
    b_put_two(AFIndex),
    b_put_four(CodeAdr),
    b_write_switch_table(TableRest).
b_write_switch_table([]).


%%% b_write_af_table/1
%%

b_write_af_table([entry(NAdr, RAdr, TAdr, Stell, Op, Prio) | TableRest]) :-
    b_put_four(NAdr),
    b_put_four(RAdr),
    b_put_four(TAdr),
    b_put_four(Stell),
    b_put_two(Op),
    b_put_two(Prio),
    b_write_af_table(TableRest).
b_write_af_table([]).


%%% b_write_text_table/1
%%
%% Aufgabe:
%%   Schreibt die Text-Tabelle
%%
%% b_write_text_table(Table)
%%
%% Eingabe:
%%   Table:  die Text-Tabelle als Liste von Bytes (Integer)
%%

b_write_text_table([Text | TableRest]) :-
    b_put_list(Text),
    b_write_text_table(TableRest).
b_write_text_table([]).


%%% b_write_relocation/1
%%

b_write_relocation([Reloc | Rest]) :-
    b_write_reloc1(Reloc),
    b_write_relocation(Rest).
b_write_relocation([]).

%%% b_write_reloc1/1
%%

b_write_reloc1(reloc(PC, Art)) :-
    b_reloc_code(Art, Code),
    put(Code),
    Val is PC * 2,
    b_put_three(Val).

%%% b_reloc_code/2
%%
%% Ermittelt zum Relocation-Typ den entsprechenden Code

b_reloc_code(symbol, 0).
b_reloc_code(code, 1).
b_reloc_code(table, 2).


%%% b_put_list/1
%%
%% Aufgabe:
%%   Schreibt eine Liste von Bytes
%%
%% b_put_list(BList)
%%
%% Eingabe:
%%   BList:  Liste von Bytes (Integer)
%%

b_put_list([B | Rest]) :-
    put(B),
    b_put_list(Rest).
b_put_list([]).


%% Zahlen werden im BIG-ENDIAN-Format geschrieben

%%% b_put_four/1
%%

b_put_four(Long) :-
    b_long_to_byte(Long, B1, B2, B3, B4),
    put(B1),
    put(B2),
    put(B3),
    put(B4).

%%% b_put_three/1
%%

b_put_three(Long) :-
    b_long_to_byte(Long, _, B2, B3, B4),
    put(B2),
    put(B3),
    put(B4).

%%% b_put_two/1
%%

b_put_two(Word) :-
    b_word_to_byte(Word, B1, B2),
    put(B1),
    put(B2).

	
%%% b_long_to_byte/5
%%
%% Aufgabe:
%%   wandelt einen Wert in 4 Bytes um
%%
%% b_long_to_byte(Long, B4, B3, B2, B1)
%%
%% Eingabe:
%%   Long
%%
%% Ausgabe:
%%   B4: hoechstwertiges Byte
%%   B3:
%%   B2:
%%   B1:
%%

b_long_to_byte(Long, B4, B3, B2, B1) :-
    B1 is Long /\ 255,
    H2 is Long >> 8,
    B2 is H2 /\ 255,
    H3 is H2 >> 8,
    B3 is H3 /\ 255,
    B4 is (H3 >> 8) /\ 255.


%%% b_word_to_byte/3
%%
%% Aufgabe:
%%   Teilt einen 16-Bit-Wert in 2 Bytes
%%
%% b_word_to_byte(Word, MSB, LSB)
%%
%% Eingabe:
%%   Word: ein 16-Bit-Wert
%%
%% Ausgabe:
%%   MSB: hoeherwertiges Byte
%%   LSB: niederwertiges Byte
%%

b_word_to_byte(Word, MSB, LSB) :-
    LSB is Word /\ 255,
    MSB is (Word >> 8) /\ 255.
