/*-------------------------------------------------------------------------*/
/* Prolog to Wam Compiler               INRIA Rocquencourt - ChLoE Project */
/* C Run-time                                           Daniel Diaz - 1991 */
/*                                                                         */
/* Atoms and Predicates Tables Management - Header file                    */
/*                                                                         */
/* atom_pred.h                                                             */
/*-------------------------------------------------------------------------*/
#include "bool.h"
#include "hash.h"




/*---------------------------------*/
/* Constants                       */
/*---------------------------------*/

#define MAX_ATOM_TBL_SIZE         16384           /* in number of elements */
#define MAX_PRED_TBL_SIZE          8192           /* in number of elements */
#define MAX_OPER_TBL_SIZE          1024           /* in number of elements */

#define MAX_MODULE                 256


#define NB_BITS_OF_ARITY           8
#define MAX_ARITY                  (1 << NB_BITS_OF_ARITY)




          /* Atom & Pred Manipulations */

#define Wam_Ptr(p)                 ((unsigned) (p)-MALLOC_START)
#define Real_Ptr(p)                ((unsigned) (p)+MALLOC_START)




          /* Characters */

#define UNKNOWN                    0              /* unknown character     */
#define LA                         1              /* layout character      */
#define SC                         2              /* solo character        */
#define QT                         4              /* quote                 */
#define DQ                         8              /* double quote          */
#define BQ                        16              /* back quote            */
#define SY                        32              /* symbol                */
#define PC                        64              /* ponctuation character */
#define DI                       128              /* digit                 */
#define UL                       256              /* underline             */
#define CL                       512              /* capital letter        */
#define SL                      1024              /* small letter          */
#define CM                      2048              /* comment character (%) */


          /* Atom */

#define NOT_AN_ATOM                0
#define IDENTIFIER_ATOM            1
#define SYMBOL_ATOM                2
#define SOLO_ATOM                  3
#define OTHER_ATOM                 4




          /* Predicate */

#define Make_Pred_Key(a,n)         Functor_Arity(a,n)
#define Atom_Of_Pred(p)            Functor_Of((p)->f_n)
#define Arity_Of_Pred(p)           Arity_Of((p)->f_n)




          /* Operator */

#define Make_Oper_Key(a,t)         ((unsigned) (a) | (t))
#define Atom_Of_Oper(o)            ((AtomInf *) (((o)->a_t) & (~0x3)))
#define Type_Of_Oper(o)            (((o)->a_t) & (0x3))


#define PREFIX                     0
#define POSTFIX                    1
#define INFIX                      2


#define MAX_PREC                   1200
#define MAX_ARG_OF_FUNCTOR_PREC     999




          /* Error Messages */

#define ERR_ALLOC_TABLE            "atom_pred: Memory allocation fault"
#define ERR_ATOM_TBL_FULL          "Atom table full"
#define ERR_PRED_TBL_FULL          "Predicate table full"
#define ERR_OPER_TBL_FULL          "Operator table full"





/*---------------------------------*/
/* Type Definitions                */
/*---------------------------------*/

typedef char *AtomTbl;
typedef char *PredTbl;
typedef char *OperTbl;
typedef char *SwtTbl;




typedef struct
    {
     char   *name;                       /* modeule name                   */
     Bool    debugged;                   /* has it been compiled with -dbg?*/
     PredTbl pred_tbl;
    }ModulInf;




typedef struct
    {
     char   *name;                       /* key is <name> (the string)     */
     int     type;                       /* IDENTIFIER, SYMBOL, SOLO, OTHER*/
     Bool    has_quote;                  /* does name contain any '        */
     Bool    needs_quote;                /* does name need ' around it ?   */
     int     length;                     /* its length (in characters)     */
     int     info;                       /* an info (e.g. g_vars)          */
    }AtomInf;




typedef struct
    {
     int     f_n;                        /* key is <functor_atom,arity>    */
     int     owner_mod_nb;               /* module nb where it is defined  */
     Bool    public;                     /* is it public ?                 */
     CodePtr codep;                      /* associated code                */
    }PredInf;




typedef struct
    {
     int     a_t;                        /* key is <atom,operator_type>    */
     int     prec;                       /* precedence of the operator     */
     int     left;                       /* precedence of the operator lhs */
     int     right;                      /* precedence of the operator rhs */
    }OperInf;




typedef struct
    {
     int     key;
     CodePtr codep;
    }SwtInf;




/*---------------------------------*/
/* Global Variables                */
/*---------------------------------*/

#ifdef ATOM_PRED


       int       nb_of_module=0;
       ModulInf  module_tbl[MAX_MODULE];

       AtomTbl   atom_tbl;
       AtomInf  *atom_nil;
       AtomInf  *atom_dot;
       AtomInf  *atom_curly_brackets;
       AtomInf  *atom_fail;
       AtomInf  *atom_compiler;
       AtomInf  *atom_wam_version;


       PredTbl   pred_tbl;

       OperTbl   oper_tbl;

    /* int       char_type[256];             see definition in atom_prec.c */


#else


extern int       nb_of_module;
extern ModulInf  module_tbl[MAX_MODULE];

extern AtomTbl   atom_tbl;
extern AtomInf  *atom_nil;
extern AtomInf  *atom_dot;
extern AtomInf  *atom_curly_brackets;
extern AtomInf  *atom_fail;
extern AtomInf  *atom_compiler;
extern AtomInf  *atom_wam_version;

extern PredTbl   pred_tbl;

extern OperTbl   oper_tbl;

extern int       char_type[];


#endif




/*---------------------------------*/
/* Function Prototypes             */
/*---------------------------------*/

void      Init_Atom_Pred        (void);

AtomInf  *Create_Allocate_Atom  (char *name);
AtomInf  *Create_Atom           (char *name);
AtomInf  *Lookup_Atom           (char *name);

PredTbl   Create_Pred_Table     (int size);
PredInf  *Create_Pred           (AtomInf *atom,int n,Bool public,
                                 int module_nb,CodePtr codep);
PredInf  *Lookup_Pred           (AtomInf *atom,int n,Bool public,
                                 int module_nb);

OperInf  *Create_Oper           (AtomInf *atom,int type,
                                 int prec,int left,int right);
OperInf  *Lookup_Oper           (AtomInf *atom,int type);
OperInf  *Delete_Oper           (AtomInf *atom,int type);

SwtTbl    Create_Swt_Table      (int size);
void      Create_Swt_Element    (SwtTbl t,int key,CodePtr codep);




/*---------------------------------*/
/* Interface with C files          */
/*---------------------------------*/

#define Init_Tables_Of_Module(module)                                       \
    {                                                                       \
     Init_Tables_Prototyp(module)                                           \
     CPP_CAT(module,_Init_Tables)();                                        \
    }




#define Init_Tables_Prototyp(module)                                        \
    void CPP_CAT(module,_Init_Tables)(void);




#define Begin_Init_Tables(module)                                           \
    void CPP_CAT(module,_Init_Tables)(void)                                 \
    {                                                                       \
     char *t;                                                               \
                                                                            \
     module_tbl[nb_of_module].name    =module_name;                         \
     module_tbl[nb_of_module].debugged=DEBUG_MODE_BOOL;                     \
     module_tbl[nb_of_module].pred_tbl=                                     \
                                    Create_Pred_Table(NB_OF_PRIVATE_PREDS); \
     module_nb= nb_of_module++;




          /* Atom Hash Table : key is the C string associated to the atom */

#define Define_Atom(atom,name)                                              \
     atom=Create_Atom(name);




          /* Pred Hash Table : key is atom info adr / arity */

#define Define_Pred(atom,n,public)                                          \
    {                                                                       \
     Prototype(Prefix(Pred_Name(atom,n)))                                   \
     Create_Pred(atom,n,public,module_nb,                                   \
                 (CodePtr) Prefix(Pred_Name(atom,n)));                      \
    }




          /* Switch Hash Table : key always is an integer     */
          /*    for constants  : key is the atom info adr     */
          /*    for integer    : key is the integer           */
          /*    for structure  : key is atom info adr / arity */

#define Define_Switch_CST_Table(tbl,size)                                   \
     t=tbl=Create_Swt_Table(size);



#define Define_Switch_CST(atom,codep)                                       \
    {                                                                       \
     Prototype(Prefix(codep))                                               \
     Create_Swt_Element(t,(int) atom,(CodePtr) Prefix(codep));              \
    }




#define Define_Switch_STC_Table(tbl,size)                                   \
     t=tbl=Create_Swt_Table(size);




#define Define_Switch_STC(atom,n,codep)                                     \
    {                                                                       \
     Prototype(Prefix(codep))                                               \
     Create_Swt_Element(t,Functor_Arity(atom,n),                            \
                        (CodePtr) Prefix(codep));                           \
    }




#define Init_Usr_File                                                       \
     Initialize_Usr();




#define End_Init_Tables                                                     \
    }
