/*
 * Copyright (C) 1990, 1993 by Ted Kim.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/*
 * image.c
 *
 * pxImages
 * pxCreateImage
 * pxDestroyImage
 *
 * pxGetImageAttribute
 * pxGetImage
 * pxPutImage
 *
 * pxSubImage
 * pxAddPixel
 *
 * pxGetPixel
 * pxSetPixel
 */

#include <stdio.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "dl.h"
#include "conn.h"
#include "ff.h"

extern char *malloc();

typedef struct _PxImage {
  struct _PxImage *next;
  int des;
  XImage *xImage;
} PxImage;

DL pxImageList = { NULL, sizeof(PxImage), 0};

/*
 * pxImages(-Next)
 *	integer: Next
 */
void
pxImages(pn)
     FFInteger *pn;
{
  if ((pxList = pxImageList.list) == NULL)
    *pn = PX_End;
  else {
    pxType = PX_List;
    *pn = PX_Cont;
  }
}

/*
 * pxCreateImage(+Connection, +Visual, +Depth, +Format, +Offset, +Width,
 *	+Height, +BitmapPad, +BytesPerLine, -Image, 0)
 *
 *	integer: Connection, Visual, Depth, Format, Offset, Width, Height,
 *		BitmapPad, BytesPerLine, Image
 */
FFInteger
pxCreateImage(pc, pv, pd, pf, po, pw, ph, pbp, pbpl, pi)
     FFInteger pc, pv, pd, pf, po, pw, ph, pbp, pbpl, *pi;
{
  PxConnection *pcp;
  PxVisual *pvp;
  PxImage *pip;
  XImage *xip;
  int pad;
  unsigned size;
  
  PX_ErrorInit("xCreateImage/10");
  *pi = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;
  if (DLLookup(pxVisualList, (int)pv, (DLHeader **)&pvp))
    PX_Error(no_such_visual);
  if ((pf != XYPixmap) && (pf != ZPixmap))
    PX_Error("bad format");

  if (pbp == 0) {
    ScreenFormat *format = pcp->xDisplay->pixmap_format;
    int i;
    
    pbp = pcp->xDisplay->bitmap_pad;
    for (i = 0; i < pcp->xDisplay->nformats; format++)
      if (format->depth == pd) {
	pbp = format->scanline_pad;
	break;
      }
  }

  /* looks like XCreateImage might do something bad on a troubled malloc */
  if ((xip = XCreateImage(pcp->xDisplay, pvp->xVisual, pd,
			  pf, NULL, po, pw, ph, pbp, pbpl)) == NULL)
    PX_Error(no_memory);

  pad = xip->bitmap_pad;	/* alloc space for the image */
  if (pf == XYPixmap) 
    size = ((((pw + po + pad - 1) / pad) * pad) >> 3) * ph * pd;
  else
    size = (((((pw + po) * xip->bits_per_pixel + pad - 1)
	      / pad) * pad) >> 3) * ph;

  if (!size)
    xip->data = NULL;
  else 
    if (((xip->data = malloc(size)) == NULL)  ||
	DLAdd(&pxImageList, (int *)pi, (DLHeader **)&pip)) {
      if (xip->obdata != NULL)
	XFree(xip->obdata);
      XFree((char *)xip);
      PX_Error(no_memory);
    }
  
  pip->xImage = xip;
  return FF_Ok;
}

/*
 * pxDestroyImage(+Image, 0)
 *	integer: Image
 */
FFInteger
pxDestroyImage(pi)
     FFInteger pi;
{
  PxImage *pip;

  PX_ErrorInit("xDestroyImage/1");

  if (DLLookup(pxImageList, (int)pi, (DLHeader **)&pip))
    PX_Error(no_such_image);

  XDestroyImage(pip->xImage);

  DLRemove(&pxImageList, (int)pi);
  return FF_Ok;
}

/*
 * pxGetImageAttribute(+Image, +Attribute, -Value, 0)
 *	integer: Image, Attribute, Value
 */

#define PX_GetAttribute(A, V) \
   case A: *pv = (pip->xImage->V); \
           break

FFInteger
pxGetImageAttribute(pim, pa, pv)
     FFInteger pim, pa, *pv;
{
  PxImage *pip;

  PX_ErrorInit("xQueryImage/2");
  *pv = PX_Invalid;

  if (DLLookup(pxImageList, (int)pim, (DLHeader **)&pip))
    PX_Error(no_such_image);

  switch (pa) {
    PX_GetAttribute(0, width);
    PX_GetAttribute(1, height);
    PX_GetAttribute(2, xoffset);
    PX_GetAttribute(3, format);
    PX_GetAttribute(4, byte_order);
    PX_GetAttribute(5, bitmap_unit);
    PX_GetAttribute(6, bitmap_pad);
    PX_GetAttribute(7, bitmap_bit_order);
    PX_GetAttribute(8, depth);
    PX_GetAttribute(9, bytes_per_line);
    PX_GetAttribute(10, bits_per_pixel);
    PX_GetAttribute(11, red_mask);
    PX_GetAttribute(12, green_mask);
    PX_GetAttribute(13, blue_mask);
    default: PX_Error(unknown_attribute);
  }
  return FF_Ok;
}

/*
 * pxGetImage(+Connection, +Drawable, +SrcX, +SrcY, +Width, +Height, +PlaneMask,
 *	+DestImage, +DestX, +DestY, 0)
 *
 *	integer: Connection, Drawable, SrcX, SrcY, Width, Height, PlaneMask,
 *		DestImage, DestX, DestY, 
 */
FFInteger
pxGetImage(pc, pd, psx, psy, pw, ph, ppm, pdi, pdx, pdy)
     FFInteger pc, pd, psx, psy, pw, ph, ppm, pdi, pdx, pdy;
{
  PxConnection *pcp;
  PxImage *pip;
  XImage *xip;

  PX_ErrorInit("xGetImage/12");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;
  if (DLLookup(pxImageList, (int)pdi, (DLHeader **)&pip))
    PX_Error(no_such_image);

  /* probably does bad things if malloc fails */
  if ((xip = XGetSubImage(pcp->xDisplay, pd, psx, psy, pw, ph, ppm,
			  pip->xImage->format, pip->xImage, pdx, pdy)) == NULL)
    return FF_Fail;
  return FF_Ok;
}

/*
 * pxPutImage(+Connection, +Drawable, +GC, +Image, +SrcX, +SrcY, +DestX, +DestY,
 *	+Width, +Height, 0)
 *
 *	integer: Connection, Drawable, GC, Image, SrcX, SrcY, DestX, DestY,
 *		Width, Height
 */
FFInteger
pxPutImage(pc, pd, pgc, pi, psx, psy, pdx, pdy, pw, ph)
     FFInteger pc, pd, pgc, pi, psx, psy, pdx, pdy, pw, ph;
{
  PxConnection *pcp;
  PxGC *pgcp;
  PxImage *pip;

  PX_ErrorInit("xPutImage/10");
  
  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;
  if (DLLookup(pxGCList, (int)pgc, (DLHeader **)&pgcp))
    PX_Error(no_such_gc);
  if (DLLookup(pxImageList, (int)pi, (DLHeader **)&pip))
    PX_Error(no_such_image);

  XPutImage(pcp->xDisplay, pd, pgcp->xGC,
	    pip->xImage, psx, psy, pdx, pdy, pw, ph);
  return FF_Ok;
}

/*
 * pxSubImage(+Image, +X, +Y, +Width, +Height, -SubImage, 0)
 *	integer: Image, X, Y, Width, Height, SubImage
 */
FFInteger
pxSubImage(pi, px, py, pw, ph, psi)
     FFInteger pi, px, py, pw, ph, *psi;
{
  PxImage *pip;
  XImage *xip;

  PX_ErrorInit("xSubImage/6");
  *psi = PX_Invalid;

  if (DLLookup(pxImageList, (int)pi, (DLHeader **)&pip))
    PX_Error(no_such_image);

  /* probably does bad things if malloc fails */
  if (((xip = XSubImage(pip->xImage, px, py, pw, ph)) == NULL) ||
      DLAdd(&pxImageList, (int *)psi, (DLHeader **)&pip))
    PX_Error(no_memory);

  pip->xImage = xip;
  return FF_Ok;
}

/*
 * pxAddPixel(+Image, +Value, 0)
 *	integer: Image, Value
 */
FFInteger
pxAddPixel(pi, pv)
     FFInteger pi, pv;
{
  PxImage *pip;

  PX_ErrorInit("xAddPixel/2");

  if (DLLookup(pxImageList, (int)pi, (DLHeader **)&pip))
    PX_Error(no_such_image);

  XAddPixel(pip->xImage, pv);
  return FF_Ok;
}

/*
 * pxGetPixel(+Image, +X, +Y, -Pixel, 0)
 *	integer: Image, X, Y, Pixel
 */
FFInteger
pxGetPixel(pi, px, py, pp)
     FFInteger pi, px, py, *pp;
{
  PxImage *pip;

  PX_ErrorInit("xGetPixel/4");
  *pp = PX_Invalid;

  if (DLLookup(pxImageList, (int)pi, (DLHeader **)&pip))
    PX_Error(no_such_image);

  /* no range checking on XGetPixel, so do it here */
  if ((px > pip->xImage->xoffset + pip->xImage->width) ||
      (py > pip->xImage->height)) 
    PX_Error(coordinate_out_of_range);

  *pp = XGetPixel(pip->xImage, px, py);
  return FF_Ok;
}

/*
 * pxSetPixel(+Image, +X, +Y, +Pixel, 0)
 *	integer: Image, X, Y, Pixel
 */
FFInteger
pxSetPixel(pi, px, py, pp)
     FFInteger pi, px, py, pp;
{
  PxImage *pip;

  PX_ErrorInit("xPutPixel/4");

  if (DLLookup(pxImageList, (int)pi, (DLHeader **)&pip))
    PX_Error(no_such_image);
  
  /* no range checking in XPutPixel, so do it here */
  if ((px > pip->xImage->xoffset + pip->xImage->width) ||
      (py > pip->xImage->height)) 
    PX_Error(coordinate_out_of_range);

  XPutPixel(pip->xImage, px, py, pp);
  return FF_Ok;
}

/*
 * eof
 */
