/*
 * Copyright (C) 1990 by Ted Kim.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/*
 * gc.c
 *
 * pxGCs
 * pxCreateGC
 * pxDestroyGC
 * pxCopyGC
 *
 * pxGetGCAttribute
 * pxQueryBestSize
 *
 * pxSetGCAttribute
 * pxSetGC
 * pxSetClips
 * pxSetDashes
 */

#include <stdio.h>
#include <X11/Xlib.h>

#include "dl.h"
#include "conn.h"
#include "ff.h"
#include "attrib.h"

/* 
 * pxGCs(-Next)
 *	integer: Next
 */
void
pxGCs(pn)
     FFInteger *pn;
{
  if ((pxList = pxGCList.list) == NULL)
    *pn = PX_End;
  else {
    pxType = PX_List;
    *pn = PX_Cont;
  }
}

/*
 * pxCreateGC(+Connection, +Drawable, -GC, 0)
 *	integer: Connection, Drawable, GC
 */

#define XGCV_Changed pxAttribChanged1
#define XGCV_Mask pxAttribMask1
#define XGCV (pxAttribBuf.xgcv)

FFInteger
pxCreateGC(pc, pd, pgc)
     FFInteger pc, pd, *pgc;
{
  PxConnection *pcp;
  PxGC *pgcp;
  GC res;
  
  PX_ErrorInit("xCreateGC/4");
  *pgc = PX_Invalid;

  if (PXLookupConnection(pc, &pcp)) {
    XGCV_Changed = False;
    XGCV_Mask = 0;
    return FF_Fail;
  }

  if (XGCV_Changed) {
    res = XCreateGC(pcp->xDisplay, pd, XGCV_Mask, &XGCV);
    XGCV_Changed = False;
    XGCV_Mask = 0;
    if (res == NULL)
      return FF_Fail;
  }
  else if ((res = XCreateGC(pcp->xDisplay, pd, 0, (XGCValues *)NULL)) == NULL)
    return FF_Fail;
  if (DLAdd(&pxGCList, (int *)pgc, (DLHeader **)&pgcp))
    return FF_Fail;
  pgcp->xGC = res;
  pgcp->conn = pc;
  return FF_Ok;
}

/*
 * pxDestroyGC(+Connection, +GC, 0)
 *	integer: Connection, GC
 */
FFInteger
pxDestroyGC(pc, pgc)
     FFInteger pc, pgc;
{
  PxConnection *pcp;
  PxGC *pgcp;

  PX_ErrorInit("xDestroyGC/2");
  
  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;
  if (DLLookup(pxGCList, (int)pgc, (DLHeader **)&pgcp))
    PX_Error(no_such_gc);

  XFreeGC(pcp->xDisplay, pgcp->xGC);

  DLRemove(&pxGCList, (int)pgc);
  return FF_Ok;
}

/*
 * pxCopyGC(+Connection, +Src, +Mask, +Dest, 0)
 *	integer: Connection, Src, Mask, Dest
 */
FFInteger
pxCopyGC(pc, ps, pm, pd)
     FFInteger pc, ps, pm, pd;
{
  PxConnection *pcp;
  PxGC *psgc, *pdgc;
  
  PX_ErrorInit("xCopyGC/4");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;
  if (DLLookup(pxGCList, (int)ps, (DLHeader **)&psgc))
    PX_Error(no_such_gc);
  if (DLLookup(pxGCList, (int)pd, (DLHeader **)&pdgc))
    PX_Error(no_such_gc);

  XCopyGC(pcp->xDisplay, psgc->xGC, pm, pdgc->xGC);
  return FF_Ok;
}

/* 
 * pxGetGCAttribute(+GC, +Attribute, -Value, 0)
 *	integer: GC, Attribute, Value
 */

#define PX_GetAttribute(A, V) \
   case A: *pv = (pgcp->xGC->V); \
           break

FFInteger
pxGetGCAttribute(pgc, pa, pv)
     FFInteger pgc, pa, *pv;
{
  PxGC *pgcp;

  PX_ErrorInit("xQueryGC/2");
  *pv = PX_Invalid;

  if (DLLookup(pxGCList, (int)pgc, (DLHeader **)&pgcp))
    PX_Error(no_such_gc);

  switch (pa) {
    PX_GetAttribute(0, values.function);
    PX_GetAttribute(1, values.plane_mask);
    PX_GetAttribute(2, values.foreground);
    PX_GetAttribute(3, values.background);
    PX_GetAttribute(4, values.line_width);
    PX_GetAttribute(5, values.line_style);
    PX_GetAttribute(6, values.cap_style);
    PX_GetAttribute(7, values.join_style);
    PX_GetAttribute(8, values.fill_style);
    PX_GetAttribute(9, values.fill_rule);
    PX_GetAttribute(10, values.tile);
    PX_GetAttribute(11, values.stipple);
    PX_GetAttribute(12, values.ts_x_origin);
    PX_GetAttribute(13, values.ts_y_origin);
    PX_GetAttribute(14, values.font);
    PX_GetAttribute(15, values.subwindow_mode);
    PX_GetAttribute(16, values.graphics_exposures);
    PX_GetAttribute(17, values.clip_x_origin);
    PX_GetAttribute(18, values.clip_y_origin);
    PX_GetAttribute(19, values.clip_mask);
    PX_GetAttribute(20, values.dash_offset);
    case 21: *pv = (unsigned char)(pgcp->xGC->values.dashes);
             break;
    PX_GetAttribute(22, values.arc_mode);
    PX_GetAttribute(23, rects);
    PX_GetAttribute(24, dashes);
    default: PX_Error(unknown_attribute);
  }
  return FF_Ok;
}

/*
 * pxQueryBestSize(+Connection, +Mode, +Screen, +Width, +Height, -ReturnWidth,
 *	-ReturnHeight, 0)
 *
 *	integer: Connection, Mode, Screen, Width, Height, ReturnWidth,
 *		ReturnHeight
 */
FFInteger
pxQueryBestSize(pc, pm, ps, pw, ph, prw, prh)
     FFInteger pc, pm, ps, pw, ph, *prw, *prh;
{
  PxConnection *pcp;
  PxScreen *psp;

  PX_ErrorInit("xQueryBestSize/7");
  *prw = *prh = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;
  if (DLLookup(pxScreenList, (int)ps, (DLHeader **)&psp))
    PX_Error(no_such_screen);

  if (!XQueryBestSize(pcp->xDisplay, pm, psp->xScreen->root, pw, ph, prw, prh))
    return FF_Fail;
  return FF_Ok;
}

/*
 * pxSetGCAttribute(+Attribute, +Value, 0)
 *	integer: Attribute, Value
 */

#define PX_SetAttribute(A, MEM, BIT) \
 case A: XGCV.MEM = pv; \
         XGCV_Mask |= BIT; \
         break

FFInteger
pxSetGCAttribute(pa, pv)
     FFInteger pa, pv;
{
  PX_ErrorInit("xSetGC/3");

  switch (pa) {
    PX_SetAttribute(0, function, GCFunction);
    PX_SetAttribute(1, plane_mask, GCPlaneMask);
    PX_SetAttribute(2, foreground, GCForeground);
    PX_SetAttribute(3, background, GCBackground);
    PX_SetAttribute(4, line_width, GCLineWidth);
    PX_SetAttribute(5, line_style, GCLineStyle);
    PX_SetAttribute(6, cap_style, GCCapStyle);
    PX_SetAttribute(7, join_style, GCJoinStyle);
    PX_SetAttribute(8, fill_style, GCFillStyle);
    PX_SetAttribute(9, fill_rule, GCFillRule);
    PX_SetAttribute(10, tile, GCTile);
    PX_SetAttribute(11, stipple, GCStipple);
    PX_SetAttribute(12, ts_x_origin, GCTileStipXOrigin);
    PX_SetAttribute(13, ts_y_origin, GCTileStipYOrigin);
    PX_SetAttribute(14, font, GCFont);
    PX_SetAttribute(15, subwindow_mode, GCSubwindowMode);
    PX_SetAttribute(16, graphics_exposures, GCGraphicsExposures);
    PX_SetAttribute(17, clip_x_origin, GCClipXOrigin);
    PX_SetAttribute(18, clip_y_origin, GCClipYOrigin);
    PX_SetAttribute(19, clip_mask, GCClipMask);
    PX_SetAttribute(20, dash_offset, GCDashOffset);
    case 21: XGCV.dashes = (char)(0xFF & pv);
             XGCV_Mask |= GCDashList;
             break;
    PX_SetAttribute(22, arc_mode, GCArcMode);
    default: PX_Error(unknown_attribute);
  }
  XGCV_Changed = True;
  return FF_Ok;
}

/*
 * pxSetGC(+Connection, +GC, 0)
 *	integer: Connection, GC
 */
FFInteger
pxSetGC(pc, pgc)
     FFInteger pc, pgc;
{
  PxConnection *pcp;
  PxGC *pgcp;

  /* need to error init here in case pxSetGCAttribute is not called */
  PX_ErrorInit("xSetGC/3");

  if (PXLookupConnection(pc, &pcp)) {
    XGCV_Changed = False;
    return FF_Fail;
  }
  if (DLLookup(pxGCList, (int)pgc, (DLHeader **)&pgcp)) {
    XGCV_Changed = False;
    XGCV_Mask = 0;
    PX_Error(no_such_gc);
  }

  if (XGCV_Changed) {
    XChangeGC(pcp->xDisplay, pgcp->xGC, XGCV_Mask, &XGCV);
    XGCV_Changed = False;
    XGCV_Mask = 0;
  }
  return FF_Ok;
}

/*
 * pxSetClips(+Connection, +GC, +XOrigin, +YOrigin, +Ordering, 0)
 *	integer: Connection, GC, XOrigin, YOrigin, Ordering
 */
FFInteger
pxSetClips(pc, pgc, pxo, pyo, po)
     FFInteger pc, pgc, pxo, pyo, po;
{
  PxConnection *pcp;
  PxGC *pgcp;

  PX_ErrorInit("xSetClips/6");

  if (PXLookupConnection(pc, &pcp))
    return PXVectorFree();
  if (DLLookup(pxGCList, (int)pgc, (DLHeader **)&pgcp)) {
    (void)PXVectorFree();
    PX_Error(no_such_gc);
  }

  XSetClipRectangles(pcp->xDisplay, pgcp->xGC, pxo, pyo,
		     pxMallocBlock, pxElements, po);
  (void)PXVectorFree();
  return FF_Ok;
}

/*
 * pxSetDashes(+Connection, +GC, +DashOffset, 0)
 *	integer: Connection, GC, DashOffset
 */
FFInteger
pxSetDashes(pc, pgc, pdo)
     FFInteger pc, pgc, pdo;
{
  PxConnection *pcp;
  PxGC *pgcp;

  PX_ErrorInit("xSetDashes/4");

  if (PXLookupConnection(pc, &pcp))
    return PXVectorFree();
  if (DLLookup(pxGCList, (int)pgc, (DLHeader **)&pgcp)) {
    (void)PXVectorFree();
    PX_Error(no_such_gc);
  }

  XSetDashes(pcp->xDisplay, pgcp->xGC, pdo, pxMallocBlock, pxElements);
  (void)PXVectorFree();
  return FF_Ok;
}

/*
 * eof
 */
 
