#open "graphics";;

(* Simple turtle graphics *)

type turtle_state =
  { mutable x : float;
    mutable y : float;
    mutable heading : float };;

let t = { x = 0.0; y = 0.0; heading = 0.0 };;

let pi180 = 4.0 *. atan 1.0 /. 180.0;;

let round x =
  if x >=. 0.0 then int_of_float(x +. 0.5) else -(int_of_float(0.5 -. x));;

let reset() =
  t.x <- float_of_int(size_x() / 2);
  t.y <- float_of_int(size_y() / 2);
  t.heading <- 0.0;
  moveto (round t.x) (round t.y)
;;

let forward d =
  t.x <- t.x +. cos(t.heading) *. d;
  t.y <- t.y +. sin(t.heading) *. d;
  lineto (round t.x) (round t.y)
;;

let turn a =
  t.heading <- t.heading +. a *. pi180
;;

(* The main drawing function *)

let spir dist dist_incr angle angle_incr =
  reset();
  let d = ref dist
  and a = ref angle
  and c = ref 0 in
  while not key_pressed() do
    forward !d;
    turn !a;
    d := !d +. dist_incr;
    a := !a +. angle_incr
  done
;;
  
(* The interaction loop *)

let print_string s =
  let (x, y) = current_point() in
  draw_string s;
  let (_, height) = text_size s in
  moveto x (y + height)
;;

let format f =
  format_float "%6.1f" f
;;

let rec loop dist dist_incr angle angle_incr =
  clear_graph();
  moveto 0 0;
  print_string "   b, B        d            f, F        h, H    to increase";
  print_string "   a, A        c            e, E        g, G    to decrease";
  print_string (format dist ^ "      " ^ format dist_incr ^ "      " ^
                format angle ^ "      " ^ format angle_incr);
  print_string
    "Distance    Dist. incr.   Angle       Angle incr.    'q' to quit";
  spir dist dist_incr angle angle_incr;
  match read_key() with
    `a` -> loop (dist -. 0.5) dist_incr angle angle_incr
  | `A` -> loop (dist -. 5.0) dist_incr angle angle_incr
  | `b` -> loop (dist +. 0.5) dist_incr angle angle_incr
  | `B` -> loop (dist +. 5.0) dist_incr angle angle_incr
  | `c` -> loop dist (dist_incr -. 0.5) angle angle_incr
  | `d` -> loop dist (dist_incr +. 0.5) angle angle_incr
  | `e` -> loop dist dist_incr (angle -. 0.5) angle_incr
  | `E` -> loop dist dist_incr (angle -. 5.0) angle_incr
  | `f` -> loop dist dist_incr (angle +. 0.5) angle_incr
  | `F` -> loop dist dist_incr (angle +. 5.0) angle_incr
  | `g` -> loop dist dist_incr angle (angle_incr -. 0.1)
  | `G` -> loop dist dist_incr angle (angle_incr -. 5.0)
  | `h` -> loop dist dist_incr angle (angle_incr +. 0.1)
  | `H` -> loop dist dist_incr angle (angle_incr +. 5.0)
  | `q` -> ()
  | _ -> loop dist dist_incr angle angle_incr
;;

let spir () =
  open_graph "";
  loop 0.0 1.0 178.5 0.0;
  close_graph()
;;

if sys__interactive then () else begin spir(); exit 0 end;;
