;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;--------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practicable And Portable Lisp Implementation
;;;            ------------------------------------------------------
;;; Funktion : Seiteneffektanalyse
;;;            Implementierung der folgenden Funktion:
;;;            Analyse-form : ZWS * EFFECT  -----> EFFEKT
;;;            wobei 
;;;            ZWS    : die Menge der Zwischensprachobjekte
;;;            EFFEKT : {(read-list,write-list)/ 
;;;                       read-list , write-list Mengen von Variablen }
;;;
;;; $Revision: 1.41 $
;;; $Log: static-effect.lisp,v $
;;; Revision 1.41  1993/07/21  12:31:11  atr
;;; Zwei Fehler korrigiert bei spec-vars-bound-to-funs und
;;; get-arg-from-param.
;;;
;;; Revision 1.40  1993/07/13  11:07:37  atr
;;; Das Setzen der Sots LEVEL passiert jetzt in der Voranalyse.
;;; Ein Aufruf von remove-duplicates durch NUNION ersetzt .
;;; Ein Test , der bis jetzt immer nil liefert, korrigiert .
;;;
;;; Revision 1.39  1993/07/06  14:31:50  atr
;;; Optimierungen wieder angeschaltet.
;;;
;;; Revision 1.38  1993/07/06  13:14:47  atr
;;; Fehler bei (Analyse-form (defined-fun)) korrigiert:
;;; union-only-global-effects durch union-all-effects ersetzt.
;;;
;;; Revision 1.37  1993/07/06  10:20:21  hk
;;; *no-let-optimizing* auf t, weil noch ein gemeiner Fehler vorliegt.
;;;
;;; Revision 1.36  1993/06/30  14:22:52  hk
;;; (require optimize) eingefuegt.
;;;
;;; Revision 1.35  1993/06/29  23:26:32  atr
;;; Kommentare korrigiert.
;;;
;;; Revision 1.34  1993/06/26  13:34:58  atr
;;; Bessere Kommentare geschrieben.
;;;
;;; Revision 1.33  1993/06/17  18:05:55  atr
;;; Kommentare verbessert.
;;;
;;; Revision 1.32  1993/06/17  08:00:09  hk
;;; Copright Notiz eingefuegt
;;;
;;; Revision 1.31  1993/06/16  18:30:47  atr
;;; Aufruf der Funktion UNION durch REMOVE-DUPLICATES und
;;; APPEND ersetzt . Jetzt entsteht keine endlose Schleife !!!!
;;;
;;; Revision 1.30  1993/06/14  20:26:13  atr
;;; Klammer zu viel entfernt.
;;;
;;; Revision 1.29  1993/06/14  19:57:31  atr
;;; In der Analyse der Applikationen wird union-only-global-effects
;;; durch union-all-effects ersetzt damit die Slots read-list write-list bei
;;; Applikationen richtig gesetzt wird also damit die lokale Seiteneffekte
;;; nicht ignoriert werden.
;;;
;;; Revision 1.28  1993/06/14  10:41:24  atr
;;; Fehler bei der Analyse von DEFCLASS korrigiert,
;;; und kleine verbesserungen an Funktionsdefinitionen gemacht.
;;;
;;; Revision 1.27  1993/06/11  17:54:01  atr
;;; Analyse der Seiteneffekten auf Daten eingebaut.
;;; In dem Slot DATA-EFFECTS bei Funktionen und bei Applikationen
;;; steht nun entweder nil oder :alloc und/oder :dest.
;;; Alloc fuer Alloziierung auf dem Heap , dest fuer destruktive Operation.
;;;
;;; Revision 1.26  1993/05/31  09:34:10  ft
;;; Tippfehler in analyse-form (class-def) beseitigt.
;;;
;;; Revision 1.25  1993/05/30  14:06:00  atr
;;; Die Analyse der Applikationen verfeinert.
;;; Jetzt werden die special-defined-functions besser analysiert.
;;;
;;; Revision 1.24  1993/05/13  16:25:28  hk
;;; *static-level* bei globalen Funktionen auf 0.
;;;
;;; Revision 1.23  1993/05/12  13:28:39  hk
;;; (clicc-message ... CALLED) entfernt
;;;
;;; Revision 1.22  1993/05/10  16:04:50  atr
;;; (require closure-analysis) wieder eingefuegt.
;;;
;;; Revision 1.21  1993/05/10  10:21:23  atr
;;; Der Aufruf von Get-closures-of-module entfernt .
;;; Zwei momentan neue Variablen *map-functions* und *error-function* eingefuegt,
;;; um die map-funktionen sowie die Funktion error auf einer sonder Art
;;; zu behandeln.
;;;
;;; Revision 1.20  1993/05/05  17:34:52  atr
;;; Eine Applikation einer Variablen hat den Effekt UNKNOWN.
;;; Das Lesen eines Funktionalen Objectes hat keine Seiteneffekte.
;;; Bei der Fixpunktiteration werden nur Funktionen reanalysiert, die noch nicht den Top-effekt haben.
;;;
;;; Revision 1.19  1993/04/27  10:55:43  atr
;;; Es wird jetzt ein 'closure-analysis' im Clicc-Baum required.
;;;
;;; Revision 1.18  1993/04/27  07:12:32  ft
;;; atr's require in sein Home-Verzeichnis verbogen.
;;;
;;; Revision 1.17  1993/04/26  12:16:05  atr
;;; Die Variable *closures* in *upward-fun-args* umbenannt.
;;;
;;; Revision 1.16  1993/04/24  16:42:11  atr
;;; Analyse der Applikationen vera?ndert , clicc-message anstatt format.
;;;
;;; Revision 1.15  1993/04/18  16:55:43  atr
;;; *side-effect-info-level* auf 1 gesetzt.
;;;
;;; Revision 1.14  1993/04/18  16:35:49  atr
;;; Fehler korrigiert bei analyse-form fuer defined-fun.
;;;
;;; Revision 1.13  1993/04/17  12:53:23  atr
;;; Messages verbessert.
;;;
;;; Revision 1.12  1993/04/15  19:06:53  atr
;;; *side-effect-info-level* Schalter fuer infos ueber die Analyse.
;;;
;;; Revision 1.11  1993/03/31  18:43:15  atr
;;; empty-message
;;;
;;; Revision 1.10  1993/03/11  02:16:39  atr
;;; Methode analyse-app-form fur
;;;
;;; Revision 1.9  1993/03/09  01:57:47  atr
;;; Applikationen der Iterationsfunktionen (map apply ...) korrigiert.
;;;
;;; Revision 1.8  1993/03/08  02:49:51  atr
;;; Fehler entfernt.
;;;
;;; Revision 1.7  1993/03/08  02:48:56  atr
;;; Fehler korrigiert.
;;;
;;; Revision 1.6  1993/03/06  04:41:28  atr
;;; kleine Verbesserungen...
;;;
;;; Revision 1.5  1993/02/16  16:14:00  hk
;;; Revision Keyword eingefuegt.
;;;
;;; Revision 1.4  1993/02/15  13:52:50  atr
;;; Fehler beseitigt.
;;;--------------------------------------------------------------------------

(in-package "CLICC")

(require "optimize")

;;;--------------------------------------------------------------------------
;;; *effect-current-function* entha"lt die gerade ,bei der Seiteneffekt-
;;; -analyse,analysierte Funktion.
;;;--------------------------------------------------------------------------
(defvar *effect-current-function* )

(defvar *map-functions* nil)

(defvar *error-function* nil)

;;;--------------------------------------------------------------------------
;;; *effect-work-set* enthaelt die zu analysierenden Funktionen waehrend
;;; der Fixpunktiteration.
;;;--------------------------------------------------------------------------
(defvar *effect-work-set* )

(defvar *effect-all-functions* nil)

(defvar *visited-functions* nil )

(defvar *special-defined-funs* nil)

;;;-----------------------------------------------------------------------------
(defvar *static-level* 0)

(defvar *Side-effect-info-level* 1)

(defvar *no-let-optimizing* nil)    ;; Schalter fuer die Optimierungen.

(require "closure-analysis")
;;;--------------------------------------------------------------------------
;;; Bei der Analyse der ZWS-Formen werden die gelesenen Variablen bzw
;;; die vera"nderten Variablen in den read-list-Slot bzw in den 
;;; write-list-Slot eingetragen.
;;;--------------------------------------------------------------------------
(defclass1 effect  ()
  (read-list   :initform nil :type list )
  (write-list  :initform nil :type list )
  (data-effects :initform nil :type (member nil :alloc :dest :alloc-dest)))



;;;-------------------------------------------------------------------------
;;; Analyse eines Modules. Hier werden alle Funktionen analysiert.
;;; Es wird die erste Funktion aus dem Work-set genommen, analysiert
;;; und gleichzeitig alle funktionen die von dieser aufgerufen werden
;;; in die Liste *history* eingetragen und analysiert falls sie noch 
;;; nicht in der *history* sind , sonst werden die Slots READ-LIST und 
;;; WRITE-LIST eingelesen. Wenn die Read-list und  die Write-list dieser 
;;; Funktion sich geaendert haben, dann wird die Liste der Aufrufer
;;; dieser Funktion  wieder in den Work-set kopiert, sonst ist der 
;;; Fixpunkt erreicht und die Funktion nicht mehr analysiert.
;;;  
;;;-------------------------------------------------------------------------
(defun analyse-module ()
  (let* ((all-functions           (append  (?fun-list *module*)
                                           (list (?toplevel-forms *module*))))
         (*effect-all-functions*  nil)
         (*effect-work-set*       nil)
         (*visited-functions*     nil)
         (*error-function*        (get-global-fun 'clicc-lisp::error))
         (*map-functions*         (append 
                                   (list (get-global-fun 'apply))
                                   (list (get-global-fun 'clicc-lisp::map))
                                   (list (get-global-fun 'clicc-lisp::mapl))
                                   (list (get-global-fun 'clicc-lisp::maplist))
                                   (list (get-global-fun 'clicc-lisp::mapc))
                                   (list (get-global-fun 'clicc-lisp::mapcar))
                                   (list (get-global-fun 'clicc-lisp::mapcan))
                                   (list (get-global-fun 'clicc-lisp::mapcon))))
         )
    
    
    ;; Hier werden die Funktionen weiter attributiert um bessere 
    ;; Ergebnisse bei der Seiteneffektanalyse zu erzielen.
    ;;----------------------------------------------------------
    (clicc-message "Preparing side effect analysis...")
    (clicc-message "---------------------------------")
    (pre-analyse-module)
    
    ;; Fixpunktiteration zur Ableitung der Seiteneffekte der Funktionen
    ;;-----------------------------------------------------------------
    (clicc-message "Side-effect analysis...")
    (clicc-message "-----------------------")
    (setq *effect-all-functions* all-functions)
    (do* ( (*effect-work-set*  *effect-all-functions*)
           *effect-current-function*
           old-read
           old-write
           old-data-effects)
         ((null *effect-work-set*))
      (setq *effect-current-function*        (pop *effect-work-set*))
      (pushnew *effect-current-function*  *visited-functions*)
      (setq old-read       (?read-list     *effect-current-function*))
      (setq old-write      (?write-list    *effect-current-function*))
      (setq old-data-effects (?data-effects 
                              *effect-current-function*))
      (when (equal *Side-effect-info-level* 2)
        (clicc-message "Analysing the function ~s"
                       (?symbol *effect-current-function*)))
      (analyse-function  *effect-current-function*  (empty-effect))
      
      ;; Wenn der Seiteneffekt der Funktion sich geaendert hat 
      ;; dann wird jeder Aufrufer dieser Funktion , dessen
      ;; Seiteneffekt noch nicht der TOP-effekt ist, nochmal
      ;; analysiert.
      ;;------------------------------------------------------
      (unless (and (equal  old-read   
                           (?read-list  *effect-current-function*))
                   (equal  old-write  
                           (?write-list *effect-current-function*))
                   (eq     old-data-effects
                           (?data-effects  *effect-current-function*)))
        
        (let ((to-reanalyse (remove-if #'has-top-effect
                                       (?mv-called-by 
                                        *effect-current-function*))))
          (setq *effect-work-set* (nunion
                                   to-reanalyse
                                   *effect-work-set* :test #'eq)))))
    
    (setq *effect-all-functions* nil)
    (unless (zerop *Side-effect-info-level*)
      (statistics))
    (setq *visited-functions* nil)
    
    ;; Optimierungen der Let-ausdruecke .
    ;; Die Optimierungen kann man ueber den Schalter *no-let-optimizing*
    ;; abschalten
    (unless *no-let-optimizing*
      (let-optimizing))
    
    (clicc-message " ---------------------------------------------------------------------------")))

(defun has-top-effect (function)
  (and (eq :unknown (?read-list function))
       (eq :unknown (?write-list function))
       (eq :alloc-dest (?data-effects function))))

;;;------------------------------------------------------------------------
;;; Zur Zeit nur diese einfache Ausgabe der Ergebnisse der Analyse.
;;;------------------------------------------------------------------------

(defun statistics ()
  (let ((se-free 0)
        (se-top  0)
        (read    0)
        (write   0)
        (alloc   0)
        (dest    0))
    (dolist (fun *visited-functions*)
      (when (and (equal :unknown (?read-list fun))
                 (equal :unknown (?write-list fun)))
        (setq se-top (1+ se-top)))
      (when (and (null (?read-list fun))
                 (null (?write-list fun)))
        (setq se-free (1+ se-free)))
      (when (equal :unknown (?read-list  fun))
        (setq read (1+ read)))
      (when (equal :unknown  (?write-list fun))
        (setq write (1+ write)))
      (when (or (equal (?data-effects  fun) :alloc)
                (equal (?data-effects  fun) :alloc-dest))
        (setq alloc (1+ alloc)))
      (when (or (equal (?data-effects  fun) :dest)
                (equal (?data-effects  fun) :alloc-dest))
        (setq dest (1+ dest))))
    
    (clicc-message "~s functions are analysed" (length *visited-functions*))
    (clicc-message "~s functions are side effect free"
                   se-free )
    (clicc-message "~s functions have an unknown side effect"
                   se-top)
    (clicc-message "~s functions have an unknown read-effect"
                   read)
    (clicc-message "~s functions have an unknown write-effect"
                   write)
    (clicc-message "~s functions may use the heap" alloc)
    (clicc-message "~s functions may be destructive" dest)))

;;;-------------------------------------------------------------------------
;;; Simple-literal hat keine Effekte.
;;;-------------------------------------------------------------------------
(defmethod analyse-form ((a-simple-lit simple-literal)  effect)
  (declare (ignore effect)))
      
;;;-------------------------------------------------------------------------
;;; Methode zur Analyse von Structured-literal
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((form structured-literal) effect)
  (declare (ignore effect)))

;;;--------------------------------------------------------------------------
;;; Ein Symbol hat keine Seiteneffekte.
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((form sym)   effect)
  (declare (ignore effect)))

;;;--------------------------------------------------------------------------
;;; Zur Zeit werden class-def nicht analysiert.
;;; -------------------------------------------------------------------------
(defmethod analyse-form ((one-class-def class-def) effect)
  (dolist (one-slot-descr (?slot-descr-list one-class-def))
    (analyse-form (?initform one-slot-descr) effect)))

;;;--------------------------------------------------------------------------
;;; Analyse von Variablen:
;;; wenn das statische Niveau dieser statisch gebundenen  Variablen kleiner
;;; ist als das aktuelle statische Niveau der Analyse , wird sie in die 
;;; Read-list eingetragen, sonst wird nichts getan. 
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((static-var static) effect)
  (when (listp (?read-list effect) )
    (pushnew static-var (?read-list effect))))
               
;;;--------------------------------------------------------------------------
;;; Die dynamisch gebundene Variablen werden in die "read-list"
;;; eingetragen.
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((dynamic-var dynamic) effect)
  (when (listp (?read-list effect))
    (pushnew dynamic-var  (?read-list effect))))

;;;--------------------------------------------------------------------------
;;; analyse-form (var-ref,r-w-list) = analyse-form ((?var var-ref),r-w-list)
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((ref var-ref) effect)
  (analyse-form (?var ref)  effect))

;;;--------------------------------------------------------------------------
;;; Seiteneffekte einer IF-Form ist das MAX ueber die Seiteneffekte
;;; der Pred- then- und else-Teile.
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((form if-form) effect)
  (analyse-form (?pred form)  effect )
  (analyse-form (?then form)  effect )
  (analyse-form (?else form)  effect ))

;;;--------------------------------------------------------------------------
;;; Analyse einer Labels-Form.
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((labels labels-form) effect)

  ;; wenn dieser labels-konstrukt zum ersten mal analysiert wird dann
  ;; wird folgendes getan:
  ;;     1:  der Slot LEVEL bei jeder lokalen Funktion gesetzt.
  ;;     2:  die Funktionen werden an *effect-work-set* 'gepusht'.
  ;;     3:  der Rumpf des Labels-konstrukts wird analysiert.
  ;; um festzustellen dass der Konstrukt analysiert wurde muss man
  ;; abfragen ob eine der lokal in diesem labels-konstrukt definierten
  ;; Funktionen in *effect-work-set* oder in *visited-functions*
  ;; vorhanden ist.
  ;;-------------------------------------------------------------------
  
  (let* ((*static-level*   (1+ *static-level*))
         (fun-list         (?fun-list labels)))
    (unless (or (member (car fun-list ) *effect-work-set* :test #'eq)
                (member (car fun-list ) *visited-functions* :test #'eq))
      (dolist (fun fun-list)
        (pushnew fun   *effect-work-set*))))
  (analyse-form (?body labels) effect))

;;;--------------------------------------------------------------------------
;;; Analyse einer PROG-Form:
;;; analyse-form (progn , effect) = Max [ (analyse-form (form1  effect))
;;;                                       (analyse-form (form2  effect))
;;;                                                    ...
;;;                                       (analyse-form (formN  effect))]
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((form progn-form) effect)
  (dolist (one-form  (?form-list form))
    (analyse-form one-form   effect )))

;;;--------------------------------------------------------------------------
;;; Analyse einer SETQ-form: (setq var form)
;;; Var wird in die WRITE-LIST eingetragen und dann wird FORM in den 
;;; neuen EFFECT analysiert.
;;;--------------------------------------------------------------------------

(defmethod analyse-form ((form setq-form) effect)
  (unless  (named-const-p (?location form)) 
    (let ((var        (?var (?location form)))
          (value      (?form form)))
      (when (listp (?write-list effect))
        (pushnew var (?write-list effect)))
      (analyse-form value  effect))))

;;;--------------------------------------------------------------------------
;;; Analyse einer Switch-Form :
;;; es werden die erste Form in der Switch, die Otherwise-form, die Form
;;; in jedem Case analysiert, und das Maximum ueber deren Seiteneffekte
;;; genommen. 
;;; -------------------------------------------------------------------------
(defmethod analyse-form ((form switch-form) effect)
  (analyse-form (?form form)      effect )
  (analyse-form (?otherwise form) effect )
  (let ((case-list (?case-list form)))
    (dolist (case case-list )
      (analyse-form  case  effect))))


(defmethod analyse-form ((form labeled-form) effect)
  (analyse-form (?value form)  effect )
  (analyse-form (?form  form)  effect ))

;;;--------------------------------------------------------------------------
;;; Analyse einer Tagbody-Form :
;;; Es wird das Maximum genommen ueber die Seiteneffekte der First-form
;;; in dem Tagbody und der Formen in jeder Tagged-form.
;;; Es werden alle tagged-formen aus der Tagged-form-list eines Tagbody
;;; analysiert.
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((form tagbody-form) effect)
  (let* ((first (?first-form  form)))
    (analyse-form first effect)
    (dolist (one-form (?tagged-form-list form))
      (analyse-form (?form one-form) effect ))))

;;;--------------------------------------------------------------------------
;;; Eine GO-FORM hat keine Seiteneffekte.
;;; Ein Sprung hat in diesem Sinne keine Seiteneffekte.
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((form tagged-form) effect)
  (declare (ignore effect)))

;;;--------------------------------------------------------------------------
;;; Analyse einer LET*-Form 
;;; analyse-form ([let* ((var1 val1)  ...  (varN valN)) body ],effect1) 
;;;  = MAX [(analyse-form val1 effect1)
;;;         (analyse-form val2 effect2)
;;;                ...
;;;         (analyse-form body effectN+1)]
;;; wobei effectI+1 der erhatene effekt nach (analyse-form (valI effectI))
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((let*form let*-form) effect)
  
  ;; Zuerst werden alle init-formen  analysiert.
  ;;-----------------------------------------------------------------
  (dolist (one-form (?init-list let*form))
    (analyse-form one-form effect))
  
  ;; nun wird der Rumpf analysiert.
  ;;-----------------------------------------------------------------
  (analyse-form (?body let*form) effect))

;;;-------------------------------------------------------------------------
;;; Hier wird eine Funktionsdefinition in einen Effekt analysiert.
;;; Aus dem  erhaltenen Effekt werden alle lokale Seiteneffekte 
;;; entfernt, und der globale Effekt wird in den Slots read-list 
;;; write-list und data-effects abgespeichert.
;;;-------------------------------------------------------------------------
(defun analyse-function (function effect)
  (let (( *static-level*            (if (local-fun-p function)
                                        (?level function)
                                        0)))
    (analyse-params (?params function) effect)
    (analyse-form   (?body   function) effect)
    (setf (?read-list function) 
          (remove-local-effect (?read-list effect)))
    (setf (?write-list function)
          (remove-local-effect (?write-list effect)))
    (setf (?data-effects function) (?data-effects effect))))

;;;-------------------------------------------------------------------------
;;; remove-local-effect dient zur versteckung lokaler Seiteneffekte.
;;; Also aus den Listen read-list und write-list werden alle Variablen 
;;; entfernt, die lokale in der Funktion gebunden sind.
;;;-------------------------------------------------------------------------
(defun remove-local-effect (list)
  (if (listp list)
      (remove-if-not  #'(lambda (s) 
                          (or (dynamic-p s)
                              (and (static-p s)
                                   (< (?level s) *static-level*))))
                      list)
      :unknown))


;;;--------------------------------------------------------------------------
;;; Analyse eines angewandten Vorkommens einer Funktion aber nicht 
;;; auf Funktionsposition (d.h als Funktionales Objekt).
;;; Hier wird der Effekt der Funktion mit dem aktuellen Effekt vereinigt.
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((function fun) effect)
  (union-all-effects  effect effect (get-effect function))
  (when (defined-fun-p function)
    (pushnew *effect-current-function* (?mv-called-by function))))

;;;--------------------------------------------------------------------------
;;; Hilfsfunktionen ...
;;;--------------------------------------------------------------------------

;;;--------------------------------------------------------------------------
;;; (STORE-EFFECT effect  fun/app)
;;; speichert den Effect  in den Slot effect der Funktion 
;;; oder der applikation 'fun/app' .
;;;---------------------------------------------------------------------------
(defun store-effect (effect fun/app)
  (setf (?read-list     fun/app)  (?read-list    effect))
  (setf (?write-list    fun/app)  (?write-list   effect))
  (setf (?data-effects  fun/app)  (?data-effects effect)))

;;;--------------------------------------------------------------------------
;;; (GET-EFFECT fun )
;;;  kopiert den Effekt einer Funktion 'fun' in eine Instance von EFFECT
;;;--------------------------------------------------------------------------
(defun get-effect (function)
  (make-instance 'effect
                 :read-list    (?read-list    function)
                 :write-list   (?write-list   function)
                 :data-effects (?data-effects function)))



(defun empty-effect ()
  (make-instance 'effect))

(defun top-effect ()
  (make-instance 'effect
                 :read-list    :unknown
                 :write-list   :unknown
                 :data-effects :alloc-dest))

;;;--------------------------------------------------------------------------
;;; Max-effect wird benutzt um die sichtbaren Effekte einer Funktion 
;;; zu berechnen.
;;; Seiteneffekte auf die lokale Variablen der Funktion sind nach aussen
;;; nicht sichtbar.
;;; Beachte den Unterschied zu unify-lists unten !!!
;;;-------------------------------------------------------------------------- 
(defun max-effect (list1 list2)
  (remove-duplicates
   (remove-if-not  #'(lambda (s) 
                       (or (dynamic-p s)
                           (and (static-p s)
                                (< (?level s) *static-level*))))
                   (append list1 list2)) :test #'eq))

(defun max-data-effects (d-effect1 d-effect2)
  (if  (and  d-effect1  d-effect2 
             (not (eq d-effect1 d-effect2)))
       :alloc-dest
       (if (eq d-effect1 d-effect2)
           d-effect1
           (if (and  d-effect1  (null d-effect2))
               d-effect1
               d-effect2))))

(defun union-only-global-effects (target-effect effect1 effect2)
  (setf (?read-list target-effect )
        (if (and (listp (?read-list effect1))
                 (listp (?read-list effect2)))
            (max-effect (?read-list effect1)
                        (?read-list effect2))
            :unknown))
  (setf (?write-list target-effect)
        (if (and (listp (?write-list effect1))
                 (listp (?write-list effect2)))
            (max-effect (?write-list effect1)
                        (?write-list effect2))
            :unknown))
  (setf (?data-effects target-effect)
        (max-data-effects (?data-effects effect1) (?data-effects effect2))))

;;;--------------------------------------------------------------------------
;;; union-all-effects wird benutzt, um die Effekte einer Applikation 
;;; zu berechnen. Sie vereinigt die read- und die write-listen der 
;;; applizierten Form mit den der Argumenten ohne den *Static-level*
;;; abzufragen (im gegenteil zu union-only-global-effects).
;;; Denn bei einer Applikation einer lokalen Funktion muessen auch die 
;;; lokale Variablen stehen die durch den Aufruf gelesen oder veraendert
;;; werden.
;;;--------------------------------------------------------------------------
(defun union-all-effects (target-effect effect1 effect2)
  (setf (?read-list  target-effect) (unify-lists (?read-list  effect1) 
                                                 (?read-list  effect2)))
  (setf (?write-list target-effect) (unify-lists (?write-list effect1)
                                                 (?write-list effect2)))
  (setf (?data-effects target-effect) 
        (max-data-effects (?data-effects effect1)
                          (?data-effects effect2))))


(defun unify-lists (list1 list2)
  (if (and (listp list1) (listp list2))
      (remove-duplicates 
       (append list1  list2)  :test #'eq)
      :unknown))

;;;--------------------------------------------------------------------------
;;; spec-vars-bound-to-funs : wird eine Funktion an der Variablen gebunden,
;;; die im Slot HAS-FUNS-AS-ARGS steht ?
;;; " Die Variablen die im Slot HAS-FUNS-AS-ARGS  stehen , sind diejenigen
;;; die im Rumpf der Funktion oder der globaleren umfassenden Funktionen
;;; nur auf Funktionsposition stehen ".
;;;--------------------------------------------------------------------------
(defun  spec-vars-bound-to-funs (app)
  (let* ((function  (?form app))
         (spec-vars (if (defined-fun-p function)
                        (remove-if-not 
                         #'(lambda (var)
                             (member var 
                                     (?all-vars (?params function)) :test #'eq))
                         (?has-funs-as-args function))
                        (?has-funs-as-args function)))
         one-spec-var)
    (loop
     (setq one-spec-var (pop spec-vars))
     
     (let ((arg (if (defined-fun-p function)
                    (get-arg-from-param one-spec-var app)
                    (get-arg-from-coded-param one-spec-var app ))))
       (if (known-funs arg)
           (if (endp spec-vars)
               (return  T)
               nil)
           (return nil))))))

;;;--------------------------------------------------------------------------
;;; get-arg-from-coded-param liefert das Argument aus der Argumentliste 
;;; einer Applikation einer imported-fun mit spec-vars.
;;; Bei imported-funs ist der Slot HAS-FUNS-AS-ARGS kodiert.
;;; Get-result-position-forms liefert dann alle Ergebnisformen, im Falle
;;; dass eine Kontrollstruktur gefunden ist.
;;;--------------------------------------------------------------------------
(defun get-arg-from-coded-param (spec-var app)
  (if (numberp spec-var)

      ;; Hier handelt es sich um die Kodierung eines required oder optionalen
      ;; Parameters durch ihre Position in der Parameterliste.
      (get-result-position-forms (nth spec-var (?arg-list app)))

      ;; Hier handelt es sich um keyword (nur das Symbol).
      ;; cadr von member liefert dann den eventuell an das keyword 
      ;; gebundene Parameter
      (get-result-position-forms 
       (cadr (member spec-var (?arg-list app)  :test #'eq)))))

;;;--------------------------------------------------------------------------
;;; get-arg-from-param ist wie die get-arg-from-coded-param, aber 
;;; wo eine defined-fun appliziert wird .
;;; Der Unterschied liegt darin, dass in diesem Fall Variablen im
;;; Slot HAS-FUNS-AS-ARGS stehen, also noch keine Kodierung .
;;;--------------------------------------------------------------------------
(defun get-arg-from-param (spec-var app)
  (let ((function (?form app)))
    (when (member spec-var (?all-vars (?params (?form app))))
      (let* ((is-key-var (find spec-var 
                               (?key-list (?params function)) 
                               :key #'?var :test #'eq)))
              
        (if is-key-var
            ;; falls die Variable fuer einen Keyword Parameter steht,
            ;; wird in der Argumentliste nach dem Keyword (symbol) 
            ;; gesucht und dann die darauffolgende Form ist dann das 
            ;; entsprechende Argument.
            (cadr (member (?sym is-key-var) (?arg-list app) :test #'eq))
            
            ;; sonst geht es um einen required oder optional Parameter.
            ;; Hier wird zunaechst die Position des Parameters in der 
            ;; parameterliste berechnet, dann wird in der Argumentliste 
            ;; an der gleichen Position nach dem Argument gesucht.
            (nth (position spec-var (?all-vars (?params function)) :test #'eq) 
                 (?arg-list app)))))))
        
;;;--------------------------------------------------------------------------
;;; Ueberprueft ob das gelieferte Argument, oder alle Ergebnisformen davon
;;; Funktionen sind .
;;;--------------------------------------------------------------------------
(defun known-funs (list-of-forms)
  (if (atom list-of-forms)
      T
      (if (endp list-of-forms)
          T
          (if (fun-p (car list-of-forms))
              (known-funs (cdr list-of-forms))
              nil))))


;;;--------------------------------------------------------------------------
;;; Analyse einer APPLICATION:
;;; Zunaechst wird der Seiteneffekt der applizierten Form berechnet
;;; FUN-EFFEKT.
;;; Dann werden die Argumente analysiert,deren Seiteneffkte in  ARG-EFFECT
;;; abgespeichert werden.
;;; Der Effekt der Applikation ist dann FUN-EFFECT vereinigt mit 
;;; ARG-EFFECT --> APP-EFFECT.
;;; APP-EFFECT wird zu dem Aktuellen EFFECT addiert.
;;;--------------------------------------------------------------------------

(defmethod analyse-form ((app-form app) effect)
  
  (let ((functional (?form      app-form))
        (fun-effect (empty-effect))
        (arg-effect (empty-effect)))
    (block calculate-app-effect
      (typecase functional 
        (defined-fun 
            
            ;; Fall 1: die applizierte Form ist eine benutzer-definierte
            ;; Funktion :
            ;; ACHTUNG  : wenn   das gerade analysierte Module das Lisp 
            ;;            Module ist, dann sind  die MAP-funktionen 
            ;;            DEFINED-FUNS. Deswegen die Abfrage  
            ;;            (member functional *map-functions*)
            ;;-----------------------------------------------------------
            (pushnew *effect-current-function* (?mv-called-by functional))
            (if (member functional *map-functions* :test #'eq)
                (let ((new-app 
                       (make-instance 'app
                                      :form      (car (?arg-list app-form))
                                      :arg-list  (cdr (?arg-list app-form))))
                      (local-effect  (empty-effect)))
                  (analyse-form new-app local-effect)
                  (store-effect  (get-effect new-app) app-form)
                  (union-all-effects effect  effect 
                                     (get-effect app-form))
                  (return-from calculate-app-effect ))
                (if (eq functional *error-function*)
                    (setf fun-effect (empty-effect))
                    (if (?has-funs-as-args functional)
                        (if (spec-vars-bound-to-funs app-form)
                            (setf fun-effect (get-effect functional))
                            (progn (print "test failed")
                                   (setf fun-effect (top-effect))))
                        (setf fun-effect (get-effect functional))))))
        (imported-fun 

         ;; Fall2 die applizierte Form ist eine importierte Funktion 
         ;; Hier weden die Aufrufe von den "MAP-FUNCTIONS" und 
         ;; der Funktion APPLY einzeln behandelt.
         ;;---------------------------------------------------------
         (if (and (special-sys-fun-p functional)
                  (?special-caller functional))
             (let ((new-app 
                    (make-instance 'app
                                   :form      (car (?arg-list app-form))
                                   :arg-list  (cdr (?arg-list app-form))))
                   (local-effect  (empty-effect)))
               (analyse-form new-app local-effect)
               (store-effect  (get-effect new-app) app-form)
               (union-all-effects effect effect (get-effect app-form))
               (return-from calculate-app-effect ))
             (if (?has-funs-as-args functional)
                 (if (spec-vars-bound-to-funs app-form)
                     (setf fun-effect (get-effect functional))
                     (progn (print "test failed ..")
                            (setf fun-effect (top-effect))))
                 (setf fun-effect (get-effect functional)))))
        (var-ref 
         
         ;; Fall3 die applizierte Form ist eine Variable :
         ;; falls die Variable schon in dem Slot HAS-FUNS-AS-ARGS
         ;; der gerade analysierten Funktion vorhanden ist,
         ;; dann  wird einfach die Variablen in die READ-LIST
         ;; eingetragen. (denn das funktionale Objekt wird an der
         ;; Aufrufstelle analysiert.
         ;; sonst wird TOP-EFFEKT.
         ;;----------------------------------------------------------
         (if (member (?var functional )
                     (?has-funs-as-args *effect-current-function*) :test #'eq)
             (pushnew (?var functional) (?read-list fun-effect))
             (setf fun-effect (top-effect))))
        
        (t
         
         ;; SONST 
         ;; werden alle Formen auf Ergebnis-position berechnet ,
         ;; mit Hilfe von GET-RESULT-POSITION-FORMS, und das 
         ;; Maximum deren Seiteneffekte als der Effekt des Funktionals
         ;; genommen.
         ;;------------------------------------------------------------
         (let ((local-effect (empty-effect)))
           (analyse-form functional local-effect)
           (let ((applied-forms (get-result-position-forms functional)))
             (dolist (one-applied-form applied-forms)
               (if (fun-p one-applied-form)
                   (progn 
                     (when (defined-fun-p one-applied-form)
                       (pushnew *effect-current-function* 
                                (?mv-called-by one-applied-form)))
                     (union-all-effects fun-effect fun-effect 
                                        (get-effect one-applied-form)))
                   (if (or (var-ref-p one-applied-form)
                           (app-p one-applied-form))
                       (setq fun-effect (top-effect))
                       (union-all-effects fun-effect fun-effect 
                                          local-effect))))))))
      
      ;; Nun wird der Effekt der Argumente berechnet, dann wird 
      ;; der Effekt der Applikation aus dem Effekt des Funktionals
      ;; und dem der Argumente berechnet und in der Applikation 
      ;; abgespeichert.
      ;; der Effekt der Applikation wird am Ende mit dem Aktuellen
      ;; Effekt der Analyse vereinigt.
      ;;----------------------------------------------------------
      (dolist (one-arg (?arg-list app-form))
        (analyse-form one-arg arg-effect))
      (union-all-effects app-form fun-effect arg-effect)
      (union-all-effects effect effect (get-effect app-form)))))

;;;--------------------------------------------------------------------------
;;; Methode zur Analyse von LET/CC
;;; Der Effekt ist der Effekt der Rumpf des Let/cc-Konstruktes.
;;;--------------------------------------------------------------------------
(defmethod analyse-form ( (let/cc let/cc-form) effect)
  (analyse-form (?body let/cc) effect))

;;;--------------------------------------------------------------------------
;;; Angewandte Vorkommen von Continuations haben keine Seiteneffkte
;;; in unserem Sinne . (Sie verursachen einen Sprung aber beeinflussen 
;;; keine Variablen).
;;;-------------------------------------------------------------------------
(defmethod analyse-form ((continuation cont) effect)
  (declare (ignore effect)))

;;;--------------------------------------------------------------------------
;;; Analyse der Parameter einer Funktion.
;;; Es werden alle Init-formen der Optionalen bzw key-Parameter analysiert.
;;;--------------------------------------------------------------------------
(defmethod analyse-params ((parameter params)  effect)
  (dolist (opt-param (?opt-list parameter))
    (analyse-form (?init opt-param)  effect))
  (dolist (key-param (?key-list parameter))
    (analyse-form (?init key-param) effect)))

;;;--------------------------------------------------------------------------
;;; Methode zur Analyse von mv-lambda.
;;;--------------------------------------------------------------------------
(defmethod analyse-form ((lambda mv-lambda) effect)
  (analyse-params (?params lambda) effect)
  (analyse-form   (?body   lambda) effect)
  (analyse-form   (?arg    lambda) effect))
;;;--------------------------------------------------------------------------

(defmethod analyse-form ((anything-else t) effect)
  (declare (ignore effect)))
;;;--------------------------------------------------------------------------
(provide "static-effect")
