;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;-----------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practicable And Portable Lisp Implementation
;;;            ------------------------------------------------------
;;; Funktion : Foreign Functions
;;;
;;; $Revision: 1.21 $
;;; $Log: p1foreign.lisp,v $
;;; Revision 1.21  1993/06/17  08:00:09  hk
;;; Copright Notiz eingefuegt
;;;
;;; Revision 1.20  1993/06/10  11:02:38  pm
;;; Quelltext bereinigt
;;;
;;; Revision 1.19  1993/05/31  17:06:02  pm
;;; Abarbeiten von call-ins eingebaut
;;;
;;; Revision 1.18  1993/05/23  17:52:44  pm
;;; p1-def-c-type um das Argument package erweitert
;;;
;;; Revision 1.17  1993/05/21  13:58:02  pm
;;; c-int in int umbenannt
;;;
;;; Revision 1.16  1993/05/12  08:36:57  pm
;;; packages verstanden und ueberarbeitet
;;;
;;; Revision 1.15  1993/04/23  09:42:35  pm
;;; Aufruf von p1-foreign-fun-call optimiert
;;;
;;; Revision 1.14  1993/04/08  12:52:32  pm
;;; Tippfehler beseitigt
;;;
;;; Revision 1.13  1993/04/08  09:16:52  pm
;;; Angefangen aufs C-Typ-System umzustellen
;;; p1-call-foreign-fun in p1-foreign-fun-call umbenannt
;;;
;;; Revision 1.12  1993/03/18  07:40:57  ft
;;; Tippfehler beseitigt.
;;;
;;; Revision 1.11  1993/03/17  14:32:11  pm
;;; struct eingebaut
;;;
;;; Revision 1.10  1993/02/18  10:22:01  kl
;;; Fehler mit den im Package FFI unbekannten Funktionen export und
;;; in-package behoben.
;;;
;;; Revision 1.9  1993/02/17  16:41:37  hk
;;; Package FFI soll KEINE anderen Packages usen, auch nicht Lisp.
;;;
;;; Revision 1.8  1993/02/16  16:58:24  hk
;;; Revision Keyword eingefuegt.
;;;
;;; Revision 1.7  1992/12/01  15:11:11  pm
;;; c-char* eingebaut
;;;
;;; Revision 1.6  1992/11/10  10:24:00  pm
;;; Fluechtigkeitsfehler behoben
;;;
;;; Revision 1.5  1992/11/05  10:52:02  pm
;;; Ueberarbeitet
;;;
;;; Revision 1.4  1992/11/04  12:41:28  pm
;;; p1-call-foreign-fun
;;;
;;; Revision 1.3  1992/10/19  14:17:44  pm
;;; kleinere Aenderungen
;;;
;;; Revision 1.2  1992/10/19  12:00:50  pm
;;; parser fuer foreign-funs
;;;
;;; Revision 1.1  1992/10/13  14:28:39  pm
;;; Initial revision
;;;-----------------------------------------------------------------------------


;;------------------------------------------------------------------------------
(in-package "CLICC")

;;------------------------------------------------------------------------------
;; Fehlermeldungen.
;;------------------------------------------------------------------------------
(defconstant NO_NAME_SPECIFIED 
  "You must specify a name in DEF-CALL-OUT")
(defconstant NO_FORWARD_REF 
  "You cannot use the call-out-function ~S before its deklaration.")
(defconstant NOT_OF_RIGHT_LENGTH
  "The length (~S) of the argument-list doesn't match its definition (~S)")

;;------------------------------------------------------------------------------
;; Meldungen.
;;------------------------------------------------------------------------------
(defconstant ANALYSE-CALL-OUT "Analyse Call-Out-Function ~S")
(defconstant ANALYSE-CALL-IN "Analyse Call-In-Function ~S")


;;------------------------------------------------------------------------------
;; Syntax einer C-Funktionsdefinition
;;
;; DEF-CALL-OUT name {option}*
;;
;; option ::= 
;;     (:name c-name)
;;   | (:arguments ({c-type}*))
;;   | (:return-type c-type)
;;------------------------------------------------------------------------------
(defun p1-def-call-out (name_options)

  (clicc-message ANALYSE-CALL-OUT (first name_options))

  (multiple-value-bind (name c-name arg-list return-type)
      (parse-foreign-fun-args name_options)

    (let ((operator-def (get-operator-def name)))

      (case (car operator-def)
        ;; Neue Definition eintragen
        ((nil)
         (set-unexpanded-foreign-fun 
          name (make-instance 'foreign-fun
                              :symbol name
                              :arg-type-list arg-list
                              :foreign-name c-name
                              :return-type return-type)))

        ;; keine Forwaertsreferenz erlaubt
        (:FORWARD
         (clicc-error NO_FORWARD_REF name))

        ;; keine Mehrfachdefinition erlaubt
        (t
         (redef-op-error (car operator-def) name)))

      (values name))))

;;------------------------------------------------------------------------------
;; DEF-CALL-IN name {option}*
;;
;; option ::=
;;     (:name c-name)
;;   | (:arguments ({c-type}*))
;;   | (:return-type c-type)
;;------------------------------------------------------------------------------
(defun p1-def-call-in (name_options)

  (clicc-message ANALYSE-CALL-IN (first name_options))

  (multiple-value-bind (name c-name arg-list return-type)
      (parse-foreign-fun-args name_options)

    (let ((call-in-fun (get-call-in-fun name)))
      ;; keine Mehrfachdefinition erlaubt
      (when call-in-fun
        (error "Call-In-Fun (~S) declared twice" call-in-fun))

      ;; Neue Definition eintragen
      (set-call-in-fun 
       name (make-instance 'call-in-fun
                           :arg-type-list arg-list
                           :foreign-name c-name
                           :return-type return-type))
      
      (values name))))

;;------------------------------------------------------------------------------
;; Syntax einer C-Typ-Definition:
;;
;; DEF-C-TYPE name data-type {option}*
;;------------------------------------------------------------------------------
(defun p1-def-c-type (name_type_options package)
  (multiple-value-bind (name type options)
      (parse-def-c-type-args name_type_options)
      (declare (ignore options))

    (set-fftype name type)

    (cond ((atom type)                  ;primitiver Typ
           ;; erzeuge die Test-Funktion (<name>-p <c-value>)
           (p1-defmacro `(,(intern-postfixed name "-P") (arg)
                          (list ',(intern-postfixed type "-P") arg)))
           (export (intern-postfixed name "-P") package)
           ;; erzeuge die Konstruktorfunktion (<name> <value>)
            (p1-defmacro `(,name (arg)
                           (list ',type arg)))
           (export name package)
           ;; (deftype name () 'typ)
           )
          (t)) ; *** Hier her kommen noch die anderen Faelle, wie Struct, ...
    ))

;;------------------------------------------------------------------------------
;; Holt den Funktionsname und die Optionen-Liste und testet einige
;; Fehlerquellen ab.
;;
;; Resultat: (MV) <name> <foreign-name> <arg-list> <return-type>
;;------------------------------------------------------------------------------
(defun parse-foreign-fun-args (name_options)
  (let (name
        options)
    
    ;; Kein Name
    (when (null name_options)
          (clicc-error NO_NAME_SPECIFIED))

    (if (atom name_options)
        (setq name name_options
              ;; Keine options
              options nil)
        (setq name (first name_options)
              options (rest name_options)))

    ;; Kein Symbol als Name
    (unless (symbolp name)
      (clicc-error NO_NAME name 'DEF-CALL-OUT))

    (multiple-value-bind (foreign-name arg-list return-type)
        (parse-ff-key-list name options)

      (values name foreign-name arg-list return-type))))


;;------------------------------------------------------------------------------
;; Parsed die Key-Liste einer Call-Out-Definition. Es werden
;; Default-Werte angelegt, wenn keine Keys angegeben sind.
;; Es werden Fehler in der Key-List abgefangen.
;;
;; Die Keyword-Liste wird von links nach rechts durchgegangen, wobei
;; das letzte Auftreten eines Keywords den endgueltigen Wert festlegt.
;; [entgegen den ueblichen LISP-Konventionen. muss noch geaendert werden.]
;;
;; Resultat: (MV) <foreign-name> <arg-list> <return-type>
;;------------------------------------------------------------------------------
(defun parse-ff-key-list (name options)
  (let ((foreign-name (string-downcase (string name)))
        (arg-list '())
        (return-type 'ffi:c-int))

    (unless (evenp (length options))
      (clicc-error ODD_LEN_KEYLIST))

    (do* ((key-list options (cddr key-list))
          (key (first key-list) (first key-list))
          (arg (second key-list) (second key-list)))
         ((null key-list))
      
      (unless (keywordp key)
        (clicc-error NO_KEYWORD key))

      (when (null arg)
        (clicc-error ODD_LEN_KEYLIST))

      (case key
        (:name 
         (if (stringp arg)
             (setq foreign-name arg)
             (clicc-error NO_STRING arg)))

        (:arguments
         (if (listp arg)
             (if (= (length arg) 1)
                 (setq arg-list (list (parse-c-type (first arg)
                                                    :could-be-void t
                                                    :could-be-vararg t)))
                 (setq arg-list 
                       (append (mapcar #'parse-c-type (butlast arg))
                               (list (parse-c-type (car (last arg))
                                                   :could-be-vararg t)))))
             (clicc-error NO_LIST arg)))

        (:return-type
         (setq return-type (parse-c-type arg)))

        (otherwise
         (clicc-error ILLEGAL_KEY '(:NAME :ARGUMENTS :RETURN-TYPE) key)))
      )
    
    (values foreign-name arg-list return-type))
)

;;------------------------------------------------------------------------------
;; Erzeugt die Typueberpruefung fuer die Argumente einer C-Funktion.
;; Diese Ueberpruefung kann durch eventuelle Optimierer schon waehrend der
;; Kompilation entfernt werden.
;;------------------------------------------------------------------------------
(defun p1-foreign-fun-call (operator-def form)
  (let* ((form-arg-list (rest form))    ; Arg-List der Applikation
         (let-arg-list '())             ; Liste der mit gensym erz. Namen
         (name (gensym))                ; Name der expandierten FF
         (operator (cdr operator-def))  ; Definition der FF
         (type-list 
          (?arg-type-list operator))    ; Liste der Foreign-Types
         )

    (when (not (= (length form-arg-list) (length type-list)))
      (clicc-error 
       NOT_OF_RIGHT_LENGTH (length form-arg-list) (length type-list)))

    (labels ((c-typep (arg typ)
               (if (atom typ)
                   `(typep ,arg (quote ,typ))
                   (clicc-error "illegal argument-type:~S ~S" arg typ)))
      
             (make-let-arg-list ()
               (let (liste)
                 (dolist (arg form-arg-list liste)
                   (let ((generated-sym (gensym)))
                     (setq let-arg-list `(,@let-arg-list ,generated-sym))
                     (setq liste `(,@liste (,generated-sym ,arg)))))))
             
             (make-and-list ()
               (let (liste)
                 (dotimes (count (length form-arg-list) liste)
                   (setq liste `(,@liste ,(c-typep 
                                           (nth count let-arg-list)
                                           (nth count type-list))))))))
      
      (set-foreign-fun name operator)

      `(let* ,(make-let-arg-list)
        (if (and ,@(make-and-list))
            (,name ,@let-arg-list)
            (error "illegal argument-type"))))))

;;------------------------------------------------------------------------------
;; Testet auf Korrektheit der Argumente einer C-Typ-Definition
;;
;; RESULTAT: (MV) <name> <type> <options>
;;------------------------------------------------------------------------------
(defun parse-def-c-type-args (name_type_options)
  (let (name
        type
        options)

    (unless (>= (length name_type_options) 2)
      (clicc-error "You must specify a name and a type"))

    (setq name (first name_type_options))

    (unless (symbolp name)
      (clicc-error NO_NAME name 'DEF-C-TYPE))

    (when (get-fftype name)
      (clicc-error "It is illegal to redefine the c-type ~S" name))

    (setq type (parse-c-type (second name_type_options)))

    (setq options (cddr name_type_options))

    (values name type options)))


;;------------------------------------------------------------------------------
(defconstant ILLEGAL_ARGUMENT
  "Not enough or to many arguments for ~S in PARSE-C-TYPE")

;;------------------------------------------------------------------------------
;; Parse eine C-Typspezifikation
;;------------------------------------------------------------------------------
(defun parse-c-type (type-spec &key could-be-void called-by-ptr could-be-vararg)
  (cond ((and (symbolp type-spec) type-spec) 
         (cond ((and (eq type-spec 'void) (not could-be-void)) 
                (clicc-error "VOID cannot be used here.")) 
               ((and (eq type-spec 'vararg) (not could-be-vararg)) 
                (clicc-error "VARARG cannot be used here.")) 
               (t 
                (let ((type (get-fftype type-spec))) 
                  (if called-by-ptr 
                      (unless type 
                        (set-fftype type-spec :not-defined)) 
                      (when (or (not type) (eq type :not-defined)) 
                        (clicc-error "Unknown type-specifier ~S" type-spec))))))
         (values (get-fftype type-spec)))

        ((listp type-spec) 
         (let ((konstruktor (first type-spec)) 
               (laenge (length type-spec))) 
           (case konstruktor 

             (ptr                       ; (ptr <typ>||void)
              (if (= laenge 2) 
                  (values 
                   `(PTR ,(parse-c-type (second type-spec) 
                           :could-be-void t 
                           :called-by-ptr t))) 
                  (clicc-error ILLEGAL_ARGUMENT 'ptr)))

;;;             (handle                    ; (handle <typ>||void)
;;;              (if (= laenge 2)
;;;                  (values (parse-c-type (second type-spec)
;;;                                         :could-be-void t
;;;                                         :called-by-ptr t))
;;;                  (clicc-error ILLEGAL_ARGUMENT 'ptr)))
             
             (fun                  ; (fun <typ>||void (<typ>* [vararg])||(void))
              (if (= laenge 3) 
                  (if (listp (third type-spec)) 
                      (if called-by-ptr ; Es muss ein Ptr vorangestellt sein.
                          (flet ((parse-fun-arg-list (type-list) 
                                        ; Wenn ein Argument, dann kann
                                        ; es void sein.
                                   (if (= (length type-list) 1) 
                                       (values 
                                        `(,(parse-c-type (first type-list)
                                            :could-be-void t
                                            :could-be-vararg t)))
                                        ; Das letzte Argument der Liste kann
                                        ; vararg sein
                                       (values 
                                        `(,@(mapcar 
                                             #'parse-c-type (butlast type-list))
                                          ,(parse-c-type (car (last type-list))
                                            :could-be-vararg t))))))
                            (values 
                             `(FUN ,(parse-c-type (second type-spec)
                                     :could-be-void t)
                               ,(parse-fun-arg-list (third type-spec)))))
                          (clicc-error "Type `fun' must be used with `ptr'."))
                      (clicc-error "Second argument of FUN must be a list."))
                  (clicc-error ILLEGAL_ARGUMENT 'fun)))

             (struct                    ; (stuct (<bezeichner> <typ>)+)
              (if (>= laenge 2) 
                  (values 
                   `(STRUCT 
                     ,@(maplist
                        #'(lambda (r-l)
                            (if (member (caar r-l) (cdr r-l) :key #'car)
                                (clicc-error "Identifier used twice")
                                (if (= (length (car r-l)) 2)
                                    (values `(,(caar r-l)
                                              ,(parse-c-type (cadar r-l))))
                                    (clicc-error ILLEGAL_ARGUMENT 'struct))))
                        (rest type-spec))))
                  (clicc-error ILLEGAL_ARGUMENT 'struct)))
             
             (union ; (union (<bezeichner> <typ>)+)
              )
             
             (enum ; (enum (bezeichner)+)
              )
             
             (array ; (vector <c-typ> {<integer>||*}+)
              (if (>= laenge 3)
                  (flet ((parse-array-arg-list (a-number)
                           (if (or (typep a-number 'fixnum)
                                   (eq a-number '*))
                               (values a-number)
                               (clicc-error "~A must be a fixnum or '*"
                                            a-number))))
                    (values
                     `(ARRAY
                       ,(parse-c-type (second type-spec))
                       ,@(mapcar #'parse-array-arg-list (cddr type-spec))
                       )))
                  (clicc-error ILLEGAL_ARGUMENT 'array)))
             
             (otherwise
              (clicc-error "Unknown constructor ~A" konstruktor)))))
        
        (t
         (clicc-error "Unknown type-spec ~S." type-spec))))


;;------------------------------------------------------------------------------
;; finalize-call-in-funs
;;------------------------------------------------------------------------------
(defun finalize-call-in-funs ()
  (let ((call-in-funs (?call-in-funs *GLOBAL-ENVIRONMENT*))
        )
    (flet ((check-call-in (name_call-in-fun)
             (let* ((name (car name_call-in-fun))
                    (call-in-fun (cdr name_call-in-fun))
                    (operator (get-operator-def name))
                    (global-fun (cdr operator)))
               (unless (eq (car operator) :GLOBAL-FUN)
                 (clc-error "No global-fun declared as call-in-fun: ~A"
                              name))
               (setf (?exported global-fun) t)
               (setf (?call-in global-fun) call-in-fun))))

      (mapcar #'check-call-in call-in-funs))))


;;------------------------------------------------------------------------------
(provide "p1foreign")
