;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;-----------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practicable And Portable Lisp Implementation
;;;            ------------------------------------------------------
;;; Inhalt   : Funktionen zur Inline-Compilation von Funktionsapplikationen.
;;;
;;; $Revision: 1.20 $
;;; $Log: inline.lisp,v $
;;; Revision 1.20  1993/07/26  14:35:39  wg
;;; Lokale Funktionen in Methoden fuer CMU global definiert.
;;;
;;; Revision 1.19  1993/07/23  13:18:49  hk
;;; Neue Variabale *inline-verbosity*.
;;;
;;; Revision 1.18  1993/07/22  12:46:45  jh
;;; Fehler in mc-form behoben.
;;;
;;; Revision 1.17  1993/07/21  12:45:06  jh
;;; structured-literals in Funktionen, die inline kompiliert werden sollen,
;;; werden in defined-named-const verpackt.
;;;
;;; Revision 1.16  1993/07/20  13:49:22  jh
;;; Die Funktion error wird nicht mehr inline kompiliert.
;;;
;;; Revision 1.15  1993/07/19  09:53:29  jh
;;; structured-literals werden wieder inline kompiliert, da sie jetzt in
;;; named-const verpackt werden.
;;;
;;; Revision 1.14  1993/06/17  08:00:09  hk
;;; Copright Notiz eingefuegt
;;;
;;; Revision 1.13  1993/06/07  08:31:15  jh
;;; Kommentar in bind-arguments eingeguegt.
;;;
;;; Revision 1.12  1993/06/05  22:27:34  hk
;;; structured-literals werden zunaechst nicht inline kompiliert.
;;;
;;; Revision 1.11  1993/06/05  18:57:28  hk
;;; In bind-arguments bei der Bearbeitung von Keyword Parametern
;;; (push a-key processed-keys) eingefuegt.
;;;
;;; Revision 1.10  1993/05/27  12:51:22  jh
;;; Inlining von tagbody-forms und let/cc-forms eingebaut.
;;;
;;; Revision 1.9  1993/05/25  13:36:36  jh
;;; Testmeldungen entfernt.
;;;
;;; Revision 1.8  1993/05/25  13:29:26  jh
;;; Inlining von Funktionen mit &key- und &rest-Parametern eingebaut.
;;;
;;; Revision 1.7  1993/05/19  11:37:24  jh
;;; *max-inline-weight* von 20 auf 10 verkleinert.
;;;
;;; Revision 1.6  1993/05/18  09:16:55  jh
;;; Die Reihenfolge der neuen Parametervariablen korrigiert.
;;;
;;; Revision 1.5  1993/05/11  09:03:04  hk
;;; inline-app, arguments: Fehler bei optionalen Paramtern behoben.
;;;
;;; Revision 1.4  1993/05/10  12:24:15  jh
;;; Schalter *no-inlining* eingebaut. mv-lambda wird jetzt unterstuetzt.
;;;
;;; Revision 1.3  1993/02/16  15:16:47  hk
;;; $ eingefuegt.
;;;
;;; Revision 1.2  1993/01/28  12:06:07  jh
;;; Fehler beseitigt.
;;;-----------------------------------------------------------------------------

(in-package "CLICC")

;;------------------------------------------------------------------------------
;; Variable zur Steuerung der Gespr"achigkeit
;; 0: nichts
;; 1: Gesamtzahl
;; 2: Fuer jede Funktion
;;------------------------------------------------------------------------------
(defvar *inline-verbosity* 1)

;;------------------------------------------------------------------------------
;; *max-inline-weight* gibt das maximale Gewicht an, bis zu dem eine Funktion
;; inline-compiliert werden soll.
;;------------------------------------------------------------------------------

(defvar *max-inline-weight* 12)

(defun set-max-inline-weight (weight)
  (setf *max-inline-weight* weight))

;;------------------------------------------------------------------------------
;; Makro zur besseren Lesbarkeit.
;;------------------------------------------------------------------------------

(defmacro inline-field (field &optional history)
  `(setf ,field (inline-form ,field ,history)))

;;------------------------------------------------------------------------------
;; inline-module untersucht die im Modul definierten Funktionen sowie die
;; toplevel-forms nach Applikationen, die inline-compiliert werden sollen.
;;------------------------------------------------------------------------------

(defvar *no-inlining* nil)

(defun inline-module (a-module)
  (unless *no-inlining*
    (mapc #'set-inline-slot (?fun-list a-module))
    (inline-fun-def-list (?fun-list a-module))
    (inline-field (?body (?toplevel-forms a-module)))))

(defun do-inline ()
  (init-inline-statistics)
  (inline-module *module*)
  (write-inline-statistics))

;;------------------------------------------------------------------------------
;; Variablen und Funktionen fuer die Inline-Statistik.
;;------------------------------------------------------------------------------

;; Tabelle zur Inline-Statistik bestehend aus Paaren
;; (Funktionsname . Anzahl der Inline-Compilationen)
(defvar *inline-statistics*)

(defun init-inline-statistics ()
  (setf *inline-statistics* nil))

(defun inc-inline-counter (fun-name)
  (let ((pair (assoc fun-name *inline-statistics*)))
    (if pair
        (incf (cdr pair))
        (push (cons fun-name 1) *inline-statistics*))))

(defun write-inline-statistics ()
  (when (> *inline-verbosity* 0)
    (let ((total-number-of-inline-compilations
           (apply #'+ (mapcar #'cdr *inline-statistics*))))
      (setf *inline-statistics*
            (sort *inline-statistics* #'(lambda (e1 e2)
                                          (> (cdr e1) (cdr e2)))))
      (clicc-message "---------------------------------------------------------~
                      -------------")
      (clicc-message "~D application~:p inline compiled." 
                     total-number-of-inline-compilations)
      (when (> *inline-verbosity* 1)
        (dolist (entry *inline-statistics*)
          (let ((fun-name (car entry))
                (number-of-inline-compilations (cdr entry)))
            (clicc-message "~A ~D time~:P inline compiled."
                           fun-name
                           number-of-inline-compilations)))
        (clicc-message "-------------------------------------------------------~
                        ---------------")))))

;;------------------------------------------------------------------------------

(defmethod inline-fun-def ((a-simple-fun simple-fun) &optional history)
  ;; Funktionen, die selbst inline compiliert werden, werden nicht veraendert.
  (unless (?inline a-simple-fun)
    (inline-field (?body a-simple-fun) (cons a-simple-fun history))
    a-simple-fun))

(defun inline-fun-def-list (fun-def-list &optional history)
  (mapc #'(lambda (fun-def) (inline-fun-def fun-def history)) fun-def-list))

(defun inline-form-list (form-list &optional history)
  (unless (endp form-list)
    (inline-field (first form-list) history)
    (inline-form-list (rest form-list) history)))

;;------------------------------------------------------------------------------
;; inline-form versucht Applikationen innerhalb eines Zwischensprachausdrucks
;; inline zu compilieren.
;;------------------------------------------------------------------------------

(defmethod inline-form ((a-form form) &optional history)
  (declare (ignore history))
  a-form)

(defmethod inline-form ((an-app app) &optional history)
  (inline-form-list (?arg-list an-app) history)
  (if (and (inline-app-p an-app)
           (not (member (?form an-app) history)))
      (progn
        (inc-inline-counter (?symbol (?form an-app)))
        (inline-form (inline-app an-app) (cons (?form an-app) history)))
      an-app))

(defmethod inline-form ((a-setq-form setq-form) &optional history)
  (inline-field (?form a-setq-form) history)
  a-setq-form)

(defmethod inline-form ((a-progn-form progn-form) &optional history)
  (inline-form-list (?form-list a-progn-form) history)
  a-progn-form)

(defmethod inline-form ((an-if-form if-form) &optional history)
  (inline-field (?pred an-if-form) history)
  (inline-field (?then an-if-form) history)
  (inline-field (?else an-if-form) history)
  an-if-form)

(defmethod inline-form ((a-switch-form switch-form) &optional history)
  (inline-field (?form a-switch-form) history)
  (inline-form-list (?case-list a-switch-form) history)
  a-switch-form)

(defmethod inline-form ((a-labeled-form labeled-form) &optional history)
  (inline-field (?form a-labeled-form) history)
  a-labeled-form)

(defmethod inline-form ((a-let*-form let*-form) &optional history)
  (inline-form-list (?init-list a-let*-form) history)
  (inline-field (?body a-let*-form) history)
  a-let*-form)

(defmethod inline-form ((a-labels-form labels-form) &optional history)
  (mapc #'set-inline-slot (?fun-list a-labels-form))
  (inline-fun-def-list (?fun-list a-labels-form) history)
  (inline-field (?body a-labels-form) history)
  a-labels-form)

(defmethod inline-form ((a-let/cc-form let/cc-form) &optional history)
  (inline-field (?body a-let/cc-form) history)
  a-let/cc-form)

(defmethod inline-form ((a-tagbody-form tagbody-form) &optional history)
  (when (?first-form a-tagbody-form)
    (inline-field (?first-form a-tagbody-form) history))
  (mapc #'(lambda (a-tagged-form) (inline-field (?form a-tagged-form) history))
        (?tagged-form-list a-tagbody-form))
  a-tagbody-form)

(defmethod inline-form ((a-tagged-form tagged-form) &optional history)
  (declare (ignore history))
  a-tagged-form)

(defmethod inline-form ((a-mv-lambda mv-lambda) &optional history)
  (inline-field (?arg a-mv-lambda) history)
  (inline-field (?body a-mv-lambda) history)
  a-mv-lambda)

;;------------------------------------------------------------------------------
;; inline-app-p entscheidet, ob eine Funktionsapplikation inline compiliert
;; werden soll.
;;------------------------------------------------------------------------------

(defmethod inline-app-p ((an-app app))
  (let ((app-form (?form an-app)))
    (when (and (fun-p app-form) (slot-boundp app-form 'body))
      (and
       (if (slot-boundp app-form 'inline)
           (?inline app-form)
           (set-inline-slot app-form))
       (keywords-ok-p (?params app-form) (?arg-list an-app))))))

(defun keywords-ok-p (params args)
  (or (null (?key-list params))
      (let ((key-args (subseq args (+ (length (?var-list params))
                                      (length (?opt-list params))))))
        (or (null key-args)
            (when (evenp (length key-args))
              (subsetp (split-args key-args)
                       (mapcar #'?sym (?key-list params))))))))
                    
;;------------------------------------------------------------------------------
;; set-inline-slot prueft die Funktion, ob sie inline compiliert werden soll.
;; Das Ergebnis dieser Pruefung wird in den inline-slot der Funktion 
;; geschrieben.
;;------------------------------------------------------------------------------

(defvar *has-structured-literals*)

(defmethod set-inline-slot ((a-defined-fun defined-fun))
  (let* ((*has-structured-literals* nil)
         (inline-it
          (and (not (eq a-defined-fun (get-function 'clicc-lisp::error)))
               (not (member a-defined-fun (?called-by a-defined-fun)))
               (and (<= (fun-weight a-defined-fun) *max-inline-weight*)
                    (not (?allow-other-keys (?params a-defined-fun)))))))
    (setf (?inline a-defined-fun) inline-it)
    (when (and inline-it *has-structured-literals*)
      (make-copyable a-defined-fun))
    inline-it))

;;------------------------------------------------------------------------------
;; make-copyable steckt jedes structured-literal in eine defined-named-const,
;; damit das structured-literal weiterhin nur ein angewandtes Vorkommen hat.
;;------------------------------------------------------------------------------

(defun make-copyable (a-fun)
  (mc-form (?params a-fun))
  (setf (?body a-fun) (mc-form (?body a-fun))))

(defmethod mc-form ((a-form form))
  a-form)

#+CMU(defun mc-opt/key (opt/key)
  (setf (?init opt/key) (mc-form (?init opt/key))))

(defmethod mc-form ((parameters params))
  (labels (#-CMU(mc-opt/key (opt/key)
             (setf (?init opt/key) (mc-form (?init opt/key)))))
    (mapc #'mc-opt/key (?opt-list parameters))
    (mapc #'mc-opt/key (?key-list parameters))))

(defmethod mc-form ((a-structured-literal structured-literal))
  (let ((const (make-instance 'defined-named-const
                              :value a-structured-literal
                              :symbol (gensym))))
    (push const (?named-const-list *module*))
    const))

(defmethod mc-form ((an-app app))
  (setf (?form an-app) (mc-form (?form an-app))
        (?arg-list an-app) (mapcar #'mc-form (?arg-list an-app)))
  an-app)

(defmethod mc-form ((a-setq-form setq-form))
  (setf (?form a-setq-form) (mc-form (?form a-setq-form)))
  a-setq-form)

(defmethod mc-form ((a-progn-form progn-form))
  (setf (?form-list a-progn-form) (mapcar #'mc-form (?form-list a-progn-form)))
  a-progn-form)

(defmethod mc-form ((an-if-form if-form))
  (setf (?pred an-if-form) (mc-form (?pred an-if-form))
        (?then an-if-form) (mc-form (?then an-if-form))
        (?else an-if-form) (mc-form (?else an-if-form)))
  an-if-form)

(defmethod mc-form ((a-switch-form switch-form))
  (setf (?form a-switch-form) (mc-form (?form a-switch-form))
        (?otherwise a-switch-form) (mc-form (?otherwise a-switch-form)))
  (mapc #'mc-form (?case-list a-switch-form))
  a-switch-form)

(defmethod mc-form ((a-labeled-form labeled-form))
  (setf (?form a-labeled-form) (mc-form (?form a-labeled-form)))
  a-labeled-form)

(defmethod mc-form ((a-let*-form let*-form))
  (setf (?init-list a-let*-form) (mapcar #'mc-form (?init-list a-let*-form))
        (?body a-let*-form) (mc-form (?body a-let*-form)))
  a-let*-form)

(defmethod mc-form ((a-labels-form labels-form))
  (mapc #'make-copyable (?fun-list a-labels-form))
  (setf (?body a-labels-form) (mc-form (?body a-labels-form)))
  a-labels-form)

(defmethod mc-form ((a-let/cc-form let/cc-form))
  (setf (?body a-let/cc-form) (mc-form (?body a-let/cc-form)))
  a-let/cc-form)

(defmethod mc-form ((a-tagbody-form tagbody-form))
  (when (?first-form a-tagbody-form)
    (setf (?first-form a-tagbody-form) (mc-form (?first-form a-tagbody-form))))
  (mapc #'(lambda (a-tagged-form)
            (setf (?form a-tagged-form) (mc-form (?form a-tagged-form))))
        (?tagged-form-list a-tagbody-form))
  a-tagbody-form)

(defmethod mc-form ((a-tagged-form tagged-form))
  a-tagged-form)

(defmethod mc-form ((a-mv-lambda mv-lambda))
  (mc-form (?params a-mv-lambda))
  (setf (?body a-mv-lambda) (mc-form (?body a-mv-lambda))
        (?arg a-mv-lambda) (mc-form (?arg a-mv-lambda)))
  a-mv-lambda)

;;------------------------------------------------------------------------------
;; inline-app bildet eine let*-form zur Bindung der Argumente an neue lokale
;; Variablen und kopiert den Funktionsrumpf unter Beachtung der neuen Bindungen.
;;------------------------------------------------------------------------------
(defmethod inline-app ((an-app app))
  (multiple-value-bind (var-list init-list subst-map)
      (bind-arguments (?params (?form an-app)) (?arg-list an-app))
    (let ((new-body (zs-copy (?body (?form an-app)) subst-map)))
      (if (null subst-map)
          new-body
          (make-instance 'let*-form
                         :var-list var-list
                         :init-list init-list
                         :body new-body)))))

(defun split-args (args)
  (if (null args)
      (values () ())
      (multiple-value-bind (rest-keys rest-args)
          (split-args (rest (rest args)))
        (values (cons (first args) rest-keys)
                (cons (first (rest args)) rest-args)))))

(defun bind-arguments (params args)
  (let ((new-var-queue (empty-queue))
        (init-queue (empty-queue))
        (subst-map ()))

    ;; Zuerst werden die benoetigten Parameter versorgt.
    ;;-------------------------------------------------- 
    (dolist (a-var (?var-list params))
      (let ((new-var (new-variable a-var)))
        (add-q new-var new-var-queue)
        (push (cons a-var new-var) subst-map)
        (add-q (pop args) init-queue)))

    ;; Dann sind die optionalen Parameter dran.
    ;;----------------------------------------- 
    (dolist (an-opt (?opt-list params))
      (let ((new-var (new-variable (?var an-opt))))
        (add-q new-var new-var-queue)
        (push (cons (?var an-opt) new-var) subst-map)
        (add-q (if args
                   (first args)
                   (zs-copy (?init an-opt) subst-map))
               init-queue)
        (when (?suppl an-opt)
          (let ((new-suppl-var (new-variable (?suppl an-opt))))
            (add-q new-suppl-var new-var-queue)
            (push (cons (?suppl an-opt) new-suppl-var) subst-map)
            (add-q (if args
                       (get-symbol-bind t)
                       empty-list)
                   init-queue)))
        (pop args)))

    ;; Und jetzt kriegen's die Keyword-Parameter.
    ;;-------------------------------------------
    (if (?key-list params)
        (multiple-value-bind (keys key-args) (split-args args)
          (let ((key-params (?key-list params))
                (processed-keys ())
                (rest-queue (empty-queue)))
            (dolist (a-key keys)
              (let* ((key-processed (member a-key processed-keys))
                     (key-param (find a-key key-params :key #'?sym))
                     (arg (first key-args))
                     (new-var (if key-processed 
                                  ;; Da der Key bereits abgearbeitet wurde, wird
                                  ;; die neue Variable nur zur Aufbewahrung des
                                  ;; berechneten Wertes fuer einen moeglichen
                                  ;; Restparameter benutzt.
                                  (make-instance 'local-static
                                                 :read 0
                                                 :write 1
                                                 :symbol (gensym)
                                                 :type (?type arg))
                                  (new-variable (?var key-param)))))
                (add-q new-var new-var-queue)
                (add-q arg init-queue)
                (unless key-processed
                  (push a-key processed-keys)
                  (push (cons (?var key-param) new-var) subst-map)
                  (when (?suppl key-param)
                    (let ((new-suppl-var (new-variable (?suppl key-param))))
                      (add-q new-suppl-var new-var-queue)
                      (push (cons (?suppl key-param) new-suppl-var)
                            subst-map)
                      (add-q (get-symbol-bind t) init-queue))))
                (when (?rest params)
                  (incf (?read new-var))
                  (add-q a-key rest-queue)
                  (add-q (make-instance 'var-ref
                                        :var new-var
                                        :type (?type arg))
                         rest-queue))
                (pop key-args)))
            (dolist (a-key-param key-params)
              (unless (member (?sym a-key-param) processed-keys)
                (let ((new-var (new-variable (?var a-key-param))))
                  (add-q new-var new-var-queue)
                  (push (cons (?var a-key-param) new-var) subst-map)
                  (add-q (zs-copy (?init a-key-param) subst-map) init-queue)
                  (when (?suppl a-key-param)
                    (let ((new-suppl-var (new-variable (?suppl a-key-param))))
                      (add-q new-suppl-var new-var-queue)
                      (push (cons (?suppl a-key-param) new-suppl-var)
                            subst-map)
                      (add-q empty-list init-queue))))))

            ;; Neben den Keyword-Param. gibt's evtl. noch einen Rest-Parameter.
            ;;-----------------------------------------------------------------
            (when (?rest params)
              (let ((new-rest-var (new-variable (?rest params)))
                    (list-fun (get-global-fun 'L::list)))
                (add-q new-rest-var new-var-queue)
                (push (cons (?rest params) new-rest-var) subst-map)
                (add-q (if (empty-queue-p rest-queue)
                           empty-list
                           (make-instance 'app
                                          :form list-fun
                                          :arg-list (queue2list rest-queue)
                                          :read-list :unknown
                                          :write-list :unknown
                                          :called-funs (list list-fun)
                                          :other-funs nil))
                       init-queue)))))
    
        ;; Nun noch der Rest-Parameter solo.
        ;;----------------------------------
        (when (?rest params)
          (let ((new-rest-var (new-variable (?rest params)))
                (list-fun (get-global-fun 'L::list)))
            (add-q new-rest-var new-var-queue)
            (push (cons (?rest params) new-rest-var) subst-map)
            (add-q (if args
                       (make-instance 'app
                                      :form list-fun
                                      :arg-list args
                                      :read-list :unknown
                                      :write-list :unknown
                                      :called-funs (list list-fun)
                                      :other-funs nil)
                       empty-list)
                   init-queue))))

    (values (queue2list new-var-queue)
            (queue2list init-queue)
            subst-map)))

(defmethod new-variable ((a-local-static local-static))
  (let ((new-local-static (make-instance 'local-static
                                         :read (?read a-local-static)
                                         :write (?write a-local-static)
                                         :symbol (?symbol a-local-static)
                                         :type (?type a-local-static))))
    (when (slot-boundp a-local-static 'level)
      (setf (?level new-local-static) (?level a-local-static)))
    new-local-static))

(defmethod new-variable ((a-var var))
  a-var)

(defun new-local-var-list (var-list subst-map)
  (let ((new-var-queue (empty-queue)))
    (dolist (old-var var-list)
      (let ((new-var (new-variable old-var)))
        (add-q new-var new-var-queue)
        (push (cons old-var new-var) subst-map)))
    (values (queue2list new-var-queue) subst-map)))

(defun new-parameters (params subst-map)
  (multiple-value-bind (new-all-vars new-subst-map)
      (new-local-var-list (?all-vars params) subst-map)
    (labels ((new-internal (var)
               (new-param-var var new-subst-map)))
      (values
       (make-instance 'params
                      :var-list (mapcar #'new-internal (?var-list params))
                      :opt-list (mapcar #'new-internal (?opt-list params))
                      :rest (if (?rest params)
                                (new-internal (?rest params))
                                nil)
                      :key-list (mapcar #'new-internal (?key-list params))
                      :allow-other-keys (?allow-other-keys params)
                      :all-vars new-all-vars)
       new-subst-map))))

(defmethod new-param-var ((a-var var) subst-map)
  (zs-copy a-var subst-map))

(defmethod new-param-var ((an-opt opt) subst-map)
  (make-instance 'opt
                 :var (zs-copy (?var an-opt) subst-map)
                 :init (zs-copy (?init an-opt) subst-map)
                 :suppl (if (?suppl an-opt)
                            (zs-copy (?suppl an-opt) subst-map)
                            nil)))

(defmethod new-param-var ((a-key key) subst-map)
  (incf (?used (?sym a-key)))
  (make-instance 'key
                 :var (zs-copy (?var a-key) subst-map)
                 :init (zs-copy (?init a-key) subst-map)
                 :suppl (if (?suppl a-key)
                            (zs-copy (?suppl a-key) subst-map)
                            nil)
                 :sym (?sym a-key)))

;;------------------------------------------------------------------------------
;; zs-copy kopiert einen Zwischensprachausdruck, wobei die neuen Variablen-
;; bindungen beachtet werden.
;;------------------------------------------------------------------------------

(defmethod zs-copy ((a-local-static local-static) subst-map)
  (let ((entry (assoc a-local-static subst-map)))
    (if entry
        (cdr entry)
        a-local-static)))

(defmethod zs-copy ((a-global-static global-static) subst-map)
  (declare (ignore subst-map))
  a-global-static)

(defmethod zs-copy ((a-imported-static imported-static) subst-map)
  (declare (ignore subst-map))
  a-imported-static)

(defmethod zs-copy ((a-dynamic dynamic) subst-map)
  (declare (ignore subst-map))
  a-dynamic)

(defmethod zs-copy ((a-var-ref var-ref) subst-map)
  (make-instance 'var-ref :var (zs-copy (?var a-var-ref) subst-map)))

(defmethod zs-copy ((a-named-const named-const) subst-map)
  (declare (ignore subst-map))
  a-named-const)

(defmethod zs-copy ((a-literal literal) subst-map)
  (declare (ignore subst-map))
  a-literal)

(defmethod zs-copy ((a-class-def class-def) subst-map)
  (declare (ignore subst-map))
  a-class-def)

(defmethod zs-copy ((a-fun fun) subst-map)
  (declare (ignore subst-map))
  a-fun)

(defmethod zs-copy ((an-app app) subst-map)
  (make-instance 'app
                 :form (zs-copy (?form an-app) subst-map)
                 :arg-list (mapcar #'(lambda (a-form)
                                       (zs-copy a-form subst-map))
                                   (?arg-list an-app))
                 :read-list (?read-list an-app)
                 :write-list (?write-list an-app)
                 :mv-used (?mv-used an-app)
                 ;; :downfun-list '() *** Anouar fragen.
                 :type (?type an-app)))

(defmethod zs-copy ((a-setq-form setq-form) subst-map)
  (make-instance 'setq-form
                 :location (zs-copy (?location a-setq-form) subst-map)
                 :form (zs-copy (?form a-setq-form) subst-map)
                 :type (?type a-setq-form)))

(defmethod zs-copy ((a-progn-form progn-form) subst-map)
  (make-instance 'progn-form
                 :form-list (mapcar #'(lambda (a-form)
                                        (zs-copy a-form subst-map))
                                    (?form-list a-progn-form))
                 :type (?type a-progn-form)))

(defmethod zs-copy ((an-if-form if-form) subst-map)
  (make-instance 'if-form
                 :pred (zs-copy (?pred an-if-form) subst-map)
                 :then (zs-copy (?then an-if-form) subst-map)
                 :else (zs-copy (?else an-if-form) subst-map)
                 :type (?type an-if-form)))

(defmethod zs-copy ((a-switch-form switch-form) subst-map)
  (make-instance 'switch-form
                 :form (zs-copy (?form a-switch-form) subst-map)
                 :case-list (mapcar #'(lambda (a-form)
                                        (zs-copy a-form subst-map))
                                    (?case-list a-switch-form))
                 :otherwise (zs-copy (?otherwise a-switch-form)
                                        subst-map)
                 :type (?type a-switch-form)))

(defmethod zs-copy ((a-labeled-form labeled-form) subst-map)
  (make-instance 'labeled-form
                 :value (zs-copy (?value a-labeled-form) subst-map)
                 :form (zs-copy (?form a-labeled-form) subst-map)
                 :type (?type a-labeled-form)))

(defmethod zs-copy ((a-let*-form let*-form) subst-map)
  (multiple-value-bind (new-var-list new-subst-map)
      (new-local-var-list (?var-list a-let*-form) subst-map)
    (make-instance 'let*-form
                   :var-list new-var-list
                   :init-list (mapcar #'(lambda (a-form)
                                          (zs-copy a-form new-subst-map))
                                      (?init-list a-let*-form))
                   :body (zs-copy (?body a-let*-form) new-subst-map)
                   :type (?type a-let*-form))))

(defmethod zs-copy ((a-let/cc-form let/cc-form) subst-map)
  (clicc-message "inlining let/cc-form")
  (let* ((cont (?cont a-let/cc-form))
         (new-cont (make-instance 'cont
                                  :read (?read cont)
                                  :write (?write cont)
                                  :type (?type cont)
                                  :mv-spec (?mv-spec cont)
                                  :unknown-caller (?unknown-caller cont)
                                  :level 0)))
    (push (cons cont new-cont) subst-map)
    (make-instance 'let/cc-form
                   :cont new-cont
                   :body (zs-copy (?body a-let/cc-form) subst-map))))
                
(defmethod zs-copy ((a-tagbody-form tagbody-form) subst-map)
  (clicc-message "inlining tagbody-form")
  (dolist (a-tagged-form (?tagged-form-list a-tagbody-form))
    (push (cons a-tagged-form
                (make-instance 'tagged-form :used (?used a-tagged-form)))
          subst-map))
  (let ((new-tagbody-form
         (make-instance 'tagbody-form
                        :first-form (zs-copy (?first-form a-tagbody-form)
                                             subst-map)
                        :tagged-form-list
                        (mapcar #'(lambda (a-tagged-form)
                                    (zs-copy a-tagged-form subst-map))
                                (?tagged-form-list a-tagbody-form)))))
    (dolist (old-tagged-form (?tagged-form-list a-tagbody-form))
      (let ((new-tagged-form (zs-copy old-tagged-form subst-map)))
        (setf (?form new-tagged-form)
              (zs-copy (?form old-tagged-form) subst-map))
        (setf (?tagbody new-tagged-form)
              new-tagbody-form)))
    new-tagbody-form))
                        
(defmethod zs-copy ((a-tagged-form tagged-form) subst-map)
  (cdr (assoc a-tagged-form subst-map)))

(defmethod zs-copy ((a-mv-lambda mv-lambda) subst-map)
  (multiple-value-bind (new-params new-subst-map)
      (new-parameters (?params a-mv-lambda) subst-map)
    (make-instance 'mv-lambda
                   :params new-params
                   :arg (zs-copy (?arg a-mv-lambda) subst-map)
                   :body (zs-copy (?body a-mv-lambda) new-subst-map)
                   :mv-spec (?mv-spec a-mv-lambda)
                   :mv-calls (?mv-calls a-mv-lambda))))

;; labels-forms werden noch nicht unterstuetzt.

;;------------------------------------------------------------------------------
;; weight berechnet das Gewicht, das eine Form bei inline-Compilation in die
;; Wagschale wirft.
;;------------------------------------------------------------------------------

(defun fun-weight (a-fun)
  (+ (weight (?params a-fun))
     (weight (?body a-fun))))

(defmethod weight ((a-form form))
  1)

#+CMU(defun weight-opt/key (opt/key)
  (weight (?init opt/key)))

(defmethod weight ((parameters params))
  (labels (#-CMU(weight-opt/key (opt/key)
             (weight (?init opt/key))))
    (+ (apply '+ (mapcar #'weight-opt/key (?opt-list parameters)))
       (apply '+ (mapcar #'weight-opt/key (?key-list parameters))))))

(defmethod weight ((a-structured-literal structured-literal))
  ;; structured-literals werden vor dem inlining in named-const verpackt.
  (setq *has-structured-literals* T)
  1)

(defmethod weight ((an-app app))
  (+
   (weight (?form an-app))
   (weight-form-list (?arg-list an-app))))

(defmethod weight ((a-setq-form setq-form))
  (1+ (weight (?form a-setq-form))))

(defmethod weight ((a-progn-form progn-form))
  (weight-form-list (?form-list a-progn-form)))

(defmethod weight ((an-if-form if-form))
  (+
   (weight (?pred an-if-form))
   (weight (?then an-if-form))
   (weight (?else an-if-form))))

(defmethod weight ((a-switch-form switch-form))
  (+
   (weight (?form a-switch-form))
   (weight-form-list (?case-list a-switch-form))
   (/ (length (?case-list a-switch-form)) 2)))

(defmethod weight ((a-labeled-form labeled-form))
  (weight (?form a-labeled-form)))

(defmethod weight ((a-let*-form let*-form))
  (+
   (weight-form-list (?init-list a-let*-form))
   (weight (?body a-let*-form))))

(defmethod weight ((a-labels-form labels-form))
  ;; Funktionen, die eine labels-form enthalten, werden vorerst nicht inline
  ;; compiliert.
  (1+ *max-inline-weight*))

(defmethod weight ((a-let/cc-form let/cc-form))
  (weight (?body a-let/cc-form)))

(defmethod weight ((a-cont cont))
  1)

(defmethod weight ((a-tagbody-form tagbody-form))
  (+
   (if (null (?first-form a-tagbody-form))
       0
       (weight (?first-form a-tagbody-form)))
   (weight-form-list (mapcar #'?form (?tagged-form-list a-tagbody-form)))))

(defmethod weight ((a-tagged-form tagged-form))
  1)

(defmethod weight ((a-mv-lambda mv-lambda))
  (+
   (weight-form-list
    (let ((params (?params a-mv-lambda)))
      (nconc (mapcar #'?init (?opt-list params))
             (mapcar #'?init (?key-list params)))))
   (weight (?body a-mv-lambda))
   (weight (?arg a-mv-lambda))))

(defun weight-form-list (form-list)
  (apply #'+ (mapcar #'weight form-list)))

;;------------------------------------------------------------------------------
(provide "inline")
