;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;-----------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practicable And Portable Lisp Implementation
;;;            ------------------------------------------------------
;;; Inhalt   : Funktionen zum Aufspueren und Beseitigen von unbenutzten
;;;            Funktionen, Symbolen und benannten Konstanten.
;;;
;;; $Revision: 1.17 $
;;; $Log: delete.lisp,v $
;;; Revision 1.17  1993/07/15  12:27:37  hk
;;; ~:*~[s~; ~:;s~] durch ~:p ersetzt
;;;
;;; Revision 1.16  1993/06/29  11:20:54  jh
;;; Schreibfehler beseitigt.
;;;
;;; Revision 1.15  1993/06/29  10:55:38  jh
;;; Das Entfernen von let/cc-forms mit unbenutzter Continuation und labels-forms
;;; mit leerer fun-list nach simplifier.lisp verlegt.
;;; Generische Funktion is-used eingebaut.
;;;
;;; Revision 1.14  1993/06/17  08:00:09  hk
;;; Copright Notiz eingefuegt
;;;
;;; Revision 1.13  1993/06/14  10:28:12  jh
;;; Nicht benutzte continuations werden entfernt.
;;;
;;; Revision 1.12  1993/03/19  08:46:42  ft
;;; sym aus *objects-to-delete* entfernt, da noch Konflikte mit dem
;;; Modul-System bestehen.
;;;
;;; Revision 1.11  1993/03/18  15:18:58  jh
;;; Fehler bei vorhandenen importierten Symbolen etc. beseitigt.
;;;
;;; Revision 1.10  1993/03/18  13:11:22  jh
;;; Fehler in inc-used-slot(let*-form) behoben.
;;;
;;; Revision 1.9  1993/03/18  11:05:02  hk
;;; Fehler in inc-used-slot(var-ref) behoben.
;;;
;;; Revision 1.8  1993/03/16  16:52:49  jh
;;; Fehler beseitigt.
;;;
;;; Revision 1.7  1993/03/16  14:14:24  jh
;;; set-used-slots benutzt jetzt eigenes Traversierverfahren.
;;;
;;; Revision 1.6  1993/02/25  13:16:44  jh
;;; Lokale Funktionen werden jetzt ebenfalls entfernt.
;;;
;;; Revision 1.5  1993/02/16  15:24:21  hk
;;; Revision Keyword eingefuegt.
;;;
;;; Revision 1.4  1993/02/16  12:44:03  jh
;;; Fehler bei structured-literal beseitigt.
;;;
;;; Revision 1.3  1993/02/11  13:27:41  jh
;;; mv-lambda eingebaut.
;;;
;;; Revision 1.2  1993/02/10  13:32:31  jh
;;; Fehler beseitigt und Ausgabe geaendert.
;;;
;;; Revision 1.1  1993/01/27  13:25:04  jh
;;; Initial revision
;;------------------------------------------------------------------------------

(in-package "CLICC")

(require "traverse")

;;------------------------------------------------------------------------------
(defvar *objects-to-delete* '(fun local-fun named-constant))
(defvar *delete-verbosity* 1)
(defvar *unused-local-funs* nil)
(defvar *delete-path* nil)

;;------------------------------------------------------------------------------
;; Die Funktion search-and-delete-unused-objects ermittelt unbenutzte Objekte
;; im aktuellen Modul, gibt diese aus und entfernt sie.
;;------------------------------------------------------------------------------

(defun search-and-delete-unused-objects ()
  (new-analyse-mark)
  (let ((*unused-local-funs* '()))
    (set-used-slots *module*)
    (write-unused-objects *module*)
    (delete-unused-objects *module*)))

;;------------------------------------------------------------------------------
;; clear-all-used-slots setzt die used slots von Funktionen, Symbolen und
;; benannten Konstanten auf 0.
;;------------------------------------------------------------------------------

(defmethod clear-used-slot ((a-zws-object zws-object))
  (setf (?used a-zws-object) 0))

(defmethod clear-read-slot ((a-named-const named-const))
  (setf (?read a-named-const) 0))

(defmethod clear-read-and-write-slot ((a-var var))
  (setf (?read a-var) 0
        (?write a-var) 0))

(defun clear-all-used-slots (a-module)
  (mapc #'clear-used-slot (?fun-list a-module))
  (when (?toplevel-forms a-module)
    (setf (?used (?toplevel-forms a-module)) 1))
  (mapc #'clear-read-slot (?named-const-list a-module))
  (mapc #'clear-used-slot (?sym-list a-module))
  (mapc #'clear-used-slot (?class-def-list a-module))
  (mapc #'clear-read-and-write-slot (?var-list a-module))
  (mapc #'clear-read-and-write-slot (?dyn-var-list a-module)))

;;------------------------------------------------------------------------------
;; is-used ermittelt, ob ein Zwischensprachobjekt benutzt wird.
;;------------------------------------------------------------------------------

(defmethod is-used ((a-zws-object zws-object))
  (plusp (?used a-zws-object)))

(defmethod is-used ((a-global-fun global-fun))
  (or (call-next-method)
      (?exported a-global-fun)
      (?call-in a-global-fun)))

(defmethod is-used ((a-sym sym))
  (or (call-next-method)
      (?exported a-sym)
      (constant-value-p a-sym)))

(defmethod is-used ((a-named-const named-const))
  (or (plusp (?read a-named-const))
      (?exported a-named-const)))

(defmethod is-used ((a-var var))
  (plusp (?read a-var)))

(defmethod is-used ((a-global-static global-static))
  (or (call-next-method)
      (?exported a-global-static)))

(defmethod is-used ((a-imported-static imported-static))
  (or (call-next-method)
      (?exported a-imported-static)))

;;------------------------------------------------------------------------------
;; inc-used-slot erhoeht den Inhalt der used slots von Funktionen, Symbolen und
;; benannten Konstanten, wenn diese beim traversieren der toplevel-forms und der
;; exportierten Funktionen erreicht werden. Werden local-funs oder continuations
;; als unbenutzt erkannt, werden die entsprechenden labels- und let/cc-forms
;; entfernt. Deshalb muss jede Methode das resultierende Zwischensprachobjekt
;; zurueckliefern.
;;------------------------------------------------------------------------------

(defmethod inc-used-slot ((anything T) &optional write)
  (declare (ignore write)))

(defmethod inc-used-slot ((a-var var) &optional write)
  (unless (analysed-p a-var)
    (mark-as-analysed a-var)
    (clear-read-and-write-slot a-var))
  (if write
      (incf (?write a-var))
      (incf (?read a-var))))

(defmethod inc-used-slot ((a-dynamic dynamic) &optional write)
  (declare (ignore write))
  (call-next-method)
  (inc-used-slot (?sym a-dynamic)))

(defmethod inc-used-slot ((a-var-ref var-ref) &optional write)
  (inc-used-slot (?var a-var-ref) write))

(defmethod inc-used-slot ((a-named-const named-const) &optional write)
  (unless (analysed-p a-named-const)
    (mark-as-analysed a-named-const)
    (inc-used-slot (?value a-named-const)))
  (unless write
    (incf (?read a-named-const))))

(defmethod inc-used-slot ((a-sym sym) &optional write)
  (declare (ignore write))
  (unless (analysed-p a-sym)
    (mark-as-analysed a-sym)
    (clear-used-slot a-sym)
    (when (constant-value-p a-sym)
      (inc-used-slot (?constant-value a-sym))))
  (incf (?used a-sym)))

(defmethod inc-used-slot ((a-structured-literal structured-literal)
                          &optional write)
  (declare (ignore write))
  (inc-used-slot (?value a-structured-literal))) ; Kommt genau einmal vor!

(defmethod inc-used-slot ((an-array array) &optional write)
  (declare (ignore write))
  (let* ((total-size (array-total-size an-array))
         (flat-array (make-array total-size
                                 :displaced-to an-array
                                 :element-type (array-element-type an-array))))
    (dotimes (index total-size)
      (inc-used-slot (aref flat-array index)))))

(defmethod inc-used-slot ((a-cons cons) &optional write)
  (declare (ignore write))
  (inc-used-slot (car a-cons))
  (inc-used-slot (cdr a-cons)))

(defmethod inc-used-slot ((a-literal-instance literal-instance) &optional write)
  (declare (ignore write))
  (inc-used-slot (?class a-literal-instance))
  (mapc #'inc-used-slot (?value-list a-literal-instance)))

(defmethod inc-used-slot ((a-class-def class-def) &optional write)
  (declare (ignore write))
  (unless (analysed-p a-class-def)
    (mark-as-analysed a-class-def)
    (clear-used-slot a-class-def)
    (mapc #'inc-used-slot (?super-list a-class-def))
    (mapc #'inc-used-slot (?slot-descr-list a-class-def)))
  (incf (?used a-class-def)))

(defmethod inc-used-slot ((a-slot-desc slot-desc) &optional write)
  (declare (ignore write))
  (inc-used-slot (?symbol a-slot-desc))
  (inc-used-slot (?initform a-slot-desc))
  (mapc #'inc-used-slot (?initargs a-slot-desc)))

(defmethod inc-used-slot ((parameters params) &optional write)
  (declare (ignore write))
  (dolist (a-var (?var-list parameters))
    (inc-used-slot a-var 'write))
  (mapc #'inc-used-slot (?opt-list parameters))
  (when (?rest parameters)
    (inc-used-slot (?rest parameters) 'write))
  (mapc #'inc-used-slot (?key-list parameters)))

(defmethod inc-used-slot ((an-opt opt) &optional write)
  (declare (ignore write))
  (inc-used-slot (?var an-opt) 'write)
  (inc-used-slot (?init an-opt))
  (inc-used-slot (?suppl an-opt) 'write))

(defmethod inc-used-slot ((a-key key) &optional write)
  (declare (ignore write))
  (call-next-method)
  (inc-used-slot (?sym a-key)))

(defmethod inc-used-slot ((a-fun fun) &optional write)
  (declare (ignore write))
  (unless (analysed-p a-fun)
    (mark-as-analysed a-fun)
    (clear-used-slot a-fun)
    (when (slot-boundp a-fun 'params)
      (inc-used-slot (?params a-fun)))
    (when (slot-boundp a-fun 'body)
      (let ((*delete-path* (cons a-fun *delete-path*)))
        (inc-used-slot (?body a-fun)))
      (when (and (defined-fun-p a-fun) (member 'local-fun *objects-to-delete*))
        (setf (?local-funs a-fun)
              (remove-if-not #'is-used (?local-funs a-fun))))))
  (incf (?used a-fun)))

(defmethod inc-used-slot ((an-app app) &optional write)
  (declare (ignore write))
  (inc-used-slot (?form an-app))
  (mapc #'inc-used-slot (?arg-list an-app)))

(defmethod inc-used-slot ((a-setq-form setq-form) &optional write)
  (declare (ignore write))
  (inc-used-slot (?location a-setq-form) 'write)
  (inc-used-slot (?form a-setq-form)))

(defmethod inc-used-slot ((a-progn-form progn-form) &optional write)
  (declare (ignore write))
  (mapc #'inc-used-slot (?form-list a-progn-form)))

(defmethod inc-used-slot ((an-if-form if-form) &optional write)
  (declare (ignore write))
  (inc-used-slot (?pred an-if-form))
  (inc-used-slot (?then an-if-form))
  (inc-used-slot (?else an-if-form)))

(defmethod inc-used-slot ((a-switch-form switch-form) &optional write)
  (declare (ignore write))
  (inc-used-slot (?form a-switch-form))
  (mapc #'inc-used-slot (?case-list a-switch-form))
  (inc-used-slot (?otherwise a-switch-form)))

(defmethod inc-used-slot ((a-labeled-form labeled-form) &optional write)
  (declare (ignore write))
  (inc-used-slot (?value a-labeled-form))
  (inc-used-slot (?form a-labeled-form)))

(defmethod inc-used-slot ((a-let*-form let*-form) &optional write)
  (declare (ignore write))
  (dolist (a-var (?var-list a-let*-form))
    (inc-used-slot a-var 'write))
  (mapc #'inc-used-slot (?init-list a-let*-form))
  (inc-used-slot (?body a-let*-form)))

(defmethod inc-used-slot ((a-labels-form labels-form) &optional write)
  (declare (ignore write))
  (dolist (a-local-fun (?fun-list a-labels-form))
    (clear-used-slot a-local-fun))
  (inc-used-slot (?body a-labels-form))
  (let ((used-local-funs '()))
    (dolist (a-local-fun (?fun-list a-labels-form))
      (if (is-used a-local-fun)
          (push a-local-fun used-local-funs)
          (push (describe-location a-local-fun) *unused-local-funs*)))
    (when (member 'local-fun *objects-to-delete*)
      (setf (?fun-list a-labels-form) used-local-funs))))

(defmethod inc-used-slot ((a-let/cc-form let/cc-form) &optional write)
  (declare (ignore write))
  (let ((cont (?cont a-let/cc-form)))
    (inc-used-slot cont 'write)
    (inc-used-slot (?body a-let/cc-form))))

(defmethod inc-used-slot ((a-tagbody-form tagbody-form) &optional write)
  (declare (ignore write))
  (dolist (a-tagged-form (?tagged-form-list a-tagbody-form))
    (clear-used-slot a-tagged-form))
  (inc-used-slot (?first-form a-tagbody-form))
  (dolist (a-tagged-form (?tagged-form-list a-tagbody-form))
    (inc-used-slot (?form a-tagged-form))))

(defmethod inc-used-slot ((a-tagged-form tagged-form) &optional write)
  (declare (ignore write))
  (incf (?used a-tagged-form)))

(defmethod inc-used-slot ((a-mv-lambda mv-lambda) &optional write)
  (declare (ignore write))
  (inc-used-slot (?params a-mv-lambda))
  (inc-used-slot (?arg a-mv-lambda))
  (inc-used-slot (?body a-mv-lambda)))

;;------------------------------------------------------------------------------
;; Die Funktion set-used-slots zaehlt die angewandten Vorkommen von Funktionen,
;; Symbolen und benannten Konstanten, die von den toplevel-forms und den
;; exportierten Funktionen aus erreichbar sind.
;;------------------------------------------------------------------------------

(defun set-used-slots (a-module)
  (clear-all-used-slots a-module)
  ;; Bei Symbolen, die einen konstanten Wert enthalten, muss dieser ebenfalls
  ;; analysiert werden, auch wenn das Symbol nicht benutzt wird. (Wegen der
  ;; moeglichen Anwendung von 'symbol-value'.)
  (dolist (a-sym (?sym-list a-module))
    (when (constant-value-p a-sym)
      (mark-as-analysed a-sym)
      (inc-used-slot (?constant-value a-sym))))
  ;; Traversierung, ausgehend von den toplevel-forms:
  (inc-used-slot (?toplevel-forms a-module))
  ;; Traversierung, ausgehend von den exportierten Funktionen:
  (dolist (a-global-fun (?fun-list a-module))
    (when (?exported a-global-fun)
      (inc-used-slot a-global-fun))))

;;------------------------------------------------------------------------------
;; Die Funktion describe-location gibt eine Liste zurueck, die die Namen der
;; Funktion a-local-fun und der Funktionen, in der sie lokal definiert ist, in
;; der Schachtelungsreihenfolge von innen nach aussen enthaelt.
;;------------------------------------------------------------------------------

(defun describe-location (a-local-fun)
  (labels ((describe-location-internal (fun-path)
             (if (local-fun-p (first fun-path))
                 (cons (?symbol (first fun-path))
                       (describe-location-internal (rest fun-path)))
                 (list (?symbol (first fun-path))))))
    (cons (?symbol a-local-fun)
          (describe-location-internal *delete-path*))))
  
;;------------------------------------------------------------------------------
;; Die folgenden Funktionen dienen zum Loeschen der als unbenutzt erkannten
;; Funktionen, Symbolen und benannten Konstanten.
;;------------------------------------------------------------------------------

(defun delete-unused-funs (a-module)
  (when (member 'fun *objects-to-delete*)
    (setf (?fun-list a-module)
          (remove-if-not #'is-used (?fun-list a-module)))))

(defun delete-unused-syms (a-module)
  (when (member 'sym *objects-to-delete*)
    (setf (?sym-list a-module)
          (remove-if-not #'is-used (?sym-list a-module)))))

(defun delete-unused-named-consts (a-module)
  (when (member 'named-constant *objects-to-delete*)
    (setf (?named-const-list a-module)
          (remove-if-not #'is-used (?named-const-list a-module)))))

(defun delete-unused-objects (a-module)
  (delete-unused-funs a-module)
  (delete-unused-syms a-module)
  (delete-unused-named-consts a-module))

;;------------------------------------------------------------------------------
;; write-unused-objects gibt die Liste der unbenutzten Funktionen, Symbolen und
;; benannten Konstanten aus.
;;------------------------------------------------------------------------------
(defun list-names (objects)
  (mapcar #'?symbol objects))

(defun list-unused-funs (a-module)
  (remove-if #'is-used (?fun-list a-module)))

(defun list-unused-syms (a-module)
  (remove-if #'is-used (?sym-list a-module)))

(defun list-unused-named-consts (a-module)
  (remove-if #'is-used (?named-const-list a-module)))

(defun write-unused-objects (a-module)
  (clicc-message "----------------------------------------------------------~
                  -----------------")
  (let ((unused-funs (list-unused-funs a-module))
        (unused-syms (list-unused-syms a-module))
        (unused-named-consts (list-unused-named-consts a-module)))
    
    (when (> *delete-verbosity* 1)
      (when unused-syms
        (clicc-message "The unused symbols are:~%~S"
                       (list-names unused-syms)))
      (when unused-named-consts
        (clicc-message "The unused named constants are:~%~S"
                       (list-names unused-named-consts)))
      (when unused-funs
        (clicc-message "The unused functions are:~%~S"
                       (list-names unused-funs)))
      (when *unused-local-funs*
        (clicc-message "The unused local functions are:~%~S"
                       *unused-local-funs*)))

    (when (> *delete-verbosity* 0)
      (clicc-message "~D unused symbol~:p found"
                     (length unused-syms))
      (clicc-message "~D unused named constant~:p found"
                     (length unused-named-consts))
      (clicc-message "~D unused function~:p found"
                     (length unused-funs))
      (clicc-message "~D unused local function~:p found"
                     (length *unused-local-funs*))))

  (clicc-message "----------------------------------------------------------~
                  -----------------"))

;;------------------------------------------------------------------------------
(provide "delete")
