;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;--------------------------------------------------------------------------
;;; Project    :CLICC ein Commmon LIsp to C Compiler
;;;             ------------------------------------
;;; Dateiname  : CLOSURE-ANALYSIS.LISP
;;; Funktion   : Die vorbereitung fuer die Seiteneffektanalyse , sowie fuer 
;;;              Die verschiedenen Optimierungsverfahren.
;;;              1) Es wird eine Liste der im Module enstandenen Closures 
;;;                 erstellt.
;;;              2) Die  Closures werden analysiert, und attributiert, mit 
;;;                 Informationen die zur Codegenerierung sowie fuer die 
;;;                 Optimierung der Tail-rekursion nuetzlich sind.
;;; Autor      : Anouar Trigui
;;; $Revision: 1.15 $
;;; $Log: closure-analysis.lisp,v $
;;; Revision 1.15  1993/07/21  12:27:37  atr
;;; Funktionen besser kommentiert.
;;;
;;; Revision 1.14  1993/07/13  16:22:06  atr
;;; Level bei dynamischen Variablen wird nicht gesetzt.
;;;
;;; Revision 1.13  1993/07/13  14:40:06  hk
;;; Lokale Aenderung von Annouar, die auch den Slot level der Variablen
;;; eines mv-lambda setzt.
;;;
;;; Revision 1.12  1993/07/13  11:08:48  atr
;;; Nun werden die Slots LEVEL bei lokalen Funktionen
;;; und bei Variablen hier gesetzt.
;;;
;;; Revision 1.11  1993/07/06  13:17:44  atr
;;; Unnoetige Ausgabe entfernt.
;;;
;;; Revision 1.10  1993/06/30  16:49:57  atr
;;; Ausgabe -preparing side effect analysis- entfernt.
;;;
;;; Revision 1.9  1993/06/26  16:01:30  atr
;;; (require static-effect) entfernt .
;;;
;;; Revision 1.8  1993/06/26  13:33:33  atr
;;; Neue Methoden geschrieben zur Attributierung der SPEC-DEFINED-FUNS
;;; als Vorbereitung zur Seiteneffektanalyse.
;;;
;;; Revision 1.7  1993/06/17  08:00:09  hk
;;; Copright Notiz eingefuegt
;;;
;;; Revision 1.6  1993/06/14  10:54:16  atr
;;; Unnoetige Ausgabe entfernt.
;;;
;;; Revision 1.5  1993/06/14  10:47:15  atr
;;; Funktionsdefinitionen verbessert.
;;;
;;; Revision 1.4  1993/06/09  12:21:30  atr
;;; delete durch remove ersetzt, denn delete ist destruktiv.
;;;
;;; Revision 1.3  1993/05/30  14:04:24  atr
;;; Analyse fuer special-defined-functions (also Funktionen
;;; die immer ein oder mehrere Parameter haben , die nur auf Funktions-
;;; position stehen) Diese Funktionen werden anderes analysiert als normale
;;; Funktionen.
;;;
;;; Revision 1.2  1993/04/30  09:37:12  hk
;;; Revision-Keyword eingetragen.
;;;
;;; Revision 1.1  1993/04/27  10:53:56  atr
;;; Initial revision
;;;
;;;--------------------------------------------------------------------------

(in-package "CLICC")

;;;--------------------------------------------------------------------------
;;; Die Globale Variablen ...
;;;--------------------------------------------------------------------------
(defvar *upward-fun-args* nil )
(defvar *current-function* nil)
(defvar *down-fun-args* nil)

(defvar *se-var-env* nil)
(defvar *se-vars-to-funs* nil)
(defvar *se-all-funs* nil)
(defvar *passed-as-argument* nil)

;;;--------------------------------------------------------------------------
;;; GET-RESULT-POSITION-FORMS traversiert eine Form f1, und liefert
;;; alle Formen, die auf Ergebnisposition stehen.
;;;--------------------------------------------------------------------------

(defmethod get-result-position-forms ((form if-form))
  (append  (get-result-position-forms  (?then form))
           (get-result-position-forms  (?else form))))

(defmethod get-result-position-forms ((form progn-form))
  (get-result-position-forms (car (last (?form-list form)))))

;;;-------------------------------------------------------------------------
;;Der tagbody-Konstrukt liefert immer nil als Resultat.
;;--------------------------------------------------------------------------
(defmethod get-result-position-forms ((form tagbody-form)))

(defmethod get-result-position-forms ((form tagged-form)))


(defmethod get-result-position-forms ((form let*-form))
  (get-result-position-forms (?body form)))

;;;-------------------------------------------------------------------------
;;; Bei einer Switch-form werden alle Ergebnis-formen jeder labelled Form
;;; zusammen mit der Ergebnis-form der Otherwise-form zurueckgegeben.
;;;------------------------------------------------------------------------- 
(defmethod get-result-position-forms ((form switch-form))
  (let ((app-nodes (get-result-position-forms  (?otherwise form))))
    (dolist (one-case (?case-list form) app-nodes)
      (setq app-nodes (append app-nodes 
                              (get-result-position-forms one-case))))))

(defmethod get-result-position-forms ((form labeled-form))
  (get-result-position-forms (?form form)))

(defmethod get-result-position-forms ((form let/cc-form))
  (get-result-position-forms (?body form)))

;;;-------------------------------------------------------------------------
;;; Die application wird an dieser Stelle nicht weiter analysiert.
;;;-------------------------------------------------------------------------
(defmethod get-result-position-forms ((form app))
  (if (or (and (equal (?form form) (get-global-fun 'return))
               (?arg-list form))
          (and (cont-p (?form form))
               (?arg-list form)))
      (?arg-list form)
      (list form)))

(defmethod get-result-position-forms ((form labels-form))
  (get-result-position-forms (?body form)))

(defmethod get-result-position-forms ((form fun))
  (list form))
  
(defmethod get-result-position-forms ((form mv-lambda))
  (get-result-position-forms (?body form)))

(defmethod get-result-position-forms ((form setq-form))
  (get-result-position-forms (?form form)))

(defmethod get-result-position-forms ((any-thing t))
 (unless (null any-thing )
   (list any-thing)))

;;;----------------------------------------------------------------------
;;; get-fun-on-result-position liefert die Liste der Closures die bei 
;;; der gegebenen Form auf Ergebnisposition stehen, wenn vorhanden, sonst
;;; nil.
;;;----------------------------------------------------------------------
(defun get-fun-on-result-position (form)
  (remove-if-not #'fun-p (get-result-position-forms form)))

;;;----------------------------------------------------------------------
;;; The-applied-function nimmt eine app und liefert die applizierte Form
;;; zurueck.
;;;----------------------------------------------------------------------
(defun the-applied-function (app)
  (let ((functional (?form app))
        (arg-list   (?arg-list app)))
   (case functional 
      (defined-fun-p functional)

      ;;; Bei einer Applikation durch FUNCALL oder APPLY oder bei einer 
      ;;; Iteration durch MAP* wird die applizierte Form zurueckgegeben
      ;;; und nicht die Funktion Funcall selbst.
      ;;; sonst wird die applizierte Funktion zurueckgegeben.
      ;;;---------------------------------------------------------------
      (imported-fun-p functional)
      (special-sys-fun-p 
       (if  (?special-caller functional )
            (the-applied-function (make-instance 'app
                                                 :form (first arg-list)
                                                 :arg-list 
                                                 (cdr (?arg-list app))))
            functional))
      
      ;;; Bei einer cont-app wird nil zurueckgegeben.
      ;;;--------------------------------------------
      (cont-p nil)
      (app-p  (the-applied-function (?form functional)))
      
      ;;; anderenfalls werden alle Formen gesammelt, die auf 
      ;;; Resultatsposition stehen .
      (t (get-result-position-forms functional)))))

;;;-----------------------------------------------------------------------
;;; Get-applied-functions traversiert eine Form, und liefert alle 
;;; applizierten Formen bei Applikationen, die auf Ergebnisposition stehen.
;;;-----------------------------------------------------------------------
(defun get-applied-functions (form)
  (let ((applied-functions nil))
    (dolist (one-res-form (get-result-position-forms form))
      (when (app-p one-res-form)
        (setq applied-functions (append applied-functions
                                        (the-applied-function one-res-form)))))
    applied-functions))
 
;;;------------------------------------------------------------------------
;;; Analyse der Closures in einem Module.
;;;------------------------------------------------------------------------
(defun get-closures-of-module ()
  (let ((*se-var-env*        nil)
        (*se-vars-to-funs*   nil)
        (*let-without-vars*  0)
        (*se-all-funs*       (?fun-list *module*)))
    (declare (special *let-without-vars*))
    (mapcar #'get-closures-of-function *se-all-funs*)
    (clicc-message "lets without vars : ~s" *let-without-vars*)))

(defun list-the-spec-defined-funs ()
  (if (null *special-defined-funs*)
      (clicc-message "there is no special defined functions")
      (dolist (fun *special-defined-funs*)
        (clicc-message "the function : ~s"  (car fun)))))
        

(defun get-closures-of-function (function)
  (find-closures (?body function)))

(defmethod find-closures ((ref var-ref)))
  
(defmethod find-closures ((form if-form))
  (find-closures (?pred form))
  (find-closures (?then form))
  (find-closures (?else form)))

(defmethod find-closures ((form progn-form))
  (dolist (one-form (?form-list form))
    (find-closures one-form)))
                
(defmethod find-closures ((form tagbody-form))
  (let ((tagged-form-list (?tagged-form-list form)))
    (dolist (one-tagged-form tagged-form-list)
      (find-closures (?form one-tagged-form)))
    (find-closures (?first-form form))))


(defmethod find-closures ((form tagged-form)))


(defmethod find-closures ((let*form let*-form))
  (find-closures (?body let*form))
  (dolist (one-init-form (?init-list let*form))
    (find-closures one-init-form)))

(defmethod find-closures ((one-class-def class-def) )
  (dolist (one-slot-descr (?slot-descr-list one-class-def))
    (find-closures (?initform one-slot-descr ))))

(defmethod find-closures ((switch switch-form))
  (find-closures (?otherwise switch))
  (let ( (case-list    (?case-list switch)))
    (dolist (one-case case-list)
      (find-closures (?form one-case)))))

(defmethod find-closures ((let/cc let/cc-form))
  (find-closures (?body let/cc)))

(defmethod find-closures ((labels labels-form))
  (dolist (one-local-fun (?fun-list labels))
    (pushnew one-local-fun *se-all-funs*)
    (find-closures (?body one-local-fun)))
  (find-closures (?body labels)))


(defmethod find-closures ((an-app app))
  (let* ((functional      (?form     an-app))
         (arg-list        (?arg-list an-app)))
    
    (when (or (member functional *map-functions*)
              (and (special-sys-fun-p  functional)
                   (?special-caller functional)))
      (setq functional (first arg-list))
      (setq arg-list (cdr arg-list)))
    (unless  (or (var-ref-p functional)
                 (fun-p functional))
      (let ((down-fun-list (remove-if-not 
                            #'defined-fun-p
                            (get-result-position-forms functional))))
        (setq *down-fun-args* (remove-duplicates 
                               (append 
                                *down-fun-args* 
                                down-fun-list)))))
    (dolist (one-arg arg-list)
      (find-closures one-arg))))

(defmethod find-closures ((function defined-fun))
  (pushnew function *upward-fun-args*))
  
(defmethod find-closures ((function local-fun))
  (pushnew  function *upward-fun-args*)) 

(defmethod find-closures ((a-mv-lambda mv-lambda))
  (find-closures (?body a-mv-lambda))
  (find-closures (?arg  a-mv-lambda)))

(defmethod find-closures ((a-setq setq-form))
  (find-closures (?form a-setq)))

(defmethod find-closures ((any-thing-else t)))

;;;-------------------------------------------------------------------------
;;; DIE VORBEREITUNG ZUR SEITENEFFEKTANALYSE.
;;; ES WIRD UNTERSUCHT OB EINER DER PARAMETER EINER FUNKTION  NUR AN 
;;; FUNKTIONSPOSITION STEHT .APPLIKATIONEN SAEMTLICHER VARIABLEN 
;;; SIND NICHT WIE APPLIKATIONEN VON GLOBALEN VARIABLEN ODER VON 
;;; LOKALEN VARIABLEN, DIE ABER  AN VERSCHIEDENEN STELLEN STEHEN, 
;;; DENN FUER DIE ERSTEN VARIABELN WIRD AN DER AUFRUFSTELLE DER 
;;; FUNKTION DAS FUNKTIONALE OBJEKT UNTERSUCHT, WAEHREND IM ZWEITEN 
;;; FALL MAN NICHT BESTIMMEN KANN WELCHES FUNKTIONALE OBJEKT 
;;; APPLIZIERT WIRD .(DIES KANN NUR MIT HILFE EINER DATENFLUSS-
;;; ANALYSE BESTIMMT WERDEN).
;;;------------------------------------------------------------------------

;;;------------------------------------------------------------------------
;;; Diese methode traversiert jede Funktion, und sammelt alle Variablen
;;; die nur auf Funktionsposition stehen.
;;; Diese Variablen werden zur Zeit in dem Slot HAS-FUNS-AS-ARGS
;;; abgespeichert.
;;;------------------------------------------------------------------------

(defun pre-analyse-module ()
  (let ((all-functions (cons
                        (?toplevel-forms *module*)
                        (?fun-list *module*)))
        (*counter*      0))
    (declare (special *counter*))
    (mapcar #'pre-analyse-function all-functions)
    (clicc-message "~s special functions found " *counter*)))

(defun pre-analyse-function  (function)
  (let* ((*current-function*  function)
         (*static-level*      0)
         (*se-var-env*          (?all-vars (?params function)))
         (*se-vars-to-funs*     nil))
    (get-fun-vars-of-function  function)))


(defun get-fun-vars-of-function (function)
  
  ;; Attributierung der Slots LEVEL bei Funktionen und bei den Variablen.
  ;;---------------------------------------------------------------------
  (let (( *static-level*            (if (local-fun-p function)
                                        (?level function)
                                        0)))
    (dolist (one-var (?all-vars (?params function)))
      (unless (dynamic-p one-var)
        (setf (?level one-var)  *static-level*)))
    
    (get-fun-vars (?body function))
    (when  *se-vars-to-funs*

      ;; Die Variablen werden in dem Slot HAS-FUNS-AS-ARGS abgespeichert.
      ;;-----------------------------------------------------------------
      (setf (?has-funs-as-args function) 
            (remove-if #'(lambda (var)
                           (< *static-level* (?level var)))
                       *se-vars-to-funs*))
      (when (?has-funs-as-args function) 
        (incf *counter*)))))

(defmethod get-fun-vars  ((ref var-ref)))


(defmethod get-fun-vars ((form if-form))
  (get-fun-vars (?pred form))
  (get-fun-vars (?then form))
  (get-fun-vars (?else form)))

(defmethod get-fun-vars ((form progn-form))
  (dolist (one-form (?form-list form))
    (get-fun-vars one-form)))
                
(defmethod get-fun-vars ((form tagbody-form))
  (let ((tagged-form-list (?tagged-form-list form)))
    (dolist (one-tagged-form tagged-form-list)
      (get-fun-vars (?form one-tagged-form)))
    (get-fun-vars (?first-form form))))

(defmethod get-fun-vars ((form tagged-form)))


(defmethod get-fun-vars ((let*form let*-form))

  ;; Setzen der Slots LEVEL bei den Variablen
  ;;-----------------------------------------
  (dolist (one-var (?var-list let*form))
    (unless (dynamic-p one-var)
      (setf (?level one-var) *static-level*)))
 
  ;; Analyse der Init-formen...
  ;;---------------------------
  (dolist (one-init-form (?init-list let*form))
    (get-fun-vars one-init-form))
  
  ;; Analyse des Rumpfes
  ;;--------------------
  (get-fun-vars  (?body let*form)))

(defmethod get-fun-vars ((one-class-def class-def) )
  (dolist (one-slot-descr (?slot-descr-list one-class-def))
    (get-fun-vars (?initform one-slot-descr ))))

(defmethod get-fun-vars ((switch switch-form))
  (get-fun-vars (?form switch))
  (let ( (case-list    (?case-list switch)))
    (dolist (one-case case-list)
      (get-fun-vars (?form one-case))))
  (get-fun-vars (?otherwise switch)))

(defmethod get-fun-vars ((let/cc let/cc-form))
  (get-fun-vars (?body let/cc)))

;;;---------------------------------------------------------------------------
;;; Bei einem Labels-Ausdruck werden zunaechst die lokale Funktionen
;;; analysiert und attributiert before die gerade analysierte 
;;; Funktion weiter analysiert wird.
;;;--------------------------------------------------------------------------
(defmethod get-fun-vars ((labels labels-form))
  (let ((*static-level* (1+ *static-level*)))
    (dolist (one-local-fun (?fun-list labels))
      (setf (?level one-local-fun) *static-level*)
      (let*  ((*se-var-env*       (append *se-var-env* 
                                          (?all-vars (?params one-local-fun))))
              (*current-function* one-local-fun))
        (get-fun-vars-of-function  *current-function*)))
    (get-fun-vars (?body labels))))


(defmethod get-fun-vars ((an-app app))
  (let* ((functional      (?form     an-app))
         (arg-list        (?arg-list an-app)))
    (block BEGIN
      (when (or (member functional *map-functions*)
                (and (special-sys-fun-p  functional)
                     (?special-caller functional)))
        (let ((new-app (make-instance 'app 
                                      :form (car (?arg-list an-app))
                                      :arg-list (cdr (?arg-list an-app)))))
          (get-fun-vars new-app)
          (return-from BEGIN)))
      
      (when  (and (var-ref-p functional)
                  (member (?var functional) *se-var-env*)
                  (static-p (?var functional)))
        (pushnew (?var functional) *se-vars-to-funs*))
      (when (and (not (var-ref-p functional))
                 (not (fun-p functional)))
        (get-fun-vars functional))
      (dolist (one-arg arg-list)
        (get-fun-vars one-arg)))))

(defmethod get-fun-vars ((function defined-fun)))

(defmethod get-fun-vars ((function local-fun)))



(defmethod get-fun-vars ((a-mv-lambda mv-lambda))
  (dolist (one-var (?all-vars (?params a-mv-lambda)))
    (unless (dynamic-p one-var)
      (setf (?level one-var) *static-level*)))
  (get-fun-vars (?body a-mv-lambda))
  (get-fun-vars (?arg  a-mv-lambda)))

;;;------------------------------------------------------------------------
;;; Wenn eine Variable neu definiert wird, wird sie nicht in dem Slot
;;; has-funs-as-args.
;;;------------------------------------------------------------------------
(defmethod get-fun-vars ((a-setq setq-form))
  (setq *se-var-env* (remove (?var (?location a-setq)) *se-var-env*))
  (setq *se-vars-to-funs* (remove (?var (?location a-setq))
                                  *se-vars-to-funs*))
  (get-fun-vars (?form a-setq)))


(defmethod get-fun-vars ((any-thing-else t)))

;;;-------------------------------------------------------------------------
(provide "closure-analysis")    
        








