;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: INTERACTORS; Base: 10 -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;         The Garnet User Interface Development Environment.      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This code was written as part of the Garnet project at          ;;;
;;; Carnegie Mellon University, and has been placed in the public   ;;;
;;; domain.  If you are using this code or any part of Garnet,      ;;;
;;; please contact garnet@cs.cmu.edu to be put on the mailing list. ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
#|
============================================================
Change log:
    5/22/92 Brad Myers - started, based on code from R J Williams
                          <rodw@cbl.leeds.ac.uk> and code from
                          opal:process.lisp
============================================================
|#

(in-package "INTERACTORS" :nicknames '("INTER") :use '("LISP" "KR"))

;;; A window waiting for a timer event

;;; A array of all interactors that have timers.  None are ever removed
;;; from this array.  An index into this array is sent with the timer
;;; event so that we will know which interactor to wake up.
(defparameter *Inters-With-Timers*
  (make-array 5 :fill-pointer 0 :adjustable T))

(defparameter *All-Timer-Processes* NIL)

(defun Reset-All-Timer-Processes ()
  (dolist (p (copy-list *All-Timer-Processes*))
    (internal-kill-timer-process p)))

(defun send-timer-event (inter)
  (let* ((wins (Get-Interactor-Windows inter))
	 (win (if (listp wins) (car wins) wins)))
    (if-debug inter (Format T "Posting Timer event for ~s~%" inter))
    (when win
      (let ((drawable (g-value win :drawable))
	    (indx (g-value inter :timer-array-index)))
	(unless indx
	  (setq indx (vector-push-extend inter *Inters-With-Timers* 10))
	  (s-value inter :timer-array-index indx))
	#+garnet-debug ;; only test when debugging
	(unless (eq (g-value inter :timer-array-index)
		    (position inter *inters-with-timers*))
	  (error "Interactor timer index not eq to position for ~s" inter))
	(xlib:send-event drawable
			 :client-message
			 nil
			 :event-window drawable
			 :type :TIMER_EVENT
			 :format 32 :data (list indx))
	(xlib:display-force-output opal::*default-x-display*)))))

;;;Sleep for appropriate time (in seconds), and then wake up and send event
(defun Timer-Process-Main-Loop (inter time once)
  (loop
   (sleep time)
   (unless (schema-p inter)  ;; if inter destroyed somehow
     (return))
   (send-timer-event inter)
   (when once (return))
   ;; now, make sure other processes run
   #+allegro (mp:process-allow-schedule)  
   #+lucid   (lcl:process-allow-schedule)
   ))

;;; Kills the timer process for the interactor, if any
(defun kill-timer-process (inter)
  (let ((timer-process (g-value inter :timer-event-process)))
    (when timer-process (internal-kill-timer-process timer-process))
    (s-value inter :timer-event-process NIL)))

(defun launch-timer-process (inter time once)
  "Spawn a process which is waiting for timer events"

  #-(or allegro lucid) ;; not possible in other lisps
  (return-from launch-timer-process)

  (let ((timer-process (g-value inter :timer-event-process)))
    (when timer-process (internal-kill-timer-process timer-process))
    (xlib:intern-atom opal::*default-x-display* ':TIMER_EVENT)
    (setf timer-process
	  #+allegro
	  (mp:process-run-function "Garnet Timer"
			   ;; Use a lambda to pass the parameters.
			   ;; This runs at priority 0, so it will be
			   ;; lower than main-event-loop-process.
			   #'(lambda ()
			       (Timer-Process-Main-Loop inter time once))
			   )
	  #+lucid 
	  (lcl:make-process :name "Garnet Timer"
			    :priority 200
			    :function
			    ;; Use a lambda to pass the parameters.
			    ;; This runs at priority 200, so it will be
			    ;; lower than main-event-loop-process.
			    #'(lambda ()
				(Timer-Process-Main-Loop inter time once))
			    )
	  #-(or allegro lucid)
	  NIL
	  )
    (if-debug inter (format T "Launching process ~s for ~s~%" timer-process
			    inter))
    (s-value inter :timer-event-process timer-process)
    (push timer-process *All-Timer-Processes*)
    timer-process))

;; This is called when a timer event occurs for the interactor
(defun Handle-Timer-Event (inter-index)
  (let ((inter (aref *Inters-With-Timers* inter-index)))
    (when (and inter (schema-p inter))
      (if-debug inter (Format T "Timer event for ~s~%" inter))
      (if (eq (g-value inter :current-state) :start)
	  (kill-timer-process inter) ; whoops, process shouldn't be running
	  (progn
	    (kr-send inter :timer-handler inter)
	    (opal:update-all))))))

;;;-------------------------------------------------------------------------

#+allegro
(defun internal-kill-timer-process (timer-process)
  (when (eq (type-of timer-process) 'mp:process)
    (mp:process-kill timer-process)
    (deleteplace timer-process *All-Timer-Processes*)
    ))

#+lucid
(defun internal-kill-timer-process (timer-process)
  (when (and timer-process (lcl:processp timer-process))
    (lcl:kill-process timer-process)
    (deleteplace timer-process *All-Timer-Processes*)
    ))


#-(or allegro lucid)
(defun internal-kill-timer-process (timer-process)
  )

