;;;-*-Mode:LISP; Package: (ITERATE (WALKER LISP)); Base:10; Syntax:Common-lisp -*-
;;;
;;; *************************************************************************
;;; Copyright (c) 1985, 1986, 1987, 1988 Xerox Corporation.
;;; All rights reserved.
;;;
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted.  Any distribution of this
;;; software or derivative works must comply with all applicable United
;;; States export control laws.
;;; 
;;; This software is made available AS IS, and Xerox Corporation makes no
;;; warranty about the software, its performance or its conformity to any
;;; specification.
;;; 
;;; Any person obtaining a copy of this software is requested to send their
;;; name and post office or electronic mail address to:
;;;   CommonLoops Coordinator
;;;   Xerox PARC
;;;   3333 Coyote Hill Rd.
;;;   Palo Alto, CA 94304
;;; (or send Arpanet mail to CommonLoops-Coordinator.pa@Xerox.arpa)
;;;
;;; Suggestions, comments and requests for improvements are also welcome.
;;; *************************************************************************
;;;
;;; This code was written by Bill vanMelle.
;;; 

(in-package "ITERATE" :use '("LISP" "WALKER"))

(export '(iterate iterate* gathering gather
	  interval list-elements list-tails
          vector-elements sequence-elements string-characters eachtime 
          while until collecting joining maximizing minimizing summing 
          *iterate-warnings*))


(defvar *iterate-warnings* :any "controls whether warnings are issued for iterate/gather forms that aren't optimized.
nil => never; :user => those resulting from user code; t => always, even if it's the iteration macro that's suboptimal."
)


;; iterate macro


(defmacro iterate (clauses &body body &environment env)
   (optimize-iterate-form 'iterate clauses body env))


(defmacro iterate* (clauses &body body &environment env)
   (optimize-iterate-form 'iterate* clauses body env))


(defun simple-expand-iterate-form (type clauses body)

   ;; expand iterate or iterate* (per type).  this is the "formal semantics"
   ;; expansion, which we never use.
   (let*
    ((block-name (gensym))
     (bound-var-lists (mapcar #'(lambda (clause)
                                       (let ((names (first clause)))
                                            (if (listp names)
                                                names
                                                (list names))))
                             clauses))
     (generator-exprs (mapcar #'second clauses))
     (generator-vars (mapcar #'(lambda (clause)
                                      (declare (ignore clause))
                                      (gensym))
                            clauses))
     (generator-bindings (mapcar #'list generator-vars generator-exprs)))
    `(block ,block-name
         (let*
          ,(ecase type
               (iterate                        ; do all the generators before
                                               ; binding anything
                  (append generator-bindings (apply #'append bound-var-lists)))
               (iterate*                       ; interleave generators and
                                               ; bindings
                  (mapcan #'(lambda (gen-binding var-list)
                                   (cons gen-binding (copy-list var-list)))
                         generator-bindings bound-var-lists)))

          ;; note bug in formal semantics: there can be declarations in front
          ;; of body; they go here
          (loop
           ,@(mapcar
              #'(lambda (var-list gen-var)     ; set each bound variable (or
                                               ; set of vars) to the result of
                                               ; calling the corresponding
                                               ; generator
                       `(multiple-value-setq
                         ,var-list
                         (funcall ,gen-var #'(lambda nil (return-from
                                                          ,block-name)))))
              bound-var-lists generator-vars)
           ,@body)))))


(defparameter *iterate-temp-vars-list*
   '(iterate-temp-1 iterate-temp-2 iterate-temp-3 iterate-temp-4 iterate-temp-5
           iterate-temp-6 iterate-temp-7 iterate-temp-8)
   "temp var names used by iterate expansions.")


(defun optimize-iterate-form (type clauses body iterate-env)
   (let*
    ((temp-vars *iterate-temp-vars-list*)
     (block-name (gensym))
     (finish-form `(return-from ,block-name))
     (bound-vars (mapcan #'(lambda (clause)
                                  (let ((names (first clause)))
                                       (if (listp names)
                                           (copy-list names)
                                           (list names))))
                        clauses))
     iterate-decls generator-decls update-forms bindings final-bindings 
     leftover-body)
    (do ((tail bound-vars (cdr tail)))
        ((null tail))                          ; check for duplicates
      (when (member (car tail)
                   (cdr tail))
          (warn "variable appears more than once in iterate: ~s" (car tail))))
    (flet
     ((get-iterate-temp nil                    ; make temporary var.  note
                                               ; that it is ok to re-use these
                                               ; symbols in each iterate,
                                               ; because they are not used
                                               ; within body.
             (or (pop temp-vars)
                 (gensym))))
     (dolist (clause clauses)
         (cond
            ((or (not (consp clause))
                 (not (consp (cdr clause))))
             (warn 
                 "bad syntax in iterate: clause not of form (var iterator): ~s"
                   clause))
            (t
             (unless (null (cddr clause))
                    (warn 
       "probable parenthesis error in iterate clause--more than 2 elements: ~s"
                          clause))
             (multiple-value-bind
              (let-body binding-type let-bindings localdecls otherdecls 
                     extra-body)
              (expand-into-let (second clause)
                     type iterate-env)

              ;; we have expanded the generator clause and parsed it into its
              ;; let pieces.
              (prog* ((vars (first clause))
                      gen-args renamed-vars)
		 (#+Genera-Release-7-1 block #+Genera-Release-7-1 punt1
		  #-Genera-Release-7-1 progn
                     (setq vars (if (listp vars)
                                    (copy-list vars)
                                    (list vars)))
                                               ; vars is now a (fresh) list of
                                               ; all iteration vars bound in
                                               ; this clause
                     (when (setq let-body (function-lambda-p let-body 1))

                         ;; we have something of the form #'(lambda
                         ;; (finisharg) ...), possibly with some let bindings
                         ;; around it.  
                         (setq let-body (cdr let-body))
                         (setq gen-args (pop let-body))
                         (when let-bindings

                          ;; the first transformation we want to perform is
                          ;; "let-eversion": turn (let* ((generator (let
                          ;; (..bindings..) #'(lambda ...)))) ..body..) into
                          ;; (let* (..bindings.. (generator #'(lambda ...)))
                          ;; ..body..).  this transformation is valid if
                          ;; nothing in body refers to any of the bindings,
                          ;; something we can assure by alpha-converting the
                          ;; inner let (substituting new names for each var). 
                          ;; of course, none of those vars can be special, but
                          ;; we already checked for that above.
                             (multiple-value-setq (let-bindings renamed-vars)
                                    (rename-let-bindings let-bindings 
                                           binding-type iterate-env 
                                           leftover-body #'get-iterate-temp))
                             (setq leftover-body nil)
                                               ; if there was any leftover
                                               ; from previous, it is now
                                               ; consumed
                             
                             )
                         (let
                          ((finish-arg (first gen-args)))

                          ;; the second transformation is substituting the
                          ;; body of the generator (lambda (finish-arg) .
                          ;; gen-body) for its appearance in the update form
                          ;; (funcall generator #'(lambda () finish-form)),
                          ;; then simplifying that form.  the requirement for
                          ;; this part is that the generator body not refer to
                          ;; any variables that are bound between the
                          ;; generator binding and the appearance in the loop
                          ;; body.  the only variables bound in that interval
                          ;; are generator temporaries, which have unique
                          ;; names so are no problem, and the iteration
                          ;; variables themselves: all of them in the case of
                          ;; iterate, only the remaining ones in the case of
                          ;; iterate*.  we'll discover the story as we walk
                          ;; the body.
                          (multiple-value-bind
                           (finishdecl other rest)
                           (parse-declarations let-body gen-args)
                           (declare (ignore finishdecl))
                                               ; pull out declares, if any,
                                               ; separating out the one(s)
                                               ; referring to the finish arg,
                                               ; which we will throw away
                           (when other         ; combine remaining decls with
                                               ; decls extracted from the let,
                                               ; if any
                               (setq otherdecls (nconc otherdecls other)))
                           (setq
                            let-body
                            (cond
                               (otherdecls     ; there are interesting
                                               ; declarations, so have to keep
                                               ; it wrapped.
                                `(let nil (declare ,@otherdecls)
                                      ,@rest))
                               ((null (cdr rest))
                                               ; only one form left
                                (first rest)))))
                          (setq
                           let-body
                           (walk-form
                            let-body iterate-env
                            #'(lambda (form context env)
                                     (declare (ignore context))

                                ;; need to substitute renamed-vars, as well as
                                ;; turn (funcall finish-arg) into the finish
                                ;; form
                                     (cond
                                        ((symbolp form)
                                         (let (renaming)
                                              (cond
                                                 ((and (eq form finish-arg)
                                                       (variable-same-p form 
                                                              env iterate-env))
                                               ; an occurrence of the finish
                                               ; arg outside of funcall
                                               ; context--i can't handle this
                                                  (maybe-warn :definition "couldn't optimize ~s because generator ~s does something with its finish arg besides funcall it."
                                                         type (second clause))
						  #-Genera-Release-7-1
						  (go punt)
						  #+Genera-Release-7-1
						  (return-from punt1 nil)
						  )
                                                 ((and (setq renaming
                                                             (assoc form 
                                                                   renamed-vars
                                                                    ))
                                                       (variable-same-p form 
                                                              env iterate-env))
                                               ; reference to one of the vars
                                               ; we're renaming
                                                  (cdr renaming))
                                                 ((and (member form bound-vars)
                                                       (variable-same-p form 
                                                              env iterate-env))
                                               ; form is a var that is bound
                                               ; in this same iterate, or
                                               ; bound later in this iterate*.
                                               ; this is a conflict.
                                                  (maybe-warn
                                                   :user "couldn't optimize ~s because generator ~s is closed over ~s, in conflict with another iteration variable.~@[  you may have meant to use iterate*~]"
                                                   type (second clause)
                                                   form
                                                   (eq type 'iterate))
						  #-Genera-Release-7-1
						  (go punt)
						  #+Genera-Release-7-1
						  (return-from punt1 nil)
						  )
                                                 (t form))))
                                        ((and (consp form)
                                              (eq (first form)
                                                  'funcall)
                                              (eq (second form)
                                                  finish-arg)
                                              (variable-same-p (second form)
                                                     env iterate-env))
                                               ; (funcall finish-arg) =>
                                               ; finish-form
                                         (unless (null (cddr form))
                                             (maybe-warn :definition 
        "generator for ~s applied its finish arg to > 0 arguments ~s--ignored."
                                                    (second clause)
                                                    (cddr form)))
                                         finish-form)
                                        (t form)))))

                          ;; gen-body is now a form which, when evaluated,
                          ;; returns updated values for the iteration
                          ;; variable(s)
                          (push (mv-setq (copy-list vars)
                                       let-body)
                                update-forms)

                          ;; note possible further optimization: if the above
                          ;; expanded into (setq var (prog1 oldvalue
                          ;; prepare-for-next-iteration)), as so many do, then
                          ;; we could in most cases split the prog1 into two
                          ;; pieces: do the (setq var oldvalue) here, and do
                          ;; the prepare-for-next-iteration at the bottom of
                          ;; the loop.  this does a slight optimization of the
                          ;; prog1 and also rearranges the code in a way that
                          ;; a reasonably clever compiler might detect how to
                          ;; get rid of redundant variables altogether (such
                          ;; as happens with interval and list-tails); that
                          ;; would make the whole thing closer to what you
                          ;; might have coded by hand.  however, to do this
                          ;; optimization, we need to assure that (a) the
                          ;; prepare-for-next-iteration refers freely to no
                          ;; vars other than the internal vars we have
                          ;; extracted from the let, and (b) that the code has
                          ;; no side effects.  these are both true for all the
                          ;; iterators defined by this module, but how shall
                          ;; we represent side-effect info and/or tap into the
                          ;; compiler's knowledge of same?
                          (when localdecls     ; there were declarations for
                                               ; the generator locals--have to
                                               ; keep them for later, and
                                               ; rename the vars mentioned
                              (setq
                               generator-decls
                               (nconc
                                generator-decls
                                (mapcar
                                 #'(lambda
                                    (decl)
                                    (let
                                     ((head (car decl)))
                                     (cons head
                                           (if (eq head 'type)
                                               (cons (second decl)
                                                     (sublis renamed-vars
                                                            (cddr decl)))
                                               (sublis renamed-vars
                                                      (cdr decl))))))
                                 localdecls))))
                          (go finish-clause)))
                     (maybe-warn :definition "could not optimize ~s clause ~s because generator not of form (let[*] ... (function (lambda (finish) ...)))"
                            type (second clause))
		     );end of block punt1 or progn
		 #-Genera-Release-7-1
	         punt
                     (let
                      ((gvar (get-iterate-temp))
                       (generator (second clause)))

                      ;; for some reason, we can't expand this guy, so go with
                      ;; the formal semantics: bind a var to the generator,
                      ;; then call it in the update section
                      (setq
                       let-bindings
                       (list
                        (list gvar
                              (cond
                                 (leftover-body; have to use this up
                                  `(progn ,@(prog1 leftover-body
                                                   (setq leftover-body nil))
                                          generator))
                                 (t generator)))))
                      (push (mv-setq (copy-list vars)
                                   `(funcall ,gvar #'(lambda nil ,finish-form))
                                   )
                            update-forms))
                 finish-clause
                     (case type
                         (iterate              ; for iterate, don't bind any
                                               ; iv's until all exprs are
                                               ; evaluated, so defer this
                            (setq final-bindings (nconc final-bindings vars))
                            (setq vars nil)
                            (when extra-body   ; have to save this for next
                                               ; clause, too
                                (setq leftover-body (nconc leftover-body 
                                                           extra-body))
                                (setq extra-body nil)))
                         (iterate*             ; pop off the vars we have now
                                               ; bound from the list of vars
                                               ; to watch out for--we'll bind
                                               ; them right now
                            (dolist (v vars)
			      ;#-Genera
			      ;(declare (ignore v))
			      (pop bound-vars))))
                     (setq
                      bindings
                      (nconc bindings let-bindings
                             (cond
                                (extra-body    ; there was some computation to
                                               ; do after the bindings--here's
                                               ; our chance
                                 (cons (list (first vars)
                                             `(progn ,@extra-body nil))
                                       (rest vars)))
                                (t vars))))))))))
    (do ((tail body (cdr tail)))
        ((not (and (consp tail)
                   (consp (car tail))
                   (eq (caar tail)
                       'declare)))             ; tail now points at first
                                               ; non-declaration.  if there
                                               ; were declarations, pop them
                                               ; off so they appear in the
                                               ; right place
         (unless (eq tail body)
             (setq iterate-decls (ldiff body tail))
             (setq body tail))))
    `(block ,block-name
         (let* ,(append bindings final-bindings)
               ,@(and generator-decls `((declare ,@generator-decls)))
               ,@iterate-decls
               ,@leftover-body
               (loop ,@(nreverse update-forms)
                     ,@body)))))


(defun expand-into-let (clause parent-name env)

   ;; return values: body, let[*], bindings, localdecls, otherdecls, extra
   ;; body, where body is a single form.  if multiple forms in a let, the
   ;; preceding forms are returned as extra body.  returns :abort if it issued
   ;; a punt warning.
   (prog ((expansion clause)
          expandedp binding-type let-bindings let-body)
     expand
         (multiple-value-setq (expansion expandedp)
                (macroexpand-1 expansion env))
         (cond
            ((not (consp expansion))           ; shouldn't happen
             
             )
            ((symbolp (setq binding-type (first expansion)))
             (case binding-type
                 ((let let*) 
                    (setq let-bindings (second expansion))
                                               ; list of variable bindings
                    (setq let-body (cddr expansion))
                    (go handle-let))))
            ((and (consp binding-type)
                  (eq (car binding-type)
                      'lambda)
                  (not (find-if #'(lambda (x)
                                         (member x lambda-list-keywords))
                              (setq let-bindings (second binding-type))))
                  (eql (length (second expansion))
                       (length let-bindings))
                  (null (cddr expansion)))     ; a simple lambda form can be
                                               ; treated as let
             (setq let-body (cddr binding-type))
             (setq let-bindings (mapcar #'list let-bindings (second expansion))
                   )
             (setq binding-type 'let)
             (go handle-let)))

    ;; fall thru if not a let 
         (cond
            (expandedp                         ; try expanding again
                   (go expand))
            (t                                 ; boring--return form as the
                                               ; body
               (return expansion)))
     handle-let
         (return (let ((locals (variables-from-let let-bindings))
                       extra-body specials)
                      (multiple-value-bind
                       (localdecls otherdecls let-body)
                       (parse-declarations let-body locals)
                       (cond
                          ((setq specials (extract-special-bindings locals 
                                                 localdecls))
                           (maybe-warn (cond
                                          ((find-if 
                                                  #'
                                            walker::variable-globally-special-p
                                                  specials)
                                               ; this could be the fault of a
                                               ; user proclamation
                                           :user)
                                          (t :definition))
                                  
          "couldn't optimize ~s because expansion of ~s binds specials ~(~s ~)"
                                  parent-name clause specials)
                           :abort)
                          (t (values (cond
                                        ((not (consp let-body))
                                               ; null body of let?  unlikely,
                                               ; but someone else will likely
                                               ; complain
                                         nil)
                                        ((null (cdr let-body))
                                               ; a single expression, which we
                                               ; hope is (function
                                               ; (lambda...))
                                         (first let-body))
                                        (t 

                          ;; more than one expression.  these are forms to
                          ;; evaluate after the bindings but before the
                          ;; generator form is returned.  save them to
                          ;; evaluate in the next convenient place.  note that
                          ;; this is ok, as there is no construct that can
                          ;; cause a let to return prematurely (without
                          ;; returning also from some surrounding construct).
                                           (setq extra-body (butlast let-body))
                                           (car (last let-body))))
                                    binding-type let-bindings localdecls 
                                    otherdecls extra-body))))))))


(defun variables-from-let (bindings)

   ;; return a list of the variables bound in the first argument to let[*].
   (mapcar #'(lambda (binding)
                    (if (consp binding)
                        (first binding)
                        binding))
          bindings))


(defun parse-declarations (tail locals)

   ;; extract the declarations from the head of tail and divide them into 2
   ;; classes: declares about variables in the list locals, and all other
   ;; declarations.  returns 3 values: those 2 lists plus the remainder of
   ;; tail.
   (let
    (localdecls otherdecls form)
    (loop
     (unless (and tail (consp (setq form (car tail)))
                  (eq (car form)
                      'declare))
         (return (values localdecls otherdecls tail)))
     (mapc
      #'(lambda
         (decl)
         (case (first decl)
             ((inline notinline optimize)      ; these don't talk about vars
                (push decl otherdecls))
             (t                                ; assume all other kinds are
                                               ; for vars
                (let*
                 ((vars (if (eq (first decl)
                                'type)
                            (cddr decl)
                            (cdr decl)))
                  (l (intersection locals vars))
                  other)
                 (cond
                    ((null l)                  ; none talk about locals
                     (push decl otherdecls))
                    ((null (setq other (set-difference vars l)))
                                               ; all talk about locals
                     (push decl localdecls))
                    (t                         ; some of each
                       (let ((head (cons 'type (and (eq (first decl)
                                                        'type)
                                                    (list (second decl))))))
                            (push (append head other)
                                  otherdecls)
                            (push (append head l)
                                  localdecls))))))))
      (cdr form))
     (pop tail))))


(defun extract-special-bindings (vars decls)

   ;; return the subset of vars that are special, either globally or because
   ;; of a declaration in decls
   (let ((specials (remove-if-not #'walker::variable-globally-special-p vars)))
        (dolist (d decls)
            (when (eq (car d)
                      'special)
                (setq specials (union specials (intersection vars (cdr d))))))
        specials))


(defun function-lambda-p (form &optional nargs)

   ;; if form is #'(lambda bindings . body) and bindings is of length nargs,
   ;; return the lambda expression
   (let (args body)
        (and (consp form)
             (eq (car form)
                 'function)
             (consp (setq form (cdr form)))
             (null (cdr form))
             (consp (setq form (car form)))
             (eq (car form)
                 'lambda)
             (consp (setq body (cdr form)))
             (listp (setq args (car body)))
             (or (null nargs)
                 (eql (length args)
                      nargs))
             form)))


(defun rename-let-bindings (let-bindings binding-type env leftover-body 
                                  &optional tempvarfn)

   ;; perform the alpha conversion required for "let eversion" of (let[*]
   ;; let-bindings . body)--rename each of the variables to an internal name. 
   ;; returns 2 values: a new set of let bindings and the alist of old var
   ;; names to new (so caller can walk the body doing the rest of the
   ;; renaming).  binding-type is one of let or let*.  leftover-body is
   ;; optional list of forms that must be eval'ed before the first binding
   ;; happens.  env is the macro expansion environment, in case we have to
   ;; walk a let*.  tempvarfn is a function of no args to return a temporary
   ;; var; if omitted, we use gensym.
   (let
    (renamed-vars)
    (values
     (mapcar #'(lambda (binding)
                      (let ((valueform (cond
                                          ((not (consp binding))
                                               ; no initial value
                                           nil)
                                          ((or (eq binding-type 'let)
                                               (null renamed-vars))
                                               ; all bindings are in parallel,
                                               ; so none can refer to others
                                           (second binding))
                                          (t   ; in a let*, have to substitute
                                               ; vars in the 2nd and
                                               ; subsequent initialization
                                               ; forms
                                             (rename-variables (second binding)
                                                    renamed-vars env))))
                            (newvar (if tempvarfn
                                        (progn ;(format T "Tempvarfn ~S" tempvarfn)
					       (funcall tempvarfn))
                                        (gensym))))
                           (push (cons (if (consp binding)
                                           (first binding)
                                           binding)
                                       newvar)
                                 renamed-vars) ; add new variable to the list
                                               ; after we have walked the
                                               ; initial value form
                           (when leftover-body

                          ;; previous clause had some computation to do after
                          ;; its bindings.  here is the first opportunity to
                          ;; do it
                               (setq valueform `(progn ,@leftover-body
                                                       ,valueform))
                               (setq leftover-body nil))
                           (list newvar valueform)))
            let-bindings)
     renamed-vars)))


(defun rename-variables (form alist env)

   ;; walks form, renaming occurrences of the key variables in alist with
   ;; their corresponding values.  env is form's environment, so we can make
   ;; sure we are talking about the same variables.
   (walk-form form env #'(lambda (form context subenv)
                                (declare (ignore context))
                                (let (pair)
                                     (cond
                                        ((and (symbolp form)
                                              (setq pair (assoc form alist))
                                              (variable-same-p form subenv env)
                                              )
                                         (cdr pair))
                                        (t form))))))


(defun mv-setq (vars expr)

   ;; produces (multiple-value-setq vars expr), except that i'll optimize some
   ;; of the simple cases for benefit of compilers that don't, and i don't
   ;; care what the value is, and i know that the variables need not be set in
   ;; parallel, since they can't be used free in expr
   (cond
      ((null vars)                             ; expr is a side-effect
       expr)
      ((not (consp vars))                      ; this is an error, but i'll
                                               ; let multiple-value-setq
                                               ; report it
       `(multiple-value-setq ,vars ,expr))
      ((and (listp expr)
            (eq (car expr)
                'values))

       ;; (mv-setq (a b c) (values x y z)) can be reduced to a parallel setq
       ;; (psetq returns nil, but i don't care about returned value).  do this
       ;; even for the single variable case so that we catch (mv-setq (a)
       ;; (values x y))
       (pop expr)                              ; values
       `(setq ,@(mapcon #'(lambda (tail)
                                 (list (car tail)
                                       (cond
                                          ((or (cdr tail)
                                               (null (cdr expr)))
                                               ; one result expression for
                                               ; this var
                                           (pop expr))
                                          (t   ; more expressions than vars,
                                               ; so arrange to evaluate all
                                               ; the rest now.
                                             (cons 'prog1 expr)))))
                       vars)))
      ((null (cdr vars))                       ; simple one variable case
       `(setq ,(car vars)
              ,expr))
      (t                                       ; general case--i know nothing
         `(multiple-value-setq ,vars ,expr))))


(defun variable-same-p (var env1 env2)
   (eq (variable-lexical-p var env1)
       (variable-lexical-p var env2)))


(defun maybe-warn (type &rest warn-args)

   ;; issue a warning about not being able to optimize this thing.  type is
   ;; one of :definition, meaning the definition is at fault, and :user,
   ;; meaning the user's code is at fault.
   (when (case *iterate-warnings*
             ((nil) nil)
             ((:user) (eq type :user))
             (t t))
       (apply #'warn warn-args)))


;; sample iterators


(defmacro interval (&whole whole &key from downfrom to downto above below by 
                          type)
   (cond
      ((and from downfrom)
       (error "can't use both from and downfrom in ~s" whole))
      ((cdr (remove nil (list to downto above below)))
       (error "can't use more than one limit keyword in ~s" whole))
      (t
       (let*
        ((down (or downfrom downto above))
         (limit (or to downto above below))
         (inc (cond
                 ((null by)
                  1)
                 ((constantp by)               ; can inline this increment
                  by))))
        `(let
          ((from ,(or from downfrom 0))
           ,@(and limit `((to ,limit)))
           ,@(and (null inc)
                  `((by ,by))))
          ,@(and type `((declare (type ,type from ,@(and limit '(to))
                                       ,@(and (null inc)
                                              `(by))))))
          #'(lambda
             (finish)
             ,@(cond
                  ((null limit)                ; we won't use the finish arg
                   '((declare (ignore finish)))))
             (prog1 
                 ,(cond
                     (limit                    ; test the limit.  if ok,
                                               ; return current value and
                                               ; increment, else quit
                            `(if (,(cond
                                      (above '>)
                                      (below '<)
                                      (down '>=)
                                      (t '<=))
                                  from to)
                                 from
                                 (funcall finish)))
                     (t                        ; no test
                        'from))
                 (setq from (,(if down
                                  '-
                                  '+)
                             from
                             ,(or inc 'by))))))))))


(defmacro list-elements (list)
   `(let ((tail ,list))
         #'(lambda (finish)
                  (if (endp tail)
                      (funcall finish)
                      (pop tail)))))


(defmacro list-tails (list)
   `(let ((tail ,list))
         #'(lambda (finish)
                  (prog1 (if (endp tail)
                             (funcall finish)
                             tail)
                      (setq tail (cdr tail))))))


(defmacro vector-elements (vector)
   `(let* ((v ,vector)
           (size (length v))
           (index 0))
          #'(lambda (finish)
                   (values (cond
                              ((< index size)
                               (aref v index))
                              (t (funcall finish)))
                          (prog1 index
                              (setq index (1+ index)))))))


(defmacro sequence-elements (sequence)
   `(let* ((s ,sequence)
           (size (length s))
           (index 0))
          #'(lambda (finish)
                   (values (cond
                              ((< size index)
                               (elt s index))
                              (t (funcall finish)))
                          (prog1 index
                              (setq index (1+ index)))))))


(defmacro string-characters (string)
   `(let* ((s ,string)
           (size (length (the string s)))
           (index 0))
          #'(lambda (finish)
                   (values (cond
                              ((< index size)
                               (char s index))
                              (t (funcall finish)))
                          (prog1 index
                              (setq index (1+ index)))))))


(defmacro eachtime (expr)
   `#'(lambda (finish)
             (declare (ignore finish))
             ,expr))


(defmacro while (expr)
   `#'(lambda (finish)
             (unless ,expr (funcall finish))))


(defmacro until (expr)
   `#'(lambda (finish)
             (when ,expr (funcall finish))))


;; GATHERING macro


(defmacro gathering (clauses &body body &environment env)
   (or ;(optimize-gathering-form clauses body env)
       (simple-expand-gathering-form clauses body env)))


(defun simple-expand-gathering-form (clauses body env)
   (declare (ignore env))

   ;; The "formal semantics" of GATHERING.  We use this only in cases that
   ;; can't be optimized.
   (let*
    ((acc-names (mapcar #'first clauses))
     (realizer-names (mapcar #'(lambda (binding)
                                      (declare (ignore binding))
                                      (gensym))
                            clauses)))
    `(multiple-value-call
      #'(lambda
         ,(mapcan #'list acc-names realizer-names)
         (flet ((gather (value accumulator)
                       (funcall accumulator value)))
               ,@body
               (values ,@(mapcar #'(lambda (rname)
                                          `(funcall ,rname))
                                realizer-names))))
      ,@(mapcar #'second clauses))))


(defvar *active-gatherers* nil
   "List of GATHERING bindings currently active during macro expansion)")


(defun optimize-gathering-form (clauses body gathering-env)
   (let*
    (acc-info leftover-body top-bindings finish-forms top-decls)
    (dolist (clause clauses)
        (multiple-value-bind
         (let-body binding-type let-bindings localdecls otherdecls extra-body)
         (expand-into-let (second clause)
                'gathering gathering-env)
         (prog* ((acc-var (first clause))
                 renamed-vars accumulator realizer)
                (when (and (consp let-body)
                           (eq (car let-body)
                               'values)
                           (consp (setq let-body (cdr let-body)))
                           (setq accumulator (function-lambda-p (car let-body))
                                 )
                           (consp (setq let-body (cdr let-body)))
                           (setq realizer (function-lambda-p (car let-body)
                                                 0))
                           (null (cdr let-body)))

                    ;; Macro returned something of the form (VALUES #'(lambda
                    ;; (value) ...) #'(lambda () ...)), a function to
                    ;; accumulate values and a function to realize the result.
                    (when binding-type

                        ;; Gatherer expanded into a LET
                        (cond
                           (otherdecls (maybe-warn
                                        :definition "Couldn't optimize GATHERING clause ~S because its expansion carries declarations about more than the bound variables: ~S"
                                        (second clause)
                                        `(declare ,@otherdecls))
                                  (go punt)))
                        (when let-bindings

                          ;; The first transformation we want to perform is a
                          ;; variant of "LET-eversion": turn (mv-bind (acc
                          ;; real) (let (..bindings..) (values #'(lambda ...)
                          ;; #'(lambda ...))) ..body..) into (let*
                          ;; (..bindings.. (acc #'(lambda ...)) (real
                          ;; #'(lambda ...))) ..body..).  This transformation
                          ;; is valid if nothing in body refers to any of the
                          ;; bindings, something we can assure by
                          ;; alpha-converting the inner let (substituting new
                          ;; names for each var).  Of course, none of those
                          ;; vars can be special, but we already checked for
                          ;; that above.
                            (multiple-value-setq (let-bindings renamed-vars)
                                   (rename-let-bindings let-bindings 
                                          binding-type gathering-env 
                                          leftover-body))
                            (setq top-bindings (nconc top-bindings let-bindings
                                                      ))
                            (setq leftover-body nil)
                                               ; If there was any leftover
                                               ; from previous, it is now
                                               ; consumed
                            
                            ))
                    (setq leftover-body (nconc leftover-body extra-body))
                                               ; Computation to do after these
                                               ; bindings
                    (push (cons acc-var (rename-and-capture-variables 
                                               accumulator renamed-vars 
                                               gathering-env))
                          acc-info)
                    (setq realizer (rename-variables realizer renamed-vars 
                                          gathering-env))
                    (push (cond
                             ((null (cdddr realizer))
                                               ; Simple (LAMBDA () expr) =>
                                               ; expr
                              (third realizer))
                             (t                ; There could be declarations
                                               ; or something, so leave as a
                                               ; LET
                                (cons 'let (cdr realizer))))
                          finish-forms)
                    (unless (null localdecls)  ; Declarations about the LET
                                               ; variables also has to
                                               ; percolate up
                        (setq top-decls (nconc top-decls (sublis renamed-vars 
                                                                localdecls))))
                    (return))
                (maybe-warn :definition "Couldn't optimize GATHERING clause ~S because its expansion is not of the form (VALUES #'(LAMBDA ...) #'(LAMBDA () ...))"
                       (second clause))
            punt
                (let
                 ((gs (gensym))
                  (expansion `(multiple-value-list ,(second clause))))
                                               ; Slow way--bind gensym to the
                                               ; macro expansion, and we will
                                               ; funcall it in the body
                 (push (list acc-var gs)
                       acc-info)
                 (push `(funcall (cadr ,gs))
                       finish-forms)
                 (setq
                  top-bindings
                  (nconc
                   top-bindings
                   (list
                    (list gs
                          (cond
                             (leftover-body
                              `(progn ,@(prog1 leftover-body (setq 
                                                                  leftover-body
                                                                   nil))
                                      ,expansion))
                             (t expansion))))))))))
    (setq body (walk-gathering-body body gathering-env acc-info))
    (cond
       ((eq body :abort)                       ; Couldn't finish expansion
        nil)
       (t `(let* ,top-bindings ,@(and top-decls
                                      `((declare ,@top-decls)))
                 ,body
                 ,(cond
                     ((null (cdr finish-forms)); just a single value
                      (car finish-forms))
                     (t `(values ,@(reverse finish-forms)))))))))


(defun rename-and-capture-variables (form alist env)

   ;; Walks FORM, renaming occurrences of the key variables in ALIST with
   ;; their corresponding values, and capturing any other free variables. 
   ;; Returns a list of the new form and the list of other closed-over vars. 
   ;; ENV is FORM's environment, so we can make sure we are talking about the
   ;; same variables.
   (let (closed)
        (list (walk-form form env
                     #'(lambda (form context subenv)
                              (declare (ignore context))
                              (let (pair)
                                   (cond
                                      ((or (not (symbolp form))
                                           (not (variable-same-p form subenv 
                                                       env)))
                                               ; non-variable or one that has
                                               ; been rebound
                                       form)
                                      ((setq pair (assoc form alist))
                                               ; One to rename
                                       (cdr pair))
                                      (t       ; var is free
                                         (pushnew form closed)
                                         form)))))
              closed)))


(defun walk-gathering-body (body gathering-env acc-info)

   ;; Walk the body of (GATHERING (...) . BODY) in environment GATHERING-ENV. 
   ;; ACC-INFO is a list of information about each of the gathering
   ;; "bindings" in the form, in the form (var gatheringfn freevars env)
   (let
    ((*active-gatherers* (nconc (mapcar #'car acc-info)
                                *active-gatherers*)))

    ;; *ACTIVE-GATHERERS* tells us what vars are currently legal as GATHER
    ;; targets.  This is so that when we encounter a GATHER not belonging to
    ;; us we can know whether to warn about it.
    (walk-form
     (cons 'progn body)
     gathering-env
     #'(lambda
        (form context env)
        (declare (ignore context))
        (let (info)
             (cond
                ((consp form)
                 (cond
                    ((not (eq (car form)
                              'gather))        ; We only care about GATHER
                     (when (and (eq (car form)
                                    'function)
                                (eq (cadr form)
                                    'gather))  ; Passed as functional--can't
                                               ; macroexpand
                         (maybe-warn :user 
                     "Can't optimize GATHERING because #'GATHER is referenced."
                                )
                         (return-from walk-gathering-body :abort))
                     form)
                    ((setq info (assoc (third form)
                                       acc-info))
                                               ; One of ours--expand (GATHER
                                               ; value var).  INFO = (var
                                               ; gatheringfn freevars)
                     (unless (null (cdddr form))
                            (warn "Extra arguments (> 2) in ~S discarded." form
                                  ))
                     (let ((fn (second info)))
                          (cond
                             ((symbolp fn)     ; Unoptimized case--just call
                                               ; the gatherer.  FN is the
                                               ; gensym that we bound to the
                                               ; list of two values returned
                                               ; from the gatherer.
                              `(funcall (car ,fn)
                                      ,(second form)))
                             (t                ; FN = (lambda (value) ...)
                                (dolist (s (third info))
                                    (unless (or (variable-same-p s env 
                                                       gathering-env)
                                                (and (variable-special-p s env)
                                                     (variable-special-p s 
                                                            gathering-env)))

                          ;; Some var used free in the LAMBDA form has been
                          ;; rebound between here and the parent GATHERING
                          ;; form, so can't substitute the lambda.  Ok if it's
                          ;; a special reference both here and in the LAMBDA,
                          ;; because then it's not closed over.
                                        (maybe-warn :user "Can't optimize GATHERING because the expansion closes over a variable ~S that is rebound around a GATHER for it."
                                               form)
                                        (return-from walk-gathering-body :abort
                                               )))

                          ;; Return ((lambda (value) ...) actual-value).  In
                          ;; many cases we could simplify this further by
                          ;; substitution, but we'd have to be careful (for
                          ;; example, we would need to alpha-convert any LET
                          ;; we found inside).  Any decent compiler will do it
                          ;; for us.
                                (list fn (second form))))))
                    ((member (third form)
                            *active-gatherers*); Some other GATHERING will
                                               ; take care of this form, so
                                               ; pass it up for now
                     form)
                    (t                         ; Nobody's going to handle it
                       (warn "No GATHERING form for ~S" form)
                                               ; Turn it into something else
                                               ; so we don't warn twice in the
                                               ; nested case
                       `(%orphaned-gather ,@(cdr form)))))
                ((and (symbolp form)
                      (setq info (assoc form acc-info))
                      (variable-same-p form env gathering-env))
                                               ; A variable reference to a
                                               ; gather binding from
                                               ; environment TEM
                 (maybe-warn :user "Can't optimize GATHERING because target variable ~S is used outside of a GATHER form"
                        form)
                 (return-from walk-gathering-body :abort))
                (t form)))))))


;; Sample gatherers


(defmacro collecting (&key initial-value)
   `(let* ((head ,initial-value)
           (tail ,(and initial-value `(last head))))
          (values #'(lambda (value)
                           (if (null head)
                               (setq head (setq tail (list value)))
                               (setq tail (cdr (rplacd tail (list value))))))
                 #'(lambda nil head))))


(defmacro joining (&key initial-value)
   `(let ((result ,initial-value))
         (values #'(lambda (value)
                          (setq result (nconc result value)))
                #'(lambda nil result))))


(defmacro maximizing (&key initial-value)
   `(let
     ((result ,initial-value))
     (values #'(lambda (value)
                      (when ,(cond
                                ((and (constantp initial-value)
                                      (not (null (eval initial-value))))
                                               ; Initial value is given and we
                                               ; know it's not NIL, so leave
                                               ; out the null check
                                 '(> value result))
                                (t '(or (null result)
                                        (> value result))))
                            (setq result value)))
            #'(lambda nil result))))


(defmacro minimizing (&key initial-value)
   `(let
     ((result ,initial-value))
     (values #'(lambda (value)
                      (when ,(cond
                                ((and (constantp initial-value)
                                      (not (null (eval initial-value))))
                                               ; Initial value is given and we
                                               ; know it's not NIL, so leave
                                               ; out the null check
                                 '(< value result))
                                (t '(or (null result)
                                        (< value result))))
                            (setq result value)))
            #'(lambda nil result))))


(defmacro summing (&key (initial-value 0))
   `(let ((sum ,initial-value))
         (values #'(lambda (value)
                          (setq sum (+ sum value)))
                #'(lambda nil sum))))


;; Easier to read expanded code if PROG1 gets left alone
;; Also some transformations are possible.

(define-walker-template prog1 (nil return walker::repeat (eval)))
