;;; -*- Package: clim-user; Syntax: Common-lisp -*-

(in-package :clim-user)
#+lucid
(eval-when (compile load eval)
  (import '(lcl::handler-case lcl::compute-restarts lcl::invoke-restart
	    lcl::*debugger-hook* lcl::invoke-restart-interactively)))
;;;---------------
;;;
;;; Menu Debugger
;;;
;;; This code hooks into the debugger to expose a menu of
;;; proceed options in case of an error.  
;;;
;;; In the words of Steele (CLtL2):  
;;;"Experience on the Lisp Machines has shown
;;; that some users who are not programmers develop a terrible
;;; phobia of debuggers.  The reason for this usually can be traced
;;; to the fact that the debugger is very foreign to them and
;;; provides an overwhelming amount of information of interest only to
;;; programmers.  "
;;;
;;; This code has proven useful as a firewall for keeping naive users 
;;; out of the real debugger when unexpected errors occur.  It
;;; is also a convenience for people who are programmers because 9/10
;;; of the time you merely need to read the error message to
;;; know what you did wrong.
;;;
;;; Jeff Morrill (jmorrill@bbn.com)

#+clim-2
(defun choose-restart (condition restarts)
  ;; Choose a restart 
  ;; Returns:
  ;; :debugger - enter debuger
  ;; a restart - a restart to restart
  (let ((port (when *application-frame*
		(port *application-frame*))))
    (cond ((not port)
	   :debugger)
	  (t
	   (let* ((*print-escape* nil)
		  (report (format nil "~A: ~A"
				  (class-name (class-of condition))
				  condition))
		  (restarts
		   `((,report :type :label)
		     (nil :type :divider)
		     ("Debug this error" :value :debugger)
		     ,@(mapcar #'(lambda (r) `(,(format nil "~A" r) :value ,r))
			       restarts))))
	     (loop
	       ;; Loop until the user selects something from the menu.
	       (let ((choice
		      (frame-manager-menu-choose (Frame-manager port)
						 restarts
						 :gesture :select
						 :label "Error")))
		 (when choice (return choice)))))))))

#-clim-2
(defun choose-restart (condition restarts)
  :debugger)

(defun invoke-menu-debugger (condition debugger-hook)
  "Pops up a menu of continue options when an error occurs.
   The function INVOKE-RESTART is called to invoke the user's choice.
   If this function returns, the standard debugger will be invoked."
  (declare (ignore debugger-hook))
  (handler-case
   (let ((restart (let ((restarts (compute-restarts)))
		    (if (null restarts) :debugger
			(choose-restart condition restarts)))))
     (case restart
       (:debugger (return-from invoke-menu-debugger nil))
       (otherwise
	(let* ((*terminal-io* #+lucid lcl::*initial-io*
			      #+allegro excl:*initial-terminal-io*)
	       ;; Interactive restarts supposedly use *query-io*.
	       (*query-io* *terminal-io*))
	  (invoke-restart-interactively restart)))))
   (error (c)
	  (format *trace-output*
		  "~%Got an error from within the menu debugger.")
	  (let ((*print-escape* nil))
	    (format *trace-output* "~%~A" c))
	  (format *trace-output* "~%Returning to the standard debugger...~%")
	  nil)))

(eval-when (load eval)
  (unless *debugger-hook*
    (setq *debugger-hook* 'invoke-menu-debugger)))

#+debug
(defun foo ()
  ;; Broken function for testing.
  (let ((x nil))
    (check-type x number)))
