;;; -*- Mode: LISP; Syntax: Common-lisp; Package: ctv; Base: 10; -*-

#+Franz-Inc
(defpackage :CTV)
#-Genera
(in-package :CTV)


;;;;	 	CLIM Emulation of Symbolics Menu Functions
;;;;
;;;;  This package emulates a number of Symbolics menu functions from the TV:
;;;;  package to faciliate porting of Symbolics code to CLIM.  Wherever possible
;;;;  the original parameters and parameter formats have been maintained.  In
;;;;  some cases, additional parameters have been added to enhance a particular
;;;;  function.
;;;;  
;;;;  Authors:
;;;;    Peter Karp
;;;;    Mabry Tyson
;;;;    David Wilkins
;;;;    Scott McKay
;;;;  
;;;;  Please report bugs, bug fixes, and enhancements to Peter Karp
;;;;  -- pkarp@ai.sri.com
;;;;  
;;;;  You are free to copy and redistribute this code, but anyone receiving
;;;;  this code from a source other than SRI should report bugs to that source.
;;;;  
;;;;  This is not supported software -- use at your own risk.  SRI
;;;;  International does not warrant the performance or results that may be
;;;;  obtained by using this software.  SRI International disclaims all
;;;;  warranties as to performance or fitness of this software for any
;;;;  particular purpose.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; 				Revisions
;
;  11/10/92 DEW: added to SUGGEST and CHOOSE VARIABLE-VALUES and CHOOSE-CLOS-SLOT-VALUES:
;    -clim:close calls to get rid of menu (sometimes stays in certain window managers)
;    -user clim:draw-line* to separate exit boxes from variables
;    -added optional X Y keywords to specify location, user can set global variables 
;     *menu-y* *menu-x* to change default locations.  When these are NIL, get default
;     behavior which is for menu to come up near mouse but this only works in openwindows at AIC,
;     not other window managers. (note give :x-position :y-position as nil to clim:accepting-values
;     which works but is not documented as part of clim)
;    also added clim:close to edit-small-pop-up 
;  11/9/92  JDL: Modified SUGGEST-VARIABLE-VALUES to set global values of variables as
;                they are changed and to reset those values on exit.
;  11/4/92  DEW: changed choose and suggest VARIABLE-VALUES as follows:
;           (1) CLIM cuts off top of first line, so add a blank line if one not already there
;           (2) always use our name for Abort option, 
;           (3) allow user to specify name of EXIT option
;           (4) allow user to specify code for execution on ABORT option
;               (option 4 does not apply to suggest-variable-values which returns nil on abort)
;     (3) and (4) are implemented using the :margin-choices symbolics syntax.
;     Thus, old Genera code runs without change, and calls without :margin-choices are unchanged.
;     For instructions for using (3) and (4), see comments before CHOOSE-VARIABLE-VALUES.
;  10/27/92 JDL: Added special proclamation for *MENU-STREAM* to fix bug in MENU-CHOOSE.
;  10/26/92 JDL: Modified SUGGEST-VARIABLE-VALUES to include variable-add-button-p;
;           Changed formatting of CHOOSE- and SUGGEST-VARIABLE-VALUE buttons.
;   6/17/92 PDK: Fixed problems where the menu of menu-multiple-choose would not be
;           displayed properly if that function was called recursively from
;           choose-variable-values.  I had to look at the source for
;           clim::format-cell-internal.  Added the ability for the user to edit
;           the final value returned by menu-multiple-choose, and added the
;           :initial-select option to let some items be selected by default when
;           the menu is entered.
;   6/11/92 JDL: Added :integer-or-nil as a choose-variable-value keyword.
;   6/1/92  PDK: Added a new presentation type :eval to those types recognized
;           by CHOOSE-VARIABLE-VALUES and its variants.  When the user clicks on
;           a value of this type in the menu, the type-arg for that item is
;           evaluated (it might create a pop-up menu, for example), and the result
;           is inserted as the value for that item.
;   4/20/92 JDL: Added :string as a choose-variable-values keyword.
;   4/17/92 DEW: previously, :multiple-choose only returned one instead of a set --
;           now use clim:subset-alist in all :multiple-choose options, 
;           CONVERT-ASSOCS which converts items for :multiple-choose wrongly 
;           put an extra list structure around the :value it returned, so removed that.
;   4/14/92 DEW+PK: ctv:menu-choose now has ABORT? and NO-SELECT? optional args.
;          if ABORT?, then add <abort> to menu and it's action mimics CLIM ^Z.
;          if NO-SELECT?, then add <no select> to menu, and it's action is to
;          return nil, which mimics the genera ability to wave off the menu.
;          Since we are mimicing genera with this function, defaults are
;          ABORT? nil, NO-SELECT? T
;   4/3/92 DEW: eliminated check for item list has length 1 in ctv:menu-choose 
;          since it didn't handle all cases correctly, and never happened in genera anyway
;   1/7/92 PDK: Added function ctv:choose-clos-slot values, which will change
;   	   the values of the slots of an instance of any CLOS class in a format
;   	   much like choose-variable-values.
; 12/11/91 PDK: Change ctv:menu-choose to call on single-choice-menu rather than
; 	   clim:menu-choose so that we can turn off scroll bars for small menus.
; 10/24/91 DEW: change menu-choose to return NIl when item-list is nil
;          (compatible with symbolics, and Grasper relies on it)
;          :choose, :choose-multiple, etc cause CLIM to garbage menu if arg nil,
;          so add a dummy entry whenever type-args is nil, skip converting this
;          case in convert-assoc.
; 10/22/91 DEW: change :choose-multiple to apply convert-assocs to type-args instead 
;          of (first type-args) since that seems to work, though I thought the other
;          would.
; 10/16/91 DEW: changed write-string to CLIM::FORMAT to make strings behave in 
;          choose-variable-values as they do in symbolics (CLIM::PRINC worked for
;          PK but noone else) also, call ctv:menu-choose-key instead of
;          ctv:menu-choose in ctv:menu-choose
; 10/11/91 DEW: changed &key to &optional in MENU-CHOOSE to maintain compatibility
;          with symbolics code, created MENU-CHOOSE-KEY which uses keywords
; 10/3/91  PDK: The item-list for MULTIPLE-MENU-CHOOSE can now be a list of dotted
; 	   pairs (as before) or a list of atomic items.
; 9/17/91  John Doleac: MENU-MULTIPLE-CHOOSE can now have two additional
;          buttons.  Clicking one will select all of the items, clicking
;          the other will deselect all of the items.  Existence of these buttons is
;          controlled with :select-all-button-p.
; 9/10/91  PDK: MENU-CHOOSE now has a :pointer-documentation argument so that users
;	   can specify a stream for pointer documentation.
; 9/10/91  PDK: MENU-CHOOSE now has keyword rather than optional arguments.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; NOTES

; User code must call function CTV:INIT-CTV before calling any of the menu
; functions herein.

; BUGS: For some of these functions, CLIM does not display the menu label
; (title) as it should, under the OLWM window manager.

(proclaim ` (special *menu-stream*))

(defvar *CLIM-ROOT* nil)
(defvar *LOCAL-HOST* "cayucos")
(defconstant EVAL-KEYWORD '*EVAL*)
(defconstant NO-SELECT-KEYWORD '*NO-SELECT*)
(defconstant FUNCALL-KEYWORD '*FUNCALL*)
(defvar CONVERTED-MENUS nil "cache are reuse converted menus")

(defvar *menu-x* nil)
(defvar *menu-y* nil)
;default is for accepting-values menu to come up near mouse.  Only works in openwindows at AIC.
;by setting above to numbers, user can change default location

(export '(init-ctv init-clim menu-choose menu-choose-key
	  choose-variable-values suggest-variable-values
	  choose-clos-slot-values
	  menu-multiple-choose multiple-choose))

;;;; Here we define a presentation type for use with choose-variable-values.
;;;; Type type takes one argument.  The accept method for the type simply
;;;; evaluates the argument.

(clim:define-presentation-type eval-form (eform))

(clim:define-presentation-method clim:presentation-typep (object (type eval-form))
  (declare (ignore object type))
  nil)

(clim:define-presentation-method clim:present
				 (eval-form (type eval-form) stream
                                   (view clim:textual-view) &key)
				 #+Lucid (declare (ignore type))
  (clim:write-string (format nil "~A" eval-form) stream)
  )

(clim:define-presentation-method clim:accept
				 ((type eval-form) stream
				  (view clim:textual-view) &key)
				 (declare (ignore stream))
  (eval eform)
  )


; ==============================================================  init-ctv

; 


(defun INIT-CTV (&optional root)
  (unless *clim-root*
    #+Genera
    (setq *clim-root* (or root (clim:open-root-window :sheet)))
    #+(or Lucid Franz-Inc)
    (if root 
	(setq *clim-root* root)
        (multiple-value-bind (host display)
			     (sri:display-env)
          (setq *clim-root* (clim:open-root-window :clx :display display :host host)) ) )
    ) )


;;this figures out what machine is being used for the display of the root -dew
;;;; modified from toplevel-clim.lisp in tacitus >code>
(defun INIT-CLIM (&optional root) (init-ctv root)) 
  ;synonym in case I can't remember ctv, also is consistent with Tyson code


  

; ==============================================================  menu-choose

; Allows the user to choose a single item from a pop-up menu.
; Emulates Symbolics TV:MENU-CHOOSE.

; Each item in Item-List is either (1) a symbol, (2) a string, or (3) a list of
; the form (name option1 .. optionN).  The selection displayed in the menu is the item
; itself in cases (1) and (2), and is the first element of the list in case (3).
; Supported options are:
; 
;  :value arg    Specifies a value to return when the user selects this item
;  :eval arg     Specifies a form to be evaled when the user selects this item
;  :funcall arg  Names a functoin to be evaled when the user selects this item
;  :no-select    On Symbolics such items are displayed in the menu but the user
;                can't select them; in this implementation they are ignored.
;  :documentation A documentation string, which CLIM ignores
;  :style arg    Must give a CLIM text style for displaying the menu item
;
; Arguments:
;  o Abort-action -- Defines what action is taken when the user clicks on the
;	<Abort> item at the bottom of the menu.  :ABORT means call (abort), which
;	pops up to the CLIM command level.  :RETURN-NIL means return a value of
;	NIL from MENU-CHOOSE.  NIL means omit that item from the command menu altogether.
;
;  o package-prefix-symbols? -- Determines if package names are displayed in the menu
;    for symbolic selections in cases (1) and (3).  If T, then package names will be
;    displayed in the case that the symbolic selections are not all in the same package.


; A keyword version of MENU-CHOOSE:

(defun MENU-CHOOSE-KEY (item-list  &key
				   label
				   near-mode
				   default-item
				   (superior *clim-root*)
				   pointer-documentation
				   abort?
				   (no-select? t)
				   package-prefix-symbols?)
  (menu-choose item-list label near-mode
	       default-item superior pointer-documentation
	       abort? no-select? package-prefix-symbols?))


(defun MENU-CHOOSE (item-list &optional
			      label			; Window title
			      near-mode
			      default-item
			      (superior *clim-root*)    ; CLIM extension
			      pointer-documentation
			      abort?			; CLIM extension
			      (no-select? t)
			      package-prefix-symbols?	; CLIM extension
			      &aux choice)

  (when package-prefix-symbols?
    (setq item-list (package-prefix-itemlist item-list :menu-choose)) )

  ;; Add ability to abort out of the menu through explicit selection of an item
  (when (and item-list abort?)
    (setq item-list (append item-list '(("  <abort>" :EVAL (ABORT) :style (:fix :bold :normal)))))
    )
  (when (and item-list no-select?)
    (setq item-list (append item-list '(("  <no select>" :VALUE NIL :style (:fix :bold :normal)))))
    )


  (cond

   ;; For an empty item-list, return nil. (symbolics does this and Grasper relies on it)
   ((null item-list) nil)

   ;; Otherwise we actually let the user select an item.
   (t
    (cond
       ((setq choice (assoc item-list CONVERTED-MENUS))
	(setq item-list (cdr choice)))
       (t (setq choice (convert-menu-items item-list))
	  (push (cons item-list choice) CONVERTED-MENUS)
	  (setq item-list choice)))

	  (setq choice (single-choice-menu item-list label))

	  ;; Get user's choice, check whether it should be evaluated, a funcall
	  ;; performed, or a value returned, or the item not selectable (in
	  ;; which case we recurse).

	  (cond ((and (consp choice) (eq eval-keyword (car choice)))
		 (eval (cadr choice)))
		((and (consp choice) (eq funcall-keyword (car choice)))
		 (funcall (cadr choice)))
		((eq no-select-keyword choice)
		 (ctv:menu-choose-key item-list :label label :near-mode near-mode
				      :default-item default-item :superior superior
				      :pointer-documentation pointer-documentation))
		(t choice))
	  ) ) )


(defun package-prefix-itemlist (itemlist itemlist-format)
  (let (new-itemlist)
    (if (loop for item in itemlist
	      for value = (item-value item)
	      for pkg1 = (if (symbolp value) (symbol-package value) nil)
	      thereis
	      (and pkg1 (not (eq *package* pkg1)))
	      )
	(progn
	  (setq new-itemlist 
		(loop for item in itemlist
		      for value = (item-value item)
		      collect
		      (if (and (symbolp value) 
			       (not (eq *package* (symbol-package value))))
			  (if (symbolp item)
			      (ecase itemlist-format
				     (:menu-choose (list (prin1-to-string value) :value value))
				     (:menu-multiple-choose (list (prin1-to-string value) value)))
			      (cons (prin1-to-string value) (cdr item)))
			    item)
		      ))
	  (if (loop for x in new-itemlist always (symbolp x))
	      (sort new-itemlist #'string-lessp)
	    new-itemlist)
	  )
      itemlist)
      ))

(defun item-value (item)
  (if (consp item)
      (first item)
    item)
  )

;; We define this function for choosing a single item from a menu instead
;; of using clim:menu-choose because clim:menu-choose insists on creating
;; menus with scroll bars, which looks stupid for small menus.

;; We have a choice between using as :PARENT either *clim-root* or
;; (clim:frame-top-level-window clim:*application-frame*).  In the former
;; case the menu is not positioned at the pointer, but it does get a label;
;; in the latter case it is positioned at the pointer, but it does not get
;; a label.  Arg!  Furthermore, in the latter case Franz CLIM 1.1 barfs on us.
;; We make one choice for Franz and one choice for Lucid because the SRI
;; preference is to have the menu positioned at the pointer, whereas the
;; Franz choice is to not have it break.  Wrong -- with proper window manager
;; settings we can have both features under Lucid if we use the window root
;; (as of CLIM 1.1), so that's what we do.

(defun single-choice-menu (item-list &optional label)
  (let ((parent (or (clim:frame-top-level-window clim:*application-frame*)
		    *clim-root*) )
	menu-window)

    ;; The following is required for Franz
    (setq parent (clim:window-root parent))

    (multiple-value-bind (x y)
			 (clim:stream-pointer-position* parent)

      (setq menu-window
	    (clim:open-window-stream :parent parent
				     :scroll-bars (if (> 25 (length item-list))
						      nil
						    :vertical)
				     :left x
				     :top y
				     :label label))

      (multiple-value-bind (item gesture)
			   (clim::menu-choose-from-drawer
			    menu-window
			    'clim::menu-item
			    #'(lambda(stream type)
				(dolist (menu-item item-list)
                                  (format stream " ")
				  (clim:with-text-style ((or (getf (cdr menu-item) :style)
							     nil)
							 stream)
                                    ;; Print the current item.  If we print "" this
				    ;; causes the CLIM window to grow much bigger with
				    ;; lots of white space at the borders, so we
				    ;; print " " in place of "".

                                    (clim:present (format nil "~A" (if (string-equal "" (car menu-item))
								       " "
								       (car menu-item) ))
						  type
						  :stream stream) )
				  (format stream " ~%"))
				)
			    :x-position x
			    :y-position y)
			   (declare (ignore gesture))

        (getf (cdr (assoc item item-list :test #'equal))
	      :value)
	) ) ) )


; ===========================================================  choose-variable-values

; Pops up a menu listing a number of Lisp variables and their current values;
; users can button existing values to change them.  Note user must type RETURN
; after typing in a new value.  Emulates TV:CHOOSE-VARIABLE-VALUES.

; Variables is a list whose elements are either:
;  o Atomic variable names
;  o Lists of the form (var prompt type type-args)
;  o Strings (whose values are printed in the menu)

; Note that this function ignores a number of its paramters.

; New functionality:
; o If parameter variable-add-button-p is set, the menu will contain a button
;   that causes a new variable to be added to the menu.

;; Bug: I can't get clim:accepting-values to not put the abort button there,
;; so abort-button-p=nil does not work.



;  11/4/92  DEW
; (3) allow user to specify name of EXIT option
; (4) allow user to specify code for execution on ABORT option
;   (option 4 does not apply to suggest-variable-values which returns nil on abort)
;(3) and (4) are implemented using the :margin-choices symbolics syntax.
;Thus, old Genera code runs without change, and calls without :margin-choices are unchanged.

; instructions for using (3) and (4):
; :margin-choices is a list, each element is (1) a string,  or 
;    (2) a list whose car is a string and whose cadr is an expression to evaluate.
; Clim has two choices: EXIT and ABORT that appear on all menus.  
; Margin-choices are processed as follows:
; (3) the first margin-choice that is a simple string and is not "Abort" becomes the name
;  for the EXIT choice, with < and > added.  Example from SIPE:
;  :margin-choices ("Do it")  will rename the <SAVE changes> button to <Do it>
; (4) The first margin-choice that is a list becomes the name of the ABORT choice,
;  with < and > added.  IE the <ABORT changes> button is renamed to car of this margin-choice, 
;  and  clicking it will revert any changes and then evaluate the code 
;  in the cadr of the margin-choice.
;This is useful since SIPE often wants to do something special after an abort, e.g.
; :margin-choices '("Do It" ("Skip Drawing" (throw 'abort nil))) 
; will execute the throw when user clicks <Skip Drawing>.

    
    
(defun CHOOSE-VARIABLE-VALUES (item-list
			       &key
			       label			    	; Window title
			       (abort-button-p t)		; CLIM extension
			       variable-add-button-p		; CLIM extension
			       width		     		; Ignored
			       function
			       superior			     	; Ignored
			       margin-choices       		; Ignored
			       (x *menu-x*)
			       (y *menu-y*)
			       )
  (declare (ignore width superior))

  (let ((m-stream (clim:open-window-stream :parent *clim-root* :label nil))
	(exit-boxes `((:exit ,(get-exit-label margin-choices))))
	abort-code 
	original-values)

    (setq item-list (add-blank-line item-list))
    (cond ((setq abort-code (get-abort-choice margin-choices))
	   (push (list :abort (concatenate 'string "<" (car abort-code) ">")) exit-boxes)
	   (setq abort-code (cadr abort-code)))
	  (abort-button-p (push '(:abort "<ABORT changes>") exit-boxes))
	  (t (push '(:abort "<ABORT changes>") exit-boxes)))
             ;since clim always puts abort button there, let's use our name

    (setq original-values (save-items item-list))
    (multiple-value-bind (v1 v2)
      (clim::catch-abort-gestures ("Abort choosing the variable values")
        (clim:accepting-values (m-stream :label label :own-window t :exit-boxes exit-boxes
					 :x-position x :y-position y)
          (loop for v in item-list
		do
		;; The user might have given either a list or an atom for
		;; each element of Item-List.  For atoms we create a full list.
		(multiple-value-bind (var prompt type type-args)
				     (if (consp v)
					 (values-list v)
				         (values v (string v) :expression))

                  ;; If the user has only supplied Var and Type, the code above will
		  ;; have put the Type keyword in Prompt.  So we check for the
		  ;; prompt being a keyword symbol, and if so we put that keyword
		  ;; in Type.
                  (when (and (symbolp prompt)
			     (eq (symbol-package prompt)
				 (find-package 'keyword)))
                    (setq type prompt)
		    (setq prompt (string var)))

		  ;; Default the type
		  (unless type (setq type :expression))
		  ;; CLIM doesn't do CRs and garbages menu if type-args nil
		  (unless type-args (setq type-args '((NONE :value nil))))

		  (cond ((symbolp var)
			 (let ((oldv (symbol-value var))
			       newv)
			   (setq newv (cvv-accept type type-args prompt oldv))
			   ;; In case var is not a legal variable name
			   (when (variable-of var)
                             (set var newv) )
			   (when (equal (symbol-value var) '(NONE :value nil))
                             (set var nil))
			   (when function
                             (funcall function m-stream var oldv (symbol-value var))))
			 )
			((stringp var)
			 ;; Mabry had used WRITE-STRING which bombed, DEW tried
			 ;; clim::princ and PK got it to work on his machine,
			 ;; everywhere else it complained that m-stream wasn't a
			 ;; stream,  so I'll try clim::format
			 (clim:format m-stream "~A" var)
			 )
			(t
			 (error "Don't understand ~S" var) )
			) )
		(clim:terpri m-stream) )
	  (multiple-value-bind (xx yy) (clim:stream-cursor-position* m-stream)
	    (declare (ignore xx))
	    (clim:draw-line* m-stream 0 (+ 5 yy) 400 (+ 5 yy) :line-thickness 2))
	  (clim:terpri m-stream)	       
	  
	  (when variable-add-button-p
            (clim:accept-values-command-button (m-stream) "<ADD item>,"
              (let (v)
		(setq v
		      (loop for symbol = (clim:accept 'clim:symbol
						      :prompt "Name of item to add")
			    for var = (variable-of symbol)
			    when var
			    return var
			    do
			    (clim:format t "~%*** ~A is not an acceptable item name~%" symbol)
			    ) )
		(clim:terpri *query-io*)
		(set v nil)
		(push v item-list)
		)))
	  ))

      (clim:close m-stream)
      ;; If the user clicked on abort, restore original variable values
      (when (and (null v1) (eq t v2))
        (loop for (var value) in original-values do (set var value))
	(eval abort-code))
      )))


#|
(progv '(x y) '(a x)
  (choose-variable-values `((x "X" :choose (a b c))
			    (y "Y")
			    ) ) )

(progv '(x y) '(a x)
  (choose-variable-values `((x "X" :choose (a b c))
			    (y "Y" :eval (ctv:menu-multiple-choose '(x y z)))
			    ) ) )
|#

(defun save-items (item-list)
  (loop for item in item-list
	for var = (variable-of item)
	when var
	  collect (list var (symbol-value var))
	    ))

(defun GET-EXIT-LABEL (margin-choices &aux choice)
  (cond ((and (consp margin-choices)
	      (setq choice (loop for x in margin-choices
				 when (and (stringp x)
					   (not (string-equal choice "Abort")))
;only recognize choices that are simple strings since CLIM does not support any user specified actions
				   return x)))
	 (concatenate 'string "<" choice ">"))
	(t "<SAVE changes>")))
;if user gives explicit margin choices, assume first simple string not named ABORT corresponds to SAVE
; also, put < and > around it (corresponds to Genera syntax, but ignores choices that are lists.
;  Example use:  :margin-choices ("Do it")  will rename the <SAVE changes> button to <Do it>

(defun GET-ABORT-CHOICE (margin-choices)
  (when (consp margin-choices)
    (loop for x in margin-choices when (and (consp x) 
					    (stringp (car x)))
;; no, let's take first consp as the abort instead of requiring name (string-equal (car x) "Abort")
				    return x)))

(defun add-blank-line (item-list)
  (unless (and (stringp (car item-list))
	       (or (string= (car item-list) " ")
		   (string= (car item-list) "")))
      (push " " item-list)) ;CLIM cuts off top of first line, so add a blank it not already there --dew
  item-list)


(defun cvv-accept (type type-args prompt default)
  (clim:accept
   (case type
     (:boolean 'clim:boolean)
     (:decimal-number 'clim:number)
     (:expression 'clim:expression)
     (:sexp 'clim:symbol)
     (:choose-multiple `(clim:subset-alist
			 ,(convert-assocs type-args) :test equal))
     (:integer 'clim:integer)
     (:assoc `(clim:member-alist ,(convert-assocs type-args) :test equal))
     (:choose `(clim:member-sequence ,type-args :test equal))
     (:subset `(clim:subset-sequence ,type-args))
     (:number 'clim:number)
     (:string (setq default (string default)) 'clim:string)
     (:eval `(eval-form ,type-args))
     (:integer-or-nil '(clim:null-or-type clim:integer))
     (otherwise (error "Unhandled type: ~S" type)))
   :prompt prompt :default default)
  )



(defun variable-of (item)
  (when (consp item)
    (setq item (first item)) )
  (if (and (symbolp item) (not (null item)) (not (eq t item)))
      item
      nil)
  )

; ==========================================================  suggest-variable-values

; The inputs of this function are the same as CHOOSE-VARIABLE-VALUES.  However,
; instead of altering the values of the item-list, we return an alist of the names
; of item-list whose values the user has changed, and their new values.

; The parameter Item-List is a list whose elements are either atomic 
; variable names, or lists of the form (var prompt type type-args).

(defun SUGGEST-VARIABLE-VALUES (item-list
				      &key
				      (abort-button-p t)
				      label
				      variable-add-button-p		; CLIM extension
				      width
				      function
				      superior
				      margin-choices
				      (x *menu-x*)
				      (y *menu-y*))
  (declare (ignore width superior))
  (let ((new-values nil)
	original-values
	(exit-boxes `((:exit ,(get-exit-label margin-choices))))
	(m-stream (clim:open-window-stream :parent *clim-root*
					   :label nil) )
	)
    (setq item-list (add-blank-line item-list)) ;add blank line, but ignore abort margin choices 
    (cond (abort-button-p (setq exit-boxes (cons '(:abort "<ABORT changes>") exit-boxes)))
	  (t (setq exit-boxes (cons '(:abort "<ABORT changes>") exit-boxes)))) 
          ;clim always puts an abort there, so use our name til we can fix this
    (setq original-values (save-items item-list))
    (multiple-value-bind (v1 v2)
      (clim::catch-abort-gestures ("Abort choosing the variable values")
        (clim:accepting-values (m-stream :label label :own-window t :exit-boxes exit-boxes
					 :x-position x :y-position y)
          (loop for v in item-list
		do
		(multiple-value-bind (var prompt type type-args)
				     (if (consp v)
					 (values-list v)
				         (values v (string v) :expression))
                  (unless type
                    (setq type :expression))
		  (cond ((symbolp var)
			 (let ((oldv (symbol-value var)))
			   (multiple-value-bind (accepted-value value-type user-input-p)
						(cvv-accept type type-args prompt oldv)
						(declare (ignore value-type))

                             (when user-input-p
                               (push (list var accepted-value)
				     new-values)
			       (set var accepted-value) )
			     (when function
                               (funcall function m-stream var oldv (symbol-value var))))
			   ) )
			((stringp var)
			 (clim:format m-stream "~A" var)
			 )
			(t
			 (error "Don't understand ~S" var) )
			) )
		(clim:terpri m-stream)
		)
	  (multiple-value-bind (xx yy) (clim:stream-cursor-position* m-stream)
	    (declare (ignore xx))
	    (clim:draw-line* m-stream 0 (+ 5 yy) 400 (+ 5 yy) :line-thickness 2))
	  (clim:terpri m-stream)
	  (when variable-add-button-p
            (clim:accept-values-command-button (m-stream) "<ADD item>,"
              (let (v)
		(setq v
		      (loop for symbol = (clim:accept 'clim:symbol
						      :prompt "Name of item to add")
			    for var = (variable-of symbol)
			    when var
			    return var
			    do
			    (clim:format t "~%*** ~A is not an acceptable item name~%" symbol)
			    ) )
		(clim:terpri *query-io*)
		(set v nil)
		(push v item-list)
		)))
	  ) )
      (clim:close m-stream)
      ;; If the user clicked on abort, there are no new values
      (when (and (null v1) (eq t v2))
	    (setq new-values nil) )
      )
    (loop for (var value) in original-values do (set var value))
    new-values
    ) )

;does not take :associated-window
; see listener.lisp in sys:clim;demo;


; ==============================================================  choose-clos-slot-values

;; type-and-prompt is a list whose elements are of the form (slot prompt type type-args)


(defun choose-clos-slot-values (instance &key
					 type-and-prompt
					 (label "Select slot values:")
					 excluded-slots
					 (guess-types-p t)
					 (x *menu-x*)
					 (y *menu-y*)
					 &aux
					 (m-stream (clim:open-window-stream :parent *clim-root*
									    :label nil) )
					 )
  (clim::catch-abort-gestures ("Abort choosing the values")
  (clim:accepting-values (m-stream :label label :own-window t
				   :x-position x :y-position y)
    (loop for slot-obj in (clos::class-direct-slots (clos::class-of instance))
	  for name = (clos::slot-definition-name slot-obj)
	  for current-value = (clos::slot-value instance name)
;;;;	  for documentation = (clos::documentation slot-obj)
	  for prompt = (or (second (assoc name type-and-prompt)) (string name))
	  for type = (or (third (assoc name type-and-prompt))
			 (if guess-types-p
			     (typecase current-value
                               (integer :integer)
			       (number :number)
			       (t :expression) )
			     :expression) )
	  for type-args = (fourth (assoc name type-and-prompt))
	  when (not (member name excluded-slots))
	  do
	  (multiple-value-bind (accepted-value value-type user-input-p)
			       (cvv-accept type type-args prompt current-value)
			       (declare (ignore value-type))

            (when user-input-p
              (setf (clos::slot-value instance name) accepted-value) )

	    (clim:terpri m-stream))
	  )
    (multiple-value-bind (xx yy) (clim:stream-cursor-position* m-stream)
	    (declare (ignore xx))
	    (clim:draw-line* m-stream 0 (+ 5 yy) 400 (+ 5 yy) :line-thickness 2))
    (clim:terpri m-stream)))
  (clim:close m-stream)
  instance
  )


; ==============================================================  convert-menu-items

; o Supports :documentation :value :eval :funcall, others ignored
; o The :style keyword and its arg is passed through so a CLIM character style
;   can be used.

(defun CONVERT-MENU-ITEMS (items)
  (let (new-item)
    (loop for item in items
	  when (setq new-item (convert-menu-item item)) collect new-item)
    ))

(defun CONVERT-MENU-ITEM (item)

; Watch for items that are atoms
  (if (atom item)
      (setq item (list item :value item)) )

; First element of item must be a string
  (if (not (stringp (first item)))
      (setq item (cons (format nil "~A" (first item)) (cdr item))) )

; Watch for items that don't use the :value keyword in front of the value
  (if (and (> (length item) 1)
	   (not (keywordp (second item))) )
      (setq item `(,(first item) :value ,@(cdr item))) )
	   
  (cons (car item)
	(loop for keys on (cdr item) by #'cddr 
	      append (case (first keys)  ;first two accepted by CLIM, second two hacked to work
			   (:value (list :value (second keys)))
			   (:documentation (list :documentation (second keys)))
			   (:eval (list :value (list eval-keyword (second keys))))
			   (:funcall (list :value (list funcall-keyword (second keys))))
			   (:no-select (list :value no-select-keyword))
			   (:style (list :style (second keys)))
			   (otherwise nil)))))



(defun CONVERT-ASSOCS (alist)
      (loop for x in alist collect
	    (if (member :value x)
		x
		(list (car x) :value (cdr x)))))
;CLIM seems to skip carriage return in menu if alist is NIL, so already set to
; dummy entry


; ==============================================================  menu-multiple-choose

; This function displays a menu of items, and allows the user to select multiple
; items from the menu before returning.  We return a list of the chosen items.

; Each element of Item-List is either:
;   o An atomic item
;   o A list of the form (Name Value Options)
; where Name is a menu-item name, Value is returned when that name is selected,
; and Options can be:
;    :NO-SELECT T       to indicate an item that is not selectable by the user
;    :INITIAL-SELECT T  to indicate that item should be preselected in the menu

; Arguments:
;  o package-prefix-symbols? -- Determines if package names are displayed in the menu
;    for symbolic selections in cases (1) and (3).  If T, then package names will be
;    displayed in the case that the symbolic selections are not all in the same package.

; BUGS: When items are selected they are displayed in bold face, but for some 
; reason the underlying nonbold face display of the item is not ereased.


(clim:define-presentation-type menu-multiple-choose-selection ())

(clim:define-presentation-type menu-multiple-choose-command ()) 
(clim:define-presentation-type no-select ()) 

(defun MENU-MULTIPLE-CHOOSE (item-list
			     &key
			     select-all-button-p     ; Set to t to add the <Select All>
						     ; and <Deselect All> buttons
			     package-prefix-symbols?
			     edit-button-p
			     default-style
			     label
			     (associated-window (clim:open-window-stream :parent *clim-root*
									 :label label))
			     (printer #'print-item)
			     max-width max-height
			     n-rows n-columns
			     inter-column-spacing inter-row-spacing
			     (cell-align-x ':left) (cell-align-y ':top)
			     &aux m-stream)

  (when package-prefix-symbols?
    (package-prefix-itemlist item-list :menu-multiple-choose) )

  ;; Convert atomic items to lists, and dotted pairs to lists.
  ;; Copy the item list because the loop below destructively changes it
  (setq item-list (copy-list item-list))
  (loop for item on item-list
	do
	(cond ((not (consp (first item)))
	       (rplaca item (list (first item) (first item))) )
	      ((and (eq 1 (length (first item)))
		    (cdr (first item)) )
	       (rplaca item (list (car (first item)) (cdr (first item)))) )
	      ) )

  (clim:with-menu (stream associated-window)
    (setq m-stream stream)
    (setf (clim:window-label stream) label)
    (clim:with-end-of-page-action (:allow stream)
      (clim:with-end-of-line-action (:allow stream)
	(clim:with-text-style (default-style stream)
	  (let ((selections (loop for x in item-list
				  for initial-value = (getf x :initial-select)
				  collect (list x initial-value) ))
		(selection-pieces ())
		;;--- Need this first-piece kludge to work around a redisplay
		;;--- bug that causes the first item to be erased whenever
		;;--- any other item is redisplayed.
		(first-piece nil))
	    ;; Display all the selections, collecting redisplay pieces as we go
	    (clim:formatting-item-list (stream :max-width max-width :max-height max-height
					       :n-rows n-rows :n-columns n-columns
					       :inter-column-spacing inter-column-spacing
					       :inter-row-spacing inter-row-spacing)
             ;;; The following fixes what appears to be a CLIM bug that prevents
	     ;;; menu-multiple-choose from working when called by choose-variable-values.
	     ;;; CLIM gets confused about which window to write to.
	     (let ((clim::*original-stream* m-stream))
	      (dolist (selection selections)
		(clim:formatting-cell (stream :align-x cell-align-x :align-y cell-align-y)
		  (let ((piece (let ((selection selection)
				     (no-select? (getf (first selection) :no-select)) )
				 (clim:updating-output (stream)
				   (clim:updating-output (stream :unique-id selection
								 :cache-value (second selection))
				     (clim:with-output-as-presentation
					 (:stream stream
					  :object selection
					  :type (if no-select?
						    'no-select
						    'menu-multiple-choose-selection) )
					 (if (second selection)
					     (clim:with-text-face (:bold stream)
                                               (funcall printer (first selection) stream))
					     (funcall printer (first selection) stream))))))))
		    (when (null first-piece)
		      (setq first-piece piece))
		    (push (list selection piece) selection-pieces))))))
	    ;; Display exit boxes and the optional select-all button
	    (let ((s-all "<Select All>")
		  (ds-all "<Deselect All>")
		  (edit "<Edit> edit and return these values")
		  (exit  "<End> uses these values")
		  (abort "<Abort> aborts"))
	      (clim:updating-output (stream :unique-id stream
					    :cache-value 'exit-boxes)
		(clim:terpri stream)
		(when select-all-button-p
                  (clim:terpri stream)
		  (clim:with-output-as-presentation(:stream stream
                                                    :type 'menu-multiple-choose-command
						    :object ':select-all)
                    (write-string s-all stream))
		  (write-string ", " stream)
		  (clim:with-output-as-presentation(:stream stream
                                                    :type 'menu-multiple-choose-command
						    :object ':deselect-all)
                    (write-string ds-all stream))
		  (write-string ", " stream)
		  (clim:terpri stream)
		  )
		(when edit-button-p
                  (clim:terpri stream)
		  (clim:with-output-as-presentation(:stream stream
                                                    :type 'clim::accept-values-exit-box
						    :object ':edit)
                    (write-string edit stream))
		  (clim:terpri stream)
		  )
		(clim:with-output-as-presentation (:stream stream
						   :type 'clim::accept-values-exit-box
						   :object ':abort)
		  (write-string abort stream))
		(write-string ", " stream)
		(clim:with-output-as-presentation (:stream stream
						   :type 'clim::accept-values-exit-box
						   :object ':exit)
		  (write-string exit stream)))
	      (clim:terpri stream)
	      )
	    ;; Size and expose the multiple-choice menu
	    (clim::size-menu-appropriately stream)
	    (multiple-value-bind (x y)
		(clim::stream-pointer-position-in-window-coordinates (clim:window-parent stream))
	      (clim::position-window-near-carefully stream x y))
	    (clim:window-expose stream)
	    ;; Now read from the menu
	    (clim:with-input-focus (stream)
	      (loop do
		(clim:with-input-context ('clim::accept-values-exit-box :override t)
					 (exit)
		     (clim:with-input-context ('menu-multiple-choose-command)
					      (button-selection)
			  (clim:with-input-context ('menu-multiple-choose-selection)
						   (selection)
			       (clim:read-gesture :stream stream)
			     (menu-multiple-choose-selection
			       (setf (second selection) (not (second selection)))
			       (let ((piece (second (assoc selection selection-pieces))))
				 (when piece
				   (clim:redisplay piece stream)
				     ))))
			(menu-multiple-choose-command
			  (ecase button-selection
			    (:select-all
			      (dolist (selection selection-pieces)
                                (unless (getf (first (first selection)) :no-select)
                                  (setf (second (car selection)) t) )
				(clim:redisplay (second selection) stream)))
			    (:deselect-all
			      (dolist (selection selection-pieces)
				(setf (second (car selection)) nil)
				(clim:redisplay (second selection) stream))))))
		   (clim::accept-values-exit-box
		     (ecase exit
		       (:abort
			 (setf (clim:window-visibility stream) nil)
			 (force-output stream)
			 (return-from menu-multiple-choose nil))
		       (:exit
			 (setf (clim:window-visibility stream) nil)
			 (force-output stream)
			 (return-from menu-multiple-choose
			   (mapcan #'(lambda (selection)
				       (and (second selection)
					    (list (second (first selection)))))
				   selections)))
		       (:edit
			 (setf (clim:window-visibility stream) nil)
			 (force-output stream)
			 (return-from menu-multiple-choose
                           (edit-small-pop-up
			    "Edit returned values"
			    (mapcan #'(lambda (selection)
					(and (second selection)
					     (list (second (first selection)))))
				    selections)
			    'clim:expression) ) )
		       )))))))))))

(defun print-item (item stream)
  (clim:format stream "~A~%" (car item))
  )

	  
;; (menu-multiple-choose (loop for i below 20 collect (cons (format nil "~R" i) i)))

#|
(menu-multiple-choose '(("Choose selection" nil :no-select t)
			("XX" X :initial-select t)
			("YY" Y)))
|#


;;; This function creates a pop-up window whose label is Prompt.  If a Default
;;; is supplied then it is displayed in the window.  The user can click on the
;;; displayed value, and either edit it using Emacs keystrokes (by first typing
;;; ctrl-meta-Y to insert it into the buffer) or replace it (by simply starting
;;; to type).  

(defun edit-small-pop-up (prompt &optional (default "   ") (type 'clim:string))
  (let (value
	(m-stream (clim:open-window-stream :parent *clim-root*
					   :label nil)))
  (clim::catch-abort-gestures ("Abort choosing the value")
    (clim:accepting-values (m-stream :label prompt :own-window t)
      (clim:terpri m-stream)
      (setq value (clim:accept type :prompt "    " :prompt-mode :raw :default default))
      (clim:format m-stream "~%~%") ))
  (clim:close m-stream)
  value))

; ============================================================  multiple-choose

; Pops up a menu with a list of items and a list of boxes next to each item.
; The user can select one or more boxes per item.  So we get a 2-D array of
; boxes.  Emulates TV:MULTIPLE-CHOOSE.

; See below for a description of parameters and return value.


(defvar *hsep* 50)		; Horizontal separation between item-list region
				; and check-box region of screen.
(defvar *title-margin* 10)	; Vertical margin around item-list and title.
(defvar *exit-margin* 10)	; Determines vertical size of DoIt/Abort region
(defvar *key-sep*  20)		; Horiz separation between keywords (column headings)
(defvar *box-size* 10)		; Maximum possible size of the check-box

(defmacro str-width (str style)
  `(clim:stream-string-width *menu-stream* ,str :text-style ,style) )

(clim:define-presentation-type check-box ())

(clim:define-presentation-method clim:presentation-typep (object (type check-box))
  (declare (ignore object type))
  nil)

; Check-boxes are displayed at an x-offset from the current cursor position.
; They are either invisible, unchecked, or checked.

(clim:define-presentation-method clim:present
				 (box (type check-box) stream 
				       (view clim:textual-view)
				       &key &allow-other-keys)
				 #+Lucid (declare (ignore type))

    (multiple-value-bind (box-x box-size i k on-off x y)
			 (values-list box)
			 (declare (ignore i k))
      (setq x (+ x box-x))

; If visible, first draw unchecked box.  We draw a background-colored filled
; rectangle inside the box so it's easier for the user to click on.

      (if on-off
	  (progn
	    (clim:draw-rectangle* stream
				  x y (+ x box-size) (+ y box-size)
				  :filled nil :line-thickness 2)
	    (clim:draw-rectangle* stream
				  (+ 2 x) (+ 2 y) (- (+ x box-size) 2) (- (+ y box-size) 2)
				  :filled t :ink clim:+background+) ) )


; If checked, draw X.

      (if (eq 'on on-off)
	  (progn
	    (clim:draw-line* stream x y (+ x box-size) (+ y box-size) :line-thickness 2)
	    (clim:draw-line* stream x (+ y box-size) (+ x box-size) y) :line-thickness 2)
	)
      ) )


; PARAMETERS:
;
; Each item in item-list is of the form  (item name keys) where:
;  item   Is used in our return value
;  name   Is what appears in the menu for this item
;  keys   Describe the allowable choices for this item.  Keys is a list, each
;         element of which is either a key name from keyword-alist, or a list
;         of the form (key-name default) where default gives the default 
;         setting of the corresponding box for this item (checked/unchecked).
;
; Each item of keyword-alist describes a column of the check-box array, and is
; a list of the form (key name i1 i2 i3 i4):
;  key    Appears in our return value
;  name   Is what appears in the menu for this item
;  iN     These are optional and describe what happens to other boxes in the
;         same row as a box that the user has just clicked.  Each i1 can be
;         T (meaning all key names), NIL (no key names), or a list of key names.
;	  i1 tells what other boxes in the row to check when the user checks a
;         box.  I2 tells what other boxes to uncheck when the user checks
;         a box.  I3 and i4 are analagous but control the behavior when the user
;         unchecks a box.

; CLIM Extensions: 
; Two new parameters determine the text style of the items, and of the title
; regions of the menu.

(defun MULTIPLE-CHOOSE (item-title	    ; A title drawn above the item list
			item-list
			keyword-alist
			&key
			(title-style '(:sans-serif :bold :large))   ; CLIM extension
			(item-style  '(:fix :roman :large))	    ; CLIM extension
			debug )


  (setq title-style
	(clim:make-text-style
	 (first title-style) (second title-style) (third title-style)))
  (setq item-style
	(clim:make-text-style
	 (first item-style) (second item-style) (third item-style)))

  (let* ((n-items    (length item-list))
	 (n-keywords (length keyword-alist))
	 (keywords (mapcar #'first keyword-alist))
	 (*menu-stream* (clim:open-window-stream :parent *clim-root*
						 :label nil) )
	 (y (+ (* 3 *title-margin*)
	       (clim:text-style-height title-style *menu-stream*)) )
	 (item-height  (clim:text-style-height item-style  *menu-stream*))
	 (boxes (make-array (list n-items n-keywords) :initial-element nil))
	 box-x-list box-size box-margin output-rec key-x)

; A few important positions:
;  y       Top of item-list region
;  exit-y  Top of DoIt/Abort region
;  key-x   Left of keyword-name/boxes region
;  max-x   Right of keyword-name/boxes region


    (setq key-x (setup-mchoose-window item-title item-list keyword-alist
				      title-style item-style y item-height) )

; Adjust box size downward if we're dealing with tiny items.

      (setq box-size (min *box-size* (- item-height 6)))
      (setq box-margin (round (/ (- item-height box-size) 2)))

; Initialize the values of the boxes array to 'on , 'off , or nil (meaning the
; user can't even select that item) based on the initializations in item-list.

      (loop for item in item-list
	    for i from 0
	    do
	    (loop for choice in (third item)
		  do (let ((default 'off)
			   keyword)
		       (if (consp choice)
			   (progn
			     (setq keyword (first choice))
			     (setq default (if (second choice) 'on 'off)) )
			   (setq keyword choice) )
		       (setf (aref boxes i (position keyword keywords :test #'eq)) default)
		       ) ) )
			       
      (if debug
	  (show-state-array "Initial state array is:" boxes) )

; Compute the X positions of the check boxes by centering them under each
; keyword name.

      (let ((x 0) key-width)
	(setq box-x-list nil)
	(loop for key in keyword-alist
	      for k from 0
	      do
	      (progn
		(setq key-width (str-width (second key) title-style))
		(push (+ x (round (/ (- key-width box-size) 2)))
		      box-x-list)
		(setq x (+ x key-width *key-sep*))
		) ) )
      (setq box-x-list (reverse box-x-list))

; Display the check boxes.  We display each row as a single cell because it
; doesn't appear that CLIM's formatting capabilities work for irregularly
; spaced items.

      (setq output-rec
	    (clim:updating-output (*menu-stream*)

	      (clim:stream-set-cursor-position* *menu-stream* key-x (+ y box-margin))

	      (clim:formatting-item-list (*menu-stream*
					  :n-rows n-items
					  :inter-row-spacing (* 2 box-margin) )
		(dotimes (i n-items)
		  (clim:formatting-cell (*menu-stream*)
		    (loop for x in box-x-list
			  for k from 0
			  do
			  (clim:updating-output (*menu-stream*
						 :unique-id (+ k (* i n-keywords))
						 :cache-value (aref boxes i k) )
                            (multiple-value-bind (xx yy)
						 (clim:stream-cursor-position* *menu-stream*)
			      (clim:present (list x box-size i k (aref boxes i k) xx yy)
					    'check-box :stream *menu-stream*) ) ) ) ) ) ) ) )


; Now repetitively read a box click, toggle the state of the selected box, and
; incrementally redisplay the boxes.  If the DoIt or Abort boxes were clicked,
; then we exit.

      (loop while
	    (let (box implications)

	      (clim:with-input-context ('check-box)
				       (box-var)
		  (clim:read-gesture :stream *menu-stream*)

		(check-box  (setq box box-var)) )

	      (multiple-value-bind (x size i k state)
				   (values-list box)
		(declare (ignore x size))

		(if debug
		    (format *terminal-io* "Event [~A,~A] state ~A~%" i k state) )

		(cond
		 ((eq 'Abort state)
		  (setf (clim:window-visibility *menu-stream*) nil)
		  (clim:force-output *menu-stream*)
		  (return-from multiple-choose nil) )
		      
		 ((eq 'DoIt state)
		  (setf (clim:window-visibility *menu-stream*) nil)
		  (clim:force-output *menu-stream*)
		  nil )
		 
		 (t
		  (setq state (if (eq 'on state) 'off 'on))
		  (setf (aref boxes i k) state)
		  (setq implications (cddr (nth k keyword-alist)))

		  (case state
			(on (toggle-boxes boxes i k (first implications)
					  'on n-keywords keywords)
			    (toggle-boxes boxes i k (second implications)
					  'off n-keywords keywords) )
			(off (toggle-boxes boxes i k (third implications)
					   'off n-keywords keywords)
			     (toggle-boxes boxes i k (fourth implications)
					   'on n-keywords keywords) )
			)

		  (if debug
		      (show-state-array "New state of state array is:" boxes) )

		  (clim:redisplay output-rec *menu-stream*)
		  t )
		 ) ) ) )

      (setf (clim:window-visibility *menu-stream*) nil)
      (clim:read-char-no-hang *menu-stream*)

; Construct our return value based on the state of the boxes array.

      (let ((results nil) result)
	(dotimes (i n-items)
	  (setq result nil)
	  (dotimes (k n-keywords)
	    (if (eq 'on (aref boxes i k))
		(push (first (nth k keyword-alist)) result) ) )
	  (if result
	      (push (cons (first (nth i item-list)) result)
		    results) ) )
	results )

      ) )


; Alter the state of the boxes in a row based on Imp, which is either T
; (meaning change all boxes), NIL (change no boxes), or a list, meaning change
; boxes named in the list.

(defun TOGGLE-BOXES (boxes i j imp new-value n-keywords keywords)
  (cond ((eq t imp)
	 (loop for k from 0 to (1- n-keywords)
	       do (if (and (aref boxes i k)
			   (not (eq j k)) )
		      (setf (aref boxes i k) new-value) ) ) )
	((consp imp)
	 (let (k)
	   (loop for key in imp
		 do (progn
		      (setq k (position key keywords))
		      (if (aref boxes i k)
			  (setf (aref boxes i k) new-value) ) ) ) ) )
	) )



; Set up static aspects of the multiple-choose window.

(defun SETUP-MCHOOSE-WINDOW (item-title item-list keyword-alist
			     title-style item-style y item-height)
  (let (key-x max-x exit-y tmp y-extent)
    (declare (special *menu-stream*))

    (clim:window-clear *menu-stream*)

; Draw the item title

    (clim:draw-text* *menu-stream*
		     item-title
		     0 *title-margin*
		     :align-y :top
		     :text-style title-style)

    (setq key-x (str-width item-title title-style))

; Display the item names (row headings)

    (clim:stream-set-cursor-position* *menu-stream* 0 y)
    (clim:formatting-item-list (*menu-stream*
				:n-columns 1)
      (clim:with-text-style (item-style *menu-stream*)
	(dolist (item item-list)
	  (clim:formatting-cell (*menu-stream*)
	    (format *menu-stream* (second item)) ) ) ) )


    (multiple-value-setq (tmp exit-y)
			 (clim:stream-cursor-position* *menu-stream*) )
    (setq key-x (+ *hsep* (max tmp key-x)))
    (setq exit-y (+ exit-y item-height *title-margin*))

; Display the column headings (keyword names)

    (clim:stream-set-cursor-position* *menu-stream* key-x *title-margin*)

    (clim:formatting-item-list (*menu-stream*
				:inter-column-spacing *key-sep*
				:n-rows 1)
      (clim:with-text-style (title-style *menu-stream*)
	(dolist (key keyword-alist)
	  (clim:formatting-cell (*menu-stream*)
	    (format *menu-stream* (second key)) ) ) ) )

; Draw horizontal lines right below the column headings and above the 
; DoIt/Abort area.

    (multiple-value-setq (max-x tmp)
			 (clim:stream-cursor-position* *menu-stream*) )

    (clim:draw-line* *menu-stream*
		     0 (- y *title-margin*)
		     max-x (- y *title-margin*))
    (clim:draw-line* *menu-stream*
		     0 exit-y
		     max-x exit-y)

    (setq y-extent (+ exit-y (* 2 *exit-margin*) y 20)) ; 20 is the bottom scrollbar

    (clim:with-text-style (title-style *menu-stream*)
      (clim:stream-set-cursor-position* *menu-stream* 40 (round (/ (+ exit-y y-extent -20) 2)))
      (clim:format *menu-stream* "Do It ")
      (multiple-value-bind (x y)
			   (clim:stream-cursor-position* *menu-stream*)

	(clim:stream-set-cursor-position* *menu-stream* (+ x 20) y)
	(clim:present (list 0 *box-size* 0 0 'DoIt (+ x 20) y)
		      'check-box :stream *menu-stream*) )

      (clim:stream-set-cursor-position* *menu-stream* 
					(- max-x (str-width "Abort " title-style) 50)
					(round (/ (+ exit-y y-extent -20) 2) ))
      (clim:format *menu-stream* "Abort ")

      (multiple-value-bind (x y)
			   (clim:stream-cursor-position* *menu-stream*)
	(clim:stream-set-cursor-position* *menu-stream* (+ x 20) y)
	(clim:present (list 0 *box-size* 0 0 'Abort (+ x 20) y)
		      'check-box :stream *menu-stream*)
	)
      )

; Crop down the window to the size we've used

    (clim::window-set-inside-size *menu-stream*
				  (+ max-x 20) ; Leave room for scroll bar
				  y-extent )

    (setf (clim:window-label *menu-stream*) "Multiple Choose Menu")
    (clim:window-expose *menu-stream*)

    key-x
    ) )


(defun SHOW-STATE-ARRAY (message boxes)
  (multiple-value-bind (n-items n-keywords)
		       (values-list (array-dimensions boxes))
    (princ message *terminal-io*)
    (terpri *terminal-io*)

    (loop for a from 0 below n-items
	  do
	  (loop for b from 0 below n-keywords
		do (format *terminal-io* "~A " (aref boxes a b)) )
	  (terpri *terminal-io*) ) ) )

#|
(defun test ()
  (setq headings '(:backup :output-binary :output-ascii :purge))
  (multiple-choose " Grasper Graph: "
		   (loop for file in '("File 1" "File 2" "File 3")
			 collect (list file file headings))
		   '((:Backup "Backup" nil nil nil nil)
		     (:Output-Binary "Output-Binary" nil nil nil nil)
		     (:Output-Ascii "Output-Ascii" nil nil nil nil)
		     (:Purge "Purge" nil nil nil nil))) )
|#
