Date: Tue, 27 Oct 1992 18:34+0200
From: Jeff Close <jclose@chesapeake.ads.com>
To: nrb!keunen
In-Reply-To: Vincent Keunen's message of Tue, 27 Oct 1992 10:10+0200 <19921027081015.2.KEUNEN@nrbmi1.ia.nrb.be>
Subject: Re: about the draw-icon examples
Reply-To: jclose@ads.com



Vincent,

Here's a fifth --  you asked me to send this:

-----
;;; -*- mode: common-lisp; package: cl-user; base:10. -*-

;;;****************************************************************
;;; icons.lisp
;;; 7/92
;;; J. Jeffrey Close
;;;
;;; This contains code for bitmap (Clim 'pattern') icons including
;;; stationary icons, mouse-movability, command-icons (buttons),
;;; and dragged-icons (drag icon and execute function on a target)
;;;
;;; Mods:
;;;  - changed dragged-icon so target form is run in place of command
;;;    form of ocmmand-icon, rather than both running on different clicks
;;;  - there is a commented-out reference in icon initialize-instance
;;;    to a function called bitmap-to-pattern; this is a set of functions
;;;    I wrote that convert X bitmaps to Clim patterns
;;;****************************************************************

(defvar *icon-list* nil)

(defvar *icon-bitmap-directory* nil)

(clim:define-command-table icon-command-table
    :inherit-from (clim:user-command-table))

(defclass icon nil
	  ((name :initform nil :initarg :name :accessor name)
	   (file :initform nil :initarg :file :accessor file)
	   (bitmap :initform nil :initarg :bitmap :accessor bitmap)
	   (presentation :initform nil :initarg :presentation :accessor presentation)
	   (label-p :initform t :initarg :label-p :accessor label-p)
	   (height :initform nil :initarg :height :accessor height)
	   (width :initform nil :initarg :width :accessor width)
	   (x :initform 0 :initarg :x :accessor x)
	   (y :initform 0 :initarg :y :accessor y)
	   (icon-window :initform nil :initarg :icon-window :accessor icon-window)
	   ))

;;; this takes care of the filename and bitmap initialization
;;; if a filename isn't given, tries to create one from the name and
;;; looks for it; if not found, asks for one.
;;; then it loads the bitmap and sets width and height info
;;; this does NOT position the icon in a window
(defmethod initialize-instance :after ((self icon) &rest ignore)
  (push self *icon-list*)
  )

#|
(defmethod initialize-instance :after ((self icon) &rest ignore)
  (let ((filename)
	)
  (unless (file self)
    (setq filename			;  hyphenate the name to produce file
      (substitute #\- #\space
		  (format nil "~A~A.bm" *icon-bitmap-directory* (name self))))
    (if (open filename :direction :probe)
	(setf (file self) filename)
      (progn
	(format t "Please enter a file name for the bitmap for icon ~A"
		(name self))
	(setq filename (read-line))
	;;; should check here for good input
	(setf (file self) filename))
      ))
  (when (file self)
    (multiple-value-bind (pattern wid ht)
	(bitmap-to-pattern (file self))
      (cond (pattern
	     (setf (bitmap self) pattern)
	     (setf (width self) wid)
	     (setf (height self) ht))
	    (t nil))
      ))
  (push self *icon-list*)
  ))
|#

;;; if displayed elsewhere, erases the icon, then
;;; sets the coordinates, places it icon in a window, and presents it
(defmethod place-in-window ((self icon) window x y)
  (when (presentation self)
    (clim:erase-output-record (presentation self) (icon-window self)))
  (setf (icon-window self) window)
  (setf (x self) x)
  (setf (y self) y)
  (clim:present self `((,(type-of self)) :with-label t)
		:stream window
		:view clim:+dialog-view+)
  )

(clim:define-presentation-type icon () :options ((with-label t)))
      
;;; present as text
(clim:define-presentation-method clim:present
    (icon (type icon) stream (view clim:textual-view) &key (with-label t))
  (clim:with-text-face (:bold)
	(format stream "~A Icon" (name icon))
	))

;;; draws the icon bitmap in its window at its coords, optionally with label
(defmethod draw ((self icon) &key (stream (icon-window self))
				  (x (x self))
				  (y (y self))
				  (with-label))
  (when stream
    (clim:draw-icon* stream (bitmap self) x y)
    (when with-label
      (draw-label self :stream stream :x x :y y)))
  )

;;; erases the icon output record and clears the slots
(defmethod erase ((self icon) &rest ignore)
    (clim:erase-output-record (presentation self) (icon-window self))
    (setf (icon-window self) nil)
    (setf (presentation self) nil))

;;; draws a bold label of the name under the icon
(defmethod draw-label ((self icon)
		       &key (stream (icon-window self))
			    (x (x self))
			    (y (y self))
			    (style '(:sans-serif :bold :normal))
			    )
  (let* ((textx (+ x (/ (width self) 2.0)))
	 (texty (+ y (height self)))
	 )
	(clim:draw-text* stream (name self)
			 textx texty
			 :align-x :center
			 :align-y :top
			 :text-style style
			 :ink clim:+flipping-ink+)
	))

;;; present in a dialog view, rather than text
(clim:define-presentation-method clim:present
    (i (type icon) stream (view clim:dialog-view) &key (with-label t))
  (setf (presentation i)
    (clim:with-output-as-presentation (:object i
					       :type (type-of i)
					       :stream stream
					       :single-box t)
      (draw i :stream stream :with-label with-label)))
  )

;;; presentation highlighting method for icons
;;; bolds the icon boundary
(clim:define-presentation-method clim:highlight-presentation ((type icon) record stream state)
  state
  (multiple-value-bind (xoff yoff)
      (clim:convert-from-relative-to-absolute-coordinates
       stream (clim:output-record-parent record))
    (clim:with-bounding-rectangle* (left top right bottom)
	(clim::draw-rectangle* stream
			       (+ left xoff)
			       (+ top yoff)
			       (+ right xoff)
			       (+ bottom yoff)
			       :ink clim:+flipping-ink+
			       :filled nil
			       :line-thickness 3)
	)))

;;; This allows the user to drag the icon on a window with the pointer.
;;; It first puts the pointer on the icon (it's already there if this is
;;; being called from a presentation action or translator), erases the old
;;; output record, then drags the old presentation around with the pointer.
;;; The move terminates when the user releases, re-presents (sic) the icon,
;;; and sets the icon's coordinates to the pointer position
(defmethod move ((self icon) &rest ignore)
  (with-slots ((icon-window icon-window)
	       (presentation presentation)
	       (x x)
	       (y y)) self
    (when presentation
      (clim:stream-set-pointer-position* icon-window x y)
      (clim:erase-output-record presentation icon-window) ; erase the current pic
      (clim:dragging-output (icon-window t t) ; draw and drag around the icon
			    (clim:present self `((,(type-of self)))
					  :stream icon-window
					  :view clim:+dialog-view+)
			    )
      (clim:erase-output-record presentation icon-window)
      (multiple-value-bind (px py)
	  (clim:stream-pointer-position* icon-window)
	(setf x px)
	(setf y py))
      (clim:present self `((,(type-of self)))
		    :stream icon-window
		    :view clim:+dialog-view+)
      (clim:stream-finish-output icon-window)
      )))

(clim:define-command (com-move-icon
		      :menu nil
		      :command-table icon-command-table)
    ((i 'icon))
  (move i)
  )

(defmethod cycle-highlight ((self icon) &optional (num 5) (delay 1))
  (dotimes (i num)
    (clim:set-highlighted-presentation (icon-window self) (presentation self))
    (sleep (/ delay 2.0))
    (clim:unhighlight-highlighted-presentation (icon-window self))
    (sleep (/ delay 2.0))
    ))

;;; *************************************************************************
;;; command-icons
;;; command icons are like buttons, and have a left-click function to execute
;;; buttons have a function  to execute when pressed, which takes the
;;; arguments (object frame window context)

(defclass command-icon (icon)		; AKA a "button"
	  ((command-form :initform nil :initarg :command-form
			     :accessor command-form)
	   (command-documentation :initform (make-string 0)
			  :initarg :command-documentation
			  :accessor command-documentation)
	   ))

(clim:define-presentation-type command-icon () :options ((with-label t)))

(clim:define-command (com-execute-icon-command
		      :menu nil
		      :command-table icon-command-table)
    ((icon 'command-icon))
  (let ((form (command-form icon)))
    (when form
      (cond ((symbolp form)
	     (funcall form))
	    ((listp form)
	     (eval form))
	    ))
  ))

(clim:define-presentation-to-command-translator execute-command
    (command-icon com-execute-icon-command icon-command-table
		  :gesture :select
		  ;;;tester is needed because of translator-ordering bug in1.1
		  :tester ((object) (eql (type-of object) 'command-icon))
		  :pointer-documentation
		  ((object presentation context-type frame stream)
		   (format stream "~A" (command-documentation object))
		   )
		  :documentation
		  ((object presentation context-type frame stream)
		   (format stream "~A" (command-documentation object))
		   )
		  :menu t)
  (object presentation context-type frame event window x y)
  (list object)
  )

;;; *************************************************************************
;;; dragged-icons
;;; dragged-icons can be selected and dragged onto another part of the screen
;;; including across multiple windows or panes;

;;; if they are dragged and dropped on something, then that object is found
;;; (actually, its presentation) and the "target function" is run on
;;; that object 
;;; I've considered making them dual-function; built on command-icons, they
;;; could execute their stationary button command if not dragged

(defclass dragged-icon (command-icon)
	  ((target :initform nil
		   :initarg :target
		   :accessor target)
	   (target-form :initform nil
			    :initarg :target-form
			    :accessor target-form)
	   ))

(clim:define-presentation-type dragged-icon () :options ((with-label t)))
;;; the drag method allows the user to drag an icon and redraws
;;; the picture as it is tracking; it does not move the original icon,
;;; and when it is done it erases the last drag picture
;;; RETURNS the WINDOW, the TERMINATING COORDINATES where the drag terminated,
;;; and the TARGET PRESENTATION it was over when it terminated
;;; in addition, the target presentation is stored in 'target'

(defun drag-icon (icon win &rest ignore)
  (when (presentation icon)
    (let* ((frame clim:*application-frame*)
	   (top-win (clim:frame-top-level-window frame))
	   (old-win win)
	   (old-x (x icon))
	   (old-y (y icon))
	   (rec (clim:with-new-output-record (win)
		  (draw icon win)))
	   (target)
	   (name)
	   )
      (clim:stream-set-pointer-position* win (x icon) (y icon))
;;; Now, start dragging
      (clim:tracking-pointer
       (win :multiple-window t)
;;; null the target because we've moved off, and redraw
       (:pointer-motion (window x y)
			(setq target nil)
			(when (or (not (eql old-win window))
				  (> (abs (- x old-x)) 5)
				  (< (abs (- y old-y)) 5))
			  (clim:stream-force-output window)
			  (when rec
			    (clim:erase-output-record rec old-win))
			  (setq old-win window old-x x old-y y)
			  (setq rec (clim:with-new-output-record (window)
				      (draw icon window x y)))
			  ))
	 ;;; record if we're over a presentation
       (:presentation (presentation window x y)
		      (setq target presentation)
		      (when (or (not (eql old-win window))
				(> (abs (- x old-x)) 5)
				(< (abs (- y old-y)) 5))
			(clim:stream-force-output window)
		      (when rec
			(clim:erase-output-record rec old-win)
			(setq rec nil))
			(setq old-win window old-x x old-y y)
			(setq rec (clim:with-new-output-record (window)
				    (draw icon window x y)))
			)
		      )
	 ;;; wrap everything up, erase the last output, and return
       (:pointer-button-press (event x y)
				(when rec
				  (clim:erase-output-record rec old-win))
				(clim:stream-finish-output
				 (clim:event-window event))
				(setf (target icon) target)
				(return
				  (values (clim:event-window event)
					  x y target))
				)
       ))
    ))

(defmethod drag ((self dragged-icon) &rest ignore)
  (drag-icon self (icon-window self))
  )

(clim:define-command (com-drag-icon
		      :menu nil
		      :command-table icon-command-table)
    ((icon 'dragged-icon))
  (multiple-value-bind (win x y target)
      (drag icon)
    (let ((form (target-form icon)))
      (when (and form target)
	(cond ((symbolp form)
	       (funcall form target))
	      ((listp form)
	       (eval form))
	      ))
      )))

(clim:define-presentation-to-command-translator drag-icon-command
    (dragged-icon com-drag-icon icon-command-table
		  :gesture :select
		  :tester ((object) (eql (type-of object) 'dragged-icon))
		   :pointer-documentation
		   ((object presentation context-type frame stream)
		    (format stream "~A" (command-documentation object))
		    )
		   :documentation
		   ((object presentation context-type frame stream)
		    (format stream "~A" (command-documentation object))
		    )
		   :menu t)
  (object)
  (list object)
  )

;;; ***********************************************************************
;;; miscellaneous icon functions

;;; returns an icon from a name
(defun find-icon (name &rest ignore)
  (find name *icon-list* :key #'name :test #'string-equal)
  )

;;; erase all icons from their windows
(defun erase-icons ()
  (dolist (i *icon-list*)
    (erase i)))

;;; erases the icon and removes it from the icon master list
(defmethod kill-icon ((self icon) &rest ignore)
  (when (and (presentation self) (icon-window self))
    (erase self))
  (setq *icon-list* (remove self *icon-list*))
  )

;;; kills all icons on the list
(defun kill-icons ()
  (dolist (i *icon-list*)
    (kill-icon i)))

;;; just a pass-through function to a Make-instance call, 
;;; avoids having to use keyword args.
(defmacro create-button (name &optional (documentation (string name)) command-form  &rest args)
  `(make-instance 'command-icon :name ,name
		  :command-form ,command-form
		  :command-documentation ,documentation
		  ,@args)
)

(defmacro create-drag-button (name &optional (documentation (string name)) target-form command-form  &rest args)
  `(make-instance 'dragged-icon :name ,name
		  :command-form ,command-form
		  :command-documentation ,documentation
		  :target-form ,target-form
		  ,@args)
)






----------------------------------
