(in-package :clim-user)

;;; A simple CLOS inspector/browser.
;;; Use the (CLOS-INSPECTOR) function to start it.
;;; Use the Inspect Class and Inspect Instance commands to start
;;; things off.  You can then navigate around by clicking on
;;; displayed objects/classes.
;;; For example, try Inspect Class RECTANGLE.
;;; Then click middle on the class RECTANGLE in any of the various
;;; places it is displayed:
;;;
;;; Click left on something to make it the "current object".
;;; Click middle on a class to get a textual display of the subclass
;;;     tree.
;;; Click shift-middle to get a superclass display.
;;; Click control-middle to get a subclass graph.
;;; Click control-shift-middle to get a superclass graph.

;;; Bugs:
;;;
;;; The Lisp printer is not integrated well enough to be able to click
;;; on the random output of calling the DESCRIBE function.
;;;
;;; Lengthly displays (such as all the subclasses of STANDARD-OBJECT)
;;; take a lot of time.  The right thing is to display only the first
;;; screenful of data until more is requested.


;;; CLOS interfaces

;;; Kludges to list all CLOS classes.  The answer is cached in
;;; *ALL-CLASSES*, which means that you lose if you define a
;;; new class and don't clear the cache.
(defvar *all-classes* nil)

(defun compute-all-classes ()
  (labels ((subclasses (class)
	     (let ((subs (apply #'append (mapcar #'subclasses
						 (class-direct-subclasses class)))))
	       (if (symbolp (class-name class))
		   (cons class subs)
		   subs))))
    (subclasses (find-class 'clos:standard-object))))

(defun all-classes ()
  (or *all-classes*
      (setq *all-classes* (compute-all-classes))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Presentation types.

;;; A type that takes as input a Lisp form and returns its value.
(define-presentation-type clos-instance ())

;;; Avoid errors in printing uninitialized instances by
;;; using our own output format.
(define-presentation-method present (object (type clos-instance) stream view
					    &key acceptably)
  (declare (ignore view acceptably))
  (format stream "Instance of class ~A" (class-name (class-of object))))

(define-presentation-method accept ((type clos-instance) stream view &key)
  (declare (ignore view))
  (let ((instance-spec (accept 'expression :stream stream :prompt nil)))
    (handler-case (let ((instance (eval instance-spec)))
		    instance)
		  (error (e) (format t "~&Got an error: ~A" e)))))

;;; A type for CLOS classes.  The accept/present methods are duals
;;; (i.e. if you give the output of the present method to the accept
;;; method to parse it will give you back the original object).
(define-presentation-type clos-class ())

;;; Just show the name on output.
(define-presentation-method present (object (type clos-class) stream view
					    &key acceptably)
  (declare (ignore view acceptably))
  (format stream "~S" (class-name object) stream))

;;; Supports completion over class names.
(define-presentation-method accept ((type clos-class) stream view &key)
  (declare (ignore view))
  (complete-input stream
		  #'(lambda (string action)
		      (clos-class-complete string action))))

;;; This uses the completion substrate over the list of all classes.
(defun clos-class-complete (string action)
  (complete-from-possibilities string (all-classes) '()
			       :action action
			       :name-key  #'(lambda (class) (string (class-name class)))
			       :value-key #'identity))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; The framework

(eval-when (compile load eval)
  (defparameter *ci-text-style* (parse-text-style '(:fix :roman :small))))

(define-application-frame clos-inspector ()
  ;; Slot to hold all the inspected things.
  ((inspectees :initform nil :accessor inspector-inspectees))
  ;; Four panes in a basic layout
  (:panes ((interactor :interactor
		       :default-text-style *ci-text-style*)
	   (inspectees :application
		       :display-function 'display-inspectees
		       :end-of-page-action ':allow
		       :default-text-style *ci-text-style*)
	   (info :application
		 :default-text-style *ci-text-style*)
	   (mouse-doc :pointer-documentation)))
  (:layout ((main (:column 1
			   (:row 1/4
				 (interactor 1/2)
				 (inspectees :rest))
			   (mouse-doc :compute)
			   (info :rest)))))
  (:command-definer t))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Display functions

;;; Run after each command to show the updated list of things
;;; that have been inspected.
(defmethod display-inspectees ((frame clos-inspector) pane)
  ;; The elements of the inspectees list are themselves lists 
  ;; of two things, the object and the type.
  (dolist (inspectee (inspector-inspectees frame))
    (present (first inspectee) (second inspectee) :stream pane)
    (terpri pane)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Utilities

;;; Use this function to set the currently-inspected object.
;;; It puts that object at the top of the inspectees list, and
;;; shows a description of it in the info window.
(defun set-inspected-object (frame object)
  ;; Modify the inspectees list.
  (cond ((find object (inspector-inspectees frame)
	       :test #'equalp)
	 (setf (inspector-inspectees frame)
	       (cons object (delete object (inspector-inspectees frame)
				    :test #'equalp))))
	(t (push object (inspector-inspectees frame))))
  ;; Display the description of the object.
  (let ((pane (get-frame-pane *application-frame* 'info)))
    (window-clear pane)
    (ecase (second object)
      (clos-class (describe-class (first object) pane))
      (clos-instance (describe-instance (first object) pane))
      (random (describe (first object) pane)))))  

;;; Simple class describer.
(defun describe-class (class stream)
  (format stream "Name:~30T")
  (present class 'clos-class :stream stream)
  (terpri stream)
  (format stream "Prototype:~30T")
  (present (class-prototype class) 'clos-instance :stream stream)
  (terpri stream)
  (when (clos:class-direct-slots class)
    (format stream "Direct slots:~30T")
    (formatting-item-list (stream)
			  (dolist (slot (clos:class-direct-slots class))
			    (formatting-cell (stream)
					     (format stream "~A" slot))))
    (terpri stream))
  (format stream "Precedence List:~30T")
  (formatting-item-list (stream)
			(dolist (super (class-precedence-list class))
			  (formatting-cell (stream)
					   (present super 'clos-class :stream stream)))))

;;; Simple instance describer.  Not much to describe, really...
(defun describe-instance (instance stream)
  (format stream "Object:~30T")
  (present instance 'clos-instance :stream stream)
  (terpri stream)
  (format stream "Class:~30T")
  (present (class-of instance) 'clos-class :stream stream)
  (terpri stream)
  (handler-case 
    (describe instance stream)
    ;; Can't print the error message, 'cause it may error as well
    ;; (e.g. when you have an unitialized slot referenced by the
    ;; PRINT-OBJECT method).
    (error (e) (format stream "~%Got an error calling DESCRIBE"))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Application commands

(define-clos-inspector-command (com-quit :name "Quit")
    ()
  (frame-exit *application-frame*))

(define-clos-inspector-command (com-clear :name "Clear")
    ()
  (let ((frame *application-frame*))
    (setf (inspector-inspectees frame) nil)
    (window-clear (get-frame-pane frame 'info))))


;;; Command to "kick off" the inspection of an instance.  It
;;; prompts for a Lisp expression, which it evaluates to get
;;; the CLOS instance.
(define-clos-inspector-command (com-inspect-instance :name "Inspect Instance")
    ((instance 'clos-instance :gesture :select))
  (let ((frame *application-frame*))
    (if (typep instance 'clos:standard-object)
	(set-inspected-object frame (list instance 'clos-instance))
	(set-inspected-object frame (list instance 'random)))))

;;; Similar command to inspect a class.  Does completion over class
;;; names.
(define-clos-inspector-command (com-inspect-class :name "Inspect Class")
    ((class 'clos-class :gesture :select))
  (let ((frame *application-frame*))
    (set-inspected-object frame (list class 'clos-class))))

(define-gesture-name :show-subclasses :button :middle)
(define-gesture-name :show-superclasses :button :middle :shifts (:shift))

;;; Show a textual description of the subclasses of a class.
(define-clos-inspector-command (com-show-subclass-tree :name "Show Subclass Tree")
    ((class 'clos-class :gesture :show-subclasses))
  (show-class-tree class #'class-direct-subclasses))

;;; Show a textual description of the superclasses of a class.
(define-clos-inspector-command (com-show-superclass-tree :name "Show Superclass Tree")
    ((class 'clos-class :gesture :show-superclasses))
  (show-class-tree class #'class-direct-superclasses))

;;; Common workhorse for above.
(defun show-class-tree (class next-generation-fun)
  (let ((pane (get-frame-pane *application-frame* 'info)))
    (window-clear pane)
    ;; Walks the class tree, up or down.
    (labels ((show-level (class level)
	       ;; Make the whole line sensitive as the class.
	       (with-output-as-presentation (:stream pane
						     :object class
						     :type 'clos-class)
		 (format pane "~vT~A~%" (* level 2) (class-name class)))
	       (dolist (x (funcall next-generation-fun  class))
		 (show-level x (1+ level)))))
      (show-level class 0))))

(define-gesture-name :graph-subclasses :button :middle :shifts (:control))
(define-gesture-name :graph-superclasses :button :middle :shifts (:control :shift))

;;; Show a graph of the subclasses of a class.
(define-clos-inspector-command (com-graph-subclass-tree :name "Graph Subclass Tree")
    ((class 'clos-class :gesture :graph-subclasses))
  (graph-class-tree class #'class-direct-subclasses))

;;; Show a graph of the superclasses of a class.
(define-clos-inspector-command (com-graph-superclass-tree :name "Graph Superclass Tree")
    ((class 'clos-class :gesture :graph-superclasses))
  (graph-class-tree class #'class-direct-superclasses))

(defun graph-class-tree (class next-generation-fun)
  (let ((pane (get-frame-pane *application-frame* 'info)))
    (window-clear pane)
    (format-graph-from-roots (list class)
			    #'(lambda (object stream)
				(present object 'clos-class :stream stream))
			    next-generation-fun
			    :merge-duplicates t
			    :move-cursor nil
			    :stream pane)))

(defvar *root* nil)
(defvar *inspector* nil)

;;; Interface function to kick things off.
(defun clos-inspector (&key reset)
  (unless *root*
    (setq *root* (open-root-window :clx)))
  (unless (or *inspector* reset)
    (setq *inspector*
	  (make-application-frame 'clos-inspector
				  :parent *root*
				  :left 50 :top 50 :width 1000 :height 700)))
  (run-frame-top-level *inspector*))

