/* File: plhip_intf.pl
 * Created:  6-Dec-93
 * Last Update: 10-Dec-93
 * Author: Afzal Ballim
 * Purpose: public interface to the plhip system.
 * Needs: plhip_int.pl
 */

/*

    The syntax of plhip rules is given below in a psuedo DCG form.  In
    addition to these definitions, each plhip_body must contain at least one
    clause that is either a normal or head rule.

    plhip_rule --> std_plhip_rule.
    plhip_rule --> delete_plhip_rule.

    std_plhip_rule    -->  atom ~~> plhip_body.
    delete_plhip_rule --> -atom ~~> plhip_body.

    plhip_body --> plhip_clause.           %
    plhip_body --> plhip_clause,plhip_body. % relavtive positioning
    plhip_body --> plhip_clause:plhip_body. % adjacency
    plhip_body --> plhip_body;plhip_body.   % disjunction

    plhip_clause --> atom.               % normal rule
    plhip_clause --> *atom.              % head rule
    plhip_clause --> -atom.              % specific delete rule
    plhip_clause --> @ terminal          % terminal
    plhip_clause --> [].                 % general delete
    plhip_clause --> (? plhip_body ?).    % optional
    plhip_clause --> !.                  % prolog cut
    plhip_clause --> {P}.                % embedded prolog code

*/

/************************************************************
 *
 * Public interface
 *
 ************************************************************/

phlip_version('1.0').

/*===========================================================
 * Predicate: plhip_phrase/2
 * Modes: plhip_phrase(+Cat,+Sent)
 * 
 * Purpose: succeeds if Sent is a sentence of type Cat in
 *          a plhip grammar. This is actually an interface.
 *          Similar syntax to the prolog DCG phrase/2.
 * 
 * Licensed Use: 
 * 
 * Created: 25-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_phrase(Cat,Sent)   
:-
    plhip_phrase(Cat,Sent,_S,_E,_C).

/*===========================================================
 * Predicate: plhip_phrase/5
 * Modes: plhip_phrase(+Cat,+Sent,-Start,-End,-Cover)
 * 
 * Purpose: As for plhip_phrase/2, but returns the span of
 *          the phrase in the sentence (indexed from zero, 
 *          End being one beyond the actual end of the span)
 *          and the number of tokens covered in this span.
 * 
 * Licensed Use: 
 * 
 * Created: 25-Nov-93
 * Last Update: 25-Nov-93
 *=========================================================*/

plhip_phrase(Cat,Sent,S,E,C)
:-
    var(Cat),!,
    format("~q illegal arguments~n",[plhip_phrase(Cat,Sent,S,E,C)]),
    fail.

plhip_phrase(Cat,Sent,SS,SE,C)
:-
    plhip_clear_chart,
    FSent =.. [s|Sent],
    length(Sent,Ntok),
    End is Ntok+1,
    plhip_clause(Cat,Goal,[FSent,1,End,SS,SE,C,_,_]),
    call(Goal).

/*===========================================================
 * Predicate: plhip_clear_chart/0
 * Modes: 
 * 
 * Purpose: clears the chart and resets the success counter.
 * 
 * Licensed Use: 
 * 
 * Created: 22-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_clear_chart :-
    retractall(plhip_known(_,_,_,_,_,_,_,_,_)),
    retractall(plhip_known_fail(_,_,_,_,_)),
    retractall(plhip_known_exhausted(_,_,_,_,_)),
    plhip_success_ref(0).

/*===========================================================
 * Predicate: plhip_phrase_rp/2,plhip_phrase_rp/5
 * Modes: 
 * 
 * Purpose: as plhip_phrase/2, but does not delete information
 *          from previous parses.
 * 
 * Licensed Use: 
 * 
 * Created: 18-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_phrase_rp(Cat,Sent)
:-
    plhip_phrase_rp(Cat,Sent,_SS,_SE,_C).

plhip_phrase_rp(Cat,Sent,SS,SE,C)
:-
    FSent =.. [s|Sent],
    length(Sent,Ntok),
    End is Ntok+1,
    plhip_clause(Cat,Goal,[FSent,1,End,SS,SE,C,_,_]),
    call(Goal).

/*===========================================================
 * Predicate: plhip_mc_phrases/4
 * Modes: plhip_mc_phrase(+Cat,+Sent,-Cover,-P)
 * 
 * Purpose: find the maximal coverage set of Sent by Cat
 * 
 * Licensed Use: 
 * 
 * Created: 18-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_mc_phrases(Cat,Sent,Cover,P)
:-
    setof((Cat,S,E,Cover),
          plhip_phrase_rp(Cat,Sent,S,E,Cover),
          P1),!,
    plhip_max_cov(P1,0,[],Cover,P).

plhip_max_cov([],C,P,C,P):- !.
plhip_max_cov([(HCat,S,E,HCov)|T],Mi,Pi,C,P)
:-
    HCov = Mi,!,
    plhip_max_cov(T,Mi,[(HCat,S,E)|Pi],C,P);
    HCov < Mi,!,
    plhip_max_cov(T,Mi,Pi,C,P);
    plhip_max_cov(T,HCov,[(HCat,S,E)],C,P).


/*===========================================================
 * Predicate: plhip_minmax_phrases/4
 * Modes: plhip_minmax_phrase(+Cat,+Sent,-Span,-MinMax)
 * 
 * Purpose: MinMax is the set of minimal spanning but maximum
 *          coverage parses of Sent under Cat.
 * 
 * Licensed Use: 
 * 
 * Created: 22-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_minmax_phrases(Cat,Sent,Span,MinMax)
:-
    setof((Cat,S,E,Cover),
          plhip_phrase_rp(Cat,Sent,S,E,Cover),
          P1),!,
    plhip_max_cov(P1,0,[],Cover,Pc),
    plhip_min_span(Pc,_,[],Span,MinMax).

plhip_min_span([],MS,P,MS,P).
plhip_min_span([(Cat,S,E)|Cats],MS,MM,Span,MinMax)
:-
    ThisSpan is E-S,
    (ThisSpan = MS,!,
     plhip_min_span(Cats,MS,[(Cat,S,E)|MM],Span,MinMax)
    ;ThisSpan < MS,!,
     plhip_min_span(Cats,ThisSpan,[(Cat,S,E)],Span,MinMax)
    ;ThisSpan > MS,!,
     plhip_min_span(Cats,MS,MM,Span,MinMax)
    ).


/*===========================================================
 * Predicate: plhip_cv_phrase/2
 * Modes: 
 * 
 * Purpose: As plhip_phrase, but ensures complete coverage
 * 
 * Licensed Use: 
 * 
 * Created: 19-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_cv_phrase(Cat,Sent)
:-
    FSent =.. [s|Sent],
    length(Sent,Ntok),
    End is Ntok+1,
    plhip_clause(Cat,Goal,[FSent,1,End,_SS,_SE,Ntok,_,_]),
    call(Goal).

/*===========================================================
 * Predicate: plhip_seq_phrase/3
 * Modes: plhip_seq_phrase(+Cat,+Sent,-Seq
 * 
 * Purpose: Seq is a seqence of parses across Sent, e.g.,
 *          Seq=[S0,S1,S2] means that S0 covers s0 to e0
 *          in Sent, etc, and ei <= s(i+1).
 * 
 * Licensed Use: 
 * 
 * Created: 22-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_seq_phrase(Cat,Sent,Seq)
:-
    setof((Cat,S,E,Cover),
          plhip_phrase_rp(Cat,Sent,S,E,Cover),
          P1),!,
    plhip_seq_in(P1,0,Seq).

plhip_seq_in(P,Ep,[(Pr,S,E,C)|Seq])
:-
    member((Pr,S,E,C),P),
    Ep =< S,
    plhip_delete_dup((Pr,S,E,C),P,Prest),
    plhip_seq_in(Prest,E,Seq).

plhip_seq_in(_,X,[]) :- X =\= 0.

plhip_delete_dup(_,[],[]).
plhip_delete_dup(P,[P|T],Td)
:-  
    !,plhip_delete_dup(P,T,Td).
plhip_delete_dup(P,[Q|T],[Q|Td])
:-
    !,plhip_delete_dup(P,T,Td).

/*===========================================================
 * Predicate: plhip_maxT_phrases/3
 * Modes: plhip_maxT_phrases(+Cat,+Sent,-MaxT)
 * 
 * Purpose: returns phrases that cover the maximum threshold.
 *          On backtracking it returns the phrases that cover
 *          the next highest threshold.
 * 
 * Licensed Use: 
 * 
 * Created: 26-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_maxT_phrases(P,Sent,MaxT)
:-
    plhip_clear_chart,
    setof((T-P),
        (A,B,C)^(plhip_phrase_rp(P,Sent,A,B,C),
                 T is C/(B-A)),
        AllPT),
    keysort(AllPT,TPs),
    plhip_maxt(TPs,0,[],MaxT).

% last
plhip_maxt([],_,MaxT,MaxT).
% first
plhip_maxt([T-P|TPs],0,[],MaxT)
:-
    !,plhip_maxt(TPs,T,[P,T],MaxT).
%next same
plhip_maxt([T-P|TPs],T,Ps,MaxT)
:-
    plhip_maxt(TPs,T,[P|Ps],MaxT).
% next different
plhip_maxt([T0-P|TPs],T,_Ps,MaxT)
:-
    T0 =\= T,
    plhip_maxt(TPs,T0,[P,T0],MaxT).
plhip_maxt([T0-_P|_TPs],T,MaxT,MaxT)
:-
    T0 =\= T.

/*===========================================================
 * Predicate: plhip_success/1
 * Modes: plhip_success(?Num)
 * 
 * Purpose: lists the successful calls to clause Num of the grammar.
 *          If Num is unbound then lists all successful clauses.
 * 
 * Licensed Use: 
 * 
 * Created: 26-Nov-93
 * Last Update:  8-Dec-93
 *=========================================================*/

plhip_success :- plhip_success(_).
plhip_success(Num)
:-
    setof((Num,Cov,So,FS,FE),
		  (Frz,Ref,Rf,Sent)^
			plhip_known(Num,Ref,Rf,Sent,FS,FE,Cov,Frz,So),
		  Succs),
    plhip_list(Succs).

/*===========================================================
 * Predicate: plhip_ms_success/0,/1,/2
 * Modes: plhip_ms_success(?Num)
 * 
 * Purpose: as plhip_success but only lists those clauses that
 *          are not used in other succesful clauses.
 * 
 * Licensed Use: 
 * 
 * Created:  6-Dec-93
 * Last Update: 
 *=========================================================*/

% plhip_ms_success/0
plhip_ms_success :- plhip_ms_success(_).

% plhip_ms_success/1
plhip_ms_success(Num)
:-
    plhip_ms_success(Num,Succs),
    plhip_list(Succs).

% this could be improved
plhip_ms_success(Num,Succs)
:-
	setof(X,
		  (A,B,C,D,E,F,G,H,I)^(
			plhip_known(A,B,C,D,E,F,G,H,I),
			member(X,C)),
		  Incs),
	plhip_success_counter(Cntr),
	plhip_ms_succ_nincs(0,Cntr,Num,Incs,SuccsN),
	sort(SuccsN,Succs).

% reached the limit
plhip_ms_succ_nincs(Limit,Limit,_,_,[]) :- !.

% current is included
plhip_ms_succ_nincs(X,Limit,Num,[X|Incs],Succs)
:-
	NewX is X+1,!,
	plhip_ms_succ_nincs(NewX,Limit,Num,Incs,Succs).

% current is not included, so a potential match
plhip_ms_succ_nincs(X,Limit,Num,[Y|Incs],Succs)
:-
	!,
	(var(Num),
	 plhip_known(TNum,X,_Rf,_Sent,FS,FE,Cov,_Frz,So),
	 Succs=[(TNum,Cov,So,FS,FE)|Sc],
	 NewX is X+1,
	 plhip_ms_succ_nincs(NewX,Limit,Num,[Y|Incs],Sc)
	;
	 plhip_known(Num,X,_Rf,_Sent,FS,FE,Cov,_Frz,So),
	 Succs=[(Num,Cov,So,FS,FE)|Sc],
	 NewX is X+1,
	 plhip_ms_succ_nincs(NewX,Limit,Num,[Y|Incs],Sc)
	;
	 NewX is X+1,
	 plhip_ms_succ_nincs(NewX,Limit,Num,[Y|Incs],Succs)).

% all rest are not included
plhip_ms_succ_nincs(X,Limit,Num,[],Succs)
:-
	!,
	(var(Num),
	 plhip_known(TNum,X,_Rf,_Sent,FS,FE,Cov,_Frz,So),
	 Succs=[(TNum,Cov,So,FS,FE)|Sc],
	 NewX is X+1,
	 plhip_ms_succ_nincs(NewX,Limit,Num,[],Sc)
	;
	 plhip_known(Num,X,_Rf,_Sent,FS,FE,Cov,_Frz,So),
	 Succs=[(Num,Cov,So,FS,FE)|Sc],
	 NewX is X+1,
	 plhip_ms_succ_nincs(NewX,Limit,Num,[],Sc)
	;
	 NewX is X+1,
	 plhip_ms_succ_nincs(NewX,Limit,Num,[],Succs)).
	
% auxillary function
plhip_list([]).
plhip_list([(N,Cov,Co,FS,FE)|T])
:-
	format("(~q) [~q--~q) /~q ~~~~> ~q~n",
		[N,FS,FE,Cov,Co]),
    plhip_list(T).

/*===========================================================
 * Predicate: plhip_failure/1
 * Modes: plhip_failure(?Num)
 * 
 * Purpose: lists the failed calls to clause Num of the grammar. If
 *          Num is unbound then lists all failed clauses.
 * 
 * Licensed Use: 
 * 
 * Created: 26-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_failure:-plhip_failure(_).
plhip_failure(Num)
:-
    setof((Num,SS,SE,Fail),
		(Sent)^plhip_known_fail(Num,Sent,SS,SE,Fail),Fails),
    plhip_list2(Fails).


/*===========================================================
 * Predicate: plhip_exhaustion/1
 * Modes: plhip_exhaustion(?Num)
 * 
 * Purpose: lists the exhausted calls to clause Num of the grammar.
 *           If Num is unbound then lists all exhausted clauses.
 * 
 * Licensed Use: 
 * 
 * Created: 26-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_exhaustion:-plhip_exhaustion(_).
plhip_exhaustion(Num)
:-
    setof((Num,SS,SE,Exh),
		(Sent)^plhip_known_exhausted(Num,Sent,SS,SE,Exh),Exhs),
    plhip_list2(Exhs).


% failure/exhaust aux. func
plhip_list2([]).
plhip_list2([(N,SS,SE,Co)|T])
:-
	format("(~q) ~q-><-~q ~~~~> ~q~n",
		[N,SS,SE,Co]),
    plhip_list2(T).

/*===========================================================
 * Predicate: plhip_threshold/1
 * Modes: plhip_threshold(?T)
 * 
 * Purpose: the global threshold value is T
 * 
 * Licensed Use: 
 *      plhip_threshold(+T)  set threshold value to T
 *      plhip_threshold(-T)  T is the current threshold value
 * 
 * Created: 25-Nov-93
 * Last Update: 
 *=========================================================*/

plhip_threshold(T)
:-
    var(T),
    plhip_threshold_value(T).

plhip_threshold(T)
:-
    nonvar(T),
    retract(plhip_threshold_value(_)),
    assert(plhip_threshold_value(T)).

