/* lqdumpwid.c -- Copyright 1989, 1990 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* lqdumpwid - dump a WID entry.
 *
 * $Id: lqmkfree.c,v 1.1 92/02/15 04:03:19 lee Exp $
 */

#include "globals.h" /* defines and declarations for database filenames */
#include "error.h"

#include <stdio.h> /* stderr, also for fileinfo.h */
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <malloc.h>
#include "emalloc.h"
#include "fileinfo.h" /* for wordinfo.h */
#include "wordinfo.h"

extern int AsciiTrace;

#ifndef LINT
static char *Revision = "@(#) $Id: lqmkfree.c,v 1.1 92/02/15 04:03:19 lee Exp $";
#endif

char *progname = "lqtext/lqmkfree";

int
main(argc, argv)
    int argc;
    char *argv[];
{
    extern int optind, getopt();
    /** extern char *optarg; (unused at present) **/
    int ch;
    int ErrorFlag = 0;

    progname = argv[0];

    SetDefaults(argc, argv);

    while ((ch = getopt(argc, argv, "Zz:ahpxVv")) != EOF) {
	switch (ch) {
	case 'z':
	case 'Z':
	    break; /* done by SetDefaults(); */
	case 'V':
	    fprintf(stderr, "%s version %s\n", progname, Revision);
	    break;
	case 'v': /* same as -t 1 */
	    AsciiTrace = 1;
	    break;
	case 'x':
	    ErrorFlag = (-1);
	    break;
	case '?':
	    ErrorFlag = 1;
	}
    }

    if (ErrorFlag) {
	if (AsciiTrace) {
	    fprintf(stderr, "%s - dump a WID Index entry\n", progname);
	}
	fprintf(stderr, "Usage: %s [options] WID [WID...]\n", progname);
	fprintf(stderr, "%s: options are:\n", progname);
	DefaultUsage();
	exit( ErrorFlag > 0 ? 1 : 0); /* 0 means -x was used */
    }
    
    Init();

    for (; optind < argc; optind++) {
	t_WID w = (t_WID) atol(argv[optind]);

	if (w <= BiggestWID) {
	    DumpWid(w);
	} else {
	    Error(E_WARN, "maximum WID in \"%s\" is %ld - %ld ignored",
		WidIndexFile,
		BiggestWID,
		w
	    );
	}
    }

    CloseFiles();
    return 0;
}

Init()
{
    struct stat statbuf;
    long FreeSize;
    t_WID CurrentWID = 0;
    t_WID BiggestWID;

    if (stat(WidIndexFile, &statbuf) < 0) {
	Error(E_FATAL|E_SYS, "Couldn't stat index file \"%s\"", WidIndexFile);
    }
    BiggestWID = (t_WID) (statbuf.st_size / WIDBLOCKSIZE);

    OpenFiles();

}

static int WIDFile;
static int FreeFile;
static int DataFile;

#define S(String) ((String) ? ( (*(String)) ? String : "[empty]") : "[null]")

OpenFiles()
{
    OpenFile(&WIDFile, "Word Index File", WidIndexFile, "reading", O_RDONLY);
    OpenFile(&DataFile, "Data File", DataBase, "reading", O_RDONLY);
    OpenFile(&FreeFile, "Free Bit Map File", FreeFileName,
					"create and write", O_CREAT|O_RDWR);
}

OpenFile(Pointer, Description, Name, ModeName, Mode)
    int *Pointer;
    char *Name;
    char *Description;
    char *ModeName;
    int Mode;
{
    if (!Pointer || !Name || !*Name || !Description || !*Description ||
					!ModeName || !*ModeName) {
	Error(E_BUG,
	    "OpenFile: Name %x=%s, Desc %x=%s, Pointer %x, Mode %x=%s, 0%o",
	    Name, S(Name),
	    Description, S(Description),
	    Pointer,
	    ModeName, S(ModeName),
	    Mode
	);
    }

    if ((*Pointer = open(Name, Mode, 0666)) < 0) {
	Error(E_FATAL|E_SYS,
	    "Couldn't open %s \"%s\" for %s",
	    Description, Name, ModeName
	);
    }
}

CloseFiles()
{
    if (close(WIDFile) < 0) {
	Error(E_WARN|E_SYS,
	    "system error closing %d=\"%s\"", WIDFile, WidIndexFile);
    }
    if (close(DataFile) < 0) {
	Error(E_WARN|E_SYS,
	    "system error closing %d=\"%s\"", DataFile, DataBase);
    }
    if (close(FreeFile) < 0) { /* this one matters... */
	Error(E_FATAL|E_SYS,
	    "system error closing %d=\"%s\"", FreeFile, FreeFileName);
    }
}

DumpWID(w)
    t_WID w;
{
    char Block[WIDBLOCKSIZE + 5];
    long l;
    long total;
    char *q;

    if (lseek(WIDFile, w * WIDBLOCKSIZE, 0) < 0) {
	Error(E_SYS|E_WARN,
	    "Couldn't lseek to %ld in \"%s\" for WID %ld",
	    w * WIDBLOCKSIZE,
	    WidIndexFile,
	    w
	);
	return;
    }

    if (read(WidFile, Block, WIDBLOCKSIZE) != WIDBLOCKSIZE) {
	Error(E_SYS|E_WARN,
	    "WID %d: Couldn't read %d bytes from \"%s\" starting at %ld",
	    w,
	    WIDBLOCKSIZE,
	    WidIndexFile,
	    w * WIDBLOCKSIZE
	);
	return;
    }
    q = Block;
    l = (int) *q;
    ++q;
    printf("%ld: %*s: ", w, l, q);
    q += l;
    l = sReadNumber(&q);
    printf("  offset: %ld", l);
    total = sReadNumber(&q);
    printf("  total: %ld", total);
}

