/* smalldb.c -- Copyright 1989,1992 Liam R. E. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* Simple interface to start and end dbm.
 * You may also need to supply dbm_store() and dbm_fetch(), but these
 * should certainly be macros.
 *
 * $Id: smalldb.c,v 1.14 92/07/30 22:37:47 lee Exp $
 */

#include "globals.h"
#include "error.h"

#include <fcntl.h> /* for O_RDONLY, etc. */
#include "emalloc.h"
#include "smalldb.h"
#include "emalloc.h"

/* Actually we don't need this, but ANSI C requires it */
#ifdef __STDC__
# include <stdio.h>
#endif

extern int strcmp();
extern char *strcpy();

/* The physical database for the list of words, and for the list
 * of files, uses ndbm.
 * The advantage of this is that it takes only two file system accesses
 * to retrieve any data item (honest!).
 * It's also reasonably fast at insertion.
 * One disadvantage is that it doesn't cope if too many words have the
 * same (32-bit) hash function, although some of the publicly available
 * replacements such as the 4.4 BSD db package fix this.
 *
 * Since starting the database is expensive (two opens and a malloc),
 * I have a cache of DBM pointers and keep them open.  Versions of the
 * dbm routines that don't support more than one database will have to
 * have a cache-size of one!
 * I am not sure what the impact of this would be on performance; for
 * adding a new file it shouldn't be too bad, as the file list is examined
 * only once for each file, during reading, and the word database is looked
 * at (at least once for each distinct word) only on writing.
 * For retrieval, however, the word database will be looked at for each
 * word in the query, and the file database for (potentially) each match
 * of each word, so the requests will be more interspersed.
 * Under no circumstances is it acceptable to dispense with the cache, as
 * otherwise you will be doing (literally) thousands of calls to
 * open() and close() per second!
 *
 */

#undef startdb

#ifndef CACHE
# define CACHE 4 /* too many and we'll run out of file descriptors */
#endif

typedef struct s_DatabaseCache {
    char *Name;
    DBM *Value;
    struct s_DatabaseCache *Next;
    short NameLength;
} t_DatabaseCache;

static int MaxInCache = CACHE;
static int NumberInCache = 0;
static t_DatabaseCache *DatabaseCache;

/* FileFlags and Mode are passed to dbm_open */
static int FileFlags = O_RDONLY;
static int FileModes = 0;

void
lqWriteAccess()
{
    FileFlags = O_RDWR|O_CREAT;
    FileModes = 0664; /* owner and group write, others read only */
}

void
lqReadOnlyAccess()
{
    FileFlags = O_RDONLY;
    FileModes = 0664; /* owner and group write, others read only */
}

void
lqGetFileModes(Flagsp, Modesp)
    int *Flagsp;
    int *Modesp;
{
    if (!Flagsp || !Modesp) {
	Error(E_BUG, "lqGetFileModes(Flagsp=0x%x, Modesp=0x%x): %s zero",
	    Flagsp, Modesp,
	    (Flagsp) ? "Flagsp" : "Modesp"
	);
    }
    *Flagsp = FileFlags;
    *Modesp = FileModes;
}

DBM *
startdb(FilePrefix)
    char *FilePrefix;
{
    extern int errno;
    t_DatabaseCache *cp;
    int NameLength = strlen(FilePrefix);

    for (cp = DatabaseCache; cp; cp = cp->Next) {
	if (cp->NameLength == NameLength && cp->Value &&
				strcmp(cp->Name, FilePrefix) == 0) {
	    return cp->Value;
	}
    }

    /* assert: cp == 0 */

    /* not in the cache */

    /* if the cache is too big, close one entry, the last one */
    if (NumberInCache > MaxInCache) {
	t_DatabaseCache **cpp;

	for (cpp = &DatabaseCache; (*cpp); cpp = &(*cpp)->Next) {
	    if (!(*cpp)->Next) break;
	}

	if (*cpp && !(*cpp)->Next) {
	    /* Actually if this isn't true, MaxInCache is probably zero! */

	    if ((*cpp)->Value) {
		(void) dbm_close((*cpp)->Value);
		(*cpp)->Value = (DBM *) 0;
	    }
	    if ((*cpp)->Name) {
		(void) efree((*cpp)->Name);
		(*cpp)->Name = (char *) 0;
	    }
	    cp = (*cpp);
	    *cpp = (t_DatabaseCache *) 0; /* so it isn't pointed to any more */
	    --NumberInCache;
	} else {
	    Error(E_BUG,
		"startdb(%s) - cache is full up, none can be discarded",
		FilePrefix
	    );
	}
    }
    
    if (!cp) {
	cp = (t_DatabaseCache *) emalloc(sizeof(t_DatabaseCache));
    }

    errno = 0;

    if ((cp->Value = dbm_open(FilePrefix, FileFlags, FileModes)) == (DBM *) 0){
	extern char *getenv();
	char *p = getenv("LQTEXTDIR");

	if (!p) {
	    Error(E_WARN|E_SYS,
		"couldn't open dbm database \"%s\"; set $LQTEXTDIR",
		FilePrefix
	    );
	} else {
	    Error(E_WARN|E_SYS,
		"couldn't open dbm database \"%s\" [$LQTEXTDIR is \"%s\"]",
		FilePrefix,
		p
	    );
	}
	return (DBM *) 0;
    }

    cp->NameLength = NameLength;
    cp->Name = emalloc(NameLength + 1);
    (void) strcpy(cp->Name, FilePrefix);
    /* Put the new element at the start of the list, since if we just called
     * dbstart, we're certain to want this dbm database almost immediately,
     * and in any case before any other database.
     */
    cp->Next = DatabaseCache; /* cp->Next was previously invalid */
    DatabaseCache = cp;
    ++NumberInCache;

    return cp->Value;
}

#undef enddb

/*ARGSUSED*/
void
enddb(db)
    DBM *db;
{
    /* no-op */
    /* This could check the named db and move it to the bottom of the list,
     * I suppose.
     */
}

void
cleanupdb()
{
    register t_DatabaseCache *cp;
    t_DatabaseCache *Next = 0;

    cp = DatabaseCache;
    while (cp) {
	if (cp->Value) {
	    (void) dbm_close(cp->Value);
	    cp->Value = 0;
	}
	if (cp->Name) {
	    (void) efree(cp->Name);
	    cp->Name = 0;
	}

	Next = cp->Next;
	(void) efree((char *) cp);
	/* can no longer refer to cp->Next...*/
	cp = Next;
    }
    NumberInCache = 0;
    DatabaseCache = (t_DatabaseCache *) 0;
}
