/* pbcache.c -- Copyright 1989 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

#ifndef LINT
static char *RcsId = "@(#) $Id: pbcache.c,v 1.5 92/07/08 22:44:35 lee Exp $";
#endif

/* Block cache for lq-text */

/* Functions defined in this file:
 *
 * OpenDataBase()
 * ReadBlock(unsigned long Offset)
 * WriteBlock(unsigned long Block; char *Data)
 *
 * FlushCache()
 *
 * OpenFreeFile()
 * WriteFreeBlock()
 * char *ReadFreeBitBlock(long Where)
 * static INLINE int FindUnsetBitInByte(unsigned int Value)
 * unsigned long FindFreeBlock(t_WID WID, unsigned char *NBlocksp)
 * void SetBlockStatus(unsigned long Offset; int Status)
 *
 * And for convenience, although they should really be elsewhere:
 * void Deletepblock(t_pblock *pblock)
 * void DeleteWordPlaces(unsigned long FirstBlock; t_WID WID)
 * (having them here means they can get at DataFile and OpenDataBase)
 *
 */

#include "globals.h" /* defines and declarations for database filenames */
#include "error.h"

#include <stdio.h> /* stderr, also for fileinfo.h */
#include <sys/types.h> /* for fileinfo.h, which uses time_t */
#include <fcntl.h> /* for O_RDONLY */
#include "fileinfo.h" /* for wordinfo.h */
#include "wordinfo.h" /* for t_WID */
#include "pblock.h"
#include "Revision.h"
#include "blkheader.h"

/** Unix system calls that need to be declared: **/
/* Some systems have a stdlib.h or stddef.h with some of these in,
 * but most don't (sigh)
 */
extern int open();
extern int read(), write();
extern long lseek();

/** C library functions that need to be declared: **/

/** lqtext library functions that need to be declared: **/

/** Functions within this file that need to be declared: **/
void FlushCache();
static void OpenDataBase();
static void OpenFreeFile();
static void WriteFreeBlock();
void SetBlockStatus();

unsigned long FindFreeBlock();
/** **/

#ifdef ASCIITRACE
extern int AsciiTrace;
#endif

static int DataFile = -1;
static unsigned long FreeBlockStart = 3L;
	/* i.e. not a multiple of 2, and hence not a valid value */

static int FreeFile = -1;
#define FREEBITLEN 1024
#define BYTESPERFREEBIT (BLOCKSIZE*8)
static unsigned char FreeBitBlock[FREEBITLEN];
static char FreeBitBlockIsDirty = 0;
static int BlockCacheIsDirty = 0;
static int LowestFreeBlock = 0L;

char *ReadBlock();
void WriteBlock();

#define CACHELEN (BLOCKSIZE * 256)

static unsigned long BlockCacheTimer = 0;

typedef struct s_CacheEntry {
    unsigned long CacheStart;
    long CacheLen;
    int TimeLastUsed;
    char ReadAheadBuffer[CACHELEN];
    short IsDirty;
} t_CacheEntry;

#define CACHEDBLOCKS 6
static t_CacheEntry BlockCache[CACHEDBLOCKS] = {
    0,
};
static t_CacheEntry *CurrentBlock = BlockCache;

#ifdef ASCIITRACE
static void
printcache()
{
    register int i;
    fprintf(stderr, "**** cache at %ld is [", BlockCacheTimer);
    for (i = 0; i < CACHEDBLOCKS; i++) {
	fprintf(stderr, " %ld-%ld",
	    BlockCache[i].CacheStart,
	    BlockCache[i].CacheStart + BlockCache[i].CacheLen
	);
    }
    fprintf(stderr, "]\n");
}
#endif

static int
IsCached(Block)
    unsigned long Block;
{
    /* see if the requested block is cached, and, if so, make
     * CurrentBlock point to it.
     */

    register t_CacheEntry *cp;

    if (DataFile < 0) {
	OpenDataBase();
    }

    for (cp = BlockCache; cp - BlockCache < CACHEDBLOCKS; cp++) {
	if (cp->CacheStart <= Block && cp->CacheStart + cp->CacheLen >=
							Block + BLOCKSIZE) {
	    /* Note: use CacheLen, not CACHELEN, in case the cache entry
	     * isn't valid, in which case CacheLen is zero.
	     */
	    CurrentBlock = cp;
	    return 1;
	}
    }

#ifdef ASCIITRACE
    if (AsciiTrace > 10) {
	fprintf(stderr, "** IsCached Miss for %ld\n", Block);
	printcache();
    }
#endif

    return 0;
}

static void
FlushOneBlock(cp)
    t_CacheEntry *cp;
{
    int i;

    if (lseek(DataFile, cp->CacheStart, 0) < 0) {
	Error(E_SYS|E_FATAL,
	    "FlushCache: lseek(%d=%s, %ld, 0) failed",
	    DataFile, DataBase, cp->CacheStart
	);
    } 
    
    i = write(DataFile, cp->ReadAheadBuffer, cp->CacheLen);

    if (i != cp->CacheLen) {
	Error(E_FATAL|E_SYS,
	    "FlushCache: write(%d=%s, 0x%s, %d) failed, returned %d",
	    DataFile, DataBase, cp->ReadAheadBuffer, cp->CacheLen, i
	);
    }

    cp->IsDirty = 0;
    /* Don't change CacheLen, so that we will still find the block
     * if we want it
     */
}

void
FlushCache(MinToFree)
    int MinToFree;
{
    register t_CacheEntry *cp;
    t_CacheEntry *Oldest = &BlockCache[0];

#ifdef ASCIITRACE
    if (AsciiTrace > 4) {
	(void) fprintf(stderr, " (flush cache) ");
	(void) fflush(stderr);
    }
#endif

    if (DataFile <= 0) {
	if (BlockCacheIsDirty) {
	    Error(E_BUG,
		"dirty cache %ld len %ld, fd %d=%s",
		CurrentBlock->CacheStart,
		CurrentBlock->CacheLen,
		DataFile,
		DataBase
	    );
	}
	BlockCacheIsDirty = 0;
	return;
    }

    if (MinToFree) {
	/* check that the cache is in fact full */
	/* find the oldest block and free it */
	for (cp = BlockCache; cp - BlockCache < CACHEDBLOCKS; cp++) {
	    if (cp->TimeLastUsed && cp->TimeLastUsed < Oldest->TimeLastUsed) {
		Oldest = cp;
	    } else if (cp->CacheLen == 0) {
		/* Only on the first few times round... */
		Oldest = cp;
	    }
	}
	CurrentBlock = Oldest;
	if (CurrentBlock->CacheLen && CurrentBlock->IsDirty) {
	    FlushOneBlock(CurrentBlock = Oldest);
	    return;
	}
    }

    for (cp = BlockCache; cp - BlockCache < CACHEDBLOCKS; cp++) {

	if (cp->TimeLastUsed && cp->TimeLastUsed < Oldest->TimeLastUsed) {
	    Oldest = cp;
	}

	if (cp->CacheLen && cp->IsDirty) {
	    FlushOneBlock(cp);
	}
    }

    BlockCacheIsDirty = 0;
    CurrentBlock = Oldest;

    if (FreeBitBlockIsDirty) {
	WriteFreeBlock();
    }
}

void
WriteBlock(Block, Data, Length)
    unsigned long Block;
    char *Data;
    int Length;
{
    t_BlockHeader *H;
    int MaxLength;

    if (DataFile < 0) {
	OpenDataBase();
    }

    if (!Data) {
	Error(E_BUG,
	    "WriteBlock %ld, 0x0: second argument (Data) is invalid",
	    Block
	);
    }

    H = (t_BlockHeader *) Data;
    MaxLength = H->NumberOfBlocks;
    while (MaxLength > Length) {
	--MaxLength;
	/* e.g. if MaxLength is 2 and Length is 1, we must free
	 * block Block+1
	 */
	SetBlockStatus(Block + (MaxLength*BLOCKSIZE), SET_BLOCK_AS_FREE);
    }


    H->NumberOfBlocks = Length;

    if (!IsCached(Block)) {
	(void) ReadBlock(Block);
	/* so that the cache gets full;
	 * this isn't so bad as we're likely to want the other blocks
	 * in the cache... and I/O is in 8K chnks anyway.   We ought to
	 * check that this block isn't 8K long, though...
	 */
    }

    if (Length > 1) {
#ifdef ASCIITRACE
	if (AsciiTrace > 2) {
	    fprintf(stderr, "Write %ld length %d\n", Block, Length);
	}
#endif
	if ((Block - CurrentBlock->CacheStart) + Length*BLOCKSIZE > CACHELEN) {
	    Error(E_BUG,
		"WriteBlock: offset %ld, length %ld overran cache at %ld by %d bytes",
		Block,
		Length,
		CurrentBlock->CacheStart,
		((Block - CurrentBlock->CacheStart) + Length*BLOCKSIZE) -
								CACHELEN
	    );
	}
    }

    bcopy(
	Data,
	&CurrentBlock->ReadAheadBuffer[Block - CurrentBlock->CacheStart],
	BLOCKSIZE * Length
    );

    CurrentBlock->TimeLastUsed = ++BlockCacheTimer;
    CurrentBlock->IsDirty = 1;
    BlockCacheIsDirty = 1;

    return;
}

static void
OpenDataBase()
{
    extern void lqGetFileModes();
    void lqCheckVersion();
    int Flags, Modes;

    lqGetFileModes(&Flags, &Modes);

    if ((DataFile = open(DataBase, Flags, Modes)) < 0) {
	Error(E_FATAL|E_SYS, "Can't open database file \"%s\"", DataBase);
    }

    /* lqWriteVersion and lqReadVersion MUST do a ReadBlock(0) immediately,
     * so that the cache gets initialised properly.
     */
    lqCheckVersion();
}

static void
WriteFreeBlock()
{
    int i;
    unsigned long OffsetInFile;

    if (FreeFile < 0 || !FreeBitBlockIsDirty) {
	return;
    }

    OffsetInFile = FreeBlockStart / (BLOCKSIZE * 8);

    if (lseek(FreeFile, OffsetInFile, 0) < 0) {
	Error(E_FATAL|E_SYS, "WriteFreeBlock: lseek(%d=\"%s\", %ld, 0) failed",
				FreeFile, FreeFileName, FreeBlockStart);
    }
    
    i = write(FreeFile, FreeBitBlock, FREEBITLEN);

    if (i != FREEBITLEN) {
	Error(E_FATAL|E_SYS,
	    "WriteFreeBlock: write(%d=\"%s\", 0x%x, %d) returned %d",
	    FreeFile, FreeFileName, FreeBitBlock, FREEBITLEN, i);
    }

    FreeBitBlockIsDirty = 0;
}

static void
ReadFreeBitBlock(Where)
    long Where;
{
    long AmountRead;
    unsigned long OffsetInFile;

    /* open the Free file if necessary... */

    if (FreeFile < 0) {
	OpenFreeFile();
    }

    OffsetInFile = Where / (BLOCKSIZE * 8);

    /* Round FREEBITLEN down to the start of the block */
    OffsetInFile /= FREEBITLEN;
    OffsetInFile *= FREEBITLEN;

    if (OffsetInFile * BLOCKSIZE * 8 == FreeBlockStart) {
	return;
    }

    if (FreeBitBlockIsDirty) {
	WriteFreeBlock();
    }

    if (lseek(FreeFile, OffsetInFile, 0) < 0) {
	Error(E_FATAL|E_SYS, "ReadFreeBlock: lseek(%d=\"%s\", %ld, 0) failed",
	    FreeFile, FreeFileName, Where);
    }

    AmountRead = read(FreeFile, FreeBitBlock, FREEBITLEN);

    if (AmountRead < 0) {
	Error(E_FATAL|E_SYS,
	    "ReadFreeBlock: read(%d=\"%s\", ...) returned %d, not %d",
			FreeFile, FreeFileName, AmountRead, FREEBITLEN
	);
    }

    FreeBlockStart = OffsetInFile * 8 * BLOCKSIZE;

    /* If we have gone past the end of the file, set the rest of the
     * block to zeros.
     */
    if (AmountRead < FREEBITLEN) {
	(void) bzero(&FreeBitBlock[AmountRead], FREEBITLEN - AmountRead);
    }
}

#define ReadNextFreeBitBlock() \
    ReadFreeBitBlock(FreeBlockStart + (8 * BLOCKSIZE * FREEBITLEN + 1))

#define SetBitInByte(Which) (01 << (Which))

static INLINE int
FindUnsetBitInByte(Value)
    unsigned int Value;
{
    register int WhichBit = 0;

    while (Value & 01) {
	Value >>= 1;
	WhichBit++;
    }

    return WhichBit;
}

static void
OpenFreeFile()
{
    extern void lqGetFileModes();
    int Flags, Modes;
    
    lqGetFileModes(&Flags, &Modes);

    if ((FreeFile = open(FreeFileName, Flags, Modes)) < 0) {
	Error(E_FATAL|E_SYS, "Can't open database free bitmap file \"%s\"",
			FreeFileName);
    }

    ReadFreeBitBlock(0L);

    if (!(FreeBitBlock[0] & 01)) {
	SetBlockStatus(0L, SET_BLOCK_AS_USED);
    }
}

/*ARGSUSED1*/
unsigned long
FindFreeBlock(WID, BlockLengthp)
    t_WID WID;
    unsigned int *BlockLengthp;
{
    register unsigned char *p;
    int GotOne = 0;
    unsigned long Here;

    if (FreeFile < 0) {
	OpenFreeFile();
	/* OpenFreeFile() calls ReadFreeBitBlock(0L) */
	LowestFreeBlock = 0L; /* actually it's probably higher, of course */
    } else {
	ReadFreeBitBlock(LowestFreeBlock);
    }

    do {
	for (p = FreeBitBlock; p - FreeBitBlock < FREEBITLEN; p++) {
	    if (*p != (unsigned char) 0xff) {
		GotOne++;
		break;
	    }
	}

	if (GotOne) break;

	ReadNextFreeBitBlock();
    } while (!GotOne);

    /* Now we've found a byte `containing' a free block... ( a zero bit)
     * so we have to identify the block.
     */

    Here = FreeBlockStart + (p - FreeBitBlock) * BYTESPERFREEBIT +
		(FindUnsetBitInByte((unsigned int)*p) * BLOCKSIZE);
    SetBlockStatus(Here, SET_BLOCK_AS_USED);

    /* count number of available blocks in the cache */
    {
	unsigned long Candidate = Here + BLOCKSIZE;
	int MaxLength;
	int i;

	MaxLength = (FREEBITLEN - (p - FreeBitBlock)) * 8;
	MaxLength -= (Here & 07); /* bit within byte */

	/* We only have one byte in which to store the length...
	 * see blockhdr.h
	 */
	if (MaxLength > 255) MaxLength = 255;
	i = (CACHELEN - (Here % CACHELEN)) / BLOCKSIZE;
	if (MaxLength > i) MaxLength = i;

	*BlockLengthp = 1;

	while (*BlockLengthp < MaxLength && BlockIsFree(Candidate)) {
	    SetBlockStatus(Candidate, SET_BLOCK_AS_USED);
	    ++*BlockLengthp;
	    Candidate += BLOCKSIZE;
	}
	*BlockLengthp *= BLOCKSIZE;
	LowestFreeBlock = Candidate + BLOCKSIZE;
    }

    return Here;
}

static int
BlockIsFree(Offset)
    unsigned long Offset;
{
    register unsigned char *p;
    register unsigned int ui;

    if (FreeFile < 0) {
	OpenFreeFile();
	LowestFreeBlock = 0L;
    }

    /* First make sure that the necessary bitmap block is loaded: */
    if (Offset >= (FreeBlockStart + (BYTESPERFREEBIT * FREEBITLEN)) ||
						(Offset < FreeBlockStart)) {
	ReadFreeBitBlock(Offset);
    }

    ui = (Offset - FreeBlockStart) / BLOCKSIZE; /* this is the bit address */
    p = &FreeBitBlock[ui >> 3]; /* i.e. ui/8, i.e. the right byte */

    /* ui & 07 is ui % 8 is the bit within the byte: */
    
    /* don't use ?: because of profiling */
    if (*p & SetBitInByte(ui & 07)) {
	return 0; /* bit is set --> block is in use */
    } else {
	return 1; /* bit not set --> block is free */
    }
}

void
SetBlockStatus(Offset, Status)
    unsigned long Offset;
    int Status;
{
    register unsigned char *p;
    register unsigned int ui;
    unsigned char OldValue ;

    if (FreeFile < 0) {
	OpenFreeFile();
	LowestFreeBlock = 0L;
    }

    /* First make sure that the necessary bitmap block is loaded: */
    if (Offset >= (FreeBlockStart + (BYTESPERFREEBIT * FREEBITLEN)) ||
						(Offset < FreeBlockStart)) {
	ReadFreeBitBlock(Offset);
    }

    ui = (Offset - FreeBlockStart) / BLOCKSIZE; /* this is the bit address */
    p = &FreeBitBlock[ui >> 3]; /* i.e. ui/8, i.e. the right byte */
    OldValue = (*p);
    if (Status == SET_BLOCK_AS_FREE) {
	/* ui & 07 is ui % 8 is the bit within the byte: */
	*p &= (unsigned char)~(unsigned char) SetBitInByte(ui & 07);
	if (Offset < LowestFreeBlock || !LowestFreeBlock) {
	    LowestFreeBlock = Offset;
	}
    } else if (Status == SET_BLOCK_AS_USED) {
	*p |= SetBitInByte(ui & 07);
	if (Offset == LowestFreeBlock) {
	    LowestFreeBlock = 0;
	}
    } else {
	Error(E_BUG, "SetBlockStatus(%ld, %d)", Offset, Status);
    }
    if (OldValue != *p) {
	FreeBitBlockIsDirty = 1;
    }
}

#define BLOCKSINREADBUFCACHE 2
static char ReadBufferCache[BLOCKSINREADBUFCACHE][CACHELEN];
static int BufferNumber = -1;
/* If sWriteNumber didn't overshoot buffers, BLOCKSINREADBUFCACHE could be
 * set to 1, and if we didn't need to go back and write to the previous
 * block to set the NextOffset, it could be zero and we could return a
 * pointer into the main cache without even the copy!
 */

/* Get a single disk block from the database */
char *
ReadBlock(Offset)
    unsigned long Offset;
{
    char *Buffer;
    t_BlockHeader *BH;

    if (DataFile < 0) {
	OpenDataBase();
    }

    if (++BufferNumber >= BLOCKSINREADBUFCACHE) BufferNumber = 0;
    Buffer = ReadBufferCache[BufferNumber];

#ifdef ASCIITRACE
    if (AsciiTrace > 15) {
	fprintf(stderr, "(R %ld) ", Offset);
	(void) fflush(stderr);
    }
#endif

    if (IsCached(Offset)) {
	BH = (t_BlockHeader *)
	    &(CurrentBlock->ReadAheadBuffer[Offset-CurrentBlock->CacheStart]);
	
	memcpy(
	    Buffer,
	    &(CurrentBlock->ReadAheadBuffer[Offset - CurrentBlock->CacheStart]),
	    BLOCKSIZE * BH->NumberOfBlocks
	);
	CurrentBlock->TimeLastUsed = ++BlockCacheTimer;
	return Buffer;
    } 

    /* So it's not cached...
     * Which means we have to read a new cache.
     * So let's make some room:
     */

    if (BlockCacheIsDirty) {
	/* The argument to FlushCache means it'll only write out a
	 * single dirty block, and set CurrentBlock top point to it.
	 */
	FlushCache(1);
    }

    /* cache only on CACHELEN (typically 8K) boundaries, as this helps the
     * file system, especially over NFS.
     */
    CurrentBlock->CacheStart = Offset / CACHELEN;
    CurrentBlock->CacheStart *= CACHELEN;

    if (lseek(DataFile, CurrentBlock->CacheStart, 0) < 0) {
	Error(E_SYS|E_WARN,
	    "ReadBlock %ld: lseek(%d=\"%s\", %ld 0) failed",
	    Offset, DataFile, DataBase, CurrentBlock->CacheStart
	);
	return (char *) 0;
    }
    /* Now we are in the right place */

    CurrentBlock->CacheLen =
		read(DataFile, CurrentBlock->ReadAheadBuffer, CACHELEN);

    if (CurrentBlock->CacheLen < 0) {
	Error(E_FATAL|E_SYS,
	    "ReadBlock: read(%d=\"%s\"...) returned %d, not %d",
	    DataFile, DataBase, CurrentBlock->CacheLen, CACHELEN
	);
    }

    BH = (t_BlockHeader *)
	    &(CurrentBlock->ReadAheadBuffer[Offset-CurrentBlock->CacheStart]);

    if (CurrentBlock->CacheLen < CACHELEN) {
	int ExtraBytes = CACHELEN - CurrentBlock->CacheLen;
	/* If CacheLen was less than CACHELN this can only mean that we
	 * have tried to read beyond the end of the file.  In this case,
	 * we might as well consider that we have the extra bytes cached
	 * as well.  The disadvantage is that the file will always grow to
	 * a multiple of CACHELEN, but the advantage is fewer reads when
	 * the file is growing.
	 */

	/* First, zero out the block if it's near the start.
	 * This is so that we can check for things stored in block zero.
	 */

	if (Offset < BLOCKSIZE * 2) {
	    bzero(
		&CurrentBlock->ReadAheadBuffer[ExtraBytes],
		CACHELEN - ExtraBytes
	    );
	} else {
	    register char *bp;
	    t_BlockHeader *Zero;
	    int i = (ExtraBytes + BLOCKSIZE - 1) / BLOCKSIZE;

	    i *= BLOCKSIZE; /* i.e. rounded up */
	    for (bp = &CurrentBlock->ReadAheadBuffer[i];
				bp - CurrentBlock->ReadAheadBuffer < CACHELEN;
							bp += BLOCKSIZE) {
		Zero = (t_BlockHeader *) bp;
		Zero->NumberOfBlocks = 1;
	    }
	}

	if (BH->NumberOfBlocks == 0) {
	    BH->NumberOfBlocks = 1;
	}

	CurrentBlock->CacheLen = CACHELEN;
    }

    (void) memcpy(
	Buffer,
	&(CurrentBlock->ReadAheadBuffer[Offset - CurrentBlock->CacheStart]),
	BLOCKSIZE * BH->NumberOfBlocks
    );
    CurrentBlock->TimeLastUsed = ++BlockCacheTimer;
    return Buffer;
}

lqWriteVersion()
{
    unsigned char *p;
    t_BlockHeader *BH;

    /* OpenDatabase depends on this function doing a ReadBlock(0) */
    p = (unsigned char *) ReadBlock(0L);
    BH = (t_BlockHeader *) p;

    p[0] = LQ_FLAG_VALUE;
    p[1] = (LQ_MAJOR_VERSION / 256);
    p[2] = (LQ_MAJOR_VERSION & 255);
    p[3] = LQ_MINOR_VERSION;

    BH->NumberOfBlocks = 1;
    WriteBlock(0L, (char *) p, 1);
}

static char *
lqFlagString(byte)
    unsigned char byte;
{
    if ((byte & 01) == 0) {
	return "[unknown, or flags corrupt]";
    }
    if (byte & 02) {
	return "-DWIDINBLOCK";
    } else {
	return "-UWIDIINBLOCK";
    }
}

void
lqCheckVersion()
{
    int MajorVersion;
    unsigned char *p;

    p = (unsigned char *) ReadBlock(0L);

    if (*p == 0) {
	lqWriteVersion();
	return;
    }

    if (*p != LQ_FLAG_VALUE) {
	Error(E_FATAL,
	    "Mismatch: Database flags [%s] compiled [%s]",
	    lqFlagString(*p),
	    lqFlagString(LQ_FLAG_VALUE)
	);
    }
    MajorVersion = ((unsigned char) p[1]) * 256 + (unsigned char) p[2];
    if (MajorVersion != LQ_MAJOR_VERSION) {
	Error(E_FATAL,
	    "Database was written with lqtext %d.%, %d.%d can't read it",
	    MajorVersion, p[3], LQ_MAJOR_VERSION, LQ_MINOR_VERSION
	);
    }
}

/*ARGSUSED1*/
void
DeleteWordPlaces(FirstBlock, WID)
    unsigned long FirstBlock;
    t_WID WID;
{
    char *p;
    t_BlockHeader *H;
    unsigned long Block = FirstBlock;

    if (!FirstBlock || !WID) {
	Error(E_BUG, "DeleteWordPlaces(%ld, %ld) invalid", FirstBlock, WID);
    }

    if (DataFile < 0) {
	OpenDataBase();
    }

    while (Block != 0L) {
	int NumberOfBlocks;

	p = ReadBlock(Block);
	H = (t_BlockHeader *) p;
	NumberOfBlocks = H->NumberOfBlocks;
#ifdef WIDINBLOCK
	if (H->WID != WID) {
	    Error(E_BUG,
		"DeleteWordPlaces(Start=%ld, WID=%ld): block %ld had WID %ld",
		FirstBlock, WID, Block, H->WID
	    );
	}
	H->WID = 0;
	/* help debugging: */
	(void) sprintf(&H->Data[0], "Deleted -- was %ld\n", WID);
	/* only need to write the block if debugging... */
	WriteBlock(Block, p, 1);
#endif
	while (NumberOfBlocks > 0) {
	    --NumberOfBlocks;
	    SetBlockStatus(Block + (NumberOfBlocks * BLOCKSIZE),
							SET_BLOCK_AS_FREE);
	}
	Block = H->NextOffset;
    }
    return;
}

void
Deletepblock(pblock)
    t_pblock *pblock;
{
    if (!pblock) {
	Error(E_BUG, "Attempt to delete null pblock (address 0x%x)", pblock);
    } else if (!pblock->ChainStart || !pblock->WID) {
	Error(E_BUG,
	    "Attempt to delete pblock 0x%x with WID %d, ChainStart %ld",
	    pblock,
	    pblock->WID,
	    pblock->ChainStart
	);
    }

    DeleteWordPlaces(pblock->ChainStart, pblock->WID);

    return;
}
