;;
;; vscmc.scm -- VSCM Bytecode compiler (compiling for the
;;              ``flat closure'' version)
;;
;; (C) m.b (Matthias Blume); HUB; Mar 1993 PU/CS
;;         Princeton University, Dept. of Computer Science
;;
;; ident "@(#) vscmc.scm (C) M.Blume, Princeton University, 2.12"
;;
;; TODO:
;; gcd/lcm based on internal gcd


;; Uninitialized definition of ``soft'' primitives:

(define (with-input-from-file file thunk) (uninitialized) (uninitialized))
(define (with-output-to-file file thunk) (uninitialized) (uninitialized))
(define (call-with-input-file file proc) (uninitialized) (uninitialized))
(define (call-with-output-file file proc) (uninitialized) (uninitialized))
(define (close-input-port port) (uninitialized) (uninitialized))
(define (close-output-port port) (uninitialized) (uninitialized))
(define (load file) (uninitialized) (uninitialized))
(define (modulo x y) (uninitialized) (uninitialized))
(define (gcd . l) (uninitialized) (uninitialized))
(define (lcm . l) (uninitialized) (uninitialized))
(define (rationalize a e) (uninitialized) (uninitialized))
(define (expt x y) (uninitialized) (uninitialized))
(define (eval x) (uninitialized) (uninitialized))
(define (open-compilation x) (uninitialized) (uninitialized))
(define (get s t) (uninitialized) (uninitialized))
(define (put! s t v) (uninitialized) (uninitialized))
(define (exit . l) (uninitialized) (uninitialized))


;; Provide definitions for ``soft'' primitives:

(define (i-sp)				; init ``soft prim.''
  (let ((* *)
	(/ /)
	(= =)
	(+ +)
	(- -)
	(< <)
	(abs abs)
	(assq assq)
	(close-port close-port)
	(eqv? eqv?)
	(exact? exact?)
	(exp exp)
	(floor floor)
	(get-properties get-properties)
	(integer? integer?)
	(log log)
	(negative? negative?)
	(null? null?)
	(open-input-file open-input-file)
	(open-output-file open-output-file)
	(odd? odd?)
	(positive? positive?)
	(quotient quotient)
	(remainder remainder)
	(set-cdr! set-cdr!)
	(set-properties! set-properties!)
	(with-input-from-port with-input-from-port)
	(with-output-to-port with-output-to-port)
	(zero? zero?))

    (define (intern-call-with-input-file file proc)
      (let* ((port (open-input-file file))
	     (result (proc port)))
	(close-port port)
	result))

    (define (intern-call-with-output-file file proc)
      (let* ((port (open-output-file file))
	     (result (proc port)))
	(close-port port)
	result))

    (define (intern-with-input-from-file file thunk)
      (let* ((port (open-input-file file))
	     (result (with-input-from-port port thunk)))
	(close-port port)
	result))

    (define (intern-with-output-to-file file thunk)
      (let* ((port (open-output-file file))
	     (result (with-output-to-port port thunk)))
	(close-port port)
	result))

    (define (gcd2 x y)
      (if (zero? y) x (gcd2 y (remainder x y))))

    (define (intern-gcd . l)
      (define (loop l r)
	(if (null? l)
	    r
	    (loop (cdr l) (gcd2 (car l) r))))
      (if (null? l)
	  0
	  (abs (loop (cdr l) (car l)))))

    (define (intern-lcm . l)
      (define (loop l x)
	(if (null? l)
	    x
	    (if (zero? (car l))
		0
		(loop (cdr l)
		      (* x
			 (/ (car l)
			    (gcd2 x (car l))))))))
      (if (null? l)
	  1
	  (if (zero? (car l))
	      0
	      (abs (loop (cdr l) (car l))))))

    (define (intern-rationalize a e)
      ;; Courtesy of Alan Bawden.
      (define (loop x y)
	(if (integer? x)
	    x
	    (let ((fx (floor x))
		  (fy (floor y)))
	      (if (= fx fy)
		  (+ fx
		     (/ (loop (/ (- y fy)) (/ (- x fx)))))
		  (+ fx 1)))))
      (define (x<y x y)
	(cond ((positive? x)
	       (loop x y))
	      ((negative? y)
	       (- (loop (- y) (- x))))
	      ((and (exact? x) (exact? y)) 0)
	      (else 0.)))
      (define (simplest-rational x y)
	(cond ((< x y) (x<y x y))
	      ((< y x) (x<y y x))
	      (else x)))
      (simplest-rational (- a e) (+ a e)))

    (define (intern-expt x y)
      (define (e-expt x y)
	(cond ((zero? y) 1)
	      ((= y 1) x)
	      (else
	       (let ((z (e-expt x (quotient y 2))))
		 (if (odd? y)
		     (* z z x)
		     (* z z))))))
      (if (and (exact? x) (exact? y) (integer? y))
	  (if (negative? y)
	      (/ (e-expt x (- y)))
	      (e-expt x y))
	  (exp (* y (log x)))))

    (define (intern-modulo x y)
      (let ((r (remainder x y)))
	(if (eqv? (negative? r) (negative? y))
	    r
	    (+ r y))))

    (define (intern-get sym tag)
      (cond ((get-properties sym) => (lambda (p)
				       (cond ((assq tag p) => cdr)
					     (else #f))))
	    (else #f)))

    (define (intern-put! sym tag val)
      (cond ((get-properties sym)
	     =>
	     (lambda (p)
	       (cond ((assq tag p)
		      =>
		      (lambda (a)
			(set-cdr! a val)))
		     (else
		      (set-properties! sym (cons (cons tag val) p))))))
	    (else
	     (set-properties! sym (cons (cons tag val) '())))))

    (set! call-with-input-file intern-call-with-input-file)
    (set! call-with-output-file intern-call-with-output-file)
    (set! with-input-from-file intern-with-input-from-file)
    (set! with-output-to-file intern-with-output-to-file)
    (set! gcd intern-gcd)
    (set! lcm intern-lcm)
    (set! rationalize intern-rationalize)
    (set! expt intern-expt)
    (set! close-input-port close-port)
    (set! close-output-port close-port)
    (set! modulo intern-modulo)
    (set! get intern-get)
    (set! put! intern-put!)
    (set! exit quit)))


;; The compiler:

(define (mcm)				; make compiler module

  (i-sp)				; init. misc. ``soft'' primitives

  ;; make private copies of all global procedures...
  (let* ((+ +)
	 (- -)
	 (< <)
	 (= =)
	 (> >)
	 (<= <=)
	 (>= >=)
	 (append append)
	 (apply apply)
	 (assq assq)
	 (assv assv)
	 (boolean? boolean?)
	 (call-with-current-continuation call-with-current-continuation)
	 (call-with-input-file call-with-input-file)
	 (car car)
	 (char=? char=?)
	 (clock clock)
	 (define-asm define-asm)
	 (disassemble disassemble)
	 (display display)
	 (eof-object? eof-object?)
	 (eq? eq?)
	 (error error)
	 (execute-asm execute-asm)
	 (for-each for-each)
	 (gc-clock gc-clock)
	 (inspect inspect)
	 (length length)
	 (list list)
	 (list->vector list->vector)
	 (list? list?)
	 (map map)
	 (member member)
	 (memq memq)
	 (negative? negative?)
	 (newline newline)
	 (null? null?)
	 (pair? pair?)
	 (quit quit)
	 (read read)
	 (reverse reverse)
	 (set-car! set-car!)
	 (set-cdr! set-cdr!)
	 (standard-port standard-port)
	 (string-append string-append)
	 (string-ci=? string-ci=?)
	 (string-length string-length)
	 (string-ref string-ref)
	 (substring substring)
	 (symbol->string symbol->string)
	 (symbol? symbol?)
	 (vector->list vector->list)
	 (vector-length vector-length)
	 (vector-ref vector-ref)
	 (vector? vector?)
	 (with-error-handler with-error-handler)
	 (with-input-from-file with-input-from-file)
	 (with-output-to-file with-output-to-file)
	 (with-output-to-port with-output-to-port)
	 (write write)
	 (zero? zero?)

	 (*open-compilation-enabled* #t)

	 (*version* (vscm-version))

	 (*error-continuation*
	  (lambda (ignore)
	    (display "Quitting..." (standard-port 2))
	    (newline (standard-port 2))
	    (quit 1))))

    (define (is-definition? expr)
      (and (pair? expr)
	   (eq? (car expr) 'define)))

    ;; returns ``normalized'' definition with a single symbol defined
    (define (expand-define form)
      (define (loop var expr-list)
	(cond ((pair? var)
	       (loop (car var)
		     (list (cons 'lambda (cons (cdr var) expr-list)))))
	      ((pair? (cdr expr-list))
	       (syntax-error "bad definition " form))
	      (else
	       (cons 'define (cons var expr-list)))))
      (cond ((not (is-definition? form)) form)
	    ((< (length form) 3)
	     (syntax-error "bad definition " form))
	    (else (loop (cadr form) (cddr form)))))

    ;; some set operations...
    (define (set-add source take? rest)
      (define (loop l r)
	(if (pair? l)
	    (loop (cdr l)
		  (if (take? (car l)) (cons (car l) r) r))
	    r))
      (loop source rest))
    (define (set-add-excl source exclusion rest)
      (set-add source (lambda (x) (not (memq x exclusion))) rest))
    (define (set-difference s1 s2)
      (set-add-excl s1 s2 '()))
    (define (set-union s1 s2)
      (set-add-excl s1 s2 s2))
    (define (set-intersection s1 s2)
      (set-add s1 (lambda (x) (memq x s2)) '()))

    ;; message... alternating sequence of messages and data
    (define (message head . l)
      (with-output-to-port
	  (standard-port 2)
	(lambda ()
	  (define (loop l now then)
	    (if (pair? l)
		(begin
		  (display " ")
		  (now (car l))
		  (loop (cdr l) then now))))
	  (display head)
	  (loop l display write)
	  (newline))))
    ;; syntax error message:
    (define (syntax-error . l)
      (apply message "SYNTAX ERROR: " l)
      (*error-continuation* '()))
    ;; semantic error message:
    (define (semantic-error . l)
      (apply message "SEMANTIC ERROR: " l)
      (*error-continuation* '()))
    ;; warning message
    (define (warning . l)
      (apply message "WARNING: " l))

    ;; this one was pulled out of ``annotate'' to re-use it in
    ;; ``compile-function-definition''
    (define (global-ensure-legal-variable var expr)
      (if (or (not (symbol? var))
	      (memq var
		    '(=> and begin case cond define delay do else if lambda
			 let let* letrec or quasiquote quote set! unquote
			 unquote-splicing)))
	  (if (eq? var expr)
	      (syntax-error "not a variable symbol" var)
	      (syntax-error "not a variable symbol" var "in" expr))))

    ;; checks syntax
    ;; replaces all internal definitions by letrec expressions
    ;; replaces all let* expressions by let expressions
    ;; annotates lambda's with sets of free variables and updatable args
    ;;  (lambda (<arg> ...) ...)
    ;;    -> (lambda (<freevar> ...) (<arg> ...) (<updatable arg> ...) ...)
    ;;  (delay <expr>)
    ;;    -> (delay (<freevar> ...) <expr>)
    ;; annotates let and letrec with sets of updatable variables
    ;;  ([let|letrec] ((<var> <expr>) ...) ...)
    ;;    -> ([let|letrec] ((<var> <expr>) ...) (<updatable var> ...) ...)
    ;;  (do ((<var> <e1> <e2>) ...) (<cnd> ...) ...)
    ;;    -> (do ((<var> <e1> <e2>) ...) (<updatable var> ...) (<cnd> ...) ...)
    ;; some other re-writes
    ;;  (and) -> #t
    ;;  (and x) -> x
    ;;  (or) -> #f
    ;;  (or x) -> x
    ;;  (or x1 x2 ... xn) -> (cond (x1) (x2) ... (else xn))
    (define (annotate expr)
      ;; extracts body, eliminates internal defines
      (define (get-body l)
	(define (loop l inits)
	  (if (pair? l)
	      (let ((first (expand-define (car l))))
		(cond ((is-definition? first)
		       (loop (cdr l) (cons (cdr first) inits)))
		      ;; this accepts a superset of R4RS programs...
		      ((and (pair? first) (eq? (car first) 'begin))
		       (loop (append (cdr first) (cdr l)) inits))
		      ((null? inits) l)
		      (else
		       (list (cons 'letrec (cons inits l))))))
	      (syntax-error "empty body in" expr)))
	(loop l '()))
      ;; get list of variables from list of formals
      (define (formals->list f lambda-expr)
	(define (add x l)
	  (if (memq x l)
	      (syntax-error "duplicate formal parameter in" lambda-expr)
	      (cons x l)))
	(define (loop f l)
	  (if (pair? f)
	      (loop (cdr f) (add (car f) l))
	      (if (null? f)
		  l
		  (add f l))))
	(loop f '()))
      ;; add a variable symbol to a list, make sure it is not already there
      (define (let/do-add-local-var v l expr)
	(if (memq v l)
	    (syntax-error "duplicate local variable in" expr)
	    (cons v l)))
      ;; extract variables from let/letrec
      (define (letdef->list l the-expr)
	(define (loop l r)
	  (if (pair? l)
	      (let ((x (car l)))
		(if (not (= (length x) 2))
		    (syntax-error "ill-formed local definition in" the-expr))
		(loop (cdr l) (let/do-add-local-var (car x) r the-expr)))
	      r))
	(loop l '()))
      ;; the same for do...
      (define (dodef->list l the-expr)
	(define (loop l r)
	  (if (pair? l)
	      (let ((x (car l)))
		(case (length x)
		  ((2 3) (loop (cdr l)
			       (let/do-add-local-var (car x) r the-expr)))
		  (else (syntax-error
			 "ill-formed local definition in" the-expr))))
	      r))
	(loop l '()))
      ;; continuation passing style: cont is a function of three arguments:
      ;;  1. the annotated expr (or expr list)
      ;;  2. the list of free variables in expr
      ;;  3. the list of updatable free variables in expr
      ;; ... on lists of expressions ...
      (define (a-list l kv cont)
	(define (loop l cont)
	  (if (pair? l)
	      (loop (cdr l)
		    (lambda (nl lfv lmv)
		      (a (car l) kv
			 (lambda (ne efv emv)
			   (cont (cons ne nl)
				 (set-union efv lfv)
				 (set-union emv lmv))))))
	      (cont '() '() '())))
	(loop l cont))
      ;; ... on let/letrec local definition lists ...
      ;; (assume it's already checked)
      (define (a-letdef l kv cont)
	(if (pair? l)
	    (let ((x (car l)))
	      (a (cadr x) kv
		 (lambda (xx xfv xmv)
		   (a-letdef (cdr l) kv
			     (lambda (ll lfv lmv)
			       (cont
				(cons (list (car x) xx) ll)
				(set-union xfv lfv)
				(set-union xmv lmv)))))))
	    (cont '() '() '())))
      ;; ... on ``do'' loop variable definitions ...
      ;; here ``cont'' takes five args (fv/mv separately for <e1> and <e2>)
      (define (a-dodef l kv1 kv2 cont)
	(if (pair? l)
	    (let ((x (car l)))
	      (a (cadr x) kv1
		 (lambda (e1 fv1 mv1)
		   (let ((c (lambda (e2 fv2 mv2)
			      (a-dodef
			       (cdr l) kv1 kv2
			       (lambda (ll lfv1 lmv1 lfv2 lmv2)
				 (cont
				  (cons (list (car x) e1 e2) ll)
				  (set-union fv1 lfv1)
				  (set-union mv1 lmv1)
				  (set-union fv2 lfv2)
				  (set-union mv2 lmv2)))))))
		     (if (pair? (cddr x))
			 (a (caddr x) kv2 c)
			 (c (car x) (list (car x)) '()))))))
	    (cont '() '() '() '() '())))
      ;; ... on lists of clauses (cond/case) ...
      (define (a-cond-case l func kv expr cont)
	(define (recur l cont)
	  (cond ((pair? (cdr l))
		 (recur (cdr l)
			(lambda (r rfv rmv)
			  (func (car l) kv
				(lambda (c cfv cmv)
				  (cont (cons c r)
					(set-union cfv rfv)
					(set-union cmv rmv)))))))
		((not (list? (car l)))
		 (syntax-error "bad cond/case clause" (car l) "in" expr))
		((eq? (caar l) 'else)
		 (if (not (pair? (cdar l)))
		     (syntax-error "no expression in else clause of" expr))
		 (a-list (cdar l) kv
			 (lambda (l fv mv)
			   (cont (list (cons 'else l)) fv mv))))
		(else (func (car l) kv
			    (lambda (l fv mv)
			      (cont (list l) fv mv))))))
	(recur l cont))
      ;; check the arity of lambda after =>
      (define (check-=>-lambda fun cl expr)
	(if (and (> (length fun) 1)
		 (eq? (car fun) 'lambda)
		 (not (symbol? (cadr fun)))
		 (not (= (length (cadr fun)) 1)))
	    (syntax-error
	     "bad explicit function" fun "in => clause" cl "of" expr)))
      ;; ... on expressions ...
      (define (a expr kv cont)
	(define (ensure-legal-variable var)
	  (global-ensure-legal-variable var expr))
	(if (pair? expr)
	    (case (car expr)
	      ((quote)
	       (if (not (= (length expr) 2))
		   (syntax-error "bad quotation" expr)
		   (cont expr '() '())))
	      ((set!)
	       (if (not (= (length expr) 3))
		   (syntax-error "bad assignment" expr))
	       (let ((v (cadr expr)))
		 (ensure-legal-variable v)
		 (a (caddr expr) kv
		    (lambda (ex fv mv)
		      (let ((nex (list 'set! v ex)))
			(if (memq v kv)
			    (let ((l (list (cadr expr))))
			      (cont nex
				    (set-union l fv)
				    (set-union l mv)))
			    (cont nex fv mv)))))))
	      ((lambda)
	       (if (< (length expr) 3)
		   (syntax-error "bad lambda expr" expr))
	       (let ((vars (formals->list (cadr expr) expr))
		     (body (get-body (cddr expr))))
		 (for-each ensure-legal-variable vars)
		 (a-list body (set-union vars kv)
			 (lambda (l fv mv)
			   (let ((fv (set-difference fv vars)))
			     (cont
			      (cons 'lambda
				    (cons fv
					  (cons (cadr expr)
						(cons (set-intersection
						       mv vars)
						      l))))
			      fv
			      (set-difference mv vars)))))))
	      ((if)
	       (if (< (length expr) 3)
		   (syntax-error "bad if conditional" expr))
	       (a-list (cdr expr) kv
		       (lambda (l fv mv)
			 (cont (cons 'if l) fv mv))))
	      ((cond)
	       (if (< (length expr) 2)
		   (syntax-error "cond without clauses" expr))
	       (let ((a-cond-clause
		      (lambda (cl kv cont)
			(let ((len (length cl)))
			  (cond ((= len 0)
				 (syntax-error "empty clause in" expr))
				((and (= len 3)
				      (eq? (cadr cl) '=>))
				 (check-=>-lambda (caddr cl) cl expr)
				 (a (car cl) kv
				    (lambda (cnd cfv cmv)
				      (a (caddr cl) kv
					 (lambda (fun ffv fmv)
					   (cont
					    (list cnd '=> fun)
					    (set-union cfv ffv)
					    (set-union cmv fmv)))))))
				(else (a-list cl kv cont)))))))
		 (a-cond-case (cdr expr) a-cond-clause kv expr
			      (lambda (l fv mv)
				(cont (cons 'cond l) fv mv)))))
	      ((case)
	       (if (< (length expr) 3)
		   (syntax-error "no clause in" expr))
	       (let ((a-case-clause
		      (lambda (cl kv cont)
			(cond ((< (length cl) 2)
			       (syntax-error "bad clause in" cl "of" expr))
			      ((not (pair? (car cl)))
			       (syntax-error
				"empty case label in" cl "of" expr))
			      ((not (list? (car cl)))
			       (syntax-error "bad label set in" cl "of" expr))
			      (else
			       (a-list (cdr cl) kv
				       (lambda (l fv mv)
					 (cont (cons (car cl) l) fv mv))))))))
		 (a-cond-case (cddr expr) a-case-clause kv expr
			      (lambda (l lfv lmv)
				(a (cadr expr) kv
				   (lambda (key kfv kmv)
				     (cont (cons 'case (cons key l))
					   (set-union lfv kfv)
					   (set-union lmv kmv))))))))
	      ((and or)
	       (if (pair? (cdr expr))
		   (if (pair? (cddr expr))
		       (a-list (cdr expr) kv
			       (lambda (l fv mv)
				 (cont (cons (car expr) l) fv mv)))
		       (a (cadr expr) kv cont))
		   (cont (eq? (car expr) 'and) '() '())))
	      ((let*)			; -> nested LETs, apply a
	       (if (< (length expr) 3)
		   (syntax-error "bad let* expression" expr))
	       (do ((l (reverse (cadr expr)) (cdr l))
		    (form (get-body (cddr expr))
			  (list (cons 'let
				      (cons (list (car l)) form)))))
		   ((not (pair? l))
		    (a (if (pair? (cdr form))
			   (cons 'begin form)
			   (car form))
		       kv cont))))
	      ((let)
	       (let ((len (length expr)))
		 (if (< len 3)
		     (syntax-error "bad let expression" expr))
		 (if (symbol? (cadr expr))
		     (begin
		       (if (< len 4)
			   (syntax-error "bad named let" expr))
		       (ensure-legal-variable (cadr expr))
		       (let ((vars (reverse (letdef->list (caddr expr) expr))))
			 (a (list 'letrec
				  (list (list (cadr expr)
					      (cons 'lambda
						    (cons vars (cdddr expr)))))
				  (cons (cadr expr)
					(map
					 ;; hack around first-class cadr...
					 (lambda (x) (cadr x))
					 (caddr expr))))
			    kv cont)))
		     (let ((vars (letdef->list (cadr expr) expr))
			   (body (get-body (cddr expr))))
		       (for-each ensure-legal-variable vars)
		       (a-letdef
			(cadr expr) kv
			(lambda (defs dfv dmv)
			  (a-list
			   body (set-union vars kv)
			   (lambda (b bfv bmv)
			     (cont
			      (cons 'let
				    (cons defs
					  (cons
					   (set-intersection vars bmv) b)))
			      (set-union dfv
					 (set-difference bfv vars))
			      (set-union dmv
					 (set-difference bmv vars)))))))))))
	      ((letrec)
	       (if (< (length expr) 3)
		   (syntax-error "bad letrec expression" expr))
	       (let ((vars (letdef->list (cadr expr) expr))
		     (body (get-body (cddr expr))))
		 (for-each ensure-legal-variable vars)
		 (let ((kv (set-union kv vars)))
		   (a-letdef
		    (cadr expr) kv
		    (lambda (defs dfv dmv)
		      (a-list
		       body kv
		       (lambda (b bfv bmv)
			 (let ((mv (set-union dmv bmv))
			       (fv (set-union dfv bfv)))
			   (cont
			    (cons
			     'letrec
			     (cons defs
				   (cons (set-intersection vars mv) b)))
			    (set-difference fv vars)
			    (set-difference mv vars))))))))))
	      ((begin)
	       (if (not (pair? (cdr expr)))
		   (syntax-error "empty sequence" expr))
	       (a-list (cdr expr) kv
		       (lambda (l fv mv)
			 (cont (cons 'begin l) fv mv))))
	      ((do)
	       (if (< (length expr) 3)
		   (syntax-error "bad do expression" expr))
	       (if (not (pair? (caddr expr)))
		   (syntax-error "bad exit condition in" expr))
	       (let ((vars (dodef->list (cadr expr) expr)))
		 (for-each ensure-legal-variable vars)
		 (let ((kv2 (set-union vars kv)))
		   (a-dodef
		    (cadr expr) kv kv2
		    (lambda (def fv1 mv1 fv2 mv2)
		      (a-list
		       (caddr expr) kv2
		       (lambda (cnd cfv cmv)
			 (a-list
			  (cdddr expr) kv2
			  (lambda (l lfv lmv)
			    (let ((ftmp (set-union fv2 (set-union cfv lfv)))
				  (mtmp (set-union mv2 (set-union cmv lmv))))
			      (cont
			       (cons 'do
				     (cons def
					   (cons (set-intersection mtmp vars)
						 (cons cnd l))))
			       (set-union fv1
					  (set-difference ftmp vars))
			       (set-union
				mv1
				(set-difference mtmp vars)))))))))))))
	      ((delay)
	       (if (not (= (length expr) 2))
		   (syntax-error "bad delayed expression" expr))
	       (a (cadr expr) kv
		  (lambda (ex fv mv)
		    (cont (list 'delay fv ex) fv mv))))
	      ((quasiquote) (a-quasiquote expr kv cont))
	      (else (a-list expr kv cont)))
	    (cond ((symbol? expr)
		   (ensure-legal-variable expr)
		   (cont expr
			 (if (memq expr kv)
			     (list expr)
			     '())
			 '()))
		  ((null? expr)
		   (syntax-error "empty combination"))
		  ((or (char? expr)
		       (string? expr)
		       (number? expr)
		       (boolean? expr))
		   (cont expr '() '()))
		  (else (syntax-error "not a legal expression: " expr)))))

      ;; quasiquote-expressions will be completely rewritten into a different
      ;; form to makes the operations necessary to evaluate them
      ;; explicit. Only a second phase actually annotates the result of the
      ;; rewriting.
      (define (a-quasiquote expr kv cont)

	(define (qq? x)
	  (and (pair? x) (eq? (car x) 'quasiquote) (= (length x) 2)))
	(define (uq? x)
	  (and (pair? x) (eq? (car x) 'unquote) (= (length x) 2)))
	(define (uqs? x)
	  (and (pair? x) (eq? (car x) 'unquote-splicing) (= (length x) 2)))

	(define (substruct2 x lev)
	  (let ((sub (construct-any (cadr x) lev)))
	    (case (car sub)
	      ((quote) (list 'quote x))
	      ((splice) (list 'cons (list 'quote (car x))
			      (cadr sub)))
	      (else (list 'cons
			  (list 'quote (car x))
			  (list 'cons
				sub
				''()))))))

	(define (construct-any x lev)
	  (cond ((qq? x) (substruct2 x (+ lev 1)))
		((uq? x)
		 (if (zero? lev)
		     (list 'eval (cadr x))
		     (substruct2 x (- lev 1))))
		((uqs? x)
		 (if (zero? lev)
		     (list 'splice (cadr x))
		     (substruct2 x (- lev 1))))
		((pair? x)
		 (construct-pair x lev))
		((vector? x)
		 (construct-vector x lev))
		(else (list 'quote x))))

	(define (q-c-h x)
	  (list 'quote (reverse x)))

	(define (construct-vector v lev)
	  (let* ((state 'quote)
		 (subs (map (lambda (y)
			      (let ((r (construct-any y lev)))
				(case (car r)
				  ((quote) r)
				  ((splice)
				   (set! state 'splice)
				   r)
				  (else
				   (if (not (eq? state 'splice))
				       (set! state 'collect))
				   r))))
			    (vector->list v))))
	    (case state
	      ((quote) (list 'quote v))
	      ((collect)
	       (cons 'vector subs))
	      (else
	       (list 'list->vector
		     (letrec
			 ((recur
			   (lambda (constant-heading rest)
			     (cond ((not (pair? rest))
				    (q-c-h constant-heading))
				   ((eq? (caar rest) 'quote)
				    (recur
				     (cons (cadar rest)
					   constant-heading)
				     (cdr rest)))
				   ((eq? (caar rest) 'splice)
				    (if (null? constant-heading)
					(list 'append
					      (list 'eval (cadar rest))
					      (recur '() (cdr rest)))
					(list
					 'append
					 (q-c-h constant-heading)
					 (list 'append
					       (list 'eval (cadar rest))
					       (recur '() (cdr rest))))))
				   ((null? constant-heading)
				    (list 'cons
					  (car rest)
					  (recur '() (cdr rest))))
				   (else
				    (list 'append
					  (q-c-h constant-heading)
					  (list 'cons
						(car rest)
						(recur '() (cdr rest)))))))))
		       (recur '() subs)))))))

	(define (construct-pair x lev)
	  (letrec
	      ((recur
		(lambda (constant-heading rest)
		  (cond ((not (pair? rest))
			 (list 'quote
			       (letrec
				   ((loop (lambda (l r)
					    (if (pair? l)
						(loop (cdr l)
						      (cons (car l) r))
						r))))
				 (loop constant-heading rest))))
			((and (zero? lev) (uq? rest))
			 (if (pair? constant-heading)
			     (list 'append
				   (q-c-h constant-heading)
				   (list 'eval (cadr rest)))
			     (list 'eval (cadr rest))))
			(else
			 (let ((item (construct-any (car rest) lev)))
			   (case (car item)
			     ((quote)
			      (recur (cons (cadr item) constant-heading)
				     (cdr rest)))
			     ((splice)
			      (if (null? constant-heading)
				  (list 'append
					(list 'eval (cadr item))
					(recur '() (cdr rest)))
				  (list 'append
					(q-c-h constant-heading)
					(list 'append
					      (list 'eval (cadr item))
					      (recur '() (cdr rest))))))
			     (else
			      (if (null? constant-heading)
				  (list 'cons
					item
					(recur '() (cdr rest)))
				  (list
				   'append
				   (q-c-h constant-heading)
				   (list 'cons
					 item
					 (recur '() (cdr rest)))))))))))))
	    (recur '() x)))

	(define (a-qq-construct x cont)
	  (case (car x)
	    ((splice) (syntax-error "ill-placed unquote-splicing in" expr))
	    ((eval)
	     (a (cadr x) kv
		(lambda (e fv mv)
		  (cont (list 'eval e) fv mv))))
	    ((quote)
	     (cont x '() '()))
	    ((cons append)
	     (a-qq-construct
	      (cadr x)
	      (lambda (e1 fv1 mv1)
		(a-qq-construct
		 (caddr x)
		 (lambda (e2 fv2 mv2)
		   (cont (list (car x) e1 e2)
			 (set-union fv1 fv2)
			 (set-union mv1 mv2)))))))
	    ((vector)
	     (letrec ((a-qq-list
		       (lambda (l cont)
			 (if (pair? l)
			     (a-qq-construct
			      (car l)
			      (lambda (x xfv xmv)
				(a-qq-list (cdr l)
					   (lambda (ll lfv lmv)
					     (cont (cons x ll)
						   (set-union xfv lfv)
						   (set-union xmv lmv))))))
			     (cont '() '() '())))))
	       (a-qq-list (cdr x)
			  (lambda (l fv mv)
			    (cont (cons 'vector l) fv mv)))))
	    ((list->vector)
	     (a-qq-construct (cadr x)
			     (lambda (e fv mv)
			       (cont (list 'list->vector e) fv mv))))
	    (else (error "internal compiler error - wrong qq-construct"))))

	(if (qq? expr)
	    (a-qq-construct
	     (construct-any (cadr expr) 0)
	     (lambda (e fv mv)
	       (cont (list 'quasiquote e) fv mv)))
	    (syntax-error "bad quasiquote syntax" expr)))

      ;; body of ``annotate''
      (a expr '() (lambda (ex fv mv) ex)))

    ;; procedure managers (used to maintain maximum stack top, list of
    ;; constants as well as label counter and pass those things around)
    (define (new-proc-manager)
      (let ((next-label 0)
	    (constants '())
	    (stack-max 0))
	(lambda (cmd)
	  ;; get back ionformation
	  (case cmd
	    ((number-of-labels) next-label)
	    ((all-constants)
	     (reverse constants))
	    ((stack-req) stack-max)
	    ((new-label!)
	     (let ((old next-label))
	       (set! next-label (+ next-label 1))
	       old))
	    ((constant-index!)
	     (lambda (item)
	       (let* ((item (list 'quote item))
		      (m (member item constants)))
		 (if m
		     (length (cdr m))
		     (begin (set! constants (cons item constants))
			    (length (cdr constants)))))))
	    ((code-index!)
	     (lambda (cod)
	       (set! constants (cons cod constants))
	       (length (cdr constants))))
	    ((stack-top!)
	     (lambda (new)
	       (if (> new stack-max)
		   (set! stack-max new))))
	    (else (error "bad command for procedure manager"))))))
    ;; proc-manager access procedures
    (define (number-of-labels pm) (pm 'number-of-labels))
    (define (all-constants pm) (pm 'all-constants))
    (define (stack-req pm) (pm 'stack-req))
    (define (new-label! pm) (pm 'new-label!))
    (define (constant-index! pm item) ((pm 'constant-index!) item))
    (define (code-index! pm cod) ((pm 'code-index!) cod))
    (define (stack-top! pm new) ((pm 'stack-top!) new))

    ;; fast code lists... (destroying operations)
    ;; pairs pointing to head and last cons..., () in head means empty
    (define (cconc! l1 l2)
      (if (null? (car l1))
	  (set-car! l1 (car l2))
	  (set-cdr! (cdr l1) (car l2)))
      (if (not (null? (car l2)))
	  (set-cdr! l1 (cdr l2)))
      l1)
    (define (cconc*! . l)
      (define (loop l)
	(if (pair? l)
	    (if (pair? (cdr l))
		(cconc! (car l) (loop (cdr l)))
		(car l))
	    (cempty)))
      (loop l))
    (define (ccons! instr l)
      (let ((h (cons instr (car l))))
	(if (null? (car l))
	    (set-cdr! l h))
	(set-car! l h)
	l))
    (define (cconc1! l instr)
      (let ((t (list instr)))
	(if (null? (car l))
	    (set-car! l t)
	    (set-cdr! (cdr l) t))
	(set-cdr! l t)
	l))
    (define (cempty) (cons '() '()))
    (define (list->code l)
      (define (last-cdr l)
	(if (pair? (cdr l)) (last-cdr (cdr l)) l))
      (if (null? l)
	  (cempty)
	  (cons l (last-cdr l))))
    (define (code . l) (list->code l))

    ;; is a function symbol in the car/cdr family?
    (define (car/cdr-combination? sym)
      (define (cadr-string? s)
	(let ((len-1 (- (string-length s) 1)))
	  (define (only-ad-in-between? i)
	    (cond ((>= i len-1) #t)
		  ((not (let ((c (string-ref s i)))
			  (or (char=? c #\a) (char=? c #\d))))
		   #f)
		  (else (only-ad-in-between? (+ i 1)))))
	  (and (> len-1 1)
	       (char=? (string-ref s 0) #\c)
	       (char=? (string-ref s len-1) #\r)
	       (only-ad-in-between? 1))))
      (and (symbol? sym)
	   (cadr-string? (symbol->string sym))))

    (define (open-compiled-when-global? sym)
      (and *open-compilation-enabled*
	   (or (memq sym '(not cons))
	       (car/cdr-combination? sym))))

    ;; The following list records all existing definitions for functions
    ;; from the car/cdr family.  Whenever a symbol out of this family
    ;; is mentioned in non-operator position while it is not defined
    ;; the compiler will automatically provide the definition.
    ;; However, if the symbol only appears in operator-position then
    ;; the call will be inlined and no actual definition is needed.
    (define existing-car/cdr-definitions '(car cdr))

    ;; compile expressions into instruction sequences
    ;; function takes:
    ;;  1. the expression
    ;;  2. the current variable bindings
    ;;  3. the current stack depth
    ;;  4. a procedure manager for the current procedure (lambda)
    ;;  5. the intended use of the result, i.e. one of:
    ;;      command push result
    ;;  6. a string containing the ``name'' of the current level
    ;; it returns:
    ;;  list of instructions (fast list, see above)
    ;; and updates:
    ;;  code manager
    ;; variable bindings: a-list with RHSs out of
    ;;   (stack <spos>)
    ;;   (stack-cell <spos>)
    ;;   (vector <spos> <vpos>)
    ;;   (vector-cell <spos> <vpos>)
    ;; stack positions are absolute

    ;; compile expression with all arguments specified (used to compile
    ;; new lambdas)
    (define (c-expr-full expr bind sd pm use fnam)
      ;; compile-expression with binding (and fnam) given (used to compile
      ;; inside new binding construct)
      (define (c-expr-bind expr bind sd use fnam)
	;; check whether function symbol is open compiled...
	(define (open-compiled? f)
	  (and (open-compiled-when-global? f)
	       (not (assq f bind))))
	;; compile expression with minimum arguments (stack depth and use)
	(define (c-expr expr sd use)
	  ;; compile a constant
	  (define (c-constant const use surround)
	    (if (eq? use 'command)
		(begin
		  (warning "unused constant" const "in" surround)
		  (cempty))
		(begin
		  (stack-top! pm (+ sd 1))
		  (ccons!
		   (cond ((assv
			   const '((#t take-true)
				   (#f take-false)
				   (() take-nil)))
			  =>
			  cdr)
			 (else (list 'take
				     (constant-index! pm const))))
		   (if (eq? use 'push)
		       (cempty)
		       (code '(exit)))))))
	  ;; compile conditional expression, handle not, and and or.
	  ;; t-lab and e-lab denote the label where to jump in the
	  ;; then- and else-case, respectively.  They can be #f to denote
	  ;; ``fall through''.
	  ;; t-retain and e-retain denote what to keep on top of the stack
	  ;; in the then- and else-case, resp.  #t means ``keep the positive
	  ;; result of the test or #t'', #f means ``keep #f'' and () means
	  ;; ``pop the result, keep nothing''.
	  (define (c-condition expr sd t-lab e-lab t-retain e-retain)
	    (c-condition-bind expr bind sd t-lab e-lab t-retain e-retain))
	  ;; with explicit bindings... (needed for DO)
	  (define (c-condition-bind expr bind sd t-lab e-lab t-retain e-retain)
	    ;; unconditional jumps
	    (define (uncond lab retain true)
	      (let ((val
		     (cond
		      ((boolean? retain)
		       (if retain
			   (c-constant true 'push expr)
			   (code (list 'take-false))))
		      (else (cempty))))
		    (jump
		     (if lab
			 (code (list 'jump-forward lab))
			 (cempty))))
		(cconc! val jump)))
	    ;; deal with ordinary expressions in conditionals:
	    (define (eval t-lab e-lab t-retain e-retain)
	      (if (not (or t-lab e-lab))
		  (cond ((null? t-retain)
			 (if (null? e-retain)
			     (c-expr-bind expr bind sd 'command fnam)
			     (error "compiler error in conditional (1)")))
			((null? e-retain)
			 (error "compiler error in conditional (2)"))
			((eq? t-retain e-retain)
			 (error "compiler error in conditional (3)"))
			(t-retain (c-expr-bind expr bind sd 'push fnam))
			(else (cconc1! (c-expr-bind expr bind sd 'push fnam)
				       (list 'not))))
		  (cconc!
		   (c-expr-bind expr bind sd 'push fnam)
		   (if t-lab
		       (if e-lab
			   (cond ((null? e-retain)
				  (case t-retain
				    ((#t)
				     (code (list 'false?jump+pop e-lab)
					   (list 'jump-forward t-lab)))
				    ((#f)
				     (code (list 'pop-false?jump e-lab)
					   (list 'take-false)
					   (list 'jump-forward t-lab)))
				    (else
				     (code (list 'pop-false?jump e-lab)
					   (list 'jump-forward t-lab)))))
				 ((null? t-retain)
				  (if (not e-retain)
				      (code (list 'true?jump+pop t-lab)
					    (list 'jump-forward e-lab))
				      (code (list 'pop-true?jump t-lab)
					    (list 'take-true)
					    (list 'jump-forward e-lab))))
				 (t-retain
				  (if e-retain
				      (code (list 'true?jump:pop t-lab)
					    (list 'take-true)
					    (list 'jump-forward e-lab))
				      (code (list 'true?jump t-lab)
					    (list 'jump-forward e-lab))))
				 (e-retain
				  (code (list 'not)
					(list 'true?jump e-lab)
					(list 'jump-forward t-lab)))
				 (else
				  (code (list 'false?jump:pop e-lab)
					(list 'take-false)
					(list 'jump-forward t-lab))))
			   (cond
			    ((null? e-retain)
			     (cond
			      ((null? t-retain)
			       ;; () ()
			       (code (list 'pop-true?jump t-lab)))
			      (t-retain
			       ;; #t ()
			       (code (list 'true?jump:pop t-lab)))
			      (else
			       ;; #f ()
			       (code '(not)
				     (list 'false?jump:pop t-lab)))))
			    (e-retain
			     (cond
			      ((null? t-retain)
			       ;; () #t
			       (code (list 'pop-true?jump t-lab)
				     '(take-true)))
			      (t-retain
			       ;; #t #t
			       (code (list 'true?jump:pop t-lab)
				     '(take-true)))
			      (else
			       ;; #f #t
			       (code '(not)
				     (list 'false?jump t-lab)))))
			    ((null? t-retain)
			     ;; () #f
			     (code (list 'true?jump+pop t-lab)))
			    (t-retain
			     ;; #t #f
			     (code (list 'true?jump t-lab)))
			    (else
			     ;; #f #f
			     (code (list 'not)
				   (list 'false?jump:pop t-lab)
				   '(take-false)))))
		       (cond
			((null? e-retain)
			 (cond ((null? t-retain)
				;; () ()
				(code (list 'pop-false?jump e-lab)))
			       (t-retain
				;; #t ()
				(code (list 'false?jump+pop e-lab)))
			       (else
				;; #f ()
				(code (list 'false?jump+pop e-lab)
				      '(take-false)))))
			(e-retain
			 (cond ((null? t-retain)
				;; () #t
				(code '(not)
				      (list 'true?jump:pop e-lab)))
			       (t-retain
				;; #t #t
				(code '(not)
				      (list 'true?jump:pop e-lab)))
			       (else
				;; #f #t
				(code '(not)
				      (list 'true?jump e-lab)))))
			((null? t-retain)
			 ;; () #f
			 (code (list 'false?jump:pop e-lab)))
			(t-retain
			 ;; #t #f
			 (code (list 'false?jump e-lab)))
			(else
			 ;; #f #f
			 (code (list 'false?jump:pop e-lab)
			       '(take-false))))))))

	    ;; body of c-condition-bind
	    (cond ((eq? expr #f)
		   (uncond e-lab e-retain #t))
		  ((symbol? expr)
		   (eval t-lab e-lab t-retain e-retain))
		  ((not (pair? expr))
		   (uncond t-lab t-retain expr))
		  ((eq? (car expr) 'quote)
		   (uncond t-lab t-retain (cadr expr)))
		  (else
		   (case (car expr)
		     ((not)
		      (if (and (= (length expr) 2)
			       (open-compiled? 'not))
			  (c-condition-bind (cadr expr) bind sd e-lab t-lab
					    e-retain t-retain)
			  (eval t-lab e-lab t-retain e-retain)))
		     ((and)
		      (let ((ne-lab (or e-lab (new-label! pm))))
			(do ((l (cdr expr) (cdr l))
			     (cod
			      (cempty)
			      (cconc!
			       cod
			       (if (pair? (cdr l))
				   (c-condition-bind (car l) bind sd
						     #f ne-lab '() e-retain)
				   (cconc!
				    (c-condition-bind (car l)  bind sd
						      t-lab e-lab
						      t-retain e-retain)
				    (if e-lab (cempty) (code ne-lab)))))))
			    ((not (pair? l)) cod))))
		     ((or)
		      (let ((nt-lab (or t-lab (new-label! pm))))
			(do ((l (cdr expr) (cdr l))
			     (cod
			      (cempty)
			      (cconc!
			       cod
			       (if (pair? (cdr l))
				   (c-condition-bind (car l) bind sd
						     nt-lab #f t-retain '())
				   (cconc!
				    (c-condition-bind (car l) bind sd
						      t-lab e-lab
						      t-retain e-retain)
				    (if t-lab (cempty) (code nt-lab)))))))
			    ((not (pair? l)) cod))))
		     (else (eval t-lab e-lab t-retain e-retain))))))

	  (define (collect-bindings vars nargs cont)
	    ;; cont is a function of 4 arguments:
	    ;;  1. code to collect bindings
	    ;;  2. code to take bindings apart again (inside new function)
	    ;;  3. bindings for new function
	    ;;  4. initial stack depth in new function
	    (define (add-pos v p l)
	      (define (loop l)
		(cond ((null? l)
		       (list (list (list v) p)))
		      ((and (pair? (caar l))
			    (= (cadar l) p))
		       (cons (list (cons v (caar l)) p) (cdr l)))
		      (else (cons (car l) (loop (cdr l))))))
	      (loop l))
	    ;; make a single binding for var at stack position pos
	    (define (sbind var pos)
	      (let ((a (assq var bind)))
		(list var
		      (case (cadr a)
			((stack vector) 'stack)
			(else 'stack-cell))
		      pos)))
	    ;; make a single binding for var at vector position pos
	    ;; this vector always sits at stack position 0
	    (define (vbind var pos)
	      (let ((a (assq var bind)))
		(list var
		      (case (cadr a)
			((stack vector) 'vector)
			(else 'vector-cell))
		      0 pos)))
	    ;; make a collection of vector-bindings, vector is located at pos
	    (define (lbind vars pos)
	      (map (lambda (v)
		     (let ((a (assq v bind)))
		       (list v
			     (case (cadr a)
			       ((vector incomplete-vector)
				'vector)
			       ((vector-cell incomplete-vector-cell)
				'vector-cell))
			     pos (cadddr a))))
		   vars))
	    ;; body of collect-bindings
	    (let ((nsd (+ nargs 1)))
	      (if (not (pair? vars))
		  (begin
		    (stack-top! pm (+ sd 1))
		    (cont (code '(take-false)) (cempty) '() nsd))
		  (do ((v vars (cdr v))
		       (p '()
			  (cond
			   ((assq (car v) bind)
			    =>
			    (lambda (b)
			      (case (cadr b)
				((incomplete-vector incomplete-vector-cell)
				 (add-pos (car v) (caddr b) p))
				((stack stack-cell)
				 (cons (list (car v) 'stack (caddr b)) p))
				(else
				 (cons (cons (car v)
					     (cons 'vector (cddr b)))
				       p)))))
			   (else
			    ;; the above clause *must* succeed
			    ;;  otherwise there is something wrong...
			    (error "compile-error in collect-bindings")))))
		      ((not (pair? v))
		       (cond
			((not (pair? (cdr p)))
			 ;; only one thing to save...
			 (stack-top! pm (+ sd 1))
			 (cond ((pair? (caar p))
				(cont
				 (code (list 'get-loc (cadar p)))
				 (cempty)
				 (lbind (caar p) 0)
				 nsd))
			       ((eq? (cadar p) 'stack)
				(cont
				 (code (list 'get-loc (caddar p)))
				 (cempty)
				 (list (sbind (caar p) 0))
				 nsd))
			       (else
				(cont
				 (code (cons 'get-vec (cddar p)))
				 (cempty)
				 (list (sbind (caar p) 0))
				 nsd))))
			(else
			 ;; many things to save...
			 (stack-top! pm (+ sd (length p)))
			 (let ((loads
				(list->code
				 (map (lambda (pos)
					(cond ((pair? (car pos))
					       (list 'get-loc (cadr pos)))
					      ((eq? (cadr pos) 'stack)
					       (list 'get-loc (caddr pos)))
					      (else
					       (cons 'get-vec (cddr pos)))))
				      p))))
			   (letrec
			       ((loop
				 (lambda (pl scod bind idx pos)
				   (cond
				    ((not (pair? pl))
				     (cont
				      (cconc1! loads
					       (list 'make-vector (length p)))
				      scod bind pos))
				    ((pair? (caar pl))
				     (loop (cdr pl)
					   (cconc1!
					    scod
					    (list 'get-vec 0 idx))
					   (append (lbind (caar pl) pos) bind)
					   (+ idx 1)
					   (+ pos 1)))
				    (else
				     (loop (cdr pl)
					   scod
					   (cons (vbind (caar pl) idx) bind)
					   (+ idx 1)
					   pos))))))
			     (loop p (cempty) '() 0 nsd))))))))))

	  (define (car/cdr-stat-list sym)
	    (let* ((strg (symbol->string sym))
		   (len (- (string-length strg) 1)))
	      (do ((i 1 (+ i 1))
		   (l '()
		      (cons (if (char=? (string-ref strg i) '#\a)
				'(car) '(cdr))
			    l)))
		  ((>= i len) l))))

	  (define (c-call fun nargs use)
	    (cond ((and (symbol? fun)
			(open-compiled? fun))
		   ;; open-compiled function symbol
		   (case fun
		     ((not)
		      (if (not (= nargs 1))
			  (semantic-error
			   "wrong number of args to procedure not:" nargs))
		      (case use
			((result) (code '(not) '(exit)))
			((push) (code '(not)))
			(else
			 (warning "useless call to not")
			 (code '(pop)))))
		     ((cons)
		      (if (not (= nargs 2))
			  (semantic-error
			   "wrong number of args to procedure cons:"
			   nargs))
		      (case use
			((result) (code '(cons) '(exit)))
			((push) (code '(cons)))
			(else
			 (warning "useless call to cons")
			 (code '(multi-pop 2)))))
		     (else		; car/cdr
		      (if (not (= nargs 1))
			  (semantic-error
			   "wrong number of args to procedure"
			   fun ":" nargs))
		      (if (eq? use 'command)
			  (begin (warning "useless call to " fun)
				 (code '(pop))))
		      (let ((cod (list->code (car/cdr-stat-list fun))))
			(if (eq? use 'push)
			    cod
			    (cconc1! cod '(exit)))))))
		  ((and (pair? fun)
			(eq? (car fun) 'lambda)
			(list? (caddr fun)))
		   (if (not (= (length (caddr fun)) nargs))
		       (semantic-error "argument list mismatch:" nargs
				       "actuals for these formals:"
				       (caddr fun)))
		   ;; explicit lambda with matching argument list
		   (let ((mv (cadddr fun))
			 (vars (caddr fun))
			 (body (cddddr fun)))
		     (define (loop v b i cod)
		       (if
			(pair? v)
			(let ((modif (memq (car v) mv)))
			  (loop
			   (cdr v)
			   (cons (list (car v) (if modif 'stack-cell 'stack) i)
				 b)
			   (- i 1)
			   (if modif (ccons! (list 'make-cell i) cod) cod)))
			(cconc*!
			 cod
			 (c-expr-bind
			  (cons 'begin body) b (+ sd nargs) use fnam)
			 (if (or (zero? nargs) (eq? use 'result))
			     (cempty)
			     (let ((pop (if (= nargs 1)
					    '(pop)
					    (list 'multi-pop nargs))))
			       (if (eq? use 'push)
				   (code (list 'put-loc sd) pop)
				   (code pop)))))))
		     (loop vars bind (+ sd nargs -1) (cempty))))
		  (else
		   (cconc! (c-expr fun (+ sd nargs) 'push)
			   (if (eq? use 'command)
			       (code (list 'call nargs) '(pop))
			       (code (list
				      (if (eq? use 'push) 'call 'call-exit)
				      nargs)))))))

	  ;; body of c-expr
	  (cond
	   ((pair? expr)
	    (case (car expr)
	      ((quote) (c-constant (cadr expr) use expr))
	      ((set!)
	       (let* ((val (c-expr (caddr expr) sd 'push))
		      (assg
		       (cond ((assq (cadr expr) bind)
			      =>
			      (lambda (b)
				(cons
				 (case (cadr b)
				   ((stack-cell)
				    (if (eq? use 'command)
					'put-loc-cell-pop
					'put-loc-cell))
				   ((vector-cell)
				    (if (eq? use 'command)
					'put-vec-cell-pop
					'put-vec-cell))
				   ((incomplete-vector-cell)
				    (semantic-error
				     "cannot assign to letrec-variable"
				     (car b)
				     "wile in letrec initialization"))
				   ;; the following case should never occur
				   (else
				    (error
				     "compiler error while compiling set!")))
				 (cddr b))))
			     (else
			      (if (open-compiled? (cadr expr))
				  (semantic-error
				   "attempt to alter value of" (cadr expr)
				   "(open-compiled)"))
			      (list (if (eq? use 'command)
					'put-glob-pop
					'put-glob)
				    (constant-index! pm (cadr expr))))))
		      (trailer
		       (if (eq? use 'result)
			   (code assg '(exit))
			   (code assg))))
		 (cconc! val trailer)))
	      ((lambda)
	       (if
		(eq? use 'command)
		(begin
		  (warning "unnecessary procedure closure creation" expr)
		  (cempty))
		(let* ((npm (new-proc-manager))
		       (args (caddr expr))
		       (len (length args))
		       (take-rest (not (list? args)))
		       (fv (cadr expr))
		       (mv (cadddr expr))
		       (body (cddddr expr))
		       (nargs (if take-rest (+ len 1) len))
		       (rargs (do ((l args (cdr l))
				   (i 1 (+ i 1))
				   (r '() (cons (cons (car l) i) r)))
				  ((not (pair? l))
				   (if take-rest (cons (cons l i) r) r)))))
		  (collect-bindings
		   fv nargs
		   (lambda (collect init nbind depth)
		     (stack-top! npm depth)
		     (let* ((cod
			     (cconc*!
			      init
			      (list->code
			       (map (lambda (v)
				      (let ((a (assq v rargs)))
					(list 'make-cell (cdr a))))
				    mv))
			      (c-expr-full
			       (cons 'begin body)
			       (append
				(map (lambda (ra)
				       (list (car ra)
					     (if (memq (car ra) mv)
						 'stack-cell 'stack)
					     (cdr ra)))
				     rargs)
				nbind)
			       depth npm 'result fnam)))
			    (fun (append
				  (list fnam
					len
					take-rest
					(list->vector
					 (reverse (map car rargs)))
					(all-constants npm)
					(stack-req npm)
					(number-of-labels npm))
				  (peephole (car cod))))
			    (lambda-stat
			     (list 'lambda (code-index! pm fun))))
		       (cconc!
			collect
			(if (eq? use 'result)
			    (code lambda-stat '(exit))
			    (code lambda-stat)))))))))
	      ((if)
	       (let ((conseq (c-expr (caddr expr) sd use)))
		 (if (= (length expr) 3)
		     (let ((lab (new-label! pm)))
		       (case use
			 ((result)
			  (cconc*!
			   (c-condition (cadr expr) sd
					lab #f '() #f)
			   (code '(exit) lab)
			   conseq))
			 (else
			  (cconc!
			   (c-condition (cadr expr) sd
					#f lab '()
					(if (eq? use 'command) '() '#f))
			   (cconc1! conseq lab)))))
		     (let ((lab (new-label! pm))
			   (altern (c-seq (cdddr expr) sd use)))
		       (if (eq? use 'result)
			   (cconc*!
			    (c-condition (cadr expr) sd #f lab '() '())
			    conseq
			    (ccons! lab altern))
			   (let ((end-lab (new-label! pm)))
			     (cconc*!
			      (c-condition (cadr expr) sd #f lab '() '())
			      conseq
			      (code (list 'jump-forward end-lab)
				    lab)
			      altern
			      (code end-lab))))))))
	      ((cond)
	       (let ((end-lab (new-label! pm)))
		 (define (rec l)
		   (let ((lab (new-label! pm)))
		     (cond ((pair? (cdr l))
			    (let ((cl (car l))
				  (rest (ccons! lab (rec (cdr l)))))
			      (cond ((not (pair? (cdr cl)))
				     (if (eq? use 'result)
					 (cconc*!
					  (c-condition (car cl) sd #f lab
						       #t '())
					  (code '(exit))
					  rest)
					 (cconc!
					  (c-condition
					   (car cl) sd end-lab #f
					   (if (eq? use 'command) '() '#t)
					   '())
					  rest)))
				    ((not (eq? (cadr cl) '=>))
				     (cconc*!
				      (c-condition (car cl) sd #f lab '() '())
				      (c-seq (cdr cl) sd use)
				      (if (eq? use 'result)
					  (cempty)
					  (code (list 'jump-forward end-lab)))
				      rest))
				    (else
				     (cconc*!
				      (c-condition (car cl) sd #f lab #t '())
				      (c-call (caddr cl) 1 use)
				      (if (eq? use 'result)
					  (cempty)
					  (code (list 'jump-forward end-lab)))
				      rest)))))
			   ((eq? (caar l) 'else)
			    (cconc1!
			     (c-seq (cdar l) sd use)
			     end-lab))
			   ;; last clause is not else-clause
			   ((not (pair? (cdar l)))
			    (cconc1!
			     (c-expr (caar l) sd use)
			     end-lab))
			   ((not (eq? (cadar l) '=>))
			    (if (eq? use 'result)
				(cconc*!
				 (c-condition (caar l) sd lab #f '() #f)
				 (code '(exit) lab)
				 (c-seq (cdar l) sd use))
				(cconc*!
				 (c-condition (caar l) sd #f end-lab '()
					      (if (eq? use 'command) '() '#f))
				 (c-seq (cdar l) sd use)
				 (code end-lab))))
			   ;; last clause is => clause
			   ((eq? use 'result)
			    (cconc*!
			     (c-condition (caar l) sd lab #f #t #f)
			     (code '(exit) lab)
			     (c-call (caddar l) 1 use)))
			   (else
			    (cconc*!
			     (c-condition (caar l) sd #f end-lab #t
					  (if (eq? use 'command) '() #f))
			     (c-call (caddar l) 1 use)
			     (code end-lab))))))
		 (rec (cdr expr))))
	      ((case)
	       (let ((switch (c-expr (cadr expr) sd 'push))
		     (end-lab (new-label! pm)))
		 (define (jump-stat case-label lab)
		   (if (pair? (cdr case-label))
		       (list 'memv?jump+pop
			     (constant-index! pm case-label)
			     lab)
		       (list 'eqv?jump+pop
			     (constant-index! pm (car case-label))
			     lab)))
		 (define (loop l jumps instr)
		   (let ((lab (new-label! pm))
			 (seq (c-seq (cdar l) sd use)))
		     (cond
		      ((pair? (cdr l))
		       (loop (cdr l)
			     (cconc1! jumps (jump-stat (caar l) lab))
			     (cconc*! instr
				      (code lab)
				      seq
				      (if (not (eq? use 'result))
					  (code (list 'jump-forward end-lab))
					  (cempty)))))
		      ((eq? (caar l) 'else)
		       (cconc*! switch
				jumps
				(code '(pop))
				(c-seq (cdar l) sd use)
				(if (not (eq? use 'result))
				    (code (list 'jump-forward end-lab))
				    (cempty))
				instr
				(code end-lab)))
		      (else
		       (cconc*! switch
				jumps
				(code (jump-stat (caar l) lab))
				(case use
				  ((command)
				   (code '(pop) (list 'jump-forward end-lab)))
				  ((result) (code '(exit)))
				  (else (code (list 'jump-forward end-lab))))
				instr
				(code lab)
				seq
				(code end-lab))))))
		 (loop (cddr expr) (cempty) (cempty))))
	      ((and)
	       (let ((lab (new-label! pm))
		     (e-retain (if (eq? use 'command) '() '#f)))
		 (do ((l (cdr expr) (cdr l))
		      (code (cempty)
			    (cconc!
			     code
			     (if (pair? (cdr l))
				 (c-condition (car l) sd #f lab '() e-retain)
				 (cconc1!
				  (c-expr (car l) sd use)
				  lab)))))
		     ((not (pair? l))
		      (if (eq? use 'result)
			  (cconc1! code '(exit))
			  code)))))
	      ((or)
	       (let ((lab (new-label! pm))
		     (t-retain (if (eq? use 'command) '() '#t)))
		 (do ((l (cdr expr) (cdr l))
		      (code (cempty)
			    (cconc!
			     code
			     (if (pair? (cdr l))
				 (c-condition (car l) sd lab #f t-retain '())
				 (cconc1!
				  (c-expr (car l) sd use)
				  lab)))))
		     ((not (pair? l))
		      (if (eq? use 'result)
			  (cconc1! code '(exit))
			  code)))))
	      ((let)			; no named let
	       (let ((mv (caddr expr))
		     (ini (cadr expr))
		     (body (cdddr expr)))
		 (define (loop l lsd b cod)
		   (if (pair? l)
		       (let ((modif (memq (caar l) mv)))
			 (loop
			  (cdr l)
			  (+ lsd 1)
			  (cons (cons (caar l)
				      (list (if modif 'stack-cell 'stack) lsd))
				b)
			  (cconc*! cod
				   (c-expr-bind
				    (cadar l) bind lsd 'push
				    (string-append fnam "."
						   (symbol->string (caar l))))
				   (if modif
				       (code (list 'make-cell lsd))
				       (cempty)))))
		       (cconc*!
			cod
			(c-expr-bind (cons 'begin body) b lsd use fnam)
			(let ((len (length ini)))
			  (cond ((or (zero? len) (eq? use 'result))
				 (cempty))
				((= len 1)
				 (code (if (eq? use 'command)
					   '(pop)
					   (list 'put-loc-pop sd))))
				(else
				 (let ((pop (list 'multi-pop len)))
				   (if (eq? use 'push)
				       (code (list 'put-loc sd) pop)
				       (code pop)))))))))
		 (loop ini sd bind (cempty))))
	      ((letrec)
	       (let ((ini (cadr expr))
		     (body (cdddr expr))
		     (lsd (+ sd 1)))
		 (cond ((not (pair? ini)) (c-seq body sd use))
		       ((not (pair? (cdr ini)))
			;; special case: only one variable -- easy
			(let ((b (cons (list (caar ini) 'stack-cell sd) bind)))
			  (stack-top! pm lsd)
			  (cconc*!
			   (code '(take-false)
				 (list 'make-cell sd))
			   (c-expr-bind
			    (cadar ini) b lsd 'push
			    (string-append fnam "."
					   (symbol->string (caar ini))))
			   (code (list 'put-loc-cell-pop sd))
			   (c-expr-bind (cons 'begin body) b lsd use fnam)
			   (case use
			     ((result) (cempty))
			     ((push) (code (list 'put-loc-pop sd)))
			     (else (code '(pop)))))))
		       (else
			(let* ((vars (map car ini))
			       (len (length ini))
			       (pos (do ((i (- len 1) (- i 1))
					 (p '() (cons i p)))
					((negative? i) p)))
			       (mv (caddr expr))
			       ;; normal bindings
			       (n-b (append
				     (map (lambda (v p)
					    (list v
						  (if (memq v mv)
						      'vector-cell 'vector)
						  sd p))
					  vars pos)
				     bind))
			       ;; incomplete bindings for collect-bindings
			       (l-b (append
				     (map (lambda (v p)
					    (list v
						  (if (memq v mv)
						      'incomplete-vector-cell
						      'incomplete-vector)
						  sd p))
					  vars pos)
				     bind)))
			  (stack-top! pm lsd)

			  (do ((cod
				(cempty)
				(cconc*!
				 (if (memq (caar l) mv)
				     (code '(take-false)
					   (list 'make-cell lsd)
					   (list 'put-vec-pop sd (car p)))
				     (cempty))
				 cod
				 (c-expr-bind
				  (cadar l) l-b lsd 'push
				  (string-append
				   fnam "." (symbol->string (caar l))))
				 (code (list (if (memq (caar l) mv)
						 'put-vec-cell-pop
						 'put-vec-pop)
					     sd (car p)))))
			       (l ini (cdr l))
			       (p pos (cdr p)))
			      ((not (pair? l))
			       (cconc*!
				(code (list 'allocate-vector len))
				cod
				(c-expr-bind
				 (cons 'begin body) n-b lsd use fnam)
				(case use
				  ((result) (cempty))
				  ((push) (code (list 'put-loc-pop sd)))
				  (else (code '(pop))))))))))))
	      ((do)
	       (let* ((mv (caddr expr))	; modified variables
		      (vars (cadr expr)) ; whole variable specification
		      (finish (cadddr expr)) ; test clause
		      (body (cddddr expr))
		      (lab (new-label! pm))
		      (end-lab (new-label! pm))
		      (len (length vars))
		      (nsd (+ sd len))	; stack depth in loop
		      (pos (do ((i (+ sd len -1) (- i 1)) ; variable locations
				(p '() (cons i p)))
			       ((< i sd) p)))
		      (nbind (append	; new bindings
			      (map (lambda (v p)
				     (list (car v)
					   (if (memq (car v) mv)
					       'stack-cell 'stack)
					   p))
				   vars pos)
			      bind))
		      (re-assigns	; re-assignments
		       (list->code
			(map (lambda (p) (list 'put-loc-pop p))
			     (reverse pos))))
		      (bodycod
		       (do ((cod (cempty)
				 (cconc! cod
					 (c-expr-bind
					  (car l) nbind nsd 'command fnam)))
			    (l body (cdr l)))
			   ((not (pair? l)) cod))))
		 (define (loop l sd1 cod1 sd2 cod2)
		   (if (pair? l)
		       (let ((modif (memq (caar l) mv)))
			 (loop (cdr l)
			       (+ sd1 1)
			       (cconc*!
				cod1
				(c-expr-bind
				 (cadar l) bind sd1 'push
				 (string-append
				  fnam ".INIT." (symbol->string (caar l))))
				(if modif
				    (code (list 'make-cell sd1))
				    (cempty)))
			       (+ sd2 1)
			       (cconc*!
				cod2
				(c-expr-bind
				 (caddar l) nbind sd2 'push
				 (string-append
				  fnam ".RE-INIT." (symbol->string (caar l))))
				(if modif
				    (code (list 'make-cell sd2))
				    (cempty)))))
		       (cconc*!
			cod1
			(code lab)
			(c-condition-bind
			 (car finish) nbind sd1 end-lab #f
			 (if (or (eq? use 'command)
				 (pair? (cdr finish)))
			     '() #t)
			 '())
			bodycod
			cod2
			re-assigns
			(code '(check)
			      (list 'jump-backward lab)
			      end-lab)
			(cond ((pair? (cdr finish))
			       (c-expr-bind (cons 'begin (cdr finish))
					    nbind sd1 use fnam))
			      ((eq? use 'result)
			       (code '(exit)))
			      (else
			       (cempty)))
			(cond ((or (zero? len) (eq? use 'result))
			       (cempty))
			      ((= len 1)
			       (code (if (eq? use 'command)
					 '(pop)
					 (list 'put-loc-pop sd))))
			      (else
			       (let ((pop (list 'multi-pop len)))
				 (if (eq? use 'push)
				     (code (list 'put-loc sd) pop)
				     (code pop))))))))
		 (loop vars sd (cempty) nsd (cempty))))
	      ((delay)
	       (if (eq? use 'command)
		   (begin
		     (warning "unnecessary promise closure creation" expr)
		     (cempty))
		   (let* ((npm (new-proc-manager))
			  (fv (cadr expr))
			  (body (caddr expr)))
		     (collect-bindings
		      fv 0
		      (lambda (collect init nbind depth)
			(stack-top! npm depth)
			(let* ((nfnam (string-append fnam ".DELAY"))
			       (cod
				(cconc!
				 init
				 (c-expr-full
				  body nbind depth npm 'result nfnam)))
			       (fun (append
				     (list nfnam 0 #f '#()
					   (all-constants npm)
					   (stack-req npm)
					   (number-of-labels npm))
				     (peephole (car cod))))
			       (delay-stat
				(list 'delay (code-index! pm fun))))
			  (cconc!
			   collect
			   (if (eq? use 'result)
			       (code delay-stat '(exit))
			       (code delay-stat)))))))))
	      ((quasiquote)
	       (letrec
		   ((cqq (lambda (part sd)
			   (case (car part)
			     ((eval)
			      (c-expr (cadr part) sd 'push))
			     ((quote)
			      (c-expr part sd 'push))
			     ((cons append)
			      (cconc*!
			       (cqq (caddr part) sd)
			       (cqq (cadr part) (+ sd 1))
			       (code (list (car part)))))
			     ((vector)
			      (let* ((vl (cdr part))
				     (len (length vl)))
				(do ((l (reverse vl) (cdr l))
				     (i (+ sd len -1) (- i 1))
				     (cod (cempty)
					  (cconc! (cqq (car l) i) cod)))
				    ((not (pair? l))
				     (cconc1!
				      cod
				      (list 'make-vector len))))))
			     ((list->vector)
			      (cconc1! (cqq (cadr part) sd)
				       '(list->vector)))
			     (else (error "compiler error - bad qq"))))))
		 (cconc!
		  (cqq (cadr expr) sd)
		  (case use
		    ((result) (code '(exit)))
		    ((command) (code '(pop)))
		    (else (cempty))))))
	      ((begin) (c-seq (cdr expr) sd use))
	      (else
	       (let ((len (length (cdr expr))))
		 (do ((l (cdr expr) (cdr l))
		      (i (+ sd len -1) (- i 1))
		      (code (cempty)
			    (cconc! (c-expr (car l) i 'push)
				    code)))
		     ((not (pair? l))
		      (cconc!
		       code
		       (c-call (car expr) len use))))))))
	   ((symbol? expr)
	    (if (eq? use 'command)
		(begin
		  (warning "unnecessary reference to variable" expr)
		  (cempty))
		(begin
		  (stack-top! pm (+ sd 1))
		  (ccons!
		   (cond ((assq expr bind)
			  =>
			  (lambda (b)
			    (cons (case (cadr b)
				    ((stack) 'get-loc)
				    ((stack-cell) 'get-loc-cell)
				    ((vector) 'get-vec)
				    ((vector-cell) 'get-vec-cell)
				    (else
				     (semantic-error
				      "cannot reference letrec variable"
				      expr
				      "while in letrec intialization")))
				  (cddr b))))
			 (else
			  (if (and (car/cdr-combination? expr)
				   (not (memq expr
					      existing-car/cdr-definitions)))
			      (define-asm
				`(define
				   ,expr
				   (,(symbol->string expr)
				    1 #f #(x) () 2 0
				    ,@(car/cdr-stat-list expr)
				    (exit)))))
			  (list 'get-glob (constant-index! pm expr))))
		   (if (eq? use 'push)
		       (cempty)
		       (code '(exit)))))))
	   (else (c-constant expr use expr))))

	(define (c-seq seq sd use)
	  (define (rec l)
	    (if (pair? (cdr l))
		(cconc! (c-expr (car l) sd 'command)
			(rec (cdr l)))
		(c-expr (car l) sd use)))
	  (rec seq))

	;; body of c-expr-bind
	(c-expr expr sd use))
      ;; body of c-expr-full
      (c-expr-bind expr bind sd use fnam))

    ;; compile a function definition
    (define (compile-function-definition def)
      (let ((edef (expand-define def))
	    (pm (new-proc-manager)))
	(if (not (and (eq? (car edef) 'define)
		      (pair? (caddr edef))
		      (eq? (caaddr edef) 'lambda)))
	    (syntax-error "not a function definition:" def))
	(global-ensure-legal-variable (cadr edef) def)
	(c-expr-full
	 (annotate (caddr edef))
	 '() 0 pm 'result (symbol->string (cadr edef)))
	(cons (cadr edef) (car (all-constants pm)))))

    ;; Simple basic peephole-``optimizer'' :-)
    (define (peephole code)
      ;; does nothing...
      (cons '(check) code))

    ;; assembly code ``pretty'' printer
    (define (write-pretty-asm asm)
      (define (write-n-spaces n)
	(if (not (zero? n))
	    (begin
	      (display " ")
	      (write-n-spaces (- n 1)))))
      (define (write-indented-nl item lev)
	(write-n-spaces lev)
	(write item)
	(newline))
      (define (startloop l lev i)
	(let ((head (car l)))
	  (define (loop l lev i)
	    (if (pair? l)
		(begin
		  (cond
		   ((= i 2)
		    (write-n-spaces lev)
		    (display (car l))
		    (newline))
		   ((= i 4)
		    (write-n-spaces lev)
		    (display "(")
		    (newline)
		    (for-each
		     (lambda (x)
		       (if (eq? (car x) 'quote)
			   (write-indented-nl x (+ lev 1))
			   (begin
			     (write-n-spaces (+ lev 1))
			     (display "(")
			     (newline)
			     (startloop x (+ lev 2) 0)
			     (write-n-spaces (+ lev 1))
			     (display ")")
			     (newline))))
		     (car l))
		    (write-n-spaces lev)
		    (display ")")
		    (newline))
		   (else
		    (if (= i 5)
			(begin
			  (write-n-spaces lev)
			  (display ";; -- body of ")
			  (write head)
			  (newline)))
		    (write-indented-nl (car l) lev)))
		  (loop (cdr l) lev (+ i 1)))))
	  (loop l lev i)))
      (display "(define ")
      (write (car asm))
      (newline)
      (display " (")
      (newline)
      (startloop (cdr asm) 2 0)
      (display "))")
      (newline))

    ;; read a function definition and compile it...
    (define (read-and-asm)
      (let ((def (read)))
	(if (eof-object? def)
	    #f
	    (begin (write-pretty-asm
		    (compile-function-definition def))
		   (newline)
		   #t))))

    ;; batch compiler loop
    (define (compile-until-eof)
      (define (loop)
	(if (read-and-asm) (loop)))
      (loop))

    ;; ... with file arguments given
    (define (compile-file from to)
      (with-input-from-file
	  from
	(lambda ()
	  (with-output-to-file
	      to
	    compile-until-eof))))

    ;; see whether x is a definition or a definition
    ;; sequence (within a ``begin'' clause)
    ;; if not -> #f; if yes -> list of definitions
    (define (definition-sequence x)
      (define (recur l)
	(cond ((not (pair? l)) '())
	      ((definition-sequence (car l))
	       =>
	       (lambda (s1)
		 (cond ((recur (cdr l))
			=>
			(lambda (s2) (append s1 s2)))
		       (else #f))))
	      (else #f)))
      (cond ((not (pair? x)) #f)
	    ((eq? (car x) 'define)
	     (list (expand-define x)))
	    ((eq? (car x) 'begin)
	     (recur (cdr x)))
	    (else #f)))

    ;; evaluate expression in toplevel environment
    ;; call cont with 5 arguments:
    ;;  1. result
    ;;  2. compilation time
    ;;  3. gc time during compilation
    ;;  4. execution time
    ;;  5. gc time during execution
    (define (with-result-and-times expr sy-sem-err cont)
      ;; the following hack only works by virtue of the fact, that user code
      ;; never returns ``into'' the compiler, not even with EVAL
      (set! *error-continuation* sy-sem-err) ; hack!!
      (cond ((definition-sequence expr)
	     =>
	     (lambda (s)
	       (define (loop l r cc cgcc ec egcc)
		 (if (pair? l)
		     (let* ((d (car l))
			    (sym (cadr d))
			    (pm (new-proc-manager))
			    (fnam (symbol->string sym))
			    (body (caddr d))
			    (c1 (clock))
			    (gcc1 (gc-clock))
			    (is-function
			     (and (pair? body) (eq? (car body) 'lambda))))
		       (global-ensure-legal-variable sym d)
		       (if (and is-function
				(open-compiled-when-global? sym))
			   (semantic-error
			    "attempt to re-define value of" sym
			    "(open compiled)"))
		       (c-expr-full
			(annotate
			 (if is-function
			     body
			     (list 'lambda '() (cons 'set! (cdr d)))))
			'() 0 pm 'result fnam)
		       (let ((c2 (clock))
			     (gcc2 (gc-clock)))
			 (if is-function
			     (define-asm (list 'define
					       sym
					       (car (all-constants pm))))
			     (execute-asm (car (all-constants pm))))
			 (loop (cdr l)
			       (cons sym r)
			       (+ (- c2 c1) cc)
			       (+ (- gcc2 gcc1) cgcc)
			       (+ (- (clock) c2) ec)
			       (+ (- (gc-clock) gcc2) egcc))))
		     (cont r cc cgcc ec egcc)))
	       (loop s '() 0 0 0 0)))
	    (else
	     (let ((pm (new-proc-manager))
		   (c1 (clock))
		   (gcc1 (gc-clock)))
	       (c-expr-full (annotate (list 'lambda '() expr))
			    '() 0 pm 'result "[unnamed]")
	       (let* ((c2 (clock))
		      (gcc2 (gc-clock))
		      (res
		       (execute-asm (car (all-constants pm)))))
		 (cont res
		       (- c2 c1)
		       (- gcc2 gcc1)
		       (- (clock) c2)
		       (- (gc-clock) gcc2)))))))

    (define (intern-eval expr)
      (with-result-and-times
       expr
       (lambda (ignore)
	 (error "eval: error during compilation"))
       (lambda (res t1 t2 t3 t4)
	 res)))

    ;; REPL
    (define (repl-until-eof k)
      (define (get)
	(display "> ")
	(read))
      (define (wrclock c gcc)
	(write c)
	(display " (")
	(write gcc)
	(display " GC)"))
      (do ((x (get) (get)))
	  ((eof-object? x))
	(with-result-and-times
	 x k
	 (lambda (res cc cgcc ec egcc)
	   (set! ## res)
	   (display ";value (after ")
	   (write (+ cc ec))
	   (display "ms... comp: ")
	   (wrclock cc cgcc)
	   (display ", exec: ")
	   (wrclock ec egcc)
	   (display "):")
	   (newline)
	   (write res)
	   (newline)))))

    (define (read-eval-until-eof port)
      (do ((x (read port) (read port)))
	  ((eof-object? x) #t)
	(intern-eval x)))

    (define (intern-load file)
      (call-with-input-file file read-eval-until-eof))

    (define (intern-open-compilation new-status)
      (let ((old-status *open-compilation-enabled*))
	(set! *open-compilation-enabled* new-status)
	old-status))

    (define (compfilt)
      (with-error-handler
       (lambda (message cont)
	 (display message (standard-port 2))
	 (newline (standard-port 2))
	 (quit 1))
       (lambda () (compile-until-eof)))
      (quit))

    (define (inspector cont)
      (let ((inspect-ccont-idx 0)
	    (inspect-shared-idx 1)
	    (inspect-stack-idx 2)
	    (inspect-prim-code-idx 3)
	    (inspect-modeid-pc-idx 4)
	    (inspect-mode-const-idx 5)
	    (inspect-env-inst-idx 6))

	(call-with-current-continuation
	 (lambda (exit)

	   (with-output-to-port
	       (standard-port 2)
	     (lambda ()

	       (define (choices . l)
		 (for-each (lambda (x)
			     (display (car x))
			     (display "/"))
			   l)
		 (display "help")
		 (display " --> ")
		 (let ((r (read (standard-port 0))))
		   (letrec ((ll (cons (list "help" "display this explanation"
					    (lambda () (help)))
				      l))
			    (help (lambda ()
				    (for-each
				     (lambda (x)
				       (display "   ")
				       (display (car x))
				       (display " - ")
				       (display (cadr x))
				       (newline))
				     ll))))
		     (if (eof-object? r)
			 (exit 'done-eof))
		     (if (symbol? r)
			 (let ((s (symbol->string r)))
			   (do ((cl ll (cdr cl))
				(c '()
				   (if (and (<= (string-length s)
						(string-length (caar cl)))
					    (string-ci=?
					     s
					     (substring
					      (caar cl) 0 (string-length s))))
				       (cons (caddar cl) c)
				       c)))
			       ((not (pair? cl))
				(if (and (pair? c) (not (pair? (cdr c))))
				    (begin ((car c))
					   (apply choices l)))))))
		     (display "... what?") (newline)
		     (help)
		     (apply choices l))))

	       (define (head s v l)
		 (display s)
		 (if (vector-ref v inspect-shared-idx)
		     (display "!"))
		 (display " {")
		 (write l)
		 (display "}; code = ")
		 (display (vector-ref v inspect-prim-code-idx))
		 (newline))

	       (define (show-stack v)
		 (let ((l (vector-length v)))
		   (do ((i 0 (+ i 1)))
		       ((>= i l) #f)
		     (display (if (> i 9) "     * " "     *  "))
		     (write i)
		     (display ": ")
		     (write (vector-ref v i))
		     (newline))))

	       (define (show-c-frame v l)
		 (head "% C-frame" v l)
		 (display "          mode = ")
		 (cond ((vector-ref v inspect-modeid-pc-idx)
			=>
			(lambda (modeid)
			  (display
			   (case modeid
			     ((0) "[INPUT-PORT: ")
			     ((1) "[OUTPUT-PORT: ")
			     ((2) "[ERROR-HANDLER: ")
			     ((3) "[GC-STRATEGY: ")
			     ((4) "[INTERRUPT-HANDLER: ")
			     ((5) "[TIMER-EXPIRATION-HANDLER: ")
			     (else "[?BOGUS?: ")))
			  (write (vector-ref v inspect-mode-const-idx))
			  (display "]")))
		       (else
			(display "[NONE]")))
		 (newline)
		 (display "          env = ")
		 (write (vector-ref v inspect-env-inst-idx))
		 (newline)
		 (choices
		  (list "stack" "show stack contents"
			(lambda ()
			  (show-stack (vector-ref v inspect-stack-idx))))
		  (list "up" "go to caller's frame"
			(lambda () (loop (+ l 1))))
		  (list "down" "go to callee's frame"
			(lambda () (loop (- l 1))))
		  (list "quit" "exit inspector loop"
			(lambda () (exit 'done-c)))))

	       (define (show-asm-stat stat cv)
		 (write (car stat))
		 (display "-")
		 (write (cadr stat))
		 (display ": ")
		 (display (caddr stat))
		 (case (caddr stat)
		   ((jump-forward
		     jump-backward true?jump false?jump true?jump+pop
		     false?jump+pop true?jump:pop false?jump:pop pop-true?jump
		     pop-false?jump get-loc get-loc-cell put-loc put-loc-pop
		     put-loc-cell put-loc-cell-pop multi-pop make-cell
		     make-vector allocate-vector call call-exit
		     unchecked-call unchecked-call-exit)
		    (display " ")
		    (write (cadddr stat)))
		   ((take-true take-false take-nil pop exit cons append
			       list->vector car cdr not check)
		    (display " -"))
		   ((memv?jump+pop eqv?jump+pop)
		    (display " ")
		    (write (vector-ref cv (cadddr stat)))
		    (display ", ")
		    (write (car (cddddr stat))))
		   ((get-vec get-vec-cell put-vec put-vec-pop put-vec-cell
			     put-vec-cell-pop)
		    (display " ")
		    (write (cadddr stat))
		    (display ", ")
		    (write (car (cddddr stat))))
		   ((get-glob put-glob put-glob-pop take lambda delay
			      unchecked-get-glob)
		    (display " ")
		    (write (vector-ref cv (cadddr stat))))
		   (else
		    (display "BOGUS: ")
		    (write stat)))
		 (newline))

	       (define (show-asm code start pc cv)
		 (do ((stat (disassemble code start)
			    (disassemble code (cadr stat)))
		      (prev start (car stat))
		      (i 0 (+ i 1)))
		     ((or (not stat) (> i 10))
		      prev)
		   (display (if (= (car stat) pc) "     % " "       "))
		   (show-asm-stat stat cv)))

	       (define (show-s-frame v l)
		 (head "$ S-frame" v l)
		 (let* ((cv (vector-ref v inspect-mode-const-idx))
			(code (vector-ref v inspect-prim-code-idx))
			(pc (vector-ref v inspect-modeid-pc-idx))
			(cstat (vector-ref v inspect-env-inst-idx))
			(start (car cstat)))
		   (display "          pc = ")
		   (write pc)
		   (display ", VM instruction = ")
		   (show-asm-stat cstat cv)
		   (choices
		    (list "stack" "show stack contents"
			  (lambda () (show-stack
				      (vector-ref v inspect-stack-idx))))
		    (list "up" "go to caller's frame"
			  (lambda () (loop (+ l 1))))
		    (list "down" "go to callee's frame"
			  (lambda () (loop (- l 1))))
		    (list "first" "show first instruction window"
			  (lambda ()
			    (set! start (show-asm code 0 pc cv))))
		    (list "next" "show next instruction window"
			  (lambda ()
			    (set! start (show-asm code start pc cv))))
		    (list "quit" "exit inspector loop"
			  (lambda () (exit 'done-s))))))

	       (define (show-info v l)
		 ((if (vector-ref v inspect-ccont-idx)
		      show-c-frame show-s-frame)
		  v l))

	       (define (loop level)
		 (if (zero? level)
		     (begin (display "*** INNERMOST LEVEL ***")
			    (newline)))
		 (cond ((negative? level)
			(loop 0))
		       ((inspect cont level)
			=>
			(lambda (info)
			  (loop (show-info info level))))
		       (else
			(display "*** NO FURTHER ANCESTORS ***")
			(newline)
			(loop (- level 1)))))

	       (loop 0)))))))

    (define (toplevel argl)
      (define (next-round k)
	;; called in tail position within error handler to allow
	;; error continuation to go away (avoid secondary space leak)
	(semi-loop k)
	(quit))
      (define (semi-loop k)
	(with-error-handler
	 (lambda (mess cont)
	   (newline (standard-port 1))
	   (with-output-to-port (standard-port 2)
	     (lambda ()
	       (display mess)
	       (newline)
	       (display "(##) to enter debugger")
	       (newline)))
	   (set! ##
	    (lambda ()
	      (display mess)
	      (newline)
	      (inspector cont)
	      (set! ## #f)))
	   (next-round k))
	 (lambda () (repl-until-eof k))))
      ;; install the ``soft'' primitives...
      (set! load intern-load)
      (set! eval intern-eval)
      (set! open-compilation intern-open-compilation)
      ;; load everything on load list
      (if (list? argl)
	  (begin
	    (set! command-line-arguments argl)
	    (for-each (lambda (file)
			(display "Loading ")
			(write file)
			(display " ...")
			(newline)
			(intern-load file))
		      argl)))
      ;; display ``welcome'' message and VSCM version/release
      (display "Welcome to VSCM (V")
      (write (car *version*))
      (display "r")
      (write (cdr *version*))
      (display ")")
      (newline)
      (display
 "[send questions and bug reports to Matthias Blume (blume@cs.princeton.edu)]")
      (newline)
      ;; go into REPL
      (call-with-current-continuation
       (lambda (k)
	 (set! *error-continuation* k)))
      (semi-loop *error-continuation*)
      (quit))

    (lambda (sel)
      (case sel
	((inspector) inspector)
	((annotate) annotate)
	((compfilt) compfilt)
	((toplevel) toplevel)
	((compile-file) compile-file)
	((compile-until-eof) compile-until-eof)
	(else 'nonsense)))))
