@part[NUMBER, Root "TMAN.MSS"]  @Comment{-*-System:TMAN-*-}
@chap[Numbers]


@iix[Numbers] in @Tau[] are objects which represent real numbers.
Numbers come in three varieties: integers, ratios, and floating point
numbers.  Integers and ratios are exact models of the corresponding
mathematical objects.  Floating point numbers give discrete approximations
to real numbers within a certain implementation-dependent range.

@dc{ Talk about external syntax of numbers. }

As expressions, numbers are self-evaluating literals (see section
@ref[EQ?]).
  @begin[ProgramExample]
-377            @ev[]  -377
72723460248141  @ev[]  72723460248141
13/5            @ev[]  13/5
34.55           @ev[]  34.55
  @end[ProgramExample]
@dc{
(define (fib)
  (do ((a 0 b)
       (b 1 (+ a b))
       (n 1 (+ n 1)))
      (nil)
    (format t " (fib ~s) = ~s~%" n b)))

(define tau (/ (+ 1 (sqrt 5.0)) 2))

(define (fib n)
  (/ (+ (expt tau n) (expt tau (- 0 n))) (sqrt 5.0)))
}

There may be many different objects which all represent the same number.
The effect of this is that @tc[EQ?] may behave in nonintuitive ways when
applied to numbers.  It is guaranteed that if two numbers have different
types or different numerical values, then they are @i[not]
@tc[EQ?], but two numbers that appear to be the same may or may not be
@tc[EQ?], even though there is no other way to distinguish them.  Use
@tc[EQUAL?] or @tc[=] (page @PageRef(EQUAL?))  for comparing numbers.


@section[Predicates]

@info[NOTES="Type predicate",EQUIV="NUMBERP"]
@desc[@el[](NUMBER? @i[object]) @yl[] @i[boolean]]
Returns true if @i[object] is a number.
@EndDesc[NUMBER?]

@info[NOTES="Type predicate",EQUIV="FIXP"]
@desc[(INTEGER? @i[object]) @yl[] @i[boolean]]
Returns true if @i[object] is an integer.
@EndDesc[INTEGER?]

@info[NOTES="Type predicate",EQUIV="FLOATP"]
@desc[(FLOAT? @i[object]) @yl[] @i[boolean]]
Returns true if @i[object] is a floating point number.
@EndDesc[FLOAT?]

@info[NOTES="Type predicate"]
@desc[(RATIO? @i[object]) @yl[] @i[boolean]]
Returns true if @i[object] is a ratio.
@EndDesc[RATIO?]

@info[EQUIV="ODDP"]
@desc[(ODD? @i[integer]) @yl[] @i[boolean]]
Returns true if @i[integer] is odd.
@EndDesc[ODD?]

@desc[(EVEN? @i[integer]) @yl[] @i[boolean]]
Returns true if @i[integer] is even.
@EndDesc[EVEN?]


@begin[group]
@section[Arithmetic]

@info[EQUIV="PLUS"]
@descN[
F1="@el[](+ . @i[numbers]) @yl[] @i[number]", FN1="+", NL1,
F2="(ADD . @i[numbers]) @yl[] @i[number]",    FN2="ADD"
]
Returns the sum of the @i[numbers].
@begin[ProgramExample]
(+ 10 27)    @ev[]  37
(+ 10 27.8)  @ev[]  37.8
(+)          @ev[]  0
(+ -35)      @ev[]  -35
(+ 1 2 3)    @ev[]  6
@end[ProgramExample]
@EndDescN[]
@end[group]

@info[EQUIV="DIFFERENCE"]
@descN[
F1="@el[](- @i[n1 n2]) @yl[] @i[number]",   FN1="-"   ,
F2="(SUBTRACT @i[n1 n2]) @yl[] @i[number]", FN2="SUBTRACT"
]
Returns the difference of @i[n1] and @i[n2].
@EndDescN[]

@info[EQUIV="MINUS"]
@descN[
F1="@el[](- @i[n]) @yl[] @i[number]",
F2="(NEGATE @i[n]) @yl[] @i[number]",  FN2="NEGATE"
]
Returns the additive inverse of @i[n].
@EndDescN[]

@info[EQUIV="TIMES"]
@descN[
F1="@el[](* . @i[numbers]) @yl[] @i[number]",   FN1="*" ,
F2="(MULTIPLY . @i[numbers]) @yl[] @i[number]", FN2="MULTIPLY"
]
Returns the product of the @i[numbers].
@EndDescN[]

@info[EQUIV="QUOTIENT"]
@descN[
F1="@el[](/ @i[n1 n2]) @yl[] @i[number]", FN1="/", NL1,
F2="(DIVIDE @i[n1 n2]) @yl[] @i[number]", FN2="DIVIDE"
]
Returns the quotient of @i[n1] and @i[n2].
@begin[ProgramExample]
(/ 20 5)    @ev[]  4
(/ 5 20)    @ev[]  1/4
(/ 5.0 20)  @ev[]  0.25
@end[ProgramExample]
@EndDescN[]

@info[EQUIV="//"]
@desc[(DIV @i[n1 n2]) @yl[] @i[integer]]
Integer division; truncates the quotient of @i[n1] and @i[n2]
towards zero.
    @BeginInset[Bug:]
    @tc[DIV] in @Timp[] 2.7 does strange things if its arguments
    aren't integers.
    @EndInset[]
@EndDesc[DIV]

@info[EQUIV="REMAINDER, \"]
@desc[(REMAINDER @i[n1 n2]) @yl[] @i[number]]
Returns the remainder of the division of @i[n1] by @i[n2], with
the sign of @i[n1].
    @BeginInset[Bug:]
    @tc[REMAINDER] in @Timp[] 2.7 does strange things
    if its arguments aren't integers.
    @EndInset[]
@EndDesc[REMAINDER]

@dc{
@desc[(FLOOR @i[n1 n2]) @yl[] @i[number]]
Returns the greatest multiple of @i[n2] which is less than or equal to @i[n1];
@qu"round down."
A particularly useful case is where @i[n2] is 1, in which case
@tc[FLOOR] returns @i[n1]'s integer part.
  @begin[ProgramExample]
(FLOOR 21 8)    @ev[]  16
(FLOOR -21 8)   @ev[]  -24
(FLOOR 2.67 1)  @ev[]  2
  @end[ProgramExample]
@EndDesc[FLOOR]

@desc[(CEILING @i[n1 n2]) @yl[] @i[number]]
Returns the least multiple of @i[n2] which is greater than or equal to @i[n1];
@qu"round up."
  @begin[ProgramExample]
(CEILING 21 8)    @ev[]  24
(CEILING -21 8)   @ev[]  16
(CEILING 2.67 1)  @ev[]  3
  @end[ProgramExample]
@EndDesc[CEILING]

@desc[(TRUNCATE @i[n1 n2]) @yl[] @i[number]]
Returns the greatest multiple of @i[n2] which is less than or equal to the
absolute value of @i[n1], with @i[n1]'s sign;
@qu"round towards zero."
  @begin[ProgramExample]
(TRUNCATE 21 8)    @ev[]  16
(TRUNCATE -21 8)   @ev[]  -24
(TRUNCATE 2.67 1)  @ev[]  1
  @end[ProgramExample]
@EndDesc[TRUNCATE]

@desc[(ROUND @i[n1 n2]) @yl[] @i[number]]
Rounds @i[n1] toward the nearest multiple of @i[n2].
@EndDesc[ROUND]
}

@desc[(MOD @i[n1 n2]) @yl[] @i[number]]
Returns a number @i[k] in the range from zero, inclusive, to @i[n2],
exclusive, such that @i[k] differs from @i[n1] by an integral
multiple of @i[n2].  If @i[n1] is positive, this behaves like
@tc[REMAINDER].
    @BeginInset[Bug:]
    @tc[MOD] in @Timp[] 2.7 does strange things if its arguments aren't integers.
    @EndInset[]
@EndDesc[MOD]

@desc[(EXPT @i[base exponent]) @yl[] @i[number]]
Returns @i[base] raised to the @i[exponent] power.
@i[Base] and @i[exponent] may be any numbers.
    @BeginInset[Bug:]
    In @Timp[] 2.7, @i[exponent] must be an integer.
    @EndInset[]
@EndDesc[EXPT]

@desc[(ABS @i[n]) @yl[] @i[number]]
Returns the absolute value of @i[n].
@EndDesc[ABS]

@desc[(GCD @i[integer1 integer2]) @yl[] @i[integer]]
Returns the greatest common divisor of @i[integer1] and @i[integer2].
  @begin[ProgramExample]
(GCD 36 60)  @ev[]  12
  @end[ProgramExample]
@EndDesc[GCD]

@AnEquivE[Tfn="ADD1",Efn="1+"]
@AnEquivE[Tfn="ADD1",Efn="ADD1"]
@descN[
F1="(ADD1 @i[n]) @yl[] @i[number]", FN1="ADD1",
F2="(1+ @i[n]) @yl[] @i[number]", FN2="1+", NL2
]
Returns @i[n] + 1.
(See also @tc[INCREMENT], page @PageRef[INCREMENT].)
@EndDescN[]

@AnEquivE[Tfn="SUBTRACT1",Efn="1-"]
@AnEquivE[Tfn="SUBTRACT1",Efn="SUB1"]
@descN[
F1="(SUBTRACT1 @i[n]) @yl[] @i[number]", FN1="SUBTRACT1",
F2="(-1+ @i[n]) @yl[] @i[number]", FN2="-1+", NL2
]
Returns @i[n] - 1.
(See also @tc[DECREMENT], page @PageRef[DECREMENT].)
@EndDescN[]

@desc[(MIN . @i[numbers]) @yl[] @i[number]]
Returns the @i[number] with the least numerical magnitude.
(There must be at least one @i[number].)
@EndDesc[MIN]

@desc[(MAX . @i[numbers]) @yl[] @i[number]]
Returns the @i[number] with the greatest numerical magnitude.
(There must be at least one @i[number].)
@EndDesc[MAX]


@section[Comparison]

@descN[
F1="@el[](= @i[n1 n2]) @yl[] @i[boolean]", FN1="=", NL1,
F2="(EQUAL? @i[n1 n2]) @yl[] @i[boolean]", FN2="EQUAL?"
]
@label[EQUAL]
Returns true if @i[n1] is numerically equal to @i[n2].
@EndDescN[]

@AnEquivE[Tfn="LESS?",Efn="LESSP"]
@descN[
F1="@el[](< @i[n1 n2]) @yl[] @i[boolean]", FN1="<", NL1,
F2="(LESS? @i[n1 n2]) @yl[] @i[boolean]",  FN2="LESS?"
]
Returns true if @i[n1] is numerically less than @i[n2].
@EndDescN[]

@AnEquivE[Tfn="GREATER?",Efn="GREATERP"]
@descN[
F1="@el[](> @i[n1 n2]) @yl[] @i[boolean]",   FN1=">", NL1,
F2="(GREATER? @i[n1 n2]) @yl[] @i[boolean]", FN2="GREATER?"
]
Returns true if @i[n1] is numerically greater than @i[n2].
@EndDescN[]

@descN[
F1="(N= @i[n1 n2]) @yl[] @i[boolean]",         FN1="N=",         NL1,
F2="(NOT-EQUAL? @i[n1 n2]) @yl[] @i[boolean]", FN2="NOT-EQUAL?"
]
Returns true if @i[n1] is not numerically equal to @i[n2].
@EndDescN[]

@descN[
F1="(>= @i[n1 n2]) @yl[] @i[boolean]",        FN1=">=", NL1,
F2="(NOT-LESS? @i[n1 n2]) @yl[] @i[boolean]", FN2="NOT-LESS?"
]
Returns true if @i[n1] is not numerically less than @i[n2].
@EndDescN[]

@descN[
F1="(<= @i[n1 n2]) @yl[] @i[boolean]",           FN1="<=", NL1,
F2="(NOT-GREATER? @i[n1 n2]) @yl[] @i[boolean]", FN2="NOT-GREATER?"
]
Returns true if @i[n1] is not numerically greater than @i[n2].
@EndDescN[]

@section[Sign predicates]

@AnEquivE[Tfn="ZERO?",Efn="ZEROP"]
@descN[
F2="(=0? @i[n]) @yl[] @i[boolean]",    FN2="=0?", NL2,
F1="(ZERO? @i[n]) @yl[] @i[boolean]",  FN1="ZERO?"
]
Returns true if @i[n] is numerically equal to zero.
@EndDescN[]

@AnEquivE[Tfn="NEGATIVE?",Efn="MINUSP"]
@descN[
F2="(<0? @i[n]) @yl[] @i[boolean]",        FN2="<0?", NL2,
F1="(NEGATIVE? @i[n]) @yl[] @i[boolean]",  FN1="NEGATIVE?"
]
Returns true if @i[n] is negative.
@EndDescN[]

@AnEquivE[Tfn="POSITIVE?",Efn="PLUSP"]
@descN[
F2="(>0? @i[n]) @yl[] @i[boolean]",        FN2=">0?", NL2,
F1="(POSITIVE? @i[n]) @yl[] @i[boolean]",  FN1="POSITIVE?"
]
Returns true if @i[n] is positive.
@EndDescN[]

@descN[
F2="(N=0? @i[n]) @yl[] @i[boolean]",    FN2="N=0?",         NL2,
F1="(NOT-ZERO? @i[n]) @yl[] @i[boolean]",  FN1="NOT-ZERO?"
]
Returns true if @i[n] is not numerically equal to zero.
@EndDescN[]

@descN[
F2="(>=0? @i[n]) @yl[] @i[boolean]",        FN2=">=0?", NL2,
F1="(NOT-NEGATIVE? @i[n]) @yl[] @i[boolean]",  FN1="NOT-NEGATIVE?"
]
Returns true if @i[n] is non-negative.
@EndDescN[]

@descN[
F2="(<=0? @i[n]) @yl[] @i[boolean]",        FN2="<=0?", NL2,
F1="(NOT-POSITIVE? @i[n]) @yl[] @i[boolean]",  FN1="NOT-POSITIVE?"
]
Returns true if @i[n] is non-positive.
@EndDescN[]

@section[Transcendental functions]

@desc[(EXP @i[float]) @yl[] @i[float]]
Exponential function (e@+[@i[x]]).
@EndDesc[EXP]

@desc[(LOG @i[float]) @yl[] @i[float]]
Natural logarithm (log@-[e]@i[x]).
@EndDesc[LOG]

@desc[(SQRT @i[float]) @yl[] @i[float]]
Returns the square root of its argument.
@EndDesc[SQRT]

@desc[(COS @i[float]) @yl[] @i[float]]
Returns the cosine of its argument.
@EndDesc[COS]

@desc[(SIN @i[float]) @yl[] @i[float]]
Returns the sine of its argument.
@EndDesc[SIN]

@desc[(TAN @i[float]) @yl[] @i[float]]
Returns the tangent of its argument.
@EndDesc[TAN]

@desc[(ACOS @i[float1]) @yl[] @i[float2]]
Arccosine.  Returns a number whose cosine is @i[float1].
@EndDesc[ACOS]

@desc[(ASIN @i[float1]) @yl[] @i[float2]]
Arcsine.  Returns a number whose sine is @i[float1].
@EndDesc[ASIN]

@desc[(ATAN2 @i[x y]) @yl[] @i[float]]
Two-argument arctangent.  This computes the
angle between the positive x-axis and the point
(@i[x], @i[y]).  For example, if (@i[x], @i[y]) is in the first or third
quadrant, @tc[(ATAN2 @i[x y])] returns the arctangent of @i[y]/@i[x].
    @BeginInset[Bug:]
    @tc[ATAN2] isn't implemented in @Timp[] 2.7.  @Timp[] 2.7 does
    implement @tc[ATAN] (arctangent), however.
    @EndInset[]
@EndDesc[ATAN2]

@section[Bitwise logical operators]
    @BeginInset[Bug:]
    In @Timp[] 2.7, the routines in this section are restricted to take
    fixnums (see page @pageref[FIXNUM?]), not arbitrary integers.
    @EndInset[]

@desc[(LOGAND @i[integer1 integer2]) @yl[] @i[integer]]
Returns the bitwise logical @i[and] of its arguments.
  @begin[ProgramExample]
(LOGAND 10 12)  @ev[]  8
  @end[ProgramExample]
@EndDesc[LOGAND]

@desc[(LOGIOR @i[integer1 integer2]) @yl[] @i[integer]]
Returns the bitwise logical inclusive @i[or] of its arguments.
@EndDesc[LOGIOR]

@desc[(LOGXOR @i[integer1 integer2]) @yl[] @i[integer]]
Returns the bitwise logical exclusive @i[or] of its arguments.
@EndDesc[LOGXOR]

@desc[(LOGNOT @i[integer]) @yl[] @i[integer]]
Returns the bitwise logical @i[not] of its argument.
@EndDesc[LOGNOT]

@desc[(ASH @i[integer amount]) @yl[] @i[integer]]
Shifts @i[integer] to the left @i[amount] bit positions.
If @i[amount] is negative, shifts @i[integer] right by the absolute
value of @i[amount] bit positions.
@EndDesc[ASH]

@desc[(BIT-FIELD @i[integer position size]) @yl[] @i[integer]]
Extracts a bit field out of @i[integer].  The field extracted begins
at bit position @i[position] from the low-order bit of the integer
and consists of @i[size] bits.  The extracted field is returned
as a positive integer.
  @begin[ProgramExample]
(BIT-FIELD 27 1 3)  @ev[]  5
(BIT-FIELD -1 4 5)  @ev[]  31
  @end[ProgramExample]
@EndDesc[BIT-FIELD]

@desc[(SET-BIT-FIELD @i[integer position size value]) @yl[] @i[integer]]
Inserts a value into a bit field of @i[integer].  The field begins
at bit position @i[position] from the low-order bit of the integer
and consists of @i[size] bits.  A new integer is returned which is the same as the
argument @i[integer] except that this field has been altered to be @i[value].
  @begin[ProgramExample]
(SET-BIT-FIELD 32 1 3 5)  @ev[]  42
  @end[ProgramExample]
@EndDesc[SET-BIT-FIELD]

@section[Coercion]

@desc[(->INTEGER @i[number]) @yl[] @i[integer]]
Coerces @i[number] to an integer, truncating towards zero if necessary.
  @begin[ProgramExample]
(->INTEGER 17)     @ev[]  17
(->INTEGER 7/4)    @ev[]  1
(->INTEGER 17.6)   @ev[]  17
(->INTEGER -17.6)  @ev[]  -17
  @end[ProgramExample]
@enddesc[->INTEGER]

@desc[(->FLOAT @i[number]) @yl[] @i[float]]
Coerces @i[number] to a floating point number.
  @begin[ProgramExample]
(->FLOAT 17)    @ev[]  17.0
(->FLOAT 7/4)   @ev[]  1.75
(->FLOAT 17.6)  @ev[]  17.6
  @end[ProgramExample]
@enddesc[->FLOAT]

@section[Assignment]

@tc[INCREMENT] and @tc[DECREMENT] are assignment forms, like @tc[SET]
and @tc[PUSH].  See section @ref[Assignmentsection] for a general
discussion.

@info[Notes "Special form"]
@desc[(INCREMENT @i[location]) @yl[] @i[number]]
Adds one to the value in @i[location], stores the sum back into @i[location],
and yields the sum.
  @begin[ProgramExample]
(LSET L 6)
(INCREMENT L)             @ev[]  7
L                         @ev[]  7
(LSET M (LIST 10 20 30))  @ev[]  (10 20 30)
(INCREMENT (CAR M))       @ev[]  11
M                         @ev[]  (11 20 30)

(INCREMENT @i[location])  @ce[]  (MODIFY @i[location] ADD1)
  @end[ProgramExample]
@EndDesc[INCREMENT]

@info[Notes "Special form"]
@desc[(DECREMENT @i[location]) @yl[] @i[number]]
Subtracts one from the value in @i[location], stores the
difference back into @i[location], and yields the difference.
  @begin[ProgramExample]
(DECREMENT @i[location])  @ce[]  (MODIFY @i[location] SUBTRACT1)
  @end[ProgramExample]
@EndDesc[DECREMENT]
