;           Virtual Scheme Machine for Scheme 84

(declare
  (special =env= =class= =temp= =temp1= =temp2= unprintable-symbols
     *initial-fnv* **comp-env** *toplevel-continuation* *initial-env*
      *toplevel-function*))

; Main Registers

(declare
  (special **ticks** **try-cont** **try-failure** **try-success** **fnv**))

; Debugging registers

(declare
  (special *error* namespacetypes *error-data* *error-res* *error-args*))

(def mk-continuation
  (macro (l)
    `(cond
       [(eq (caar (setq =temp2= ,(cadr l))) '*re) ,(cadddddr l)]
       [t (cons
	    (cons =temp2= (cons ,(caddr l) (cons ,(cadddr l) ,(caddddr l))))
	    ,(cadddddr l))])))

(def mk-save-continuation
  (macro (l)
    `(cons
       (cons ,(cadr l) (cons ,(caddr l) (cons ,(cadddr l) ,(caddddr l))))
       ,(cadddddr l))))

(def synthetic-id?
  (macro (l)
    (let ([z (cadr l)]) `(and (dtpr ,z) (eq (car ,z) '&synthetic-id)))))

(def &frame (macro (l) `(car ,(cadr l))))
(def &pc (macro (l) `(car ,(cadr l))))
(def &args (macro (l) `(cadr ,(cadr l))))
(def &env (macro (l) `(caddr ,(cadr l))))
(def &fnv (macro (l) `(cdddr ,(cadr l))))
(def &cont (macro (l) `(cdr ,(cadr l))))
(def top-stack (macro (l) '(car **args**)))
(def 2nd-stack (macro (l) '(cadr **args**)))
(def 3rd-stack (macro (l) '(caddr **args**)))
(def 4th-stack (macro (l) '(cadddr **args**)))

; S-code interpreter

(def vsm
  (lambda (**pc** **cont** **env** **args** **result**)
    (prog (=opcode= =frame= =funtype= =n= =rib-loc= =pair= =op= =opargs=
	    =inst= =rib= =rib-depth= =offset=)
   (go redirected-pc-loop)
   
 loop
   (setq **pc** (cdr **pc**))
 redirected-pc-loop
   (cond [(or (null **ticks**) (plusp (setq **ticks** (1- **ticks**))))
	  (go opcode-dispatch)])
   (setq **args**
     (list
       (cons '&cont
	 (mk-continuation **pc** **args** **env** **fnv** **cont**))
       **result**))
   (setq **result** **try-failure**)
 try-return
   (setq **cont** **try-cont**)
   (setq **ticks** nil)
   (setq =opargs= 2)
   (go *application)
   
 %exit-try
   (cond [(null **ticks**) (scherror "Sorry, no try running")])
   (setq **ticks** 1)
   (go loop)

 %try
   (cond 
     [**ticks** (scherror "Sorry, can't nest trys")]
     [(or (not (fixp (setq =temp= (2nd-stack))))
	  (and (not (zerop =temp=)) (not (plusp =temp=))))
      (raise (list 'SE%vsm 0 t '|illegal ticks for try:| =temp=))]
     [(zerop =temp=) ; this is just for efficiency
      (setq **cont**
        (mk-continuation
	  (cdr **pc**) (cddddr **args**) **env** **fnv** **cont**))
      (setq =temp= (list **result** (top-stack)))
      (setq **result** (4th-stack))
      (setq **args** =temp=)
      (setq =opargs= 2)]
     [t (setq **ticks** (1+ =temp=))
	(setq **try-success** (3rd-stack))
        (setq **try-failure** (4th-stack))
	(setq **try-cont**
          (mk-continuation
	    (cdr **pc**) (cddddr **args**) **env** **fnv** **cont**))
	(setq **cont** nil)
	(setq **args** (list (top-stack)))
	(setq =opargs= 1)])
   (go *application)

 %C
   (setq **cont** 
      (mk-continuation (cdr **pc**) **args** **env** **fnv** **cont**))
   (setq **args** (list (cons '&cont **cont**)))
   (setq =opargs= 1)
   (go *application)

 %partial-cont
   (let ([first **result**] [second (top-stack)])
      (cond [(not (and (eq '&cont (car first)) (eq '&cont (car second))))
	     (scherror "Arguments to partial-cont must be continuations")])
      (setq **result** (cons '&cont (copy-cont (cdr first) (cdr second))))
      (go pop1))

 %append-cont
   (let ([first **result**] [second (top-stack)])
      (cond [(not (and (eq '&cont (car first)) (eq '&cont (car second))))
             (scherror "Both arguments to append-cont must be continuations")])
      (setq **result** (cons '&cont (append (cdr first) (cdr second))))
      (go pop1))

 %T
   (setq **cont** nil)
   (setq **fnv** *initial-fnv*)
   (setq =opargs= 1)
   (go *application)

 apply-continuation
   (cond [(not (= =opargs= 1)) (go wrong-number-of-args-to-continuation)])
   (setq **cont** (cond [**cont** (append (cdr **result**) **cont**)]
			[t (cdr **result**)]))
   (setq **result** (top-stack))
 *restore
   (cond
     [**cont**
       (setq =frame= (&frame **cont**))
       (setq **cont** (&cont **cont**))
       (setq **pc** (&pc =frame=))
       (setq **args** (&args =frame=))
       (setq **env** (&env =frame=))
       (setq **fnv** (&fnv =frame=))
       (go redirected-pc-loop)]
     [(null **ticks**)
      (setq **pc** '((*at . 1)))
      (setq **env** *initial-env*)
      (setq **fnv** *initial-fnv*)
      (setq **args** (list **result**))
      (setq **result** *toplevel-function*)
      (go redirected-pc-loop)]
     [t (setq **args** (list **result** (- **ticks** 2)))
	(setq **result** **try-success**)
	(go try-return)])

 %apply
   (setq **cont**
     (mk-continuation (cdr **pc**) (cdr **args**) **env** **fnv** **cont**))
   (setq **args** (properargs (top-stack)))
   (cond ((null **args**) (go *thunk-invocation)))
   (setq =opargs= (length **args**))
   (go *application)

 *application-tail (setq =opargs= (cdr =inst=))
 *application
   (cond [(not (dtpr **result**)) (go bad-function)])
   (setq =funtype= (car **result**))
   (cond ((eq =funtype= '&closure) (go apply-closure))
	 ((eq =funtype= '&rest-closure) (go apply-rest-closure))
	 ((eq =funtype= '&sys) (go system-call))
	 ((eq =funtype= '&cont) (go apply-continuation))
	 ((eq =funtype= '&vector) (go apply-vector))
	 (t (go bad-function)))

 %execute
   (setq **cont**
      (mk-continuation (cdr **pc**) **args** **env** **fnv** **cont**))
   (setq **pc** **result**)
   (setq **env** nil)
   (go redirected-pc-loop)

 *thunk-invocation
   (cond [(not (dtpr **result**)) (go bad-function)])
   (setq =funtype= (car **result**))
   (cond ((eq =funtype= '&closure) (go apply-thunk))
	 ((eq =funtype= '&rest-closure) (go apply-rest-thunk))
	 ((eq =funtype= '&sys) (go system-call))
	 ((eq =funtype= '&cont) (go apply-continuation))
	 ((eq =funtype= '&vector) (go apply-vector))
	 (t (go bad-function)))

 opcode-dispatch
   (setq =inst= (car **pc**))
   (setq =opcode= (car =inst=))
   (cond
     ((eq =opcode= '*pu) (go *push))
     ((eq =opcode= '*lr) (go *local-lookup))
     ((eq =opcode= '*pr) (go *primitive))
     ((eq =opcode= '*at) (go *application-tail))
     ((eq =opcode= '*ti) (go *thunk-invocation))
     ((eq =opcode= '*co) (go *constant))
     ((eq =opcode= '*if) (go *if-tail))
     ((eq =opcode= '*le) (go *let-lambda))
     ((eq =opcode= '*sa) (go *save))
     ((eq =opcode= '*re)
      (cond [(numberp **ticks**) (setq **ticks** (1+ **ticks**))])
      (go *restore))
     ((eq =opcode= '*la) (go *lambda))
     ((eq =opcode= '*gr) (go *global-lookup))
     ((eq =opcode= '*ls) (go *local-set))
     ((eq =opcode= '*rla) (go *rest-lambda))
     ((eq =opcode= '*rle) (go *rest-let-lambda))
     ((eq =opcode= '*gs) (go *global-set))
     ((eq =opcode= '*lf) (go *local-fluid))
     ((eq =opcode= '*gf) (go *global-fluid))
     ((eq =opcode= '*ra) (go *result->args))
     (t (go bad-vsm-opcode)))

 *constant
   (setq =opargs= (cdr =inst=))
   (cond [(and (dtpr =opargs=) (eq (car =opargs=) '&unassigned-constant))
	  (rplacd =inst= (setq **result** (lookup-constant (cdr =opargs=))))]
	 [t (setq **result** =opargs=)])
   (go loop)

 *lambda
   (setq **result** (cons '&closure (cons (cdr =inst=) **env**)))
   (go loop)

 *rest-lambda
   (setq **result** (cons '&rest-closure (cons (cdr =inst=) **env**)))
   (go loop)

 *if-tail
   (setq **pc** (cond [**result** (cadr =inst=)] [t (cddr =inst=)]))
   (go redirected-pc-loop)

 *result->args (setq **args** **result**) (go loop)

 *let-lambda
   (setq **env** (cons (cons **args** (cdr =inst=)) **env**))
   (setq **args** nil)
   (go loop)

 *rest-let-lambda
   (setq **env**
     (cons
       (cons  ;;; this nthcdr should not need to be done!
	 (cons (nthcdr (length (cdr (cdr =inst=))) **args**) **args**)
	 (cdr =inst=))
       **env**))
   (go loop)

 apply-thunk
   (cond [(not (zerop (caadr **result**)))
	  (go wrong-number-of-args-to-closure)])
   (setq **env** (cddr **result**))
   (setq **pc** (cddadr **result**))
   (go redirected-pc-loop)

 apply-closure
   (cond [(not (= =opargs= (caadr **result**)))
	  (go wrong-number-of-args-to-closure)])
   (setq **env**
     (cons (cons **args** (cadadr **result**)) (cddr **result**)))
   (setq **pc** (cddadr **result**))
   (setq **args** nil)
   (go redirected-pc-loop)

 apply-rest-thunk
   (cond [(not (zerop (caadr **result**)))
	  (go wrong-number-of-args-to-closure)])
   (setq **env**
     (cons (cons (cons nil nil) (cadadr **result**)) (cddr **result**)))
   (setq **pc** (cddadr **result**))
   (go redirected-pc-loop)

 apply-rest-closure
   (cond [(lessp =opargs= (caadr **result**))
	  (go wrong-number-of-args-to-closure)])
   (setq **env**
     (cons
       (cons 
	 (cons (nthcdr (caadr **result**) **args**) **args**)
	 (cadadr **result**))
       (cddr **result**)))
   (setq **pc** (cddadr **result**))
   (setq **args** nil)
   (go redirected-pc-loop)
    
 system-call
   (setq **result** (apply (cdr **result**) **args**))
   (setq **args** nil)
   (go *restore)

 apply-vector
   (cond ((not (= =opargs= 1)) (go wrong-number-args-to-vector)))
   (setq **result** (vactor-ref **result** (top-stack)))
   (setq **args** nil)
   (go *restore)

 *push
   (setq **args** (cons **result** **args**))
   (go loop)

 *save
   (setq **cont**
     (mk-save-continuation (cdr =inst=) **args** **env** **fnv** **cont**))
   (setq **args** nil)
   (go loop)

 *local-set
   (setq =rib-depth= (cadr =inst=))
   (setq =offset= (cddr =inst=))
   (setq =temp=
     (cond ((= 0 =rib-depth=) (caar **env**))
	   ((= 1 =rib-depth=) (caadr **env**))
	   ((= 2 =rib-depth=) (caaddr **env**))
	   (t (car (nthelem (1+ =rib-depth=) **env**)))))
   (cond [(= 0 =offset=) =temp=]
	 [(= 1 =offset=) (setq =temp= (cdr =temp=))]
	 [(= 2 =offset=) (setq =temp= (cddr =temp=))]
	 [t (setq =temp= (nthcdr =offset= =temp=))])
   (cond
     [(setq =temp2= (assq =temp= **fnv**))
      (cond
	[(synthetic-id? (setq =temp1= (cdr =temp2=)))
	 (setq **cont**
	   (mk-continuation (cdr **pc**) **args** **env** **fnv** **cont**))
	 (setq **args** (list **result**))
	 (setq **result** (cdr (cdr =temp1=)))
	 (setq =opargs= 1)
	 (go apply-closure)])
      (rplacd =temp2= **result**)]
     [(synthetic-id? (setq =temp1= (car =temp=)))
      (setq **cont**
	(mk-continuation (cdr **pc**) **args** **env** **fnv** **cont**))
      (setq **args** (list **result**))
      (setq **result** (cdr (cdr =temp1=)))
      (setq =opargs= 1)
      (go apply-closure)]
     [t (rplaca =temp= **result**)])
   (go loop)

 *local-fluid
   (setq =rib-depth= (cadr =inst=))
   (setq =offset= (cddr =inst=))
   (setq =temp=
     (cond ((= 0 =rib-depth=) (caar **env**))
	   ((= 1 =rib-depth=) (caadr **env**))
	   ((= 2 =rib-depth=) (caaddr **env**))
	   (t (car (nthelem (1+ =rib-depth=) **env**)))))
   (cond [(= 0 =offset=) =temp=]
	 [(= 1 =offset=) (setq =temp= (cdr =temp=))]
	 [(= 2 =offset=) (setq =temp= (cddr =temp=))]
	 [t (setq =temp= (nthcdr =offset= =temp=))])
   (setq **fnv** (cons (cons =temp= **result**) **fnv**))
   (go loop)
 
 *global-set
   (setq =opargs= (cdr =inst=))
   (cond
     [(setq =temp= (assq =opargs= **fnv**)) (setq =opargs= =temp=)]
     [(eq (cdr =opargs=) 'unassigned)
      (setq =temp= (global-namespacetype (car =opargs=)))
      (cond
	[(or (not =temp=) (eq =temp= 'base-identifier)
	     (eq =temp= 'scheme-primitive) (eq =temp= 'system-function))
	 (rplacd =opargs= **result**)
	 (cond [(memq =temp= '(scheme-primitive system-function))
		(remprop (car =opargs=) =temp=)])]
	[(and (eq =temp= 'constant)
	      (setq =pair=
		(getl (car =opargs=)
		      '(scheme-constant constant-primitive
			 constant-system-function)))
	      (eq (cadr =pair=) 'unassigned-constant))
	 (rplaca (cdr =pair=) **result**)]
	[t (setq *error* =opargs=)
	   (scherror (concat =opargs= " already declared as a " =temp=))])
      (go loop)])
   (cond
     [(synthetic-id? (setq =temp1= (cdr =opargs=)))
      (setq **cont**
	(mk-continuation (cdr **pc**) **args** **env** **fnv** **cont**))
      (setq **args** (list **result**))
      (setq **result** (cddr =temp1=))
      (setq =opargs= 1)
      (go apply-closure)]
     [t (rplacd =opargs= **result**)
	(cond
	  [(lookupinbase 'scheme-global-note)
	   (princ "[Redefining ") (print (car =opargs=))
	   (princ "]") (terpri)])
	(go loop)])

 *global-fluid
   (setq =opargs= (cdr =inst=))
   (setq **fnv** (cons (cons =opargs= **result**) **fnv**))
   (go loop)
    
 *local-lookup
   (setq =rib-depth= (cadr =inst=))
   (setq =offset= (cddr =inst=))
   (setq =temp=
      (cond ((= 0 =rib-depth=) (caar **env**))
	    ((= 1 =rib-depth=) (caadr **env**))
	    ((= 2 =rib-depth=) (caaddr **env**))
	    (t (car (nthelem (1+ =rib-depth=) **env**)))))
   (cond ((= 0 =offset=) =temp=)
	 ((= 1 =offset=) (setq =temp= (cdr =temp=)))
	 ((= 2 =offset=) (setq =temp= (cddr =temp=)))
	 (t (setq =temp= (nthcdr  =offset= =temp=))))
   (setq **result**
     (cond [(setq =temp1= (assq =temp= **fnv**)) (cdr =temp1=)]
	   [t (car =temp=)]))
 id-out
   (cond
      [(synthetic-id? **result**)
       (setq **cont**
	  (mk-continuation (cdr **pc**) **args** **env** **fnv** **cont**))
       (setq **result** (cadr **result**))
       (go *thunk-invocation)])
   (go loop)

 *global-lookup
   (setq =opargs= (cdr =inst=))
   (cond
     [(setq =temp= (assq =opargs= **fnv**)) (setq **result** (cdr =temp=))]
     [(eq (cdr =opargs=) 'unassigned)
      (setq *error-data* (car =opargs=))
      (setq *error* **cont**)
      (scherror "Unassigned identifier:")]
     [t (setq **result** (cdr =opargs=))])
   (go id-out)

 *primitive (setq =opargs= (cdr =inst=))

; Scheme primitives

; Scheme primitives take their arguments from the top of **args**,
; popping them off and leaving the result in **result**

; When apply-primitive is called, the top two elements of the =opargs=
; are the class and name of the primitive

 *primitive-apply
   (setq =class= (car =opargs=))
   (setq =op= (cdr =opargs=))
   (cond ((eq =class= '&se) (go &select))
	 ((eq =class= '&ls) (go &list))
	 ((eq =class= '&pr) (go &predicate))
	 ((eq =class= '&tp) (go &type))
	 ((eq =class= '&no) (go &numeric))
	 ((eq =class= '&s4) (go &select-four))
	 ((eq =class= '&ve) (go &vector))
	 ((eq =class= '&np) (go &npred))
	 ((eq =class= '&io) (go &input/output))
	 ((eq =class= '&si) (go &system-interface))
	 ((eq =class= '&tr) (go &transcendental))
	 ((eq =class= '&ms) (go &misc))
	 (t (go bad-class)))

 &select
   (cond ((eq =op= 'car) (go %car))
	 ((eq =op= 'cdr) (go %cdr))
	 ((eq =op= 'caaar) (go %caaar))
	 ((eq =op= 'caadr) (go %caadr))
	 ((eq =op= 'caar) (go %caar))
	 ((eq =op= 'cadar) (go %cadar))
	 ((eq =op= 'caddr) (go %caddr))
	 ((eq =op= 'cadr) (go %cadr))
	 ((eq =op= 'cdaar) (go %cdaar))
	 ((eq =op= 'cdadr) (go %cdadr))
	 ((eq =op= 'cdar) (go %cdar))
	 ((eq =op= 'cddar) (go %cddar))
	 ((eq =op= 'cdddr) (go %cdddr))
	 ((eq =op= 'cddr) (go %cddr)))

 &list
   (cond ((eq =op= 'cons) (go %cons))
	 ((eq =op= 'set-car!) (go %set-car!))
	 ((eq =op= 'set-car!!) (go %set-car!!))
	 ((eq =op= 'set-cdr!) (go %set-cdr!))
	 ((eq =op= 'copy-no-constant) (go %copy-no-constant))
	 ((eq =op= 'delete) (go %delete))
	 ((eq =op= 'delq) (go %delq))
	 ((eq =op= 'last-pair) (go %last-pair))
	 ((eq =op= 'length) (go %length))
	 ((eq =op= 'member) (go %member))
	 ((eq =op= 'nth) (go %nth))
	 ((eq =op= 'list-ref) (go %list-ref))
	 ((eq =op= 'list-tail) (go %list-tail))
	 ((eq =op= 'reverse) (go %reverse))
	 ((eq =op= 'reverse!) (go %reverse!))
	 ((eq =op= 'transpose) (go %transpose)))

 &predicate
   (cond ((eq =op= 'null?) (go %null?))
	 ((eq =op= 'eq?) (go %eq?))
	 ((eq =op= 'eqv?) (go %eqv?))
	 ((eq =op= 'memv) (go %memv))
	 ((eq =op= 'assv) (go %assv))
	 ((eq =op= 'assoc) (go %assoc))
	 ((eq =op= 'assq) (go %assq))
	 ((eq =op= 'port?) (go %port?))
	 ((eq =op= 'equal?) (go %equal?))
	 ((eq =op= 'memq) (go %memq))
	 (t (go bad-op)))

 &type
  (cond ((eq =op= 'atom?) (go %atom?))
	((eq =op= 'number?) (go %number?))
	((eq =op= 'string?) (go %string?))
	((eq =op= 'proc?) (go %proc?))
	((eq =op= 'fix?) (go %fix?))
	((eq =op= 'float?) (go %float?))
	((eq =op= 'pair?) (go %pair?))
	((eq =op= 'ref?) (go %ref?))
	((eq =op= 'symbol?) (go %symbol?))
	((eq =op= 'syntactic-extension?) (go %syntactic-extension?))
	((eq =op= 'scheme-constant?) (go %scheme-constant?))
	(t (go bad-op)))

 &numeric
   (cond ((eq =op= 'add1) (go %add1))
	 ((eq =op= 'sub1) (go %sub1))
	 ((eq =op= '+) (go %+))
	 ((eq =op= '*) (go %*))
	 ((eq =op= '-) (go %-))
	 ((eq =op= 'minus) (go %minus))
	 ((eq =op= '/) (go %/))
	 ((eq =op= 'abs) (go %abs))
	 ((eq =op= 'factorial) (go %factorial))
	 ((eq =op= 'fix) (go %fix))
	 ((eq =op= 'ceiling) (go %ceiling))
	 ((eq =op= 'truncate) (go %truncate))
	 ((eq =op= 'round) (go %round))
	 ((eq =op= 'float) (go %float))
	 ((eq =op= 'quotient) (go %quotient))
	 ((eq =op= 'remainder) (go %remainder))
	 ((eq =op= 'random) (go %random))
	 ((eq =op= 'sqrt) (go %sqrt))
	 (t (go bad-op)))

 &select-four
   (cond ((eq =op= 'caaaar) (go %caaaar))
	 ((eq =op= 'caaadr) (go %caaadr))
	 ((eq =op= 'caadar) (go %caadar))
	 ((eq =op= 'caaddr) (go %caaddr))
	 ((eq =op= 'cadaar) (go %cadaar))
	 ((eq =op= 'cadadr) (go %cadadr))
	 ((eq =op= 'caddar) (go %caddar))
	 ((eq =op= 'cadddr) (go %cadddr))
	 ((eq =op= 'cdaaar) (go %cdaaar))
	 ((eq =op= 'cdaadr) (go %cdaadr))
	 ((eq =op= 'cdadar) (go %cdadar))
	 ((eq =op= 'cdaddr) (go %cdaddr))
	 ((eq =op= 'cddaar) (go %cddaar))
	 ((eq =op= 'cddadr) (go %cddadr))
	 ((eq =op= 'cdddar) (go %cdddar))
	 ((eq =op= 'cddddr) (go %cddddr)))
	  
 &vector
   (cond ((eq =op= 'vector-ref) (go %vector-ref))
	 ((eq =op= 'vector-set!) (go %vector-set!))
	 ((eq =op= 'primitive-make-vector) (go %primitive-make-vector))
	 ((eq =op= 'vector-length) (go %vector-length))
	 ((eq =op= 'vector->list) (go %vector->list))
	 ((eq =op= 'list->vector) (go %list->vector))
	 ((eq =op= 'vector-fill!) (go %vector-fill!))
	 ((eq =op= 'vector?) (go %vector?)))

 &npred
   (cond ((eq =op= '=0) (go %=0))
	 ((eq =op= '=) (go %=))
	 ((eq =op= '<) (go %<))
	 ((eq =op= '>) (go %>))
	 ((eq =op= '>0) (go %>0))
	 ((eq =op= '<0) (go %<0))
	 ((eq =op= '<=) (go %<=))
	 ((eq =op= '>=) (go %>=))
	 ((eq =op= 'alpha<) (go %alpha<))
	 (t (go bad-op)))
 
 &input/output
   (cond  ((eq =op= 'close) (go %close))
	  ((eq =op= 'current-column) (go %current-column))
	  ((eq =op= 'display) (go %display))
	  ((eq =op= 'display&) (go %display&))
	  ((eq =op= 'file-exists?) (go %file-exists?))
	  ((eq =op= 'flush-input) (go %flush-input))
	  ((eq =op= 'flush-output) (go %flush-output))
	  ((eq =op= 'line-length) (go %line-length))
	  ((eq =op= 'newline) (go %newline))
	  ((eq =op= 'open) (go %open))
	  ((eq =op= 'print) (go %print))
	  ((eq =op= 'print-length) (go %print-length))
	  ((eq =op= 'print&) (go %print&))
	  ((eq =op= 'prompt-read) (go %prompt-read))
	  ((eq =op= 'read) (go %read))
	  ((eq =op= 'read-atom) (go %read-atom))
	  ((eq =op= 'read-char) (go %read-char))
	  ((eq =op= 'read/port) (go %read/port))
	  ((eq =op= 'read-atom/port) (go %read-atom/port))
	  ((eq =op= 'read-char/port) (go %read-char/port))
	  ((eq =op= 'set-line-length!) (go %set-line-length!)))

 &system-interface
   (cond  ((eq =op= 'pretty) (go %pretty))
	  ((eq =op= 'compile) (go %compile))
	  ((eq =op= 'mkmac-match?) (go %mkmac-match?))
	  ((eq =op= 'add-to-syntax-table) (go %add-to-syntax-table))
	  ((eq =op= 'beta-expand) (go %beta-expand))
	  ((eq =op= 'expand-once) (go %expand-once))
	  ((eq =op= 'pp-exp) (go %pp-exp))
	  ((eq =op= 'copying-intern*) (go %copying-intern*))
	  ((eq =op= 'beta-tag) (go %beta-tag))
	  ((eq =op= 'exit) (go %exit))
	  ((eq =op= 'reset) (go %reset))
	  ((eq =op= 'declare-constant) (go %declare-constant))
	  ((eq =op= 'undeclare-constant) (go %undeclare-constant))
	  ((eq =op= 'function-alias) (go %function-alias))
	  ((eq =op= 'gc) (go %gc))
	  ((eq =op= 'top-level-ids) (go %top-level-ids))
	  ((eq =op= 'scheme-constants) (go %scheme-constants))
	  ((eq =op= 'beta-transforms) (go %beta-transforms))
	  ((eq =op= 'base-identifiers) (go %base-identifiers))
	  ((eq =op= 'scheme-primitives) (go %scheme-primitives))
	  ((eq =op= 'system-functions) (go %system-functions))
	  ((eq =op= 'reify) (go %reify))
	  ((eq =op= 'global-namespace-type) (go %global-namespace-type))
	  ((eq =op= 'import) (go %import))
	  ((eq =op= 'lisp-eval) (go %lisp-eval))
	  ((eq =op= 'make-printable) (go %make-printable))
	  ((eq =op= 'make-unprintable) (go %make-unprintable))
	  ((eq =op= 'ptime) (go %ptime))
	  ((eq =op= 'remove-from-namespace) (go %remove-from-namespace))
	  ((eq =op= 'scheme-reset) (go %scheme-reset))
	  ((eq =op= 'transcript-off) (go %transcript-off))
	  ((eq =op= 'transcript-on) (go %transcript-on))
	  ((eq =op= 'add-mkmac-name) (go %add-mkmac-name))
	  ((eq =op= 'remove-mkmac-name) (go %remove-mkmac-name))
	  ((eq =op= 'set-lexical-semantics) (go %set-lexical-semantics))
	  ((eq =op= 'set-application-semantics)
	   (go %set-application-semantics))
	  ((eq =op= 'set-literal-semantics) (go %set-literal-semantics))
	  ((eq =op= 'scoped?) (go %scoped?))
	  (t (go bad-op)))

 &transcendental
  (cond  ((eq =op= 'arccos) (go %arccos))
	 ((eq =op= 'arcsin) (go %arcsin))
	 ((eq =op= 'arctan) (go %arctan)) ; 1-Mar-85
	 ((eq =op= 'atan) (go %atan))
	 ((eq =op= 'cos) (go %cos))
	 ((eq =op= 'exp) (go %exp))
	 ((eq =op= 'expt) (go %expt))
	 ((eq =op= 'log) (go %log))
	 ((eq =op= 'sin) (go %sin))
	 ((eq =op= 'tan) (go %tan))
	 (t (go bad-op)))
  
 &misc
   (cond
     ((eq =op= 'C) (go %C))
     ((eq =op= 'append-cont) (go %append-cont))
     ((eq =op= 'partial-cont) (go %partial-cont))
     ((eq =op= 'T) (go %T))
     ((eq =op= 'try) (go %try))
     ((eq =op= 'exit-try) (go %exit-try))
     ((eq =op= 'execute) (go %execute))
     ((eq =op= 'apply) (go %apply))
     ((eq =op= 'result) (go %result))
     ((eq =op= 'synthetic-identifier) (go %synthetic-identifier))
     ((eq =op= 'ascii->symbol) (go %ascii->symbol))
     ((eq =op= 'symbol->ascii) (go %symbol->ascii))
     ((eq =op= 'deref) (go %deref))
     ((eq =op= 'explode) (go %explode))
     ((eq =op= 'genbase) (go %genbase))
     ((eq =op= 'gensym) (go %gensym))
     ((eq =op= 'string->uninterned) (go %string->uninterned))
     ((eq =op= 'getprop) (go %getprop))
     ((eq =op= 'global-binding) (go %global-binding))
     ((eq =op= 'implode) (go %implode))
     ((eq =op= 'proplist) (go %proplist))
     ((eq =op= 'putprop) (go %putprop))
     ((eq =op= 'ref) (go %ref))
     ((eq =op= 'remprop) (go %remprop))
     ((eq =op= 'set-ref!) (go %set-ref!))
     ((eq =op= 'subst) (go %subst))
     ((eq =op= 'substring) (go %substring))
     ((eq =op= 'swap-ref!) (go %swap-ref!))
     (t (go bad-op)))
   
 %car (setq **result** (car **result**)) (go loop)
 %cdr (setq **result** (cdr **result**)) (go loop)
 %caaar (setq **result** (caaar **result**)) (go loop)
 %caadr (setq **result** (caadr **result**)) (go loop)
 %caar (setq **result** (caar **result**)) (go loop)
 %cadar (setq **result** (cadar **result**)) (go loop)
 %caddr (setq **result** (caddr **result**)) (go loop)
 %cadr (setq **result** (cadr **result**)) (go loop)
 %cdaar (setq **result** (cdaar **result**)) (go loop)
 %cdadr (setq **result** (cdadr **result**)) (go loop)
 %cdar (setq **result** (cdar **result**)) (go loop)
 %cddar (setq **result** (cddar **result**)) (go loop)
 %cdddr (setq **result** (cdddr **result**)) (go loop)
 %cddr (setq **result** (cddr **result**)) (go loop)
 %caaaar (setq **result** (caaaar **result**)) (go loop)
 %caaadr (setq **result** (caaadr **result**)) (go loop)
 %caadar (setq **result** (caadar **result**)) (go loop)
 %caaddr (setq **result** (caaddr **result**)) (go loop)
 %cadaar (setq **result** (cadaar **result**)) (go loop)
 %cadadr (setq **result** (cadadr **result**)) (go loop)
 %caddar (setq **result** (caddar **result**)) (go loop)
 %cadddr (setq **result** (cadddr **result**)) (go loop)
 %cdaaar (setq **result** (cdaaar **result**)) (go loop)
 %cdaadr (setq **result** (cdaadr **result**)) (go loop)
 %cdadar (setq **result** (cdadar **result**)) (go loop)
 %cdaddr (setq **result** (cdaddr **result**)) (go loop)
 %cddaar (setq **result** (cddaar **result**)) (go loop)
 %cddadr (setq **result** (cddadr **result**)) (go loop)
 %cdddar (setq **result** (cdddar **result**)) (go loop)
 %cddddr (setq **result** (cddddr **result**)) (go loop)
 %cons (setq **result** (cons **result** (top-stack))) (go pop1)
 %set-car!!
   (cond
      [(not (dtpr **result**))
       (raise (list 'SE%vsm 0 t '|Bad argument to set-car!:| **result**))])
   (setq **result** (rplaca **result** (top-stack))) (go pop1)
 %set-car!
   (cond
     [(dtpr **result**)
      (cond
	[(proc? **result**)
	 (raise
	   (list 'SE%vsm 0 t '|Illegal argument to set-car! :| **result**))]
	[t (setq **result** (rplaca **result** (top-stack))) (go pop1)])]
     [t (raise (list 'SE%vsm 0 t '|Bad argument to set-car!:| **result**))])
 %set-cdr!
   (cond
     [(dtpr **result**)
      (cond
	[(proc? **result**)
	 (raise
	   (list 'SE%vsm 0 t '|Illegal argument to set-cdr! :| **result**))]
	[t (setq **result** (rplacd **result** (top-stack))) (go pop1)])]
     [t (raise (list 'SE%vsm 0 t '|Bad argument to set-cdr! :| **result**))])

 %copy-no-constant (setq **result** (copy-no-constant **result**)) (go loop)
 %delete (setq **result** (delete **result** (top-stack))) (go pop1)
 %delq (setq **result** (delq **result** (top-stack))) (go pop1)
 %last-pair (setq **result** (last **result**)) (go loop)
 %length (setq **result** (length **result**)) (go loop)
 %member (setq **result** (member **result** (top-stack))) (go pop1)
 %nth (setq **result** (nthelem **result** (top-stack))) (go pop1)
 %list-ref (setq **result** (nthelem (1+ (top-stack)) **result**)) (go pop1)
 %list-tail (setq **result** (nthcdr (top-stack) **result**)) (go pop1)
 %reverse (setq **result** (reverse **result**)) (go loop)
 %reverse! (setq **result** (nreverse **result**)) (go loop)
 %transpose (setq **result** (transpose **result**)) (go loop)
 %null? (setq **result** (null **result**)) (go loop)
 %eq? (setq **result** (eq **result** (top-stack))) (go pop1)
 %eqv? (setq **result** (eqv **result** (top-stack))) (go pop1)
 %memv (setq **result** (memv **result** (top-stack))) (go pop1)
 %assv (setq **result** (assv **result** (top-stack))) (go pop1)
 %assoc (setq **result** (assoc **result** (top-stack))) (go pop1)
 %assq (setq **result** (assq **result** (top-stack))) (go pop1)
 %port? (setq **result** (port? **result**)) (go loop)
 %equal? (setq **result** (equal **result** (top-stack))) (go pop1)
 %memq (setq **result** (memq **result** (top-stack))) (go pop1)
 %atom? (setq **result** (atom **result**)) (go loop)
 %number? (setq **result** (numberp **result**)) (go loop)
 %string? (setq **result** (stringp **result**)) (go loop)
 %proc? (setq **result** (proc? **result**)) (go loop)
 %fix? (setq **result** (fixp **result**)) (go loop)
 %float? (setq **result** (floatp **result**)) (go loop)
 %pair?
   (setq **result**
     (and (dtpr **result**)
	  (not (memq (car **result**) unprintable-symbols))))
   (go loop)
 %ref?
   (setq **result** (and (dtpr **result**) (eq '&ref (car **result**))))
   (go loop)
 %symbol? (setq **result** (symbolp **result**)) (go loop)
 %syntactic-extension?
   (setq **result** (syntactic-extension? **result**)) (go loop)
 %scheme-constant? (setq **result** (scheme-constant? **result**)) (go loop)
 %add1 (setq **result** (add1 **result**)) (go loop)
 %sub1 (setq **result** (sub1 **result**)) (go loop)
 %+ (setq **result** (plus **result** (top-stack))) (go pop1)
 %* (setq **result** (times **result** (top-stack))) (go pop1)
 %- (setq **result** (difference **result** (top-stack))) (go pop1)
 %minus (setq **result** (minus **result**)) (go loop)
 %/ (setq **result** (quotient **result** (top-stack))) (go pop1)
 %abs (setq **result** (abs **result**)) (go loop)
 %factorial (setq **result** (fact **result**)) (go loop)
 %fix (setq **result** (fix **result**)) (go loop)
 %ceiling
   (cond
     ((fixp **result**))
     ((zerop (difference **result** (fix **result**))))
     (t (setq **result** (fix (add1 **result**)))))
   (go loop)
 %truncate
   (cond [(not (fixp **result**))
	  (setq **result**
	    (* (cond [(plusp **result**) 1] [t -1]) (fix (abs **result**))))])
   (go loop)
 %round (setq **result** (fix (plus **result** 0.5))) (go loop)
 %float (setq **result** (float **result**)) (go loop)
 %quotient (setq **result** (/ **result** (top-stack))) (go pop1)
 %remainder (setq **result** (mod **result** (top-stack))) (go pop1)
 %random (setq **result** (random **result**)) (go loop)
 %sqrt (setq **result** (sqrt **result**)) (go loop)
 %vector? (setq **result** (vactor? **result**)) (go loop)
 %list->vector (setq **result** (list->vactor **result**)) (go loop)
 %vector->list (setq **result** (vactor->list **result**)) (go loop)
 %vector-ref (setq **result** (vactor-ref **result** (top-stack))) (go pop1)
 %vector-set!
   (setq **result** (vactor-set! **result** (top-stack) (2nd-stack)))
   (go pop2)
 %vector-length (setq **result** (vactor-length **result**)) (go loop)
 %vector-fill!
   (setq **result** (vactor-fill! **result** (top-stack))) (go pop1)
 %primitive-make-vector (setq **result** (make-vactor **result**)) (go loop)
 %=0 (setq **result** (and (numberp **result**) (zerop **result**))) (go loop)
 %= (setq **result** (zerop (difference **result** (top-stack)))) (go pop1)
 %< (setq **result** (lessp **result** (top-stack))) (go pop1)
 %> (setq **result** (greaterp **result** (top-stack))) (go pop1)
 %>0 (setq **result** (plusp **result**)) (go loop)
 %<0 (setq **result** (minusp **result**)) (go loop)
 %>= (setq **result** (not (lessp **result** (top-stack)))) (go pop1)
 %<= (setq **result** (not (greaterp **result** (top-stack)))) (go pop1)
 %alpha< (setq **result** (alphalessp **result** (top-stack))) (go pop1)
 %close (setq **result** (schclose **result**)) (go loop)
 %current-column (setq **result** (current-column)) (go loop)
 %display (setq **result** (display **result**)) (go loop)
 %display& (setq **result** (display& **result**)) (go loop)
 %file-exists? (setq **result** (probef **result**)) (go loop)
 %flush-input
   (setq **result** nil) (drain (lport (lookupinbase 'input-port))) (go loop)
 %flush-output
   (setq **result** nil) (drain (lport (lookupinbase 'output-port))) (go loop)
 %line-length
   (setq **result** (line-length (lookupinbase 'output-port))) (go loop)
 %newline (setq **result** (new-line)) (go loop)
 %open (setq **result** (open **result** (top-stack))) (go pop1)
 %print (setq **result** (schprint **result**)) (go loop)
 %print-length (setq **result** (pntlen **result**)) (go loop)
 %print& (setq **result** (schprint& **result**)) (go loop)
 %prompt-read (setq **result** (prompt-read **result**)) (go loop)
 %read (setq **result** (schread)) (go loop)
 %read-atom (setq **result** (read-atom)) (go loop)
 %read-char (setq **result** (read-char)) (go loop)
 %read/port (setq **result** (schread/port **result**)) (go loop)
 %read-atom/port (setq **result** (read-atom/port **result**)) (go loop)
 %read-char/port (setq **result** (read-char/port **result**)) (go loop)
 %set-line-length!
   (setq **result** (set-line-length! (lookupinbase 'output-port) **result**))
   (go loop)
 %pretty (setq **result** (pretty **result**)) (go loop)
 %compile (setq **result** (compile **result**)) (go loop)
 %mkmac-match?
   (setq **result** (mkmac-match? **result** (top-stack) (2nd-stack)))
   (go pop2)
 %declare-constant (declare-constant **result**) (go loop)
 %undeclare-constant (undeclare-constant **result**) (go loop)
 %exit (exit)
 %add-to-syntax-table
   (setq **result** (add-to-syntax-table **result** (top-stack))) (go pop1)
 %beta-expand (setq **result** (beta-expand **result**)) (go loop)
 %expand-once (setq **result** (expand-once **result**)) (go loop)
 %pp-exp ($prpr **result**) (setq **result** (terpri)) (go loop)
 %copying-intern* (setq **result** (copying-intern* **result**)) (go loop)
 %beta-tag (setq **result** (tag-frees **result** nil)) (go loop)
 %function-alias
   (setq **result** (function-alias **result**(top-stack))) (go pop1)
 %gc (setq **result** (gc)) (go loop)
 %top-level-ids (setq **result** (top-level-assigned-identifiers)) (go loop)
 %scheme-constants (setq **result** (symbols-bound-to-constants)) (go loop)
 %beta-transforms (setq **result** (beta-transforms)) (go loop)
 %base-identifiers (setq **result** (base-identifiers))  (go loop)
 %scheme-primitives (setq **result** (scheme-primitives)) (go loop)
 %system-functions (setq **result** (system-functions)) (go loop)
 %reify (setq **result** (reify **result**)) (go loop)
 %global-namespace-type
   (setq **result** (global-namespacetype **result**)) (go loop)
 %import (setq **result** (import **result**)) (go loop)
 %lisp-eval (setq **result** (eval **result**)) (go loop)
 %make-printable (setq **result** (make-printable **result**)) (go loop)
 %make-unprintable
   (setq **result** (make-unprintable **result** (top-stack))) (go pop1)
 %ptime (setq **result** (ptime)) (go loop)
 %remove-from-namespace
   (setq **result** (remove-from-namespace **result**)) (go loop)
 %reset (setq **result** (reset)) (go loop)
 %scheme-reset (setq **result** (scheme-reset)) (go loop)
 %transcript-off (setq **result** (transcript-off)) (go loop)
 %transcript-on (setq **result** (transcript-on **result**)) (go loop)
 %add-mkmac-name (setq **result** (add-mkmac-name **result**)) (go loop)
 %remove-mkmac-name (setq **result** (remove-mkmac-name **result**)) (go loop)
 %set-lexical-semantics
   (set-lexical-semantics **result**) (setq **result** t) (go loop)
 %set-application-semantics
   (set-application-semantics **result**) (setq **result** t) (go loop)
 %set-literal-semantics
   (set-literal-semantics **result**) (setq **result** t) (go loop)
 %scoped? (setq **result** (memq **result** (car **comp-env**))) (go loop)
 %arccos (setq **result** (acos **result**)) (go loop)
 %arcsin (setq **result** (asin **result**)) (go loop)
 %arctan (setq **result** (atan **result** 1)) (go loop) ; 1-Mar-85
 %atan (setq **result** (atan **result** (top-stack))) (go pop1)	     
 %cos (setq **result** (cos **result**)) (go loop)
 %exp (setq **result** (exp **result**)) (go loop)
 %expt (setq **result** (expt **result** (top-stack))) (go pop1)
 %log (setq **result** (log **result**)) (go loop)
 %sin (setq **result** (sin **result**)) (go loop)
 %tan (setq **result** (quotient (sin **result**) (cos **result**))) (go loop)
 %synthetic-identifier
  (setq **result** (cons '&synthetic-id (cons **result** (top-stack))))
  (go pop1)
 %result (return **result**)
 %ascii->symbol (setq **result** (ascii->symbol **result**)) (go loop)
 %symbol->ascii (setq **result** (symbol->ascii **result**)) (go loop)
 %deref (setq **result** (cdr **result**)) (go loop)
 %explode (setq **result** (aexplodec **result**)) (go loop)
 %genbase (setq **result** (genbase)) (go loop)
 %gensym (setq **result** (gensym **result**)) (go loop)
 %string->uninterned (setq **result** (uconcat **result**)) (go loop)
 %getprop (setq **result** (get **result** (top-stack))) (go pop1)
 %global-binding (setq **result** (global-binding **result**)) (go loop)
 %implode (setq **result** (implode **result**)) (go loop)
 %proplist (setq **result** (plist **result**)) (go loop)
 %putprop
   (setq **result** (putprop **result** (top-stack) (2nd-stack))) (go pop2)
 %ref (setq **result** (cons '&ref **result**)) (go loop)
 %remprop (setq **result** (remprop **result** (top-stack))) (go pop1)
 %set-ref! (setq **result** (rplacd **result** (top-stack))) (go pop1)
 %subst
   (setq **result** (subst **result** (top-stack) (2nd-stack))) (go pop2)
 %substring
   (setq **result** (substring **result** (top-stack) (2nd-stack))) (go pop2)
 %swap-ref! (setq =temp= (cdr **result**))
            (rplacd **result** (top-stack))
	    (setq **result** =temp=)
	    (go pop1)

 pop1 (setq **args** (cdr **args**)) (go loop)
 pop2 (setq **args** (cddr **args**)) (go loop)
 pop3 (setq **args** (cdddr **args**)) (go loop)

 bad-class (setq *error* =class=) (scherror "Bad primitive class")
 bad-op (setq *error* =op=) (scherror "Bad primitive op")
 bad-vsm-opcode (setq *error* =opcode=) (scherror "Bad vsm opcode")

 bad-function
   (setq *error* **cont**)
   (setq *error-args* **args**)
   (setq *error-data* **result**)
   (scherror "Bad function")

 wrong-number-of-args-to-closure
   (setq *error* **cont**)
   (setq *error-res* **result**)
   (setq *error-data* **args**)
   (scherror "Wrong number of arguments to closure")

 wrong-number-args-to-vector
   (setq *error* **result**)
   (setq *error-data* **args**)
   (scherror "Wrong number of arguments to vector")

 wrong-number-of-args-to-continuation
   (setq *error* **cont**)
   (setq *error-data* **args**)
   (scherror "Wrong number of arguments to continuation"))))

(def properargs
  (lambda (l)
    (cond
      [(proc? l)
       (raise (list 'SE%vsm 0 t '|Illegal argument to apply:| l))]
      [t (good-args l l)])))

(def good-args
  (lambda (l args)
    (cond
      [(dtpr l) (cons (car l) (good-args (cdr l) args))]
      [(null l) nil]
      [t (raise (list 'SE%vsm 0 t '|Illegal argument to apply:| args))])))

(def copy-cont
   (lambda (k1 k2)
      (cond
	 [(eq k1 k2) nil]
	 [(null k1) (scherror "Error 53 (partial-cont)")]
	 [t (cons (car k1) (copy-cont (cdr k1) k2))])))


