/* PORTS.C
 ************************************************************************
 *									*
 *		PC Scheme/Geneva 4.00 Borland C code			*
 *									*
 * (c) 1985-1988 by Texas Instruments, Inc. See COPYRIGHT.TXT		*
 * (c) 1992 by L. Bartholdi & M. Vuilleumier, University of Geneva	*
 *									*
 *----------------------------------------------------------------------*
 *									*
 *		 Basic manipulations on port Object			*
 *									*
 *----------------------------------------------------------------------*
 *									*
 * Created by: Marc Vuilleumier		Date: Jan 1993			*
 *             (get_port written by John Jensen 1985)			*
 * Revision history:							*
 * - 18 Jun 92:	Renaissance (Borland Compilers, ...)			*
 *									*
 *					``In nomine omnipotentii dei''	*
 ************************************************************************/

////////////// Don't forget to kill spopen, spclose & get_mode ////////////
/////////////////// when new ports will be working well ///////////////////

#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	<fcntl.h>
#include	<sys\stat.h>
#include	<share.h>
#include	<io.h>
#include	"scheme.h"

static char	*types[] = {
	"WINDOW", "SOFTWARE", "STRING", "FILE", NULL, "MAKE-PORT" };

typedef enum	{ 
	IS_WINDOW, IS_SOFTWARE, IS_STRING, IS_FILE
	} KIND_OF_PORT;

static char	*attributes[] = {
	"TYPE", "SOURCE", "READ", "WRITE", "BINARY?", "WRAP?", "NEW?"
	"TRANSCRIPT?", "LOCK?", "HANDLE", "BORDER", "TEXT", "LINE",
	"COLUMN", "TOP", "LEFT", "HEIGHT", "WIDTH", 
	NULL, "PORT-GET/SET-ATTRIBUTE/!" };

typedef	enum	{
	TYPE, SOURCE, READ, WRITE, BINARY, WRAP, NEW, TRANSCRIPT, LOCK,
	HANDLE, BORDER, TEXT, LINE, COLUMN, TOP, LEFT, HEIGHT, WIDTH
	} KIND_OF_ATTRIB;

static char	*modes[] = {
	"CLOSED", "PROTECTED", "SHARED", "EXCLUSIVE", NULL, "PORT-SET-ATTRIBUTE!" };

typedef	enum	{
	CLOSED, PROTECTED, SHARED, EXCLUSIVE
	} KIND_OF_MODE;
	
/************************************************************************/
/* Give enum equivalents of most importants flags			*/
/************************************************************************/
void	port_get_info( PORT far *p, KIND_OF_PORT *type, KIND_OF_MODE *read_mode, 
							KIND_OF_MODE *write_mode)
{
	switch( p->flags & PORT_TYPE ) {
		case TYPE_WINDOW:
			*type = IS_WINDOW; break;
		case TYPE_SOFTWARE:
			*type = IS_SOFTWARE; break;
		case TYPE_STRING:
			*type = IS_STRING; break;
		case TYPE_FILE:
			*type = IS_FILE; break;
	}
	switch( p->flags & READ_MODE ) {
		case READ_CLOSED:
			*read_mode = CLOSED; break;
		case READ_PROTECTED:
			*read_mode = PROTECTED; break;
		case READ_SHARED:
			*read_mode = SHARED; break;
		case READ_EXCLUSIVE:
			*read_mode = EXCLUSIVE; break;
	}
	switch( p->flags & WRITE_MODE ) {
		case WRITE_CLOSED:
			*write_mode = CLOSED; break;
		case WRITE_PROTECTED:
			*write_mode = PROTECTED; break;
		case WRITE_SHARED:
			*write_mode = SHARED; break;
		case WRITE_EXCLUSIVE:
			*write_mode = EXCLUSIVE; break;
	}
}

/************************************************************************/
/* Determine Port							*/
/*									*/
/* Purpose:  To determine is a register contains a valid port object	*/
/* representation and to return the appropriate port			*/
/* pointer in "tmp_reg".						*/
/************************************************************************/
int	get_port(REGPTR reg, int mode)
{
	unsigned	disp;	/* displacement component of a pointer */
	unsigned	page;	/* page number component of a pointer */

	/* fetch page and displacement portions of port pointer */
	page = CORRPAGE(reg->page);
	disp = reg->disp;

	/* check to see if port pointer is nil-- if so, search fluid env */
	if (!page) {
		if (mode)
			intern(&tmp_reg, "OUTPUT-PORT", 11);
		else
			intern(&tmp_reg, "INPUT-PORT", 10);

		/* search fluid environment for interned symbol */
		fluid_lookup(&tmp_reg);
		page = CORRPAGE(tmp_reg.page);
		disp = tmp_reg.disp;
	}
	/* page & disp should point to a port, or the symbol 'console */
	if (ptype[page] != PORTTYPE) {
		if (CORRPAGE(console_reg.page) != page || console_reg.disp != disp)
			return	1;
		tmp_reg.page = ADJPAGE(SPECPOR);
		tmp_reg.disp = (mode ? OUT_DISP : IN_DISP);
	} else {
		tmp_reg.page = ADJPAGE(page);
		tmp_reg.disp = disp;
	}
	return	0;
}

/************************************************************************/
/* Make a new port							*/
/*									*/
/* Purpose: to allocate a new port object, of given type and based on	*/
/*	source (source type depend of the given type)			*/
/************************************************************************/
int	make_port( REGPTR port, REGPTR source )
{
	KIND_OF_PORT	type;
	PORT	far	*p;

	type = (KIND_OF_PORT) match( port, types );
	switch( type ) {
		case IS_SOFTWARE:
			if( ptype[CORRPAGE(source->page)] != CLOSTYPE ) {
				set_src_error("MAKE-PORT", 2, port, source);
				return -1;
			}
			break;
		case IS_WINDOW:
		case IS_STRING:
			if( eq(source, &nil_reg) ) break;
		case IS_FILE:
			if( ptype[CORRPAGE(source->page)] != STRTYPE ) {
				set_src_error("MAKE-PORT", 2, port, source);
				return -1;
			}
	}

	alloc_block(port, PORTTYPE, sizeof(PORT)-BLK_OVHD );
	zero_blk( CORRPAGE(port->page), port->disp );
	p = &reg2c(port)->port;

	load( &(p->ptr), source );
	p->flags = PORT_BINARY | PORT_LOCKED;
	p->nlines = BUFFSIZE;
	p->ncols = 1;

	switch( type ) {
		case IS_WINDOW:
			p->flags |= TYPE_WINDOW | PORT_WRAP;
			p->nlines = get_max_rows();
			p->ncols = get_max_cols();
			p->border = -1;
			p->text = 7;
			break;
		case IS_SOFTWARE:
			p->flags |= TYPE_SOFTWARE;
			break;
		case IS_STRING:
			p->flags |= TYPE_STRING;
			if( eq( source, &nil_reg ) )
				p->flags |= PORT_NEW;
			break;
		case IS_FILE: 
			p->flags |= TYPE_FILE;
		{
			REG	tmp(p->ptr.disp, p->ptr.page);
			char	*name = string_asciz(&tmp);
			int	err = sopen( name, O_RDONLY | SH_DENYNO, 0);

			if( err < 0 )
				p->flags |= PORT_NEW;
			else
				close( err );
			rlsstr( name );
		}
	}
	return 0;
}

/************************************************************************/
/* Get one of the port attributes					*/
/*									*/
/************************************************************************/
int	port_get_attribute( REGPTR port, REGPTR symbol )
{
	PORT	far	*p;
	KIND_OF_PORT	type;
	KIND_OF_MODE	read_mode, write_mode;
	KIND_OF_ATTRIB	attr;

	if( get_port(port, OUTPUT_PORT) )
	{
		set_src_error("PORT-GET-ATTRIBUTE", 2, port, symbol);
		return	-1;
	} else
		*port =	tmp_reg;

	attr = (KIND_OF_ATTRIB) match( symbol, attributes );
	p = &reg2c(port)->port;
	port_get_info( p, &type, &read_mode, &write_mode );

	switch( attr ) {
		case TYPE:
			intern( port, types[type], strlen(types[type]) );
			return 0;
		case SOURCE:
			load( port, &(p->ptr) );
			return 0;
		case READ:
			intern( port, modes[read_mode], strlen(modes[read_mode]) );
			return 0;
		case WRITE:
			intern( port, modes[write_mode], strlen(modes[write_mode]) );
			return 0;
		case BINARY:
			bool2scm( port, p->flags & PORT_BINARY );
			return 0;
		case WRAP:
			bool2scm( port, p->flags & PORT_WRAP );
			return 0;
		case NEW:
			bool2scm( port, p->flags & PORT_NEW );
			return 0;
		case TRANSCRIPT:
			bool2scm( port, p->flags & PORT_TRANSCRIPT );
			return 0;
		case LOCK:
			bool2scm( port, p->flags & PORT_LOCKED );
			return 0;
		case HANDLE:
			if ( type == IS_FILE || type == IS_SOFTWARE ) {
				long2int( port, p->handle );
				return 0;
			} else	break;
		case BORDER:
			if ( type == IS_WINDOW || type == IS_SOFTWARE ) {
				long2int( port, p->border );
				return 0;
			} else	break;
		case TEXT:
			if ( type == IS_WINDOW || type == IS_SOFTWARE ) {
				long2int( port, p->text );
				return 0;
			} else	break;
		case LINE:
			long2int( port, p->curline );
			return 0;
		case COLUMN:
			long2int( port, p->curcol );
			return 0;
		case TOP:
			long2int( port, p->ulline );
			return 0;
		case LEFT:
			long2int( port, p->ulcol );
			return 0;
		case HEIGHT:
			long2int( port, p->nlines );
			return 0;
		case WIDTH:
			long2int( port, p->ncols );
			return 0;
	}
	set_src_error("PORT-GET-ATTRIBUTE", 2, port, symbol);
	return -1;
}

/************************************************************************/
/* Calculate offset of upper-left position of a port			*/
/************************************************************************/
inline	long	poffset( PORT far *p )
{
	return p->ulcol + p->ulline * p->ncols;
}

/************************************************************************/
/* Calculate length of active window of a port				*/
/************************************************************************/
inline	long	plength( PORT far *p )
{
	return p->nlines * p->ncols;
}

/************************************************************************/
/* Calculate offset from upper-left position of a port			*/
/************************************************************************/
inline	long	pcurrent( PORT far *p )
{
	return (p->ulcol + p->curcol) + (p->ulline + p->curline) * p->ncols;
}

/************************************************************************/
/* Lock a port 								*/
/************************************************************************/
int	plock( REGPTR port )
{
	PORT	far	*p = &reg2c(port)->port;

	if( p->flags & PORT_OPEN && p->flags & PORT_SHARED && p->flags & PORT_LOCKED ) {
		if( (p->flags & PORT_TYPE) == TYPE_FILE)
			if( !lock( p->handle, poffset(p), plength(p) ) )
				return 0;
		else {
			/* ensure string locking accepted */
			return 0;
		}
		/* signal error */
		return -1;
	}
	return 0;
}

/************************************************************************/
/* Unlock a port							*/
/************************************************************************/
void	punlock( REGPTR port )
{
	PORT	far	*p = &reg2c(port)->port;

	if( p->flags & PORT_OPEN && (p->flags & PORT_TYPE) == TYPE_FILE && p->flags & PORT_LOCKED )
		unlock( p->handle, poffset(p), plength(p) );
}

/************************************************************************/
/* Open a port 								*/
/************************************************************************/
int	popen( REGPTR port )
{
	PORT	far	*p = &reg2c(port)->port;

	switch( p->flags & PORT_TYPE ) {
		case TYPE_FILE: {
			REG	tmp(p->ptr.disp, p->ptr.page);
			int	att = O_BINARY;
			char	*name = string_asciz(&tmp);

			if( p->flags & WRITE_OPEN ) {
				if( p->flags & PORT_NEW ) {
					int	err = creat( name, S_IREAD|S_IWRITE );
					if( err < 0 ) {
						/* handle errors */
						rlsstr( name );
						return -1;
					} else
						close( err );
				}

				if( p->flags & READ_OPEN )
					att |= O_RDWR;
				else
					att |= O_WRONLY;
			} else
				att |= O_RDONLY;

			if( p->flags & WRITE_PRIVATE )
				if( p->flags & READ_PRIVATE )
					att |= SH_DENYRW;
				else
					att |= SH_DENYWR;
			else
				if( p->flags & READ_PRIVATE )
					att |= SH_DENYRD;
				else
					att |= SH_DENYNONE;
			{
				int err = sopen( name, att, 0 );

				rlsstr( name );
				if( err < 0 ) {
					/* handle errors */
					return -1;
				} else
					p->handle = err;
			}
			break;
		}
		case TYPE_STRING:
			/* test nil string -> create, like files */
			/* verify access */
			return -1;
	}
	return plock( port );
}

/************************************************************************/
/* Close a port 							*/
/************************************************************************/
void	pclose( REGPTR port )
{
	PORT	far	*p;

	punlock( port );	

	p = &reg2c(port)->port;
	if( (p->flags & PORT_TYPE) == TYPE_FILE )
		close( p->handle );
}

/************************************************************************/
/* Set one of the port attributes					*/
/*									*/
/************************************************************************/
int	port_set_attribute( REGPTR port, REGPTR symbol, REGPTR value )
{
	PORT	far	*p;
	KIND_OF_PORT	type;
	KIND_OF_MODE	read_mode, write_mode, new_mode;
	KIND_OF_ATTRIB	attr;

	if( get_port(port, OUTPUT_PORT) )
	{
		set_src_error("PORT-SET-ATTRIBUTE!", 2, port, symbol);
		return	-1;
	} else
		*port =	tmp_reg;

	attr = (KIND_OF_ATTRIB) match( symbol, attributes );
	p = &reg2c(port)->port;
	port_get_info( p, &type, &read_mode, &write_mode );

	switch( attr ) {
		case READ:
			new_mode = (KIND_OF_MODE) match( value, modes );
			p = &reg2c(port)->port;
			p->flags &= ~READ_MODE;
			switch( new_mode ) {
				case CLOSED:
					p->flags |= READ_CLOSED; break;
				case PROTECTED:
					p->flags |= READ_PROTECTED; break;
				case SHARED:
					p->flags |= READ_SHARED; break;
				case EXCLUSIVE:
					p->flags |= READ_EXCLUSIVE;
			}
			if( read_mode != CLOSED || write_mode != CLOSED )
				pclose( port );
			if( new_mode != CLOSED || write_mode != CLOSED )
				if( popen( port ) ) {
					p->flags &= ~PORT_OPEN;
					return -1;
				}
			intern( port, modes[read_mode], strlen(modes[read_mode]) );
			return 0;
		case WRITE:
			new_mode = (KIND_OF_MODE) match( value, modes );
			p = &reg2c(port)->port;
			p->flags &= ~WRITE_MODE;
			switch( new_mode ) {
				case CLOSED:
					p->flags |= WRITE_CLOSED; break;
				case PROTECTED:
					p->flags |= WRITE_PROTECTED; break;
				case SHARED:
					p->flags |= WRITE_SHARED; break;
				case EXCLUSIVE:
					p->flags |= WRITE_EXCLUSIVE; break;
			}
			if( read_mode != CLOSED || write_mode != CLOSED )
				pclose( port );
			if( read_mode != CLOSED || new_mode != CLOSED )
				if( popen( port ) ) {
					p->flags &= ~PORT_OPEN;
					return -1;
				}
			intern( port, modes[write_mode], strlen(modes[write_mode]) );
			return 0;
		case BINARY:
			bool2scm( port, p->flags & PORT_BINARY );
			if( scm2bool(value) )
				p->flags |= PORT_BINARY;
			else
				p->flags &= ~PORT_BINARY;
			return 0;
		case WRAP:
			bool2scm( port, p->flags & PORT_WRAP );
			if( scm2bool(value) )
				p->flags |= PORT_WRAP;
			else
				p->flags &= ~PORT_WRAP;
			return 0;
		case NEW:
			bool2scm( port, p->flags & PORT_NEW );
			if( scm2bool(value) )
				p->flags |= PORT_NEW;
			else
				p->flags &= ~PORT_NEW;
			return 0;
		case TRANSCRIPT:
			bool2scm( port, p->flags & PORT_TRANSCRIPT );
			if( scm2bool(value) )
				p->flags |= PORT_TRANSCRIPT;
			else
				p->flags &= ~PORT_TRANSCRIPT;
			return 0;
		case LOCK:
			bool2scm( port, p->flags & PORT_LOCKED );
			if( scm2bool(value) ) {
				if( plock( port ) )
					return -1;
				else
					p->flags |= PORT_LOCKED;
			} else {
				p->flags &= ~PORT_LOCKED;
				punlock( port );
			}
			return 0;
		case HANDLE:
			if ( type == IS_SOFTWARE ) {
				long2int( port, p->handle );
				p->handle = int2long( value );
				return 0;
			} else	break;
		case BORDER:
			if ( type == IS_WINDOW || type == IS_SOFTWARE ) {
				long2int( port, p->border );
				p->border = int2long( value );
				if( type == IS_WINDOW && p->border != 0xffff ) {
					char	*string;

					load( &tmp_reg, &(p->ptr) );
					string = string_asciz(&tmp_reg);
					zborder( p->ulline, p->ulcol, p->nlines,
						p->ncols, p->border, string);
					rlsstr(string);
				}
				return 0;
			} else	break;
		case TEXT:
			if ( type == IS_WINDOW || type == IS_SOFTWARE ) {
				long2int( port, p->text );
				p->text = int2long( value );
				return 0;
			} else	break;
		case LINE: {
			long	old = p->curline, val = int2long(value);
			if ( val < 0 )
				val += p->nlines;

			if ( val > 0 && val < p->nlines ) {
				p->curline = val;
				if( type == IS_FILE )
					lseek( p->handle, pcurrent(p), SEEK_SET );
				p->bufpos = p->bufend;
				long2int( port, old );
				return 0;
			}
		}
		case COLUMN: {
			long	old = p->curcol, val = int2long(value);
			if ( val < 0 )
				val += p->ncols;

			if ( val > 0 && val < p->ncols ) {
				p->curcol = val;
				if( type == IS_FILE )
					lseek( p->handle, pcurrent(p), SEEK_SET );
				p->bufpos = p->bufend;
				long2int( port, old );
				return 0;
			}
		}
		case TOP: {
			int	old = p->ulline, val = int2long(value);

			punlock( port );
			if( val < 0 )	switch( type ) {
				case IS_WINDOW:
					val += get_max_rows();
					break;
				case IS_FILE:
					val += ( filelength(p->handle) / plength(p) );
					break;
				case IS_STRING:
					val += ( regstrlen(&(REG)p->ptr) / plength(p) );
					p = &reg2c(port)->port;
			};
			p->ulline = val + 1;
			if( type == IS_FILE )
				lseek( p->handle, pcurrent(p), SEEK_SET );
			plock( port );
			p->bufpos = p->bufend;
			long2int( port, old );
			return 0;
		}
		case LEFT: {
			int	old = p->ulcol, val = int2long(value);

			punlock( port );
			if( val < 0 )	switch( type ) {
				case IS_WINDOW:
					val += get_max_cols();
					break;
				case IS_FILE:
					val += filelength(p->handle);
					break;
				case IS_STRING:
					val += regstrlen(&(REG)p->ptr);
					p = &reg2c(port)->port;
			};
			p->ulcol = val + 1;
			if( type == IS_FILE )
				lseek( p->handle, pcurrent(p), SEEK_SET );
			plock( port );
			p->bufpos = p->bufend;
			long2int( port, old );
			return 0;
		}
		case HEIGHT: {
			int	old = p->nlines, val = int2long(value);

			if( val >= 0 ) {
				punlock( port );
				p->nlines = val;
				if( p->curline >val ) {
					p->curline = val;
					if( type == IS_FILE )
						lseek( p->handle, pcurrent(p), SEEK_SET );
				}
				plock( port );
				p->bufpos = p->bufend;
				long2int( port, old );
				return 0;
			}
		}
		case WIDTH: {
			int	old = p->ncols, val = int2long(value);

			if( val >= 0 ) {
				punlock( port );
				p->ncols = val;
				if( p->curcol >val ) {
					p->curcol = val;
					if( type == IS_FILE )
						lseek( p->handle, pcurrent(p), SEEK_SET );
				}
				plock( port );
				p->bufpos = p->bufend;
				long2int( port, old );
				return 0;
			}
		}
	}
	set_src_error("PORT-SET-ATTRIBUTE!", 3, port, symbol, value);
	return -1;
}

/************************************************************************/
/* Match a symbolic parameter to a string table				*/
/************************************************************************/
int	match( REGPTR symbol, char **str )
{
	int	count = 0;

	if (ptype[CORRPAGE(symbol->page)] == SYMTYPE)
		while( *str )
		{
			intern( &tmp_reg, *str, strlen(*str) );
			if ( eq( &tmp_reg, symbol ) )
				return	count;
			count++;
			str++;
		}
	str++;
	set_src_error(*str, 1, symbol);
	scheme_error();		/* we won't return from this call */
	return -1;
}



///////////////////////////////////////////////////////////////////////////
//////// Following procedures are to be destroyed ! ///////////////////////
///////////////////////////////////////////////////////////////////////////

/************************************************************************/
/* Open a Port								*/
/************************************************************************/
#define FILE_NOT_FOUND	2	/* MS-DOS error code */
#define NON_RESTART	1	/* Operation not restartable */
#define READ		0
#define WRITE		1
#define	APPEND		2

int	spopen(REGPTR file, REGPTR mode)
{
	extern int	prn_handle;	/* handle assigned to printer */
	int		direction;	/* 'read, 'write, 'append code */
	unsigned	disp;
	int		handle;
	int		i;
	int		len;	/* length of file's pathname (plus 1) */
 	unsigned	page;
	int		retstat = 0;
	int		stat;	/* status returned from open request */
	char		*string;	/* file pathname buffer pointer */
	unsigned long	fsize;	/* file size - dbs */
	SCHEMEOBJ	o;

	/* identify mode value */
	if ((direction = get_mode(mode)) == -1)
		goto src_err;

	page = CORRPAGE(file->page);
	disp = file->disp;
	o = reg2c(file);

	switch (ptype[page]) {
	case STRTYPE:
		len = o->string.len;
		if (len < 0)	/* Adjust for small string */
			len = len + BLK_OVHD;
		else
			len = len - BLK_OVHD;

		if (!(string = (char *) malloc(len + 1)))
			malloc_error("spopen");
		get_str(string, page, disp);
		string[len] = '\0';
		for (i = 0; i < len; i++)
			string[i] = toupper(string[i]);
		switch (direction) {
		case READ:
			if ((stat = zopen(&handle, string, direction, &fsize)) != 0) {
		open_error:
				rlsstr(string);
				stat += (IO_ERROR_START - 1);
				dos_error(NON_RESTART, stat, file);
			}
			break;
		case WRITE:
			if ((stat = zcreate(&handle, string)) != 0)
				goto open_error;
			if (((stat = strcmp(string, "PRN")) == 0) ||
				((stat = strcmp(string, "LST")) == 0))
				prn_handle = handle;
			break;
		case APPEND:
			if ((stat = zopen(&handle, string, direction, &fsize)) == FILE_NOT_FOUND) {
				if ((stat = zcreate(&handle, string)) != 0)
					goto open_error;
				break;
			}
			if (stat)
				goto open_error;
			/*
			 * do { if (zread(handle, buffer, &length)) break; }
			 * while (length);
			 */
			if (((stat = strcmp(string, "PRN")) == 0) ||
				((stat = strcmp(string, "LST")) == 0))
				break;
			mov_fptr(handle);
		}
		tmp_reg = *file;
		alloc_block(file, PORTTYPE, sizeof(PORT)-BLK_OVHD );
		page = CORRPAGE(file->page);
		disp = file->disp;
		zero_blk(page, disp);
		o = reg2c(file);

		if (direction == WRITE)
			o->port.ulline = 1;
		else if (direction == APPEND) {	/* update the chunk# and
						 * buffer position */
			o->port.ulline = (fsize >> 8) + 1;
			o->port.bufpos = fsize & 0xff;
			direction = WRITE;	/* unsets read flag - dbs */
		}
		switch (direction) {
		case READ:
			o->port.flags = READ_EXCLUSIVE; break;
		case WRITE:
			o->port.flags = WRITE_EXCLUSIVE; break;
		case APPEND:
			o->port.flags = READ_EXCLUSIVE + WRITE_EXCLUSIVE; break;
		}
		o->port.flags |= TYPE_FILE;
		o->port.ncols = 80;
		o->port.handle = handle;
		o->port.nlines = fsize >> 16;
		o->port.border = fsize & 0xffff;
		/* put pointer to pathname into port object */
		o->port.ptr.page = tmp_reg.page;
		o->port.ptr.disp = tmp_reg.disp;
		rlsstr(string);
		break;

	case SYMTYPE:
		if (file->page != console_reg.page || file->page != console_reg.disp)
			goto src_err;
		break;

	case PORTTYPE:
		if( o->port.flags & (READ_EXCLUSIVE | WRITE_EXCLUSIVE) )
			break;

src_err:
	default:
		set_src_error("OPEN-PORT", 2, file, mode);
		retstat = -1;
	}
	return	retstat;
}

/************************************************************************/
/* Close a Port				 				*/
/************************************************************************/
int	spclose(REGPTR port)
{
	SCHEMEOBJ	o;

	if( get_port(port, INPUT_PORT) )
	{
		set_src_error("CLOSE-PORT", 1, port);
		return	-1;
	}

	o = reg2c(&tmp_reg);

	if( o->port.flags & PORT_OPEN && (o->port.flags & PORT_TYPE) == TYPE_FILE )
	{
		int	stat;

		if ((stat = zclose(o->port.handle)) != 0)
		{
			stat += (IO_ERROR_START - 1);
	io_err:
			dos_error(NON_RESTART, stat, port);
		}
		o->port.bufpos = BUFFSIZE;

		o->port.flags &= ~(READ_MODE | WRITE_MODE);
		return	1;
	}
	o->port.flags &= ~(READ_MODE | WRITE_MODE);
	return	0;
}

/************************************************************************/
/* Local Support:	Determine Input/Output Mode Value		*/
/************************************************************************/
int	get_mode(REGPTR reg)
{
	char	*modes[] = {"READ", "WRITE", "APPEND", NULL };

	if (ptype[CORRPAGE(reg->page)] == SYMTYPE)
	for( int i = 0; modes[i]; i++ )
	{
		intern(&tmp_reg, modes[i], strlen(modes[i]) );
		if (tmp_reg.disp == reg->disp && tmp_reg.page == reg->page)
			return	i;
	}
	return	-1;
}
