; mini-fx.scm - an implementation of the Mini-FX language

; Revision 2, 13 September 89

; *** This file is specific to MIT Scheme.

; We assume that the character ~ is not used in names in user Mini-FX
; programs.  That way this file can avoid conflicts with user names by
; putting a ~ in names that are for its own use.  This is a kludge.

; The following is specific to C Scheme.

(define-macro (define-macro-at-load-time pattern . body)
  `(syntax-table-define
       user-initial-syntax-table
       ',(car pattern)
     (macro ,(cdr pattern) ,@body)))

; SYMBOL

; The SYMBOL construct is implemented as a synonym for quotation (').
; The distinction between SYMBOL and ' exists for the benefit of
; static type inference.

(define-macro-at-load-time (symbol x)
  (if (symbol? x)
      `',x
      (error "invalid syntax for symbol" `(symbol ,x))))

; DEFINE-PRODUCT

; Records are implemented as vectors.  Element 0 of a vector is a list
; (*record* label1 label2 ...), and the other elements are the
; component values.

(define-macro-at-load-time (define-product name labels)
  (define constructor-name
    (string->symbol (string-append (symbol->string 'make-)
				   (symbol->string name))))
  (define marker-name
    (string->symbol (string-append "~"
				   (symbol->string name)
				   (symbol->string '-marker))))
  (define (make-projector-name label)
    (string->symbol (string-append (symbol->string name)
				   "-"
				   (symbol->string label))))
  (define (make-matcher-name name)
    (string->symbol (string-append (symbol->string '~match-)
				   (symbol->string name))))
  (define temp-names
    (map (lambda (name)
	   (string->symbol (string-append (symbol->string '~temp-)
					  (symbol->string name))))
	 labels))
  `(begin (define ,name (vector '*product* ',name ',labels))
	    (define ,marker-name
	      '(*record* ,@labels))
	    ,@(do ((i 1 (+ i 1))
		   (labels labels (cdr labels))
		   (ds '()
		       (cons `(define ,(make-projector-name (car labels))
				(~make-projector ,marker-name ,i))
			     ds)))
		  ((null? labels) (reverse ds)))
	    (define ,constructor-name
	      (lambda ,temp-names	;A hack
		(vector ,marker-name ,@temp-names)))
	    (define ,(make-matcher-name constructor-name)
	      (~make-record-matcher ,marker-name))
	    '*product*))

; DEFINE-SUM

; Tagged values are implemented as vectors.  Element 0 of the vector
; is a two-element list (*tagged* label), and element 1 is the
; underlying value.

(define-macro-at-load-time (define-sum name labels)
  (define (make-injector-name label)
    (string->symbol (string-append (symbol->string label)
				   "->"
				   (symbol->string name))))
  (define (make-matcher-name name)
    (string->symbol (string-append (symbol->string '~match-)
				   (symbol->string name))))
  `(begin (define ,name (vector '*sum* ',name ',labels))
	  ,@(map (lambda (label)
		   (let ((inj (make-injector-name label)))
		     `(begin (define ,inj
			       (~make-injector ',label))
			     (define ,(make-matcher-name inj)
			       (~make-tag-matcher ',label)))))
		 labels)
	  '*sum*))

; MATCH

; This is not the most efficient possible strategy for implementing
; ML-style pattern matching.
;
; (match foo ((make-foo x y) ...) ...)
;   ==>  (~match-make-foo foo 2 (lambda (x y) ...) (lambda () ...))
;
; Don't try to understand the implementation of this macro without
; proper supervision.

(define-macro-at-load-time (match thing . clauses)

  ; The matcher for injector FOO is named ~MATCH-FOO
  (define (make-matcher-name name)
    (string->symbol (string-append (symbol->string '~match-)
				   (symbol->string name))))

  (define counter 0)

  (define (gentemp)
    (set! counter (+ counter 1))
    (string->symbol (string-append (symbol->string '~temp)
				   (number->string counter '(heur)))))

  (define (expand thing clauses)
    (if (null? clauses)
	`(error "invalid match - no pattern matched" ,thing)
	(let ((clause (car clauses)))
	  (if (pair? thing)
	      `(let ((~thing ,thing))
		 ,(expand-clause '~thing clause (expand '~thing (cdr clauses))))
	      ;Optimization
	      (expand-clause thing clause (expand thing (cdr clauses)))))))

  (define (expand-clause thing clause fail)
    (if (and (pair? clause)
	     (pair? (cdr clause))
	     (null? (cddr clause)))
	(expand-pattern thing (car clause) (cadr clause) fail)
	(else (error "invalid match clause syntax" clause))))

  ; succ and fail are expressions
  (define (expand-pattern thing pat succ fail)
    (cond ((eq? pat '_) succ)
	  ((symbol? pat)
	   (if (eq? pat thing)
	       succ			;Optimization
	       `(let ((,pat ,thing)) ,succ)))
	  ((number? pat)
	   `(if (= ,thing ,pat) ,succ ,fail))
	  ((boolean? pat)
	   `(if (eq? ,thing ,pat) ,succ ,fail))
	  ((string? pat) (char? pat)
	   `(if (string=? ,thing ,pat) ,succ ,fail))
	  ((char? pat)
	   `(if (char=? ,thing ,pat) ,succ ,fail))
	  ((not (and (pair? pat) (symbol? (car pat))))
	   (error "invalid match pattern" pat))
	  ((eq? (car pat) 'symbol)
	   `(if (eq? ,thing ,pat) ,succ ,fail))
	  ((eq? (car pat) 'quote)
	   (let ((pred (if (symbol? (cadr pat))
			   'eq?		;Optimization
			   'equal?)))
	     `(if (,pred ,thing ,pat) ,succ ,fail)))
	  ((eq? (car pat) 'quasiquote)
	   (expand-pattern thing (expand-quasiquote (cadr pat) 0) succ fail))
	  (else
	   (expand-compound-pattern thing pat succ fail))))

  ; Expand a pattern of the form (op arg ...).
  ; op is assumed to be an injection or construction procedure.
  ; If it's not, you'll get a Scheme error of the form
  ; "unbound variable ~MATCH-OP".

  (define (expand-compound-pattern thing pat succ fail)
    (let* ((names (map (lambda (pat)
			 (if (and (symbol? pat) (not (eq? pat '_)))
			     pat	;Optimization
			     (gentemp)))
		       (cdr pat)))
	   (matcher
	    `(,(make-matcher-name (car pat))
	      ,thing
	      ,(length (cdr pat))
	      (lambda ,names
		,(let loop ((names names)
			    (pats (cdr pat)))
		   (if (null? pats)
		       succ
		       (expand-pattern (car names)
				       (car pats)
				       (loop (cdr names) (cdr pats))
				       '(~fail)))))
	      ~fail)))
      (if (equal? fail '(~fail))
	  matcher			;Optimization
	  `(let ((~fail (lambda () ,fail)))
	     ,matcher))))

  ; Expand backquote forms into expressions using CONS, QUOTE, and
  ; APPEND

  (define (expand-quasiquote x level)
    (descend-quasiquote x level finalize-quasiquote))

  (define (descend-quasiquote x level return)
    (cond ((not (pair? x))
	   (return 'quote x))
	  ((and (not (null? (cdr x))) (null? (cddr x)))
	   (case (car x)
	     ((unquote)
	      (if (= level 0)
		  (return 'unquote (cadr x))
		  (descend-interesting x (- level 1) 'unquoted->sexpr return)))
	     ((unquote-splicing)
	      (if (= level 0)
		  (return 'unquote-splicing (cadr x))
		  (descend-interesting x (- level 1) 'unquoted-splicing->sexpr return)))
	     ((quasiquote)
	      (descend-interesting x (+ level 1) 'quasiquoted->sexpr return))
	     ((quote)
	      (descend-interesting x level 'quoted->sexpr return))
	     (else
	      (descend-quasiquote-list x level return))))
	  (else (descend-quasiquote-list x level return))))

  (define (descend-interesting x level inject return)
    (descend-quasiquote (cadr x) level
      (lambda (mode arg)
	(if (eq? mode 'quote)
	    (return 'quote x)
	    (return 'unquote `(,inject ,(finalize-quasiquote mode arg)))))))

  (define (descend-quasiquote-list x level return)
    (descend-quasiquote-tail x level
      (lambda (mode arg)
	(if (eq? mode 'quote)
	    (return 'quote x)
	    (return 'unquote `(list->sexpr ,arg))))))

  (define (descend-quasiquote-tail x level return)
    (if (null? x)
	(return 'quote x)
	(descend-quasiquote-tail (cdr x) level
	  (lambda (cdr-mode cdr-arg)
	    (descend-quasiquote (car x) level
	      (lambda (car-mode car-arg)
		(cond ((and (eq? car-mode 'quote) (eq? cdr-mode 'quote))
		       (return 'quote x))
		      ((eq? car-mode 'unquote-splicing)
		       (cond ((and (eq? cdr-mode 'quote) (null? cdr-arg))
			      ;; (,@mumble)
			      (return 'unquote car-arg))  ;Type must be a list!
			     (else
			      ;; (,@mumble ...)
			      (return 'unquote
				      `(append ,car-arg
					       ,(finalize-quasiquote
						   cdr-mode cdr-arg))))))
		      (else
		       (return 'unquote
			       `(cons ,(finalize-quasiquote car-mode car-arg)
				      ,(finalize-quasiquote cdr-mode cdr-arg)))))))))))

  (define (finalize-quasiquote mode arg)
    (case mode
      ((quote) `',arg)
      ((unquote) arg)
      ((unquote-splicing)
       (error ",@ in illegal context" arg)) ;`,@x or ``,,@x or `(y . ,@x)
      (else
       (error "quasiquote bug" mode arg))))

  (expand thing clauses))

; The run-time library.

; NULL returns an empty list.

(define (null) '())

; VECTOR-UPDATE is an applicative version of VECTOR-SET!.  Not useful
; for real programs, but perhaps useful pedagogically.

(define (vector-update vec i val)
  (let* ((len (vector-length vec))
	 (new (make-vector len)))
    (do ((j 0 (+ j 1)))
	((= j len)
	 (vector-set! new i val)
	 new)
      (vector-set! new j (vector-ref vec j)))))

; INTEGER->STRING has a more pleasant type and interface than
; Scheme's built-in NUMBER->STRING.

(define (integer->string n)
  (number->string n '(heur)))

; The "unit" (written #u in FX) is the only member of the type unit.

(define ~the-unit '#((*unit*)))

; References are implemented as 3-element vectors
;  #((*ref*) uid value)
; Uid is a unique identifier whose purpose is purely for debugging.
; When the Scheme REP loop prints two references, you can tell whether
; or not they're the same, even if they refer to the same value, by
; looking at their uids.  Probably the value shouldn't print at all,
; but implementing that behavior would require a separate table to
; simulate a store, which would be relatively much slower.

(define ~ref-marker '(*ref*))
(define ~*ref-counter* 0)

(define (ref initial-value)
  (let ((r (vector ~ref-marker ~*ref-counter* initial-value)))
    (set! ~*ref-counter* (+ ~*ref-counter* 1))
    r))

(define (~ref? r)
  (and (vector? r)
       (= (vector-length r) 3)
       (eq? (vector-ref r 0) ~ref-marker)))

(define (^ r)				;ref-ref
  (if (~ref? r)
      (vector-ref r 2)
      (error "invalid argument to ref-ref" r)))

(define (:= r new-value)		;ref-set!
  (if (~ref? r)
      (begin (vector-set! r 2 new-value)
	     ~the-unit)
      (error "invalid argument to ref-ref" r)))

(define (same-ref? r1 r2)
  (if (~ref? r1)
      (if (~ref? r2)
	  (eq? r1 r2)
	  (error "invalid argument to same-ref?" r2))
      (error "invalid argument to same-ref?" r1)))

; Auxiliaries for DEFINE-PRODUCT

; ~MAKE-PROJECTOR creates a projection procedure.

(define (~make-projector marker i)
  (lambda (r)
    (if (and (vector? r)
	     (> (vector-length r) 0))
	(if (equal? (vector-ref r 0) marker)
	    (vector-ref r i)  ;optimization
	    ; The following is a kludge to try to implement structural
	    ; equivalence instead of name equivalence of types.
	    (let ((label (list-ref marker i))
		  (marker (vector-ref r 0)))
	      (if (and (pair? marker)
		       (eq? (car marker) '*record*)
		       (= (length marker) (vector-length r)))
		  (let loop ((i 1) (l (cdr marker)))
		    (if (null? l)
			(error "invalid projection" r)
			(if (eq? (car l) label)
			    (vector-ref r i)
			    (loop (+ i 1) (cdr l)))))
		  (error "invalid projection" r))))
	(error "invalid projection" r))))

; ~MAKE-RECORD-MATCHER creates a matcher for a record constructor.

(define (~make-record-matcher marker)
  (lambda (thing nargs succ fail)
    nargs fail ;ignored
    (if (and (vector? thing)
	     (> (vector-length thing) 0)
	     (equal? (vector-ref thing 0) marker))
	(apply succ (cdr (vector->list thing)))
	(error "invalid record match" thing marker))))

; Auxiliaries for DEFINE-SUM

; ~MAKE-INJECTOR creates an injection procedure.

(define (~make-injector label)
  (let ((marker `(*tagged* ,label)))
    (lambda (thing)
      (vector marker thing))))

; ~MAKE-TAG-MATCHER creates a matcher for an injection procedure.

(define (~make-tag-matcher label)
  (lambda (tagged nargs succ fail)
    (cond ((not (= nargs 1)) (fail))
	  ((and (vector? tagged)
		(= (vector-length tagged) 2))
	   (let ((marker (vector-ref tagged 0)))
	     (if (and (pair? marker)
		      (eq? (car marker) '*tagged*))
		 (if (eq? (cadr marker) label)
		     (succ (vector-ref tagged 1))
		     (fail))
		 (error "invalid match - not a tagged object" tagged))))
	  (else
	   (error "invalid match - not a tagged object" tagged)))))

; Matchers for CONS, NULL, and LIST

(define (~match-cons thing nargs succ fail)
  (if (= nargs 2)
      (cond ((pair? thing) (succ (car thing) (cdr thing)))
	    ((null? thing) (fail))
	    (else (error "invalid match - not a list" thing)))
      (error "wrong number of arguments in match" 'cons)))

(define (~match-null thing nargs succ fail)
  (if (= nargs 0)
      (cond ((null? thing) (succ))
	    ((pair? thing) (fail))
	    (else (error "invalid match - not a list" thing)))
      (error "wrong number of arguments in match" 'null)))

(define (~match-list thing nargs succ fail)
  (if (or (pair? thing) (null? thing))
      (if (= nargs (length thing))
	  (apply succ thing)
	  (fail))
      (error "invalid match - not a list" thing)))

; S-expressions

; We simulate
;   (define-sum sexpr
;     (boolean integer string symbol list quoted))
; by defining appropriate injection and match procedures.

(define sexpr
  (vector '*sum*
	  'sexpr
	  '(boolean integer string symbol list quoted)))

(define (boolean->sexpr x)
  (if (boolean? x) x (error "invalid argument to boolean->sexpr" x)))
(define (integer->sexpr x)
  (if (integer? x) x (error "invalid argument to integer->sexpr" x)))
(define (string->sexpr x)
  (if (string? x) x (error "invalid argument to string->sexpr" x)))
(define (symbol->sexpr x)
  (if (symbol? x) x (error "invalid argument to symbol->sexpr" x)))

(define (list->sexpr x) 
  (if (or (null? x) (pair? x))
      x
      (error "invalid argument to list->sexpr" x)))

(define (quoted->sexpr x) (list 'quote x))

; The following are second-class summands, for now, because they won't
; work with MATCH.  Fix later.

(define (unquoted->sexpr          x) (list 'unquote          x))
(define (unquoted-splicing->sexpr x) (list 'unquote-splicing x))
(define (quasiquoted->sexpr	  x) (list 'quasiquote	     x))

; Matchers for S-expression injections

(define (~make-sexpr-matcher pred whoami)
  (lambda (thing nargs succ fail)
    (if (= nargs 1)
	(if (pred thing)
	    (succ thing)
	    (fail))
	(error "wrong number of arguments in match" whoami))))

(define ~match-boolean->sexpr
  (~make-sexpr-matcher boolean? 'boolean->sexpr))

(define ~match-integer->sexpr
  (~make-sexpr-matcher integer? 'integer->sexpr))

(define ~match-symbol->sexpr
  (~make-sexpr-matcher symbol? 'symbol->sexpr))

(define ~match-string->sexpr
  (~make-sexpr-matcher string? 'string->sexpr))

(define ~match-list->sexpr
  (~make-sexpr-matcher (lambda (thing)
			 (or (null? thing)
			     (and (pair? thing)
				  (not (~quoted? thing)))))
		       'list->sexpr))

(define (~match-quoted->sexpr thing nargs succ fail)
  (if (= nargs 1)
      (if (~quoted? thing)
	  (succ (cadr thing))
	  (fail))
      (error "wrong number of arguments in match" whoami)))

(define (~quoted? thing)
  (and (pair? thing)
       (eq? (car thing) 'quote)
       (pair? (cdr thing))
       (null? (cddr thing))))
