;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus"))



;;;; List hashing (for tuple consing and canonical caches)

;;; This implements a data structure which maps LISTS into
;;; HASH-LIST-ENTRYs; if the elments of two lists are EQ?, the lists
;;; map into the same hash-list-entry.  A hash-list-entry for a list `A'
;;; is a `weak-cons' whose CAR is available to store some arbitary
;;; object associated with `A' and whose CDR can be used for finding
;;; hash list entries for lists having `A' as an initial prefix.  In
;;; finding a hash-list-entry for such an extension of `A', we look up
;;; the first element of the extension in the a-list stored in the CDR
;;; of the hash-list-entry.  The result of this lookup is a
;;; hash-list-entry which may be the hash list entry for the list (if
;;; the list was one longer than `A') or is at least the hash list
;;; entry for another initial prefix of the list we want, on which we
;;; recur. 

(define (hash-list-entry list table)
  (if (null? list)
      ;; If the list to lookup is (), we have the right hash-list-entry:
      table
      ;; Otherwise, extract the car of the list and try to find the
      ;; corresponding HASH-LIST-ENTRY:
      (let* ((key (car list)) (entry (weak-assq key (weak-cdr table))))
	(if (null? entry)		; If there is no such entry, create one
	    (begin (set! entry (cons key (weak-cons () ())))
		   (set-weak-cdr! table (cons entry (weak-cdr table)))))
	;; And recur on this entry.
	(hash-list-entry (cdr list) (cdr entry)))))


;;;; Tuple consing.

;;; Tuples are structures which are EQ? if their elements are all EQ?.
;;; A tuple has a list of elements and a type-cache (for use by
;;; TYPICALs indexer, in the future).

(define-structure tuple (type-cache elements)
  ;; Tuples print out surrounded by `<#'/`#>' pairs and have their
  ;; elements separated by colons.
  (let ((list (tuple-elements tuple)))
    (if (null? list) (display "<##>")
	(sequence (display "<#") (printout-printer (first list))
		  (for-each (lambda (x) (display ":") (printout-printer x))
			    (rest list))
		  (display "#>")))))

;;; This is the hash-list-entry table for mapping lists into tuples.
(define tuple-table (list ()))

;;; This gets or constructs a tuple, ensuring EQ?ness.
(define (make-tuple elements)
  (let ((ref (hash-list-entry elements tuple-table)))
    (let ((entry (weak-car ref)))
      (if (not (null? entry)) entry
	  (let ((tuple (cons-tuple (make-empty-type-cache 200)
				   (apply list elements))))
	    (set-weak-car! ref tuple)
	    tuple)))))

;;; Other ways to make tuples:
(define (tuple . elements) (make-tuple elements))
(define (map-tuple fcn t) (make-tuple (map fcn (tuple-elements t))))
;;; Referencing elements of tuples is easy, but note that tuples are
;;; 1-based rather than 0-based.
(definline (tuple-ref tuple n) (list-ref (tuple-elements tuple) (-1+ n)))

(define the-empty-tuple (make-tuple ()))
(definline (empty-tuple? x) (eq? x the-empty-tuple))
(definline (non-empty-tuple? x) (not (eq? x the-empty-tuple)))


;;;; Canonical caches.

;;; Canonical caches provide caches for procedures with multiple
;;; arguments by standardizing the list of arguments into EQ? form and
;;; then using a simple cache on these arguments.

;;; A standardizer is a function which returns a copy of a list so
;;; that for any two lists with EQ? elements, the same copy is always
;;; returned. 
(define (make-standardizer)
  (let ((table (list ())))
    (define (standard-form from-x)
      (let* ((ref (hash-list-entry from-x table))
	     (entry (weak-car ref)))
	(if (null? entry)
	    (begin (set-weak-car! ref from-x) from-x)
	    entry)))
    standard-form))

(define (make-canonical-cache procedure)
  (let ((input-list ()))
    ;; We keep a list of inputs to keep our inputs (which are keys)
    ;; from going away.  Since both the standardizer and the cache's
    ;; mutable use weak conses, we might be tossing away perfectly
    ;; good cached values.
    (define (simplified-procedure x)
      (set! input-list (cons x input-list))
      (apply procedure x))
    (let ((inner-cache (simple-cache simplified-procedure))
	  (standardize (make-standardizer)))
      (let ((actual-cache (memoizer-cache inner-cache)))
	(define (cached-version . args) (inner-cache (standardize args)))
	(define (given-cache . x) (actual-cache (standardize x)))
	((modifier memoizer-cache) cached-version given-cache)
	((modifier memoizer-function) cached-version procedure)
	(copy-properties actual-cache given-cache (list mutator inverse))
	cached-version))))

(define canonical-cache (simple-cache make-canonical-cache))


