;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

(declare (usual-integrations)
	 (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/utility/tuple")
	 (integrate-external "/u/kwh/programs/typical/kernel")
	 (integrate-external "/u/kwh/programs/cyrano/rel"))


;;;; Defining procedural relations.

(define (generated-set underneath generator from-space)
  (let* ((type (generated-collection underneath))
	 (generator (collection-generator type generator)))
    (add-generator! type (make-combiner type generator from-space))
    type))
    
(define application-implementation
  (function! (td-property 'application-implementation)
	     relations procedures))
(set-property-name+format! application-implementation "Lisp Implementation" $function)

(define implemented-operations
  (type "Implemented Operations" simple-type
	(lambda (x) (and (type-description? x)
			 (defined? (application-implementation x))))
	relations))
(define lisp-application-types
  (type "Lisp Application Types" generated-collection implemented-operations))
(add-type-property! application-implementation implemented-operations)

(define (make-raw-lisp-application-type from-procedure)
  (declarable-type
   (tuple-product (function-range from-procedure) (function-domain from-procedure))))

(define construct-lisp-application-type
  (collection-generator lisp-application-types make-raw-lisp-application-type))

(define (make-lisp-application-type procedure)
  (let ((domain (function-domain procedure)) (range (function-range procedure)))
    (let* ((appl-type (type (list "#'" procedure)
			    construct-lisp-application-type procedure))
	   (declare-instance! (satisfaction-modifier appl-type)))
      (define (generate-application input)
	(let ((instance (tuple (procedure input) input)))
	  (declare-instance! instance #T)
	  instance))
      (let ((combiner (make-combiner appl-type generate-application domain)))
	((modifier application-implementation) appl-type procedure)
	(procedure! combiner ($procedure procedure)
		    "Use the implementation " ($procedure procedure)
		    " on " domain)
	(add-generator! range (make-combiner range left appl-type))
	(add-generator! appl-type combiner)
	appl-type))))
(define lisp-application-type (simple-cache make-lisp-application-type))

(add-daemon! (inhibitor collect-related-sources) (<AND> instance-sources implemented-operations))
(add-daemon! (inhibitor try-related-types) (<AND> instance-sources implemented-operations))


;;;; Example Generation advice.

;;; Don't generate random tuples for implemented operations.
(add-daemon! (inhibitor (use-generator random-tuple-generator))
	     (<AND> implemented-operations constrained-tuples instance-sources))
(procedure! (inhibitor (use-generator random-tuple-generator))
	    'INHIBIT-RANDOM-TUPLE-GENERATION
	    "Don't bother generating random tuples --- you can use the definition.")

;;; Don't do anything fancy with application pairs
(add-daemon! (inhibitor (use-generator right-triangle-generator))
	     (<AND> instance-sources (subtypes-of right-equals)
		    (image-constraint right-constraints implemented-operations)
		    (image-constraint left-constraints implemented-operations)))
(add-daemon! (inhibitor (use-generator br-corner-generator))
	     (<AND> instance-sources (subtypes-of right-equals)
		    (image-constraint right-constraints implemented-operations)
		    (image-constraint left-constraints implemented-operations)))
(add-daemon! (inhibitor (use-generator tr-corner-generator))
	     (<AND> instance-sources (subtypes-of right-equals)
		    (image-constraint right-constraints implemented-operations)
		    (image-constraint left-constraints implemented-operations)))

(define (operation-pair-generator type)
  (let ((top-operation (left-constraints type))
	(bottom-operation (right-constraints type)))
    (let ((top-implementation (application-implementation top-operation))
	  (bottom-implementation (application-implementation bottom-operation))
	  (top-declare! (satisfaction-modifier top-operation))
	  (bottom-declare! (satisfaction-modifier bottom-operation)))
      (define (generate-calls input)
	(let ((top-pair (tuple (top-implementation input) input))
	      (bottom-pair (tuple (bottom-implementation input) input)))
	  (if (defined? top-declare!)    (top-declare! top-pair #T))
	  (if (defined? bottom-declare!) (bottom-declare! bottom-pair #T))
	  (tuple top-pair bottom-pair)))
      (make-combiner type generate-calls
		     (<AND> (right-constraints top-operation)
			    (right-constraints bottom-operation))))))

;;; Generate application pairs directly.
(add-daemon! (use-generator operation-pair-generator)
	     (<AND> instance-sources (subtypes-of right-equals)
		    (image-constraint right-constraints implemented-operations)
		    (image-constraint left-constraints implemented-operations)))


;;;; Combining operations.

;;; A combining operation takes two objects of the same type and
;;; combines them into a second object of some other type.
(define combining-operations
  (type "Combining Operations" <AND>
	implemented-operations relations
	(image-constraint right-constraints pairings)))

(define (make-mirror-class operation)
  (let ((fcn (application-implementation operation))
	(domain (right-constraints operation)))
    (define (mirror-function input) (fcn (tuple (right input) (left input))))
    (let ((left-in (left-constraints domain)) (right-in (right-constraints domain)))
      (declare-function! mirror-function
			 (tuple-product right-in left-in)
			 (left-constraints operation)
			 ($procedure fcn) "-MIRROR")
      (type (list "Mirror Class for " operation) tuple-product
	    operation (lisp-application-type mirror-function)))))

;;; If a combining operation is deterministic regardless of argument
;;; order, it is abelian.
(define abelian-operations
  (type "Abelian Operations" empirical-class combining-operations
	make-mirror-class (lambda (ignore) left-equals)))

(define (make-coalesced-operation operation)
  (let ((implementation (application-implementation operation))
	(domain (right-constraints operation))
	(range (left-constraints operation)))
    (define (coalesced-operation x) (implementation (tuple x x)))
    (declare-function! coalesced-operation
		       (<AND> (right-constraints domain) (left-constraints domain))
		       range
		       "COALESCED-" operation)
    (type (list "Coalesced " operation) lisp-application-type coalesced-operation)))

(define coalesce-operation
  (procedure! (simple-cache make-coalesced-operation) 'COALESCE-OPERATION
	      "Convert an operation on pairs to an operation on singletons."))


;;;; Binary operations

(define binary-operations
  (type "Binary Operations" <AND>
	(subtypes-of (tuple-product lisp-objects pairs))
	implemented-operations))

(define (right-generator? operation)
  (and (type-description? operation)
       (in? operation binary-operations) 
       (let* ((lt (left-constraints operation))
	      (rt (right-constraints operation))
	      (rrt (and rt (right-constraints rt))))
	 (and lt rrt (<<? lt rrt)))))
(define right-generators
  (type "Right Generators" simple-type right-generator? binary-operations))
(define (left-generator? operation)
  (and (type-description? operation)
       (in? operation binary-operations) 
       (let* ((lt (left-constraints operation))
	      (rt (right-constraints operation))
	      (lrt (and rt (left-constraints rt))))
	 (and lt lrt (<<? lt lrt)))))
(define left-generators
  (type "Left Generators" simple-type left-generator? binary-operations))
	
(define closed-operations
  (type "Closed Operation" <AND> right-generators left-generators))


;;;; Testing for Associativity

(define (make-association-comparison operation)
  (let ((fcn (application-implementation operation))
	(domain (right-constraints operation))
	(range (left-constraints operation)))
    (define (right-association input)
      (fcn (tuple (tuple-ref input 1)
		  (fcn (tuple (tuple-ref input 2)
			      (tuple-ref input 3))))))
    (define (left-association input)
      (fcn (tuple (fcn (tuple (tuple-ref input 1) (tuple-ref input 2)))
		  (tuple-ref input 3))))
    (let ((left-in (left-constraints domain)) (right-in (right-constraints domain)))
      (tuple-product (lisp-application-type
		      (function! left-association
				 (tuple-product left-in right-in right-in)
				 range))
		     (lisp-application-type
		      (function! right-association
				 (tuple-product left-in left-in right-in)
				 range))))))

;;; A closed combiner c is associative if c(x,c(y,z))=c(c(x,y),z)
;;;  We define cr(x,y,z)=c(x,c(y,z)) and cl(x,y,z)=c(c(x,y),z) so that
;;;  c is associative 
(define associative-operations
  (type "Associative Operations" empirical-class closed-operations
	(lambda (op) (<AND> (make-association-comparison op) right-equals))
	(lambda (ignore) left-equals)))


;;;; Operations -> Relations

(define (convert-operation-into-relation op)
  (let* ((fcn (application-implementation op))
	 (domain (right-constraints op))
	 (predicate (td-predicate domain)))
    (define (eq-by-operation? pair)
      (and (doubleton? pair) (in? (right pair) domain)
	   (eq? (left pair) (fcn (right pair)))))
    (let ((rel (type (list "=" op "=") simple-type eq-by-operation? op)))
      (if (defined? (satisfies? op left-deterministic))
	  (given! rel (if (satisfies? op left-deterministic) left-deterministic
			  (complement left-deterministic))))
      (if (defined? (satisfies? op right-deterministic))
	  (given! rel (if (satisfies? op right-deterministic) right-deterministic
			  (complement right-deterministic))))
      ((modifier application-implementation) rel fcn)
      rel)))
(define operation->relation
  (procedure! (simple-cache convert-operation-into-relation) 'OPERATION->RELATION
	      "Converts deterministic operations into relations."))


;;;; Using operations for distance measures.

(define (use-operation-as-distance op)
  (let ((fcn (application-implementation op))
	(domain (right-constraints op))
	(range  (left-constraints op)))
    (define (pair-fcn x) (tuple (fcn (left x)) (fcn (right x))))
    (declare-function! pair-fcn
		      (tuple-product domain domain)
		      (tuple-product range range))
    (let ((hole (<AND> range (complement domain))))
      (type (list "Same " op " distance")
	    inductive-definition
	    (tuple-product hole hole)
	    (tuple-product domain domain)
	    (list pair-fcn)))))

(define operation->distance
  (procedure! (simple-cache use-operation-as-distance) 'OPERATION->DISTANCE
	      "Produce a relation which measures `same applicability distance' for an operation"))


;;;; Iterative compositions

(define iterated-compositions
  (type "Iterative Compositions" generated-collection lisp-application-types))

(define (generate-iterative-composers from-reducer)
  (function!
   (lambda (composer) (iterating-composer from-reducer composer))
   (<AND> implemented-operations right-deterministic
	  (image-constraint tuple-type->type-tuple <<?-rel)
	  (complement (subtypes-of (tuple-product lisp-objects tuples))))
   binary-operations
   "ITERATE-COMPOSERS-USING-" from-reducer))

(define (make-iterating-composer reducer composer)
  (let ((reduce (application-implementation reducer))
	(reduce-while (right-constraints reducer))
	(compose (application-implementation composer))
	(compose-while (right-constraints composer)))
    (define (iterate x y)
      (if (in? x reduce-while)
	  (if (in? y compose-while)
	      (iterate (reduce x) (compose y))
	      (%undefined))
	  y))
    (define (iterate-tuple x.y) (iterate (left x.y) (right x.y)))
    (define (check-instance x)
      (and (tuple? x) (tuple? (right x)) (eq? (left x) (iterate-tuple (right x)))))
    (let* ((domain (tuple-product reduce-while compose-while))
	   (range  (left-constraints composer))
	   (result (type (list "ITERATE-" composer "-WHILE-" reducer)
			simple-type check-instance (tuple-product range domain))))
      ((modifier application-implementation) result iterate-tuple)
      (if (and (in? reducer right-deterministic) (in? composer right-deterministic))
	  (given! result right-deterministic))
      result)))
(define iterating-composer
  (collection-generator iterated-compositions
			(canonical-cache make-iterating-composer)))


;;;; Iterative combination

(define iterated-combinations
  (type "Iterative Combinations" generated-collection lisp-application-types))

(define (generate-iterative-combiners from-reducer)
  (function!
   (lambda (combiner) (iterating-combiner from-reducer combiner))
   (<AND> right-generators right-deterministic)
   binary-operations))

(define (make-iterating-combiner reducer combiner)
  (let ((reduce (application-implementation reducer))
	(reduce-while (right-constraints reducer))
	(combine (application-implementation combiner))
	(combine-while (left-constraints (right-constraints combiner))))
    (define (iterate x y c)
      (if (in? x reduce-while)
	  (if (in? y combine-while)
	      (iterate (reduce x) (combine (tuple y c)) c)
	      (%undefined))
	  y))
    (define (iterate-tuple x.y)
      (iterate (left x.y) (right x.y) (right x.y)))
    (define (check-instance x)
      (and (tuple? x) (tuple? (right x)) (eq? (left x) (iterate-tuple (right x)))))
    (let* ((domain (tuple-product reduce-while combine-while))
	   (range  (left-constraints combiner))
	   (result (type (list "ITERATE-" combiner "-WHILE-" reducer)
			simple-type check-instance (tuple-product range domain))))
      ((modifier application-implementation) result iterate-tuple)
      (if (and (in? reducer right-deterministic) (in? combiner right-deterministic))
	  (given! result right-deterministic))
      result)))
(define iterating-combiner
  (collection-generator iterated-combinations (canonical-cache make-iterating-combiner)))

