#| -*-Scheme-*-

$Header: btree.scm,v 1.1 87/10/16 17:21:02 GMT cph Exp $

Copyright (c) 1987 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. |#

;;;; Balanced Trees
;;;  Knuth, Donald E., "The Art of Computer Programming",
;;;  volume 3, "Sorting and Searching",
;;;  section 6.2.3, "Balanced Trees".

(declare (usual-integrations))

(define make-btree)
(let ()

(declare (integrate-operator make-node wrapped-key set-wrapped-key!
			     get-link set-link! left-link right-link
			     balance-factor set-balance-factor!
			     balanced tipped-left tipped-right
			     +a -a get-link+ get-link- set-link+! set-link-!
			     cons-path path-components))

(define (make-node wrapped-key left-link right-link balance-factor)
  (declare (integrate wrapped-key left-link right-link balance-factor))
  (list->vector (list wrapped-key left-link right-link balance-factor)))

(define (wrapped-key node)
  (declare (integrate node))
  (vector-ref node 0))

(define (set-wrapped-key! node key)
  (declare (integrate node key))
  (vector-set! node 0 key))

(define get-link vector-ref)
(define set-link! vector-set!)
(define left-link 1)
(define right-link 2)

(define (balance-factor node)
  (declare (integrate node))
  (vector-ref node 3))

(define (set-balance-factor! node b)
  (declare (integrate node b))
  (vector-set! node 3 b))

(define balanced 'BALANCED)                            ; Knuth's 0
(define tipped-left 'TIPPED-LEFT)                      ; Knuth's -1
(define tipped-right 'TIPPED-RIGHT)                    ; Knuth's +1

(define left-d
  (list->vector (list tipped-left tipped-right left-link right-link)))

(define right-d
  (list->vector (list tipped-right tipped-left right-link left-link)))

(define (-d d)
  (cond ((eq? d left-d) right-d)
	((eq? d right-d) left-d)
	(else (error "Bad argument" '-D d))))

(define (+a d)
  (declare (integrate d))
  (vector-ref d 0))

(define (-a d)
  (declare (integrate d))
  (vector-ref d 1))

(define (get-link+ p d)
  (declare (integrate p d))
  (get-link p (vector-ref d 2)))

(define (get-link- p d)
  (declare (integrate p d))
  (get-link p (vector-ref d 3)))

(define (set-link+! p d l)
  (declare (integrate p d l))
  (set-link! p (vector-ref d 2) l))

(define (set-link-! p d l)
  (declare (integrate p d l))
  (set-link! p (vector-ref d 3) l))

(define (cons-path p d path)
  (declare (integrate p d path))
  (list p d path))

(define (path-components path receiver)
  (declare (integrate path receiver))
  (apply receiver path))

;;;; Rebalancing

(define (rebalance! A d if-shorter if-same-height)
  (let ((B (get-link+ A d)))
    (define (case-1)
      (set-link+! A d (get-link- B d))
      (set-balance-factor! A balanced)
      (set-link-! B d A)
      (set-balance-factor! B balanced)
      (if-shorter B))

    (define (case-2 X)
      (set-link-! B d (get-link+ X d))
      (set-link+! X d B)
      (set-link+! A d (get-link- X d))
      (set-link-! X d A)
      (cond ((eq? (balance-factor X) balanced)
	     (set-balance-factor! A balanced)
	     (set-balance-factor! B balanced))
	    ((eq? (balance-factor X) (+a d))
	     (set-balance-factor! A (-a d))
	     (set-balance-factor! B balanced))
	    (else
	     (set-balance-factor! A balanced)
	     (set-balance-factor! B (+a d))))
      (set-balance-factor! X balanced)
      (if-shorter X))

    (define (case-3)
      (set-link+! A d (get-link- B d))
      (set-balance-factor! A (+a d))
      (set-link-! B d A)
      (set-balance-factor! B (-a d))
      (if-same-height B))

    (cond ((eq? (balance-factor B) (+a d))
	   (case-1))
	  ((eq? (balance-factor B) (-a d))
	   (case-2 (get-link- B d)))
	  (else
	   (case-3)))))

;;;; Auxiliary Procedures for :DELETE!

(define (remove-successor! p path)
  (if (null? (get-link p left-link))
      (begin (replace-node! path (get-link p right-link))
	     p)
      (remove-successor! (get-link p left-link)
			 (cons-path p left-d path))))

(define (replace-node! path new-node)
  (path-components path
    (lambda (pl-1 dl-1 rest)
      (set-link+! pl-1 dl-1 new-node)
      (adjust-balance-factors! pl-1 dl-1 rest))))

(define (adjust-balance-factors! pk dk path)
  (cond ((null? path) 'DONE)
	((eq? (balance-factor pk) balanced)
	 (set-balance-factor! pk (-a dk)))
	(else
	 (path-components path
	   (lambda (pk-1 dk-1 rest)
	     (if (eq? (balance-factor pk) (+a dk))
		 (begin (set-balance-factor! pk balanced)
			(adjust-balance-factors! pk-1 dk-1 rest))
		 (rebalance! pk (-d dk)
			     (lambda (new-pk)
			       (set-link+! pk-1 dk-1 new-pk)
			       (adjust-balance-factors! pk-1 dk-1 rest))
			     (lambda (new-pk)
			       (set-link+! pk-1 dk-1 new-pk)))))))))

(set! make-btree
  (named-lambda (make-btree < wrap-key unwrap-key)

(declare (integrate-operator key))
(define (key p)
  (declare (integrate p))
  (unwrap-key (wrapped-key p)))

(define head
  (make-node 'DUMMY-KEY '() '() balanced))

(define (:insert! k if-found if-not-found)
  (define (search t s-link s p)
    (define (move-once link q)
      (cond ((null? q)
	     (let ((wrapped-key (wrap-key k)))
	       (let ((q (make-node wrapped-key '() '() balanced)))
		 (define (adjust-balance-factors! p)
		   (cond ((eq? p q) 'DONE)
			 ((< k (key p))
			  (set-balance-factor! p tipped-left)
			  (adjust-balance-factors! (get-link p left-link)))
			 (else
			  (set-balance-factor! p tipped-right)
			  (adjust-balance-factors! (get-link p right-link)))))

		 (set-link! p link q)
		 (let ((d (if (< k (key s)) left-d right-d)))
		   (adjust-balance-factors! (get-link+ s d))
		   (balancing-act! d)))
	       (if-not-found wrapped-key)))
	    ((eq? (balance-factor q) balanced)
	     (search t s-link s q))
	    (else
	     (search p link q q))))

    (define (balancing-act! d)
      (cond ((eq? (balance-factor s) balanced)
	     (set-balance-factor! s (+a d)))
	    ((eq? (balance-factor s) (-a d))
	     (set-balance-factor! s balanced))
	    (else
	     (rebalance! s d
			 (lambda (new-s)
			   (set-link! t s-link new-s))
			 (lambda (new-s)
			   (error "Tree shouldn't be same height!"
				  'BALANCING-ACT!))))))

    (cond ((< k (key p))
	   (move-once left-link (get-link p left-link)))
	  ((< (key p) k)
	   (move-once right-link (get-link p right-link)))
	  (else
	   (if-found (wrapped-key p)))))

  (let ((p (get-link head right-link)))
    (if (null? p)
	(let ((wrapped-key (wrap-key k)))
	  (set-link! head right-link (make-node wrapped-key '() '() balanced))
	  wrapped-key)
	(search head right-link p p))))

(define (:delete! k if-found if-not-found)
  (let loop ((p (get-link head right-link))
	     (path (cons-path head right-d '())))
    (cond ((null? p)
	   (if-not-found k))
	  ((< k (key p))
	   (loop (get-link p left-link)
		 (cons-path p left-d path)))
	  ((< (key p) k)
	   (loop (get-link p right-link)
		 (cons-path p right-d path)))
	  (else
	   (let ((result (wrapped-key p)))
	     (cond ((null? (get-link p left-link))
		    (replace-node! path (get-link p right-link)))
		   ((null? (get-link p right-link))
		    (replace-node! path (get-link p left-link)))
		   (else
		    (set-wrapped-key!
		     p
		     (wrapped-key
		      (remove-successor! (get-link p right-link)
					 (cons-path p right-d path))))))
	     (if-found result))))))

(define (:lookup k if-found if-not-found)
  (define (loop p)
    (cond ((null? p)
	   (if-not-found k))
	  ((< k (key p))
	   (loop (get-link p left-link)))
	  ((< (key p) k)
	   (loop (get-link p right-link)))
	  (else
	   (if-found (wrapped-key p)))))
  (loop (get-link head right-link)))

(define (:fringe)
  (define (loop p tail)
    (if (null? p)
	tail
	(loop (get-link p left-link)
	      (cons (wrapped-key p)
		    (loop (get-link p right-link)
			  tail)))))
  (loop (get-link head right-link)
	'()))

;; end MAKE-BTREE.
(vector :insert! :delete! :lookup :fringe)))

)

(define (btree-insert! btree key if-found if-not-found)
  ((vector-ref btree 0) key if-found if-not-found))

(define (btree-delete! btree key if-found if-not-found)
  ((vector-ref btree 1) key if-found if-not-found))

(define (btree-lookup btree key if-found if-not-found)
  ((vector-ref btree 2) key if-found if-not-found))

(define (btree-fringe btree)
  ((vector-ref btree 3)))