#| -*-Scheme-*-

Written January 11, 1993 by Max Hailperin <max@nic.gac.edu>
to provide a purely functional style of graphics.  The code
here presupposes the PostScript graphics provided by Schematik.
Updated February 19, 1993 to use PostScript features far more
sparingly, to ease porting and to avoid problems with gsave
being nested too deeply in complex images.

Copyright (c) 1993 Gustavus Adolphus College.  All rights reserved.

This software was developed by Gustavus Adolphus College (GAC).
Permission to copy this software, to redistribute it, and to use it
for any purpose is granted, subject to the following restrictions and
understandings.

1. Any copy made of this software must include this copyright notice in full.

2. Users of this software agree to make their best efforts (a) to
return to the GAC Mathematics and Computer Science Department any
improvements or extensions that they make, so that these may be
included in future releases; and (b) to inform GAC of noteworthy uses
of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. GAC makes no express or implied warranty or representation of any
kind with respect to this software, including any warranty that the
operation of this software will be error-free.  ANY IMPLIED WARRANTY
OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE IS HEREBY
DISCLAIMED.  GAC is under no obligation to provide any services, by
way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of Gustavus Adolphus College nor of
any adaptation thereof in any advertising, promotional, or sales
literature without prior written consent from GAC in each case. |#

(declare (usual-integrations))

(define fg/repl-write
  (let ((old/repl-write
         (access hook/repl-write (->environment '(runtime rep)))))
    (named-lambda (fg/repl-write repl object)
      (if (and (image? object)
               (graphics-type-available? schematik-style-graphics-device-type)
               ((access cmdl/io-to-console?
                        (->environment '(runtime emacs-interface)))
                repl))
          (let ((win (make-graphics-device schematik-style-graphics-device-type
                                           'points 
                                           (image-width object)
                                           (image-height object))))
            (repl-history/record! (repl/printer-history repl) object)
            (dynamic-wind (lambda () #f)
                          (lambda () ((image-proc object) win
                                                          (lambda (x y)
                                                            (values x y))))
                          (lambda () (graphics-close win))))
          (old/repl-write repl object)))))

(define-structure (image
                   (print-procedure
                    (unparser/standard-method 'image
                                              (lambda (state obj)
                                                (unparse-object state
                                                                (image-width
                                                                 obj))
                                                (unparse-char state #\*)
                                                (unparse-object state
                                                                (image-height
                                                                 obj))))))
  proc width height)

(define (install-fg/repl-write)
  (set! (access hook/repl-write (->environment '(runtime rep)))
        fg/repl-write))

(install-fg/repl-write)

(add-event-receiver! event:after-restart install-fg/repl-write)

(define default-image-size 288)

(define (line x0 y0 x1 y1 #!optional width height)
  (if (not (real? x0))
      (error:illegal-datum x0 'line))
  (if (not (<= -1 x0 1))
      (error:datum-out-of-range x0 'line))
  (if (not (real? x1))
      (error:illegal-datum x1 'line))
  (if (not (<= -1 x1 1))
      (error:datum-out-of-range x1 'line))
  (if (not (real? y0))
      (error:illegal-datum y0 'line))
  (if (not (<= -1 y0 1))
      (error:datum-out-of-range y0 'line))
  (if (not (real? y1))
      (error:illegal-datum y1 'line))
  (if (not (<= -1 y1 1))
      (error:datum-out-of-range y1 'line))
  (cond ((default-object? width)
         (set! width default-image-size))
        ((not (exact-integer? width))
         (error:illegal-datum width 'line)))
  (cond ((default-object? height)
         (set! height width))
        ((not (exact-integer? height))
         (error:illegal-datum height 'line)))
  (make-image (lambda (win transform)
                (with-values (lambda () (transform x0 y0))
                  (lambda (x0 y0)
                    (with-values (lambda () (transform x1 y1))
                      (lambda (x1 y1)
                        (graphics-draw-line win x0 y0 x1 y1))))))
              width height))

(define (ps-image ps #!optional width height)
  (if (not (string? ps))
      (error:illegal-datum ps 'ps-image))
  (cond ((default-object? width)
         (set! width default-image-size))
        ((not (exact-integer? width))
         (error:illegal-datum width 'ps-image)))
  (cond ((default-object? height)
         (set! height width))
        ((not (exact-integer? height))
         (error:illegal-datum height 'ps-image)))
  (make-image (lambda (win transform)
                (with-values (lambda () (transform 0 0))
                  (lambda (tx ty)
                    (with-values (lambda () (transform 1 0))
                      (lambda (x->x x->y)
                        (let ((x->x (- x->x tx))
                              (x->y (- x->y ty)))
                          (with-values (lambda () (transform 0 1))
                            (lambda (y->x y->y)
                              (let ((y->x (- y->x tx))
                                    (y->y (- y->y ty)))
                                (graphics-operation
                                 win 'draw-postscript
                                 (string-append "save ["
                                                (number->string x->x) " "
                                                (number->string x->y) " "
                                                (number->string y->x) " "
                                                (number->string y->y) " "
                                                (number->string tx) " "
                                                (number->string ty)
                                                "] concat " ps 
                                                " restore")))))))))))            
              width height))

(define (quarter-turn-right image)
  (if (not (image? image))
      (error:illegal-datum image 'quarter-turn-right))
  (make-image (lambda (win transform)
                ((image-proc image) win
                                    (lambda (x y)
                                      (transform y (- x)))))
              (image-height image)
              (image-width image)))

(define (mirror-image image)
  (if (not (image? image))
      (error:illegal-datum image 'mirror-image))
  (make-image (lambda (win transform)
                ((image-proc image) win
                                    (lambda (x y)
                                      (transform (- x) y))))
              (image-width image)
              (image-height image)))

(define (invert image)
  (if (not (image? image))
      (error:illegal-datum image 'invert))
  (make-image (lambda (win transform)
                (with-values (lambda () (transform -1 -1))
                  (lambda (x0 y0)
                    (let ((x0 (string-append (number->string x0) " "))
                          (y0 (string-append (number->string y0) " ")))
                      (with-values (lambda () (transform 1 1))
                        (lambda (x1 y1)
                          (let ((x1 (string-append (number->string x1) " "))
                                (y1 (string-append (number->string y1) " ")))
                              (graphics-operation
                               win 'draw-postscript
                               (string-append "newpath " x0 y0 "moveto "
                                              x0 y1 "lineto "
                                              x1 y1 "lineto "
                                              x1 y0 "lineto closepath fill gsave 1 currentgray sub setgray"))))))))
                ((image-proc image) win transform)
                (graphics-operation win 'draw-postscript "grestore"))
              (image-width image)
              (image-height image)))

(define (overlay image . images)
  (if (not (image? image))
      (error:illegal-datum image 'overlay))
  (let ((w (image-width image))
        (h (image-height image)))
    (for-each
     (lambda (i)
       (if (not (image? i))
           (error:illegal-datum i 'overlay))
       (if (not (and (= (image-width i) w)
                     (= (image-height i) h)))
           (error "Only images of equal size can be overlayed"
                  (cons image images))))
     images)
    (make-image
     (lambda (win transform)
       (for-each
        (lambda (image) ((image-proc image) win transform))
        (cons image images)))
     w h)))

(define (resize-image image #!optional width height)
  (if (not (image? image))
      (error:illegal-datum image 'resize-image))
  (cond ((default-object? width)
         (set! width default-image-size))
        ((not (exact-integer? width))
         (error:illegal-datum width 'resize-image)))
  (cond ((default-object? height)
         (set! height width))
        ((not (exact-integer? height))
         (error:illegal-datum height 'resize-image)))
  (make-image (image-proc image) width height))

(define (stack top . rest)
  (define (stack2 top bottom)
    (if (not (image? top))
        (error:illegal-datum top 'stack))
    (if (not (image? bottom))
        (error:illegal-datum bottom 'stack))
    (if (not (= (image-width top) (image-width bottom)))
        (error "Attempt to stack images of different widths" (list top bottom))
        (let ((th (image-height top))
              (bh (image-height bottom)))
          (let* ((h (+ th bh))
                 (inexact-h (exact->inexact h)))
            (let ((tscale (/ th inexact-h))
                  (bscale (/ bh inexact-h)))
              (make-image
               (lambda (win transform)
                 ((image-proc top) win
                                   (lambda (x y)
                                     (transform x (+ (* tscale y) bscale))))
                 ((image-proc bottom) win
                                      (lambda (x y)
                                        (transform x (- (* bscale y) tscale)))))
               (image-width top)
               h))))))
  (let loop ((image top)
             (images rest))
    (if (null? images)
        image
        (loop (stack2 image (car images)) (cdr images)))))