#| -*-Scheme-*-

$Header$

Copyright (c) 1989 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. |#

;;;; Scheduler

(declare (usual-integrations)
	 (integrate-external "struct"))

(define (initialize-constraint-system!)
  (set! *contradiction-queue* (make-queue))
  (set! *detector-queue* (make-queue))
  (set! *vanilla-queue* (make-queue))
  (set! *nogood-queue* (make-queue))
  (set! *defer-queue* (make-queue))
  (set! *rebel-queue* (make-queue))
  (set! *inhibit-rebels?* false)
  (set! *punt-queue* (make-queue))
  unspecific)

(define *contradiction-queue*)
(define *detector-queue*)
(define *vanilla-queue*)
(define *nogood-queue*)
(define *defer-queue*)
(define *rebel-queue*)
(define *inhibit-rebels?*)
(define *punt-queue*)

(define (run!)
  (cond ((or (run-one-queue-item! *contradiction-queue*)
	     (run-one-queue-item! *detector-queue*)
	     (run-one-queue-item! *vanilla-queue*)
	     (run-one-queue-item! *nogood-queue*))
	 (run!))
	((not (queue-empty? *defer-queue*))
	 (queue-move! *contradiction-queue* *defer-queue*)
	 (run!))
	((and (not *inhibit-rebels?*)
	      (not (queue-empty? *rebel-queue*)))
	 (set! *inhibit-rebels?* true)
	 (queue-map! *rebel-queue* (lambda (queue-item) (queue-item)))
	 (run!))
	((and (not (queue-empty? *punt-queue*))
	      (y-or-n? "Process punted contradictions"))
	 (queue-move! *contradiction-queue* *punt-queue*)
	 (run!))))

(define (run-one-queue-item! queue)
  (and (not (queue-empty? queue))
       (begin
	 ((dequeue! queue))
	 true)))

(define (queue-move! target source)
  (queue-map! source (lambda (item) (enqueue! target item))))

(define (enqueue-contradiction/constraint! alist)
  (enqueue! *contradiction-queue*
	    (letrec ((queue-item
		      (lambda ()
			(run-contradiction/constraint! alist queue-item))))
	      queue-item)))

(define (enqueue-contradiction/node! cell cell*)
  (enqueue! *contradiction-queue*
	    (letrec ((queue-item
		      (lambda ()
			(run-contradiction/node! cell cell* queue-item))))
	      queue-item)))

(define (enqueue-contradiction/resolution! alist)
  (enqueue! *contradiction-queue*
	    (letrec ((queue-item
		      (lambda ()
			(run-contradiction/resolution! alist queue-item))))
	      queue-item)))

(define (enqueue-rule/add! rule constraint)
  (rule/enqueue!
   (let ((output (constraint/output constraint rule)))
     (cond ((not output)
	    *detector-queue*)
	   ((or (rule/nogood? rule)
		(and (rule/nogoodbeg? rule)
		     (not (node/bound? output))))
	    *nogood-queue*)
	   (else
	    *vanilla-queue*)))
   rule
   constraint))

(define (enqueue-rule/forget! rule constraint)
  (rule/enqueue! (if (rule/assumption? rule) *nogood-queue* *vanilla-queue*)
		 rule
		 constraint))

(define (enqueue-rule/nogood! rule constraint)
  (if (not (and (rule/nogoodbeg? rule)
		(node/bound? (constraint/output constraint rule))))
      (rule/enqueue! *nogood-queue* rule constraint)))

(define (rule/enqueue! queue rule constraint)
  (constraint/rule-queued! constraint rule)
  (enqueue! queue
	    (lambda ()
	      (constraint/rule-dequeued! constraint rule)
	      (rule/run! rule constraint))))

(define (rule/run! rule constraint)
  (let ((inputs (constraint/inputs constraint rule))
	(output (constraint/output constraint rule)))
    (if (for-all? inputs
	  (lambda (input)
	    (case (cell/state input)
	      ((KING FRIEND REBEL) (rule/assumption? (cell/rule input)))
	      ((SLAVE) (node/bound? input))
	      ((DUPE) true)
	      ((PUPPET) false)
	      (else (error "bad cell state" input)))))
	(let ((result
	       (apply (rule/procedure rule)
		      constraint output (map cell/value inputs)))
	      (input-pairs
	       (lambda (connective)
		 ($map ($nested " " connective " "
				($pluralize "input" (length inputs)) " ")
		       ", "
		       ""
		       (lambda (input)
			 ($nested ($literal (cell/normal-name input))
				  "="
				  ($literal (cell/value input))))
		       inputs))))
	  (cond ((eq? result rule-result:contradiction)
		 (signal-contradiction! constraint inputs))
		((eq? result rule-result:dismiss)
		 (ctrace ($literal (constraint/id constraint)) " dismissed "
			 ($literal (rule/id rule)) (input-pairs "given") "."))
		(else
		 (ctrace ($literal (constraint/id constraint)) " computed "
			 (cell/normal-name output) "=" ($literal result)
			  (input-pairs "from") " using "
			  ($literal (rule/id rule)) ".")
		 (if (not output)
		     (error "rule has no output pin but computed value"))
		 (process-setc! constraint output result rule)))))))

(define (process-setc! constraint cell contents rule)
  (let ((new-state!
	 (lambda (state)
	   (set-cell/state! cell state)
	   (set-cell/contents! cell contents)
	   (set-cell/rule! cell rule))))
    (let ((king!
	   (lambda ()
	     (new-state! 'KING)
	     (for-each (lambda (cell*)
			 (if (not (eq? cell* cell))
			     (awaken-cell/add! cell*)))
		       (node/cells cell)))))
      (case (cell/state cell)
	((PUPPET)
	 (king!))
	((KING FRIEND REBEL)
	 (cond ((value=? (cell/contents cell) contents)
		(if (subsetv? (rule/inputs rule)
			      (rule/inputs (cell/rule cell)))
		    (set-cell/rule! cell rule)))
	       ((rule/nogoodbeg? (cell/rule cell))
		(ctrace ($literal rule) " overrides value "
			($literal (cell/contents cell)) " of "
			($literal (cell/rule cell)) " with "
			($literal value))
		(forget! cell)
		(process-setc! constraint cell contents rule))
	       (else
		(error "disagreeing rules on value for cell" cell))))
	((SLAVE DUPE)
	 (cond ((not (node/bound? cell))
		(usurper! cell)
		(king!))
	       ((value=? contents (node/value cell))
		(new-state! 'FRIEND))
	       (else
		(new-state! 'REBEL)
		(increment-node/n-contradicting! cell)
		(if constraint
		    (ctrace-contradiction
		     constraint
		     (cons (cons cell (node/value cell))
			   (map (lambda (input)
				  (cons input (cell/value input)))
				(constraint/inputs constraint rule)))
		     ($nested ";" $nl-ctrace "it computed "
			      ($literal (cell/normal-name cell)) "="
			      ($literal (cell/contents cell))
			      " by " (rule/id rule) ".")))
		(enqueue-contradiction/node! cell (node/supplier cell)))))
	(else (error "bad cell state" cell))))))

(define (usurper! cell)
  (let ((supplier (node/supplier cell)))
    (point-links-toward! cell)
    (let ((sx (cell/state supplier)))
      (set-cell/state! supplier (cell/state cell))
      (set-cell/state! cell sx))))

(define (point-links-toward! cell)
  (let ((supplier (node/supplier cell)))
    (let loop ((x cell) (y false))
      (if (eq? x supplier)
	  (set-cell/link! x y)
	  (let ((link (cell/link x)))
	    (set-cell/link! x y)
	    (loop link x)))))
  (set-node/supplier! cell cell))

(define (signal-contradiction! constraint inputs)
  (ctrace-contradiction constraint
			(map (lambda (input)
			       (cons input (cell/value input)))
			     inputs)
			".")
  (enqueue-contradiction/constraint!
   (map (lambda (input) (cons input (cell/value input))) inputs)))

(define (ctrace-contradiction constraint pairs rest)
  (ctrace "Contradiction in " ($literal (constraint/id constraint))
	  ($map " among these cells: " ", " ""
		(lambda (pair)
		  ($nested ($literal (cell/normal-name (car pair)))
			   "="
			   ($literal (cdr pair))))
		pairs)
	  rest))

(define (*disallow! cells)
  (enqueue-contradiction/resolution!
   (map (lambda (premise) (cons premise (cell/value premise)))
	(with-values (lambda () (cells/premises cells))
	  (lambda (defaults parameters assumptions trees links)
	    trees links
	    (append defaults parameters assumptions))))))

(define (run-contradiction/node! cell cell* queue-item)
  (set! *inhibit-rebels?* false)
  (if (and (node=? cell cell*)
	   (eq? (cell/true-supplier cell) cell)
	   (eq? (cell/true-supplier cell*) cell*)
	   (not (and (node/bound? cell)
		     (node/bound? cell*)
		     (value=? (cell/contents cell) (cell/contents cell*)))))
      (process-contradiction! (list cell cell*) queue-item)))

(define (run-contradiction/constraint! alist queue-item)
  (set! *inhibit-rebels?* false)
  (if (cells-retain-values? alist)
      (process-contradiction! (map car alist) queue-item)))

(define (run-contradiction/resolution! alist queue-item)
  (set! *inhibit-rebels?* false)
  (if (cells-retain-values? alist)
      (process-contradiction! (map car alist) queue-item)
      (install-nogood-set!
       (map (lambda (association)
	      (cons (cell/repository (car association))
		    (cdr association)))
	    alist))))

(define (cells-retain-values? alist)
  (for-all? alist
    (lambda (association)
      (let ((cell (car association)))
	(and (node/bound? cell)
	     (value=? (cell/contents cell) (cdr association)))))))

(define (process-contradiction! cells queue-item)
  (with-values (lambda () (cells/premises cells))
    (lambda (defaults parameters assumptions trees links)
      links				;ignore
      (if (not (null? assumptions))
	  (begin
	    (let ((cell (car assumptions)))
	      (ctrace "Deeming " ($literal (cell/good-name cell))
		      "=" ($literal (cell/value cell))
		      " (computed by " ($literal (rule/id (cell/rule cell)))
		      ") to be the culprit."))
	    (form-nogood-set! (append assumptions parameters trees))
	    (*retract! (car assumptions)))
	  (let ((premises (append defaults parameters assumptions)))
	    (if (null? premises)
		(error "hard-core contradiction!"))
	    (if (not (null? parameters))
		(form-nogood-set! (append parameters trees)))
	    (if (null? (cdr premises))
		(*retract! (car premises))
		(let ((choice (choose-culprit premises)))
		  (case choice
		    ((DEFER) (enqueue! *defer-queue* queue-item))
		    ((PUNT) (enqueue! *punt-queue* queue-item))
		    (else (*retract! choice))))))))))

(define (choose-culprit losers)
  (newline)
  (write-string ";;; These are the premises that seem to be at fault:")
  (for-each (lambda (loser)
	      (newline)
	      (write-string ";;;    ")
	      (write loser)
	      (for-each (lambda (cell)
			  (if (and (cell/global? cell)
				   (eq? loser (cell/true-supplier cell)))
			      (begin
				(write-string " == ")
				(write (cell/good-name cell)))))
			(node/cells loser)))
	    losers)
  (newline)
  (write-string ";;; Choose one of these to retract and PROCEED with it.")
  (let ((culprit (bkpt "Choose Culprit")))
    (or (and (memq culprit '(DEFER PUNT))
	     culprit)
	(and (cell? culprit)
	     (let ((supplier (cell/true-supplier culprit)))
	       (and (memq supplier losers)
		    supplier)))
	(begin
	  (newline)
	  (write-string "Value must be one of the given premises.")
	  (choose-culprit losers)))))

(define-integrable (awaken-cells/add! cells)
  (for-each awaken-cell/add! cells))

(define-integrable (awaken-cells/nogood! cells)
  (for-each awaken-cell/nogood! cells))

(define (awaken-cell/add! cell)
  (awaken-cell! cell constraint-type/add-rules enqueue-rule/add!))

(define (awaken-cell/nogood! cell)
  (awaken-cell! cell constraint-type/nogood-rules enqueue-rule/nogood!))

(define (awaken-cell! cell constraint-type/rules enqueue-rule!)
  (let ((constraint (cell/owner cell)))
    (if constraint
	(for-each
	 (lambda (rule)
	   (if (not (constraint/rule-queued? constraint rule))
	       (let ((inputs (constraint/inputs constraint rule)))
		 (if (for-all? inputs node/bound?)
		     (if (there-exists? inputs
			   (lambda (input)
			     (memq (cell/state input) '(REBEL DUPE))))
			 (enqueue! *rebel-queue*
				   (lambda ()
				     (enqueue-rule! rule constraint)))
			 (enqueue-rule! rule constraint))))))
	 (vector-ref (constraint-type/rules (constraint/type constraint))
		     (cell/name cell))))))