#| -*-Scheme-*-

$Header$

Copyright (c) 1989 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. |#

;;;; Network Editing

(declare (usual-integrations)
	 (integrate-external "struct"))

(define (equate! x y)
  (if (not (eq? x y))
      (begin
	(add-cell/equivalent! y x)
	(add-cell/equivalent! x y)))
  (if (not (node=? x y))
      (let ((rx (cell/repository x))
	    (ry (cell/repository y))
	    (bx? (node/bound? x))
	    (by? (node/bound? y)))
	(let ((r (merge-values! x y)))
	  (let ((rcells (append (repository/cells rx) (repository/cells ry)))
		(newcomers
		 (if bx?
		     (if by? '() (repository/cells ry))
		     (if by? (repository/cells rx) '())))
		(r* (if (eq? r rx) ry rx)))
	    (set-repository/cells! r rcells)
	    (for-each (lambda (cell)
			(set-cell/repository! cell r))
		      (repository/cells r*))
	    (let ((fcells (alter-nogoods-repository! r* r)))
	      (set-repository/nogoods!
	       r
	       (merge-nogood-sets (repository/nogoods r)
				  (repository/nogoods r*)))
	      (awaken-cells/nogood! fcells))
	    (awaken-cells/add! newcomers))))))

(define (add-cell/equivalent! cell equivalent)
  (let ((equivalents (cell/equivalents cell)))
    (if (not (memq equivalent equivalents))
	(set-cell/equivalents! cell (cons equivalent equivalents)))))

(define (merge-values! x y)
  (cond ((not (node/bound? x))
	 (merge-one-value! x y))
	((not (node/bound? y))
	 (merge-one-value! y x))
	(else
	 (let ((c
		(cond ((rule/constant? (node/rule x)) x)
		      ((rule/constant? (node/rule y)) y)
		      ((cell/ancestor? x y) x)
		      ((cell/ancestor? y x) y)
		      ((positive? (node/n-contradicting y)) x)
		      (else y))))
	   (merge-two-values! c (if (eq? c x) y x))))))

(define (merge-one-value! x y)
  (set-cell/state! (node/supplier x) 'SLAVE)
  (point-links-toward! x)
  (set-cell/link! x y)
  (cell/repository y))

(define (merge-two-values! king deposed)
  (let ((rk (cell/repository king))
	(rd (cell/repository deposed)))
    (let ((sk (repository/supplier rk))
	  (sd (repository/supplier rd)))
      (let ((value (cell/contents sk)))
	(let ((consistent?
	       (lambda (cell)
		 (value=? (cell/contents cell) value))))
	  (if (and (zero? (repository/n-contradicting rd))
		   (consistent? sd))
	      (set-cell/state! sd 'FRIEND)
	      (begin
		(ctrace "Contradiction when merging "
			($literal (cell/good-name king)) "="
			($literal (cell/value king)) " and "
			($literal (cell/good-name deposed)) "="
			($literal (cell/value deposed)) ".")
		(for-each (lambda (cell)
			    (case (cell/state cell)
			      ((SLAVE)
			       (if (not (consistent? (node/supplier cell)))
				   (begin
				     (set-cell/state! cell 'DUPE)
				     (set-cell/contents! cell sd))))
			      ((REBEL KING FRIEND)
			       (if (consistent? cell)
				   (set-cell/state! cell 'FRIEND)
				   (begin
				     (set-cell/state! cell 'REBEL)
				     (increment-node/n-contradicting! king)
				     (enqueue-contradiction/node! cell sk))))
			      ((DUPE)
			       (if (consistent? (cell/contents cell))
				   (set-cell/state! cell 'SLAVE)))
			      (else (error "bad cell state" cell))))
			  (repository/cells rd)))))))
    (point-links-toward! deposed)
    (set-cell/link! deposed king)
    rk))

(define (forget! cell)
  (ctrace-forget! cell false false)
  (forget-internal! cell))

(define (ctrace-forget! cell source via)
  (ctrace "Removing " ($literal (cell/contents cell))
	  " from " ($literal (cell/good-name cell))
	  (if source
	      ($nested " because "
		       (if (and via (not (eq? via source)))
			   ($nested ($literal (cell/good-name via)) "==")
			   "of ")
		       ($literal (cell/good-name source)))
	      "")
	  "."))

(define (forget-internal! cell)
  (case (cell/state cell)
    ((FRIEND)
     (reset-cell/state! cell 'SLAVE))
    ((REBEL)
     (set-cell/state! cell 'SLAVE)
     (decrement-node/n-contradicting! cell)
     (awaken-cell/add! cell)
     (let ((dupes
	    (list-transform-positive (node/cells cell)
	      (lambda (c)
		(and (cell/dupe? c)
		     (eq? (cell/contents c) cell))))))
       (for-each (lambda (cell)
		   (set-cell/state! cell 'SLAVE)
		   (awaken-cell/add! cell))
		 dupes)
       (forget-consequences! cell (forget-consequences dupes))))
    ((KING)
     (let ((friend (list-search-positive (node/cells cell) cell/friend?)))
       (if (not friend)
	   (forget-friendless-king! cell)
	   (begin
	     (usurper! friend)
	     (reset-cell/state! cell 'SLAVE)
	     (if (not (zero? (node/n-contradicting cell)))
		 (for-each (lambda (cell)
			     (enqueue-contradiction/node! cell friend))
			   (list-transform-positive (node/cells cell)
			     cell/rebel?)))))))
    ((SLAVE PUPPET DUPE) unspecific)
    (else (error "bad cell state" cell))))

(define (forget-friendless-king! cell)
  (let ((nogoods (nogood-assoc (cell/contents cell) (node/nogoods cell))))
    (if nogoods
	(for-each
	 (lambda (nogood)
	   (let ((winner?
		  (lambda (entry)
		    (let ((supplier (repository/supplier (car entry))))
		      (and (node/bound? supplier)
			   (value=? (node/value supplier)
				    (cdr entry)))))))
	     (let loop ((entries (cdr nogood)))
	       (cond ((null? entries)
		      unspecific)
		     ((winner? (car entries))
		      (loop (cdr entries)))
		     ((for-all? (cdr entries) winner?)
		      (awaken-cells/nogood!
		       (repository/cells (caar entries))))))))
	 (cdr nogoods))))
  (let ((cells (node/cells cell)))
    (let ((consequences
	   (forget-consequences (list-transform-positive cells cell/slave?)))
	  (rebels (list-transform-positive cells cell/rebel?)))
      (if (null? rebels)
	  (begin
	    (reset-cell/state! cell 'PUPPET)
	    (awaken-cells/nogood! cells))
	  (let ((rebel (car rebels)))
	    (usurper! rebel)
	    (reset-cell/state! cell 'SLAVE)
	    (decrement-node/n-contradicting! cell)
	    (awaken-cell/add! cell)
	    (let ((value (cell/contents rebel)))
	      (for-each (lambda (c)
			  (case (cell/state c)
			    ((SLAVE) (awaken-cell/add! c))
			    ((REBEL)
			     (if (value=? (cell/contents c) value)
				 (begin
				   (set-cell/state! c 'FRIEND)
				   (decrement-node/n-contradicting! cell))
				 (enqueue-contradiction/node! c rebel)))
			    ((DUPE)
			     (if (value=? (cell/contents (cell/contents c))
					  value)
				 (set-cell/state! c 'SLAVE)))
			    ((KING))
			    (else (error "bad cell state" c))))
			cells))))
      (forget-consequences! cell consequences))))

(define (forget-consequences cells)
  (map (lambda (cell)
	 (cons cell
	       (let ((constraint (cell/owner cell)))
		 (if (not constraint)
		     '()
		     (list-transform-positive
			 (vector->list (constraint/cells constraint))
		       (let ((name (cell/name cell)))
			 (lambda (v)
			   (and (not (eq? v cell))
				(memq (cell/state v) '(KING FRIEND REBEL))
				(memv name (rule/inputs (cell/rule v)))))))))))
       cells))

(define (forget-consequences! cell consequences)
  (for-each (lambda (q)
	      (for-each (lambda (f)
			  (ctrace-forget! f cell (car q))
			  (forget-internal! f))
			(cdr q)))
	    consequences))

(define (*dissolve! cell)
  (expunge-nogoods! cell)
  (let ((supplier (node/supplier cell))
	(cells (node/cells cell)))
    (for-each (lambda (c)
		(set-cell/link! c false)
		(set-cell/equivalents! c '())
		(if (not (eq? c supplier))
		    (let ((r (make-repository)))
		      (case (cell/state c)
			((FRIEND REBEL)
			 (set-cell/state! c 'KING))
			((DUPE SLAVE)
			 (if (not (node/bound? supplier))
			     (set-cell/state! c 'PUPPET)))
			(else
			 (error "bad cell state" c)))
		      (set-repository/supplier! r c)
		      (set-cell/repository! c r)
		      (set-repository/cells! r
					     (cons c (repository/cells r))))))
	      cells)
    (set-node/cells! supplier (list supplier))
    (reset-node/n-contradicting! supplier)
    (if (node/bound? supplier)
	(let ((slaves
	       (list-transform-positive cells
		 (lambda (cell)
		   (memq (cell/state cell) '(SLAVE DUPE))))))
	  (for-each (lambda (slave) (set-cell/state! slave 'PUPPET)) slaves)
	  (forget-consequences! cell (forget-consequences slaves))))))

(define (expunge-nogoods! cell)
  (let ((do-subcells
	 (lambda (loop cell)
	   (for-each (lambda (cell)
		       (let ((constraint (cell/owner cell)))
			 (if constraint
			     (for-each-vector-element
			      (constraint/cells constraint)
			      loop))))
		     (node/cells cell)))))
    (let loop ((cell cell))
      (if (eq? (cell/mark cell) 'UNMARKED)
	  (begin
	    (set-cell/mark! cell 'MARKED)
	    (if (not (null? (node/nogoods cell)))
		(begin
		  (awaken-cells/nogood! (node/cells cell))
		  (set-node/nogoods! cell '())))
	    (do-subcells loop cell))))
    (let loop ((cell cell))
      (if (not (eq? (cell/mark cell) 'UNMARKED))
	  (begin
	    (set-cell/mark! cell 'UNMARKED)
	    (do-subcells loop cell))))))

(define (*detach! cell)
  (for-each (lambda (c)
	      (set-cell/equivalents! c (delq! cell (cell/equivalents c))))
	    (cell/equivalents cell))
  (set-cell/equivalents! cell '())
  (reconstruct-node! cell))

(define (*disconnect! cell)
  (let ((equivalents (cell/equivalents cell)))
    (for-each (lambda (c)
		(set-cell/equivalents!
		 c
		 (unionq (delq c equivalents)
			 (delq! cell (cell/equivalents c)))))
	      equivalents))
  (set-cell/equivalents! cell '())
  (reconstruct-node! cell))

(define (*disequate! x y)
  (if (eq? (cell/repository x) (cell/repository y))
      (begin
	(set-cell/equivalents! x (delq! y (cell/equivalents x)))
	(set-cell/equivalents! y (delq! x (cell/equivalents y)))
	(if (or (eq? x (cell/link y)) (eq? y (cell/link x)))
	    (reconstruct-node! x)))))

(define (reconstruct-node! cell)
  (for-each (lambda (equate) (equate! (car equate) (cdr equate)))
	    (let ((equates
		   (mapcan (lambda (cell)
			     (map (lambda (equivalent) (cons cell equivalent))
				  (cell/equivalents cell)))
			   (node/cells cell))))
	      (*dissolve! cell)
	      equates)))

(define (*retract! cell)
  (ctrace "Retracting the premise " ($literal (cell/good-name cell))
	  "=" (cell/value cell) ".")
  (forget! cell))

(define (*change! cell value)
  (let ((supplier (cell/true-supplier cell)))
    (let ((rule (cell/rule supplier)))
      (if (not (or (rule/default? rule) (rule/parameter? rule)))
	  (error "can only change the value of a default or parameter" cell))
      (if (or (not (forbidden? supplier value))
	      (y-or-n? "That value is contradictory; do it anyway"))
	  (begin
	    (*retract! supplier)
	    (process-setc! false supplier value rule))))))