/*
 *	memory allocation system with fixed-sized buckets
 *
 *	the system maintains buckets of sizes 4,8,...,4*FREEBLOCKS.
 *	a user memory request is converted to an internal request which
 *	includes a header and is rounded up to a multiple of 4; call this L.
 *	if the corresponding bucket is nonempty the memory is obtained from
 *	there.  otherwise it is peeled off the end of the pe's once only
 *	allocation pool (MEM[pe]).  when the memory is freed it is appended
 *	to its bucket for later reuse.
 *
 *	buckets are indexed by the total size of the memory blocks they contain
 *	divided by 4.  the size of a memory block is:
 *		(user_request_size + internal_header_size) rounded up to 4.
 *
 *	statistics kept: for each block size, # requests, # returns, # reused.
 */

#include "cksys.h"
#include "host.h"
#include <signal.h>

#define FREEBLOCKS 1000
#define HASHBLOCKS 113

#define MISCBLOCK 0
#define BLOCK_TO_SIZE(b) (b<<2)
#define SIZE_TO_BLOCK(s) (s>>2)
#define TRACE(n,p) if(MemTrace >= n) p

#define HDRLTH (sizeof(int))
#define DATA(frp) (((char *)frp)+HDRLTH)
#define MEMADDR(pe,index) (MemBase + pe*SysMem + index)

int BytesFreed[FREEBLOCKS];
int FreeInd[MaxPe];       /* Index into MEM where free space begins */
int SysMem = MaxMem;
char *MemBase;
int MemTrace;
int HashTableEntries;
int Trash;
int PrintMemStat = 0;

struct fr {
	int size;
	struct fr *next;
} *FreeBlock[MaxPe][FREEBLOCKS], HashBlock[MaxPe][HASHBLOCKS];

struct stat {
	int nrequest, nreturn, nreuse;
} BlockStat[MaxPe][FREEBLOCKS];

/*---------------------------------------------------------------- MemInit */

MemInit()
{
	/* initialize chare kernel memory allocator.
	 * the first time MemInit is called, a contiguous block of SysMem*numPe
	 * bytes of shared memory is requested from the operating system.
	 * subsequent calls to MemInit will reset the chare kernel memory allocator
	 * without requesting further memory from the operating system.
	 */
	register int i,j;
	extern int numPe;

	for( i=0; i<MaxPe; i++ ) FreeInd[i] = 0;

	for( i=0; i<MaxPe; i++ ) {
		for( j=0; j<FREEBLOCKS; j++ )
			FreeBlock[i][j] = NULL;
		for (j=0; j<HASHBLOCKS; j++) {
			HashBlock[i][j].size = 0;
			HashBlock[i][j].next = NULL;
		}
	}

	for( i=0; i<FREEBLOCKS; i++ )
		for( j=0; j<MaxPe; j++ ) {
		BlockStat[j][i].nrequest = 0;
		BlockStat[j][i].nreturn = 0;
		BlockStat[j][i].nreuse = 0;
	}

	MemBase = (char *)OsSharedMalloc(SysMem*numPe);
	if( MemBase == NULL ) {
		printf("CK: can't allocate %d bytes of memory\n", SysMem*numPe);
		exit(1);
	}
	HashTableEntries = 0;
	Trash = 0;
}

/*---------------------------------------------------------------- OsAlloc */

void * OsAlloc(size)
register int size;
{
	/* return pointer to size bytes of shared memory.  pointer is word aligned.
	 * causes fatal chare kernel error if out of memory.
	 */
	register struct fr *frp;
	struct fr *prevfr;
	register int pe;
	int i;
	int bucket;

	/* form internal request size.
	 * add header length to request size and round up to multiple of 4
	 */
	if (size <= 0) return NULL;
	size = (size + HDRLTH + 3) & (~3);
	pe = MyPenum;
	if ( (bucket = SIZE_TO_BLOCK(size)) >= FREEBLOCKS)
		bucket = MISCBLOCK;
	frp = NULL;

	if( bucket != MISCBLOCK && (frp = FreeBlock[pe][bucket]) != NULL ) {
		/* size is in stock; remove from free chain
		 */
		FreeBlock[pe][bucket] = frp->next;
		BlockStat[pe][bucket].nrequest++;
		BlockStat[pe][bucket].nreuse++;
		TRACE(20,OsPrintf("OsAlloc: free list, pe(%d) size(%d) &(%d)\n", pe, size, ((char *)frp)-MemBase));
	}

	else if ( bucket == MISCBLOCK ) {
		int hashnum;
		if ((hashnum = HashBucket(SIZE_TO_BLOCK(size))) >= 0) {
			if ((frp = HashBlock[pe][hashnum].next) != NULL) {
				HashBlock[pe][hashnum].next = frp->next;
				OsPrintf("OsAlloc: got size %d from hash bucket %d\n", size, hashnum);
			}
		}
	}

	/* process request sizes which are not bucket sizes maintained by the
	 * memory allocator, and request sizes which correspond to buckets but
	 * have empty free chains
	 */
	if ( frp == NULL && (i = FreeInd[pe]) + size < SysMem ) {
		/* size is not in stock, but can be allocated from main storage array.
		 */
		FreeInd[pe] += size;
		frp = (struct fr *)MEMADDR(pe,i);
		frp->size = size;
		BlockStat[pe][bucket].nrequest++;
		TRACE(20,OsPrintf("OsAlloc: pe(%d) size(%d) &(%d)\n", pe, size, ((char *)frp)-MemBase ));
	}

	else if (frp == NULL) {
		/* size is not in stock and cannot be allocated from main storage array.
		 * cannibalize the miscellaeous list in first-fit mode
		 */
		frp = FreeBlock[pe][MISCBLOCK];
		prevfr = NULL;
		while (frp != NULL && frp->size < size) {
			prevfr = frp;
			frp = frp->next;
		}

		if( frp != NULL ) {
			if( prevfr == NULL )
				FreeBlock[pe][MISCBLOCK] = frp->next;
			else
				prevfr->next = frp->next;
			BlockStat[pe][MISCBLOCK].nrequest++;
			BlockStat[pe][MISCBLOCK].nreuse++;
			OsPrintf("OsAlloc: cannibalized size %d\n", size);
		}
		else {
			OsPrintf("OsAlloc: fatal memory overflow (PE %d)\n", pe);
			OsKillSys();
			MemStats();
			kill(0,SIGINT);
			exit(1);
		}
	}
	return (void *)DATA(frp);
}

/*---------------------------------------------------------------- OsFree */

OsFree(p)
char *p;
{ 
	/* return memory allocated by OsAlloc.
	 * append to specific linked list if size < FREEBLOCKS, otherwise append to
	 * miscellaneous list.
	 */
	register struct fr *frp;
	register int size, pe;
	int bucket, hashnum;

	frp = (struct fr *)(p - HDRLTH);
	size = frp->size;
	bucket = SIZE_TO_BLOCK( size );
	pe = MyPenum;
	TRACE(20,OsPrintf("OsFree: return from pe(%d) &(%d) of size %d\n", pe, p-HDRLTH-MemBase, size));
	if( bucket < FREEBLOCKS ) {
		frp->next = FreeBlock[pe][bucket];
		FreeBlock[pe][bucket] = frp;
		BlockStat[pe][bucket].nreturn++;
	}
	else {
		if ((hashnum = HashBucket(bucket)) >= 0) {
			frp->next = HashBlock[pe][hashnum].next;
			HashBlock[pe][hashnum].next = frp;
		}
		else {
			frp->next = FreeBlock[pe][MISCBLOCK];
			FreeBlock[pe][MISCBLOCK] = frp;
			BlockStat[pe][MISCBLOCK].nreturn++;
		}
	}
}

/*----------------------------------------------------------- MemAdviseHash */

MemAdviseHash(size)
int size;
{
	/* entry point to create a hash bucket for requests of 'size' bytes.
	 * returns 1 if successful, 0 if failed.  no action is taken on failure.
	 * reasons for failure are: size <= 0; size within range of bucket array;
	 * hash table full.
	 */
	int bucket;

	if (size <= 0) return 0;    /* bogus size */
	size = (size + HDRLTH + 3) & (~3);
	if ( (bucket = SIZE_TO_BLOCK(size)) < FREEBLOCKS )
		return 0;    /* no need to hash small sizes */
	if (HashBucketOrCreate( bucket ) == -1)
		return 0;    /* hash table full */
	return 1;
}

/*----------------------------------------------------------- Hash functions */

HashBucket(value)
int value;
{
	/* return hash bucket for 'value', -1 if no hash value.
	 */
	int hashindex;

	hashindex = HashProbe(value);
	if (hashindex < 0) return -1;
	return (hashindex < HASHBLOCKS ? hashindex : -1);
}

HashBucketOrCreate(value)
int value;
{
	/* return the hash bucket for 'value'.  if it does not exist, attempt to
	 * create it; return its index if successful.  otherwise return -1.
	 */
	int hashindex;

	hashindex = HashProbe(value);
	if (hashindex < 0) return -1;
	if (hashindex < HASHBLOCKS) return hashindex;

	hashindex -= HASHBLOCKS;
	HashBlock[MyPenum][hashindex].size = value;
	HashBlock[MyPenum][hashindex].next = NULL;
	TRACE(10,OsPrintf("HashBucketOrCreate: new entry for size %d at %d\n", value, hashindex));
	HashTableEntries++;
	return hashindex;
}

HashProbe(value)
int value;
{
	/* return hash index of 'value' if it exists; else HASHBLOCKS + a free slot
	 * in the hash table; else -1 if there is no free slot.
	 *
	 * the hash function and rehash strategy are implemented in this procedure.
	 * the simplest scheme is used at present: closed hashing with linear
	 * probing, and a trivial hash function.
	 */
	int hashify, hashsearch, pe, i;

	/* hash an integer to 0..HASHBLOCKS
	 */
	hashify = hashsearch = value % HASHBLOCKS;

	pe = MyPenum;
	do {
		i = HashBlock[pe][hashsearch].size;
		if (i == value)
			return hashsearch;
		if (i == 0)
			return hashsearch + HASHBLOCKS;
		hashsearch = (++hashsearch) % HASHBLOCKS;
	} while (hashsearch != hashify);
	return -1;
}

/*---------------------------------------------------------------- MemStats */

MemStats()
{
	/*TRACE(10,FreeDisplay());*/
	if (PrintMemStat)
		FreeDisplay();
}

static FreeDisplay()
{
	int i, j, stock, total_mem = 0;
	struct fr *frp;
	int col, lrequest, lrelease, lreuse;

	OsPrintf("Memory Status:\n");
	OsPrintf("PE[FreeInd(FreeBytes)]: ");
	for( i=0; i<MaxPe; i++ )
		if( FreeInd[i] > 0 ) {
			total_mem += FreeInd[i];
			stock = 0;
			for( j=0; j<FREEBLOCKS; j++ ) {
				for( frp = FreeBlock[i][j]; frp != NULL; frp = frp->next ) 
					stock += BLOCK_TO_SIZE(j);
			}
			OsPrintf("%d[%d(%d)] ", i, FreeInd[i],stock);
		}
	OsPrintf("\n");

	OsPrintf("Total Memory Usage: %d\n", total_mem);

	OsPrintf("[Bucket,Inuse,Request,Reuse]:\n");
	col = 0;
	for( i=0; i<FREEBLOCKS; i++ ) {
		lrequest = lrelease = lreuse = 0;
		for( j=0; j<MaxPe; j++ ) {
			lrequest += BlockStat[j][i].nrequest;
			lrelease += BlockStat[j][i].nreturn;
			lreuse   += BlockStat[j][i].nreuse;
		}
		if( lrequest | lrelease | lreuse ) {
			OsPrintf("[%d,%d,%d,%d]", i, lrequest-lrelease, lrequest, lreuse );
			if( (++col % 4 ) == 0 ) OsPrintf("\n"); else OsPrintf("\t");
		}
	}
	OsPrintf("\n");
}

GarbageMan()
{
}

void * OsAllocInit(size)
register int size;
{
	return (void *)OsSharedMalloc(size);
}
