/****************************************************************
Copyright (C) The University of Melbourne 1993
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name of The University of Melbourne 
or any of its entities not be used in advertising or publicity
pertaining to distribution of the software without specific,
written prior permission.

THE UNIVERSITY OF MELBOURNE DISCLAIMS ALL WARRANTIES WITH REGARD
TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY
OF MELBOURNE OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.

AUTHORS : Jason Lee (jasonl@cs.mu.oz.au)
	  Andrew Davison (ad@cs.mu.oz.au)

COMMENTS : This file contains the error handling function for bebop.
	   That is the function yyerror the yacc calls when an error
	   occurs during parsing.
	   This code is based on an idea from fjh, and prints out
	   the line containing the error and highlights it, like 
	   the C compiler 'cc'.
****************************************************************/

/* Include all the header files needed */
#include "bebop.h"
#include "y.tab.h"

/* Extern declarations */
extern int nerrors;
extern int yylineno;
extern FILE *yyin;
extern char *char_line;
extern char *prev_line;
extern int yyleng;
extern int column;
extern char yytext[];
extern int bec_col;
extern int iam_col;
extern int line_size;

/* Used to signal error at end of file */
char endflag;
char var_error;
char error_becomes;
char error_iam;

/* This is the definition of yyerror that yacc uses when a parse error
   occurs.
*/
void yyerror(const char *mesg)
{
	int j, i, col;
	char prefix[14];
	char *str;


	if (endflag) return;

	/* One more error */
	nerrors++;

	/* Print out line number and error */
#ifdef SPRT
	sprintf(prefix, "Line %d : ", yylineno);
	j = strlen(prefix);
#else
	j = sprintf(prefix, "Line %d : ", yylineno);
#endif
	fprintf(stderr, "\n\n%s ", prefix);

	if (feof(yyin))
	{
		fprintf(stderr, "error at end-of-file\n\n");
		endflag = TRUE;
		return;
	}

	/* Now column will either be at the end of the token which is the 
	   error or to the right of it due to look ahead nature of yacc.
	   Search for beginning of the error token.
	   But if we have a forced error with becomes or i_am we may have to
	   search the previous line for the becomes/i_am
	*/
	if (error_becomes)
	{
		if (bec_col < 0 )
		{
			bec_col = 0;
		}
		i = line_size - LEN_BEC;
		for(col = bec_col;  col < i && strncmp(&char_line[col], BECOMES, LEN_BEC); col++)
			; /* Nothing */
		if(col != i)
		{
			str = char_line;
		}
		else
		{
			for(col = bec_col; prev_line[col] != EOS && strncmp(&prev_line[col], BECOMES, LEN_BEC); col++)
				;/* Nothing */
			str = prev_line;
		}
		yyleng = LEN_BEC;	/*strlen of becomes */
	}
	else
	{
		if (error_iam)
		{
			if (iam_col < 0 )
			{
				iam_col = 0;
			}
			i = line_size - LEN_IAM;
			for(col = iam_col;  col < i && strncmp(&char_line[col], IAM, LEN_IAM); col++)
				; /* Nothing */
			if(col != i)
			{
				str = char_line;
			}
			else
			{
				for(col = iam_col; prev_line[col] != EOS && strncmp(&prev_line[col], IAM, LEN_IAM); col++)
					;/* Nothing */
				str = prev_line;
			}
			yyleng = LEN_IAM;	/*strlen of iam*/
		}
		else
		{
			str = char_line;
			if (!var_error)
			{
				for(col = column; col >= 0 && strncmp(&char_line[col],yytext, yyleng); col--)
				{
					;/* Empty for loop */
				}
			}
			else
			{
				i = strlen(yylval.var->varname);
				for(col = column; col >= 0 && strncmp(&char_line[col], yylval.var->varname,i); col--)
				{
					;/* Empty for loop */
				}
			}
		}
	}

	/* If error not on this line just point out line and
	   make error at the start
	*/
	if (col < 0) col = 0;


	/* Now print out line highlight error */
	if(col >= 0 && yyleng > 0 && yytext[0] != EOS)
	{
		fprintf(stderr, "%s", str);

		for(i = 0; i <= j; i++)
		{
			fprintf(stderr, " ");
		}
		for(i = 0; i < col; i++)
		{
			fprintf(stderr, "-");
		}
		for(i = 0; i < yyleng; i++)
		{
			fprintf(stderr, "^");
		}

	}

	fprintf(stderr,"\n");

	/* Make sure user sees errors */
	fflush(stderr);
}

/* Print out warnings about possible errors to do with predicates called and
   not defined and with declarations that may be wrong
*/
void print_warnings(void)
{

	extern FAPTR predicate_table[];
	int i, arity;
	FAPTR faptr;

	for(i = 0; i < HASHSIZE; i++)
	{
		faptr = predicate_table[i];

		while(faptr)
		{
			arity = (faptr->arity < 0) ? 0 : faptr->arity;
			if (faptr->called && !faptr->defined)
			{

				fprintf(stderr,"\nWarning : predicate %s, arity %d called but not defined within class.\n\n", faptr->faname, arity);
			}

			if (faptr->decl && !faptr->defined)
			{
				fprintf(stderr,"\nWarning : declaration of predicate %s, arity %d without being defined within class.\n\n", faptr->faname, arity);
			}
			faptr = faptr->next;
		}
	}
	fflush(stderr);
}
