/****************************************************************
Copyright (C) The University of Melbourne 1993
All Rights Reserved

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby
granted, provided that the above copyright notice appear in all
copies and that both that the copyright notice and this
permission notice and warranty disclaimer appear in supporting
documentation, and that the name of The University of Melbourne 
or any of its entities not be used in advertising or publicity
pertaining to distribution of the software without specific,
written prior permission.

THE UNIVERSITY OF MELBOURNE DISCLAIMS ALL WARRANTIES WITH REGARD
TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY
OF MELBOURNE OR ANY OF ITS ENTITIES BE LIABLE FOR ANY
SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
THIS SOFTWARE.

AUTHORS : Jason Lee (jasonl@cs.mu.oz.au)
	  Andrew Davison (ad@cs.mu.oz.au)

COMMENTS : This file contains all the compilation functions
****************************************************************/

/* Include all the header files needed */
#include "bebop.h"

#include "y.tab.h"

/* Some macros instead of function calls*/

/* Print out vis/invis variables modes i.e input/output
   Given the list of variables and file to print to
*/
#define	PRINTVARSMODES(ptr, fptr)  for(ptr = ptr##_hd; ptr; ptr = ptr->next)\
			           {\
			             fprintf(fptr,",%c",((ptr->variable->mode == INPUT || ptr->variable->mode == UNKNOWN) ? 'i' : 'o'));\
			           }

/* Print out vis/invis variables modes i.e input/output
   Given the list of variables and files to print to
   This is used just for the initial section, see below.
*/
#define	PRINTVARSMODES2(ptr, fptr1, fptr2) \
			for(ptr = ptr##_hd; ptr; ptr = ptr->next)\
			{\
			  if (ptr->variable->mode == INPUT || \
			      ptr->variable->mode == UNKNOWN)\
			  {\
			    fprintf(fptr1, ",%c", 'i');\
			    fprintf(fptr2, ",%c", 'i');\
			  }\
			  else\
			  {\
			    fprintf(fptr1, ",%c", 'o');\
			    fprintf(fptr2, ",%c", 'o');\
			  }\
			}

/* Print out variables names
   Given list of variables and file to print to
*/
#define	PRINTVARS(ptr, fptr)  for(ptr = ptr##_hd; ptr; ptr = ptr->next)\
			      {\
			         fprintf(fptr,", %s", ptr->variable->varname);\
			      }

/* Print out variables names
   Given list of variables and files to print to
   This is used just for the initial section, see below
*/
#define	PRINTVARS2(ptr, fptr1, fptr2) \
			      for(ptr = ptr##_hd; ptr; ptr = ptr->next)\
			      {\
			         fprintf(fptr1,", %s", ptr->variable->varname);\
			         fprintf(fptr2,", %s", ptr->variable->varname);\
			      }


/* Print out variable name and assisgn it to value
   This is used just for the initial section, see below
*/
#define	PRINTVARVAL2(ptr, fptr1, fptr2) \
			  for(ptr = ptr##_hd; ptr; ptr = ptr->next)\
			  {\
			    if(ptr->variable->value && !ptr->variable->dbs)\
			    {\
			      if (!pv2)\
			      {\
				pv2 = TRUE;\
				fprintf(fptr2, "\t%s", ptr->variable->varname);\
			      }\
			      else\
			      {\
			        fprintf(fptr2, ",\n\t%s", ptr->variable->varname);\
			      }\
			      fprintf(fptr1, ",\n\t%s", ptr->variable->varname);\
			      if(use_eq_or_is(ptr->variable->value)== EQ)\
			      {\
				fprintf(fptr1, " = ");\
				fprintf(fptr2, " = ");\
			      }\
			      else\
			      {\
				fprintf(fptr1, " is ");\
				fprintf(fptr2, " is ");\
			      }\
			      print_tree2(ptr->variable->value, fptr1, fptr2);\
			      free_tree(ptr->variable->value);\
			      ptr->variable->value = NULL;\
			    }\
			  }

/* Print either the variable name or it's value */
#define	PRINTVARORVAL(ptr, fptr) if (ptr->variable->value)\
			         {\
			           print_tree(ptr->variable->value, fptr);\
			           free_tree(ptr->variable->value);\
			           ptr->variable->value = NULL;\
			         }\
			         else\
			         {\
			           fprintf(fptr, ptr->variable->varname);\
			         }

/* Print either the variable name or it's value 
   This is used just for the initial section, see below
*/
#define	PRINTVARORVAL2(ptr, fptr1, fptr2) \
			if (ptr->variable->value)\
			{\
			  print_tree2(ptr->variable->value, fptr1, fptr2);\
			  free_tree(ptr->variable->value);\
			  ptr->variable->value = NULL;\
			}\
			else\
			{\
			  fprintf(fptr1, ptr->variable->varname);\
			  fprintf(fptr2, ptr->variable->varname);\
			}

/* Print out the recursive call to c_<classname> */
#define	PRINTRECCALL(vis, invis, fptr, tt) \
		if (forget)\
		{\
		  fprintf(fptr, "\n");\
		  print_tabs(fptr, tt);\
		  fprintf(fptr, "c_%s(%s, %s", classname, INPUTSTR, ID);\
		  forget = FALSE;\
		}\
		else\
		{\
		  fprintf(fptr, ",\n");\
		  print_tabs(fptr, tt);\
		  fprintf(fptr, "c_%s(%s, %s", classname, INPUTSTR, ID);\
		}\
		for(vis = vis##_hd; vis; vis = vis->next)\
		 {\
		   fprintf(fptr, ", ");\
		   PRINTVARORVAL(vis, fptr);\
		 }\
		 for(invis=invis##_hd;invis;invis=invis->next)\
		 {\
		   fprintf(fptr, ", ");\
		   PRINTVARORVAL(invis, fptr);\
		 }\
		 if (dbs = dbs_hd)\
		 {\
		   PRINTVARS(dbs, fptr);\
		 }\
		 fprintf(fptr, ", %s)", MESG);

/* Print out the recursive call to c_<classname> 
   This is used just for the initial section, see below
*/
#define	PRINTRECCALL2(vis, invis, fptr1, fptr2, tt) \
		if (forget)\
		{\
		  fprintf(fptr1, "\n");\
		  fprintf(fptr2, "\n");\
		  print_tabs2(fptr1, fptr2, tt);\
		  fprintf(fptr1, "c_%s(%s, %s", classname, INPUTSTR, ID);\
		  fprintf(fptr2, "c_%s(%s, %s", classname, INPUTSTR, ID);\
		  forget = FALSE;\
		}\
		else\
		{\
		  fprintf(fptr1, ",\n");\
		  print_tabs(fptr1, tt);\
		  fprintf(fptr1, "c_%s(%s, %s", classname, INPUTSTR, ID);\
		  if (!pv2)\
		  {\
		    pv2 = TRUE;\
		    fprintf(fptr2, "\n");\
		    print_tabs(fptr2, tt);\
		    fprintf(fptr2, "c_%s(%s, %s", classname, INPUTSTR, ID);\
		  }\
		  else\
		  {\
		    fprintf(fptr2, ",\n");\
		    print_tabs(fptr2, tt);\
		    fprintf(fptr2, "c_%s(%s, %s", classname, INPUTSTR, ID);\
		  }\
		}\
		for(vis = vis##_hd; vis; vis = vis->next)\
		 {\
		   fprintf(fptr1, ", ");\
		   fprintf(fptr2, ", ");\
		   PRINTVARORVAL2(vis, fptr1, fptr2);\
		 }\
		 for(invis=invis##_hd;invis;invis=invis->next)\
		 {\
		   fprintf(fptr1, ", ");\
		   fprintf(fptr2, ", ");\
		   PRINTVARORVAL2(invis, fptr1, fptr2);\
		 }\
		 if (dbs = dbs_hd)\
		 {\
		   PRINTVARS2(dbs, fptr1, fptr2);\
		 }\
		 fprintf(fptr1, ", %s)", MESG);\
		 fprintf(fptr2, ", %s)", MESG);

/* Just free up memory */
#define	 FREE(ptr)	while(ptr##_hd)\
			{\
			  ptr = ptr##_hd;\
			  ptr##_hd = ptr##_hd->next;\
			  ptr->next = NULL;\
			  free(ptr);\
			}


/* Extern declarations */
extern char *file_nl;
extern char *fstr;
extern char *classname;
extern char pnpf;
extern TREEPTR prog_root;
extern const char *token_table[];
extern VARPTR variable_table[HASHSIZE];
extern int var_counter;

/* Prototypes */
void free_tree(TREEPTR root);
void interrupt(int sig);
void free_list(TLISTPTR flist);
VARPTR var_insert(const char *var, int length);
TREEPTR build_var(VARPTR var);
static void print_tree(TREEPTR root, FILE *fp);
static void print_tree2(TREEPTR root, FILE *fp1, FILE *fp2);
static char use_eq_or_is(TREEPTR root);
static void process_bec(TLISTPTR beclist, FILE *fp, int n);
static void process_bec2(TLISTPTR beclist, FILE *fp1, FILE *fp2, int n);
static char process_iam(TLISTPTR iamlist, FILE *fp);
static char process_iam2(TLISTPTR iamlist, FILE *fp1, FILE *fp2);
static void print_tree_format2(TREEPTR root, FILE *fp1, FILE *fp2, int tabs);
static void print_tree_format(TREEPTR root, FILE *fp, int tabs);
static void print_tabs2(FILE *fp1, FILE *fp2, int n);
static void print_tabs(FILE *fp, int n);

/* These will hold the tmp file names */
static char *tempname = NULL, *temp2 = NULL;
static FILE *tmp, *tmp2;

/* Keep a list of becomes and i_am's to process */
static TLISTPTR bec_hd, iam_hd, bec, iam;

/* Forget about printing a comma */
static char forget = FALSE;

/* Need lists of inv/vis variables and dbs*/
static VARLISTPTR vis_vars_hd, vis_vars;
static VARLISTPTR invis_vars_hd, invis_vars;
static VARLISTPTR dbs_hd, dbs;


/* Compile : takes the parse tree (global variable prog_root) and
   compiles (traverses the tree) to produce the NU-Prolog file
   "file.nl" and may or may not (depends on option p) a "file.pnp"
   file with pnp code and nu-prolog code in it for the user to look
   at.
   If compile succeds returns TRUE otherwise FALSE.
*/
char compile(void)
{
	/* root of parse tree */
	TREEPTR root = prog_root;

	/* Just variables to use */
	pid_t pid;
	time_t t = time(NULL);
	char *strtime;
	int i, j, k, m;
	FILE  *filenl;
	VARLISTPTR varlptr;
	VARPTR varptr;
	VARLISTPTR varlptr2, varlptr3;
	TREEPTR tptr, tptr2;
	TLISTPTR lptr;

	/* Pv2 is a dumb name to mean prevent varies errors to do
	   with 2 commas appearing when only one should
	*/
	char pv2 = FALSE;

	/* Various char pointers for names commands etc */
	char *pnp_comm = NULL;
	const char *str = APP, *str2 = PNP_SUFFIX;

	/* Make dummy start node of inv/vis variables and dbs */
	vis_vars_hd = NEW(VARLIST);
	vis_vars = vis_vars_hd;
	invis_vars_hd = NEW(VARLIST);
	invis_vars = invis_vars_hd;
	dbs_hd = NEW(VARLIST);
	dbs = dbs_hd;
	vis_vars->variable = invis_vars->variable = NULL;
	vis_vars->next = invis_vars->next = NULL;
	dbs->variable = NULL;
	dbs->next = NULL;

	/* Create unique tmp file name, this is a minor tmp
	   file and is always removed
	*/
	if (!temp2)
	{
		int length_pid;
		pid = getpid();
		i = pid;
		for(length_pid = 0; i; length_pid++)
		{
			i /= 10;
		}
		temp2 = copy_string(PREFIXTMP, (length_pid + LEN_PREFTMP));
		length_pid = LEN_PREFTMP;
		itoa(temp2, length_pid, pid);
	}

	/* Create the file.pnp name (this may be removed)
	   Use it as the main temp file
	*/
	k = strlen(fstr);
	tempname = copy_string(fstr, (k + LEN_PNP_SUFFIX));
	for(i = 0; i < LEN_PNP_SUFFIX; i++, k++)
	{
		tempname[k] = str2[i];
	}
	tempname[k] = EOS;

	/* Create and open for writing temp files */
	if (!(tmp = fopen(tempname, "w")))
	{
		perror("\nCompiler error : unable to create temporary pnp file");
		return FALSE;
	}
	if (!(tmp2 = fopen(temp2, "w")))
	{
		perror("\nCompiler error : unable to create temporary file");
		return FALSE;
	}

	/* Catch interrupt and quit signals so as to remove tmp file */
	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
	{
		signal(SIGINT, interrupt);
	}
	if (signal(SIGQUIT, SIG_IGN) != SIG_IGN)
	{
		signal(SIGINT, interrupt);
	}

	/* Create list of vis/invis variables and dbs which aren't system
	   ones
	*/
	for(i = 0; i < HASHSIZE; i++)
	{
		varptr = variable_table[i];
		while(varptr)
		{
			if(varptr->type == VISIBLE && !varptr->sys)
			{
				if ((!vis_vars->variable) ||
				    (vis_vars->variable->position <
				        varptr->position)
				   )
				{
					vis_vars->next = NEW(VARLIST);
					vis_vars = vis_vars->next;
					vis_vars->variable = varptr;
					vis_vars->next = NULL;
				}
				else
				{
					varlptr2 = vis_vars_hd;
					while (varlptr2->next && varlptr2->next->variable->position < varptr->position)
					{
						varlptr2 = varlptr2->next;
					}
					varlptr3 = NEW(VARLIST);
					varlptr3->next = varlptr2->next;
					varlptr3->variable = varptr;
					varlptr2->next = varlptr3;
				}
			}
			if(varptr->type == INVISIBLE && !varptr->sys)
			{
				invis_vars->next = NEW(VARLIST);
				invis_vars = invis_vars->next;
				invis_vars->variable = varptr;
				invis_vars->next = NULL;
			}
			if (varptr->dbs)
			{
				dbs->next = NEW(VARLIST);
				dbs = dbs->next;
				dbs->variable = varptr;
				dbs->next = NULL;
			}
			varptr = varptr->next;
		}
	}


        /* Remove dummy start node of variable lists and dbs */
        vis_vars = vis_vars_hd;
        vis_vars_hd = vis_vars_hd->next;
        free(vis_vars);
        invis_vars = invis_vars_hd;
        invis_vars_hd = invis_vars_hd->next;
        free(invis_vars);
	dbs = dbs_hd;
	dbs_hd = dbs_hd->next;
	free(dbs);

	/* Now before we begin write a little header at the start */
	if (!(filenl = fopen(file_nl, "w")))
	{
		fprintf(stderr,"\nCompiler error : unable to open '%s", file_nl);
		perror("'");
		return FALSE;
	}
	strtime  = ctime(&t);
	fprintf(filenl,"%c File created by bebop compiler %s", '%', strtime);
	fprintf(tmp,"%c File created by bebop compiler %s", '%', strtime);
	fprintf(filenl,"%c Do not edit this file, re-compile class %s\n", '%',classname);
	fprintf(tmp,"%c Do not edit this file, re-compile class %s\n", '%',classname);
	fprintf(filenl,"%c The next two predicates are the entry points into this class\n\n", '%');
	fprintf(tmp,"%c The next two predicates are the entry points into this class\n\n", '%');

	/* Close for the moment */
	fclose(filenl);

	/* From here on we compile. That is we create the file in tmp
	   performing the necessary fprintf's
	*/

	/* Print out declaration of class predicate
	   And the link predicate in minor tmp file.
	   That is, because the entry point to the class can either
	   be through calling the class directly ie class(ID) or
	   via an i_am call from another class (link we call it)
	   these predicates are virtual the same. So a psuedo-parallel
	   write is done print to tmp and tmp2 until these two predicates
	   are finished and merge the two files together into tmp.
	*/
	/*fprintf(tmp, "%s %s %s(o,o", tok_lookup(TOK_PROLOG), tok_lookup(TOK_LAZYDET), classname);*/
	fprintf(tmp, "%s %s %s(i,o", tok_lookup(TOK_PROLOG), tok_lookup(TOK_LAZYDET), classname);
	fprintf(tmp2, "%s %s l_%s(%s, %s, %s", tok_lookup(TOK_PROLOG), tok_lookup(TOK_LAZYDET), classname, "i", "i", "i");
	PRINTVARSMODES2(vis_vars, tmp, tmp2);
	fprintf(tmp, ").\n");
	fprintf(tmp2, ").\n");


	/*Print out head of class predicate 
	  And link predicate
	*/
	fprintf(tmp, "%s(%s, %s", classname, MESG, ID);
	fprintf(tmp2, "l_%s(%s, %s, %s", classname, MESG, INPUTSTR, ID);
	PRINTVARS2(vis_vars, tmp, tmp2);
	fprintf(tmp, ") %s\n", tok_lookup(TOK_RULE));
	fprintf(tmp2, ") %s\n", tok_lookup(TOK_RULE));

	/*Print out call to create id. This is the only difference (sort of)
	  between the class predicate and the link one
	*/
	fprintf(tmp, "\t%s", CREATEID);

	/* Now print out variables initial values */
	PRINTVARVAL2(vis_vars, tmp, tmp2);
	PRINTVARVAL2(invis_vars, tmp, tmp2);

	/* Now print out database variables and dbs */
	for(dbs = dbs_hd, varlptr = NULL; dbs;)
	{
		/* Pv2 is a dumb name to mean prevent varies errors to do
		   with 2 commas appearing when only one should
		*/
		if (!pv2)
		{
			fprintf(tmp2, "\t%s = [", dbs->variable->varname);
			pv2 = TRUE;
		}
		else
		{
			fprintf(tmp2, ",\n\t%s = [", dbs->variable->varname);
		}
		fprintf(tmp, ",\n\t%s = [", dbs->variable->varname);
		print_tree2(dbs->variable->value, tmp, tmp2);
		fprintf(tmp, "]");
		fprintf(tmp2, "]");
		free_tree(dbs->variable->value);
		dbs->variable->value = NULL;

		/* Now if the database variable is also an invis/vis variable
		   then remove it from dbs list and just use the case of it
		   in other list
		*/
		if (dbs->variable->type != UNKNOWN)
		{
			if (varlptr)
			{
				varlptr->next = dbs->next;
				dbs->next = NULL;
				free(dbs);
				dbs = varlptr;
			}
			else
			{
				varlptr = dbs;
				dbs = dbs->next;
				dbs_hd = dbs;
				varlptr->next = NULL;
				free(varlptr);
				varlptr = NULL;
				continue;
			}
		}
		varlptr = dbs;
		dbs = dbs->next;
	}

	/* Now have dummy start nodes for bec/iam list */
	bec_hd = NEW(TLIST);
	iam_hd = NEW(TLIST);
	bec = bec_hd;
	iam = iam_hd;
	iam->next = NULL;
	iam->tree = NULL;
	bec->next = NULL;
	bec->tree = NULL;

	/* Start at the initial section */
	tptr = OP1(root);
	/* Now print out initial section, if any
	   Again note the psuedo-parallel write method to two files
	*/
	if (tptr)
	{
		fprintf(tmp, ",\n");
		if (!pv2)
		{
			fprintf(tmp2, "\n");
			pv2 = TRUE;
		}
		else
		{
			fprintf(tmp2, ",\n");
		}

		/* Take care of if then else as the guard */
		if (OP(tptr) == TOK_THEN)
		{
			if (!OP1(OP1(tptr)))
			{
				fprintf(tmp, "\t(");
				fprintf(tmp2, "\t(");
			}
			else
			{
				fprintf(tmp, "\n");
				fprintf(tmp2, "\n");
			}
			print_tree2(OP1(tptr), tmp, tmp2);
			fprintf(tmp, " then\n");
			fprintf(tmp2, " then\n");
			tptr = OP2(tptr);
			if (tptr && OP(tptr) == TOK_ELSE)
			{
				print_tree_format2(OP1(tptr), tmp, tmp2, 2);
				process_bec2(bec_hd->next, tmp, tmp2, 2);
				bec_hd->next = NULL;
				bec = bec_hd;
				if (!process_iam2(iam_hd->next, tmp, tmp2))
				{
					PRINTRECCALL2(vis_vars, invis_vars, tmp, tmp2, 2);
				}
				iam_hd->next = NULL;
				iam = iam_hd;
				fprintf(tmp, "\n\telse\n");
				fprintf(tmp2, "\n\telse\n");
				if (OP(OP2(tptr)) == TOK_CPAR)
				{
					print_tree_format2(OP1(OP2(tptr)), tmp, tmp2, 2);
				}
				else
				{
					print_tree_format2(OP2(tptr), tmp, tmp2, 2);
				}
				process_bec2(bec_hd->next, tmp, tmp2, 2);
				bec_hd->next = NULL;
				bec = bec_hd;
				if (!process_iam2(iam_hd->next, tmp, tmp2))
				{
					PRINTRECCALL2(vis_vars, invis_vars, tmp, tmp2, 2);
				}
				iam_hd->next = NULL;
				iam = iam_hd;
				fprintf(tmp, "\n\t).\n\n\n");
				fprintf(tmp2, "\n\t).\n\n\n");
			}
			else
			{
				if (OP(tptr) == TOK_CPAR)
				{
					print_tree_format2(OP1(tptr), tmp, tmp2, 2);
				}
				else
				{
					print_tree_format2(tptr, tmp, tmp2, 2);
				}
				process_bec2(bec_hd->next, tmp, tmp2, 2);
				bec_hd->next = NULL;
				bec = bec_hd;
				if (!process_iam2(iam_hd->next, tmp, tmp2))
				{
					PRINTRECCALL2(vis_vars, invis_vars, tmp, tmp2, 2);
				}
				iam_hd->next = NULL;
				iam = iam_hd;
				fprintf(tmp, "\n\t).\n\n\n");
				fprintf(tmp2, "\n\t).\n\n\n");
			}
		}
		else
		{
			/* Take care of non-logical if the else (i.e ->) */
			if (OP(tptr) == TOK_NIF)
			{
				fprintf(tmp, "\t");
				fprintf(tmp2, "\t");
				print_tree2(OP1(tptr), tmp, tmp2);
				fprintf(tmp, " ->\n");
				fprintf(tmp2, " ->\n");
				tptr = OP2(tptr);
				if (tptr && OP1(tptr) && OP(OP1(tptr)) == TOK_SEMICOLON)
				{
					tptr = OP1(tptr);
					print_tree_format2(OP1(tptr), tmp, tmp2, 2);
					process_bec2(bec_hd->next, tmp, tmp2, 2);
					bec_hd->next = NULL;
					bec = bec_hd;
					if (!process_iam2(iam_hd->next, tmp, tmp2))
					{
						PRINTRECCALL2(vis_vars, invis_vars, tmp, tmp2, 2);
					}
					iam_hd->next = NULL;
					iam = iam_hd;
					fprintf(tmp, "\n\t;\n");
					fprintf(tmp2, "\n\t;\n");
					print_tree_format2(OP2(tptr), tmp, tmp2, 2);
					process_bec2(bec_hd->next, tmp, tmp2, 2);
					bec_hd->next = NULL;
					bec = bec_hd;
					if (!process_iam2(iam_hd->next, tmp, tmp2))
					{
						PRINTRECCALL2(vis_vars, invis_vars, tmp, tmp2, 2);
					}
					iam_hd->next = NULL;
					iam = iam_hd;
					fprintf(tmp, "\n\t).\n\n\n");
					fprintf(tmp2, "\n\t).\n\n\n");
				}
				else
				{
					print_tree_format2(OP1(tptr), tmp, tmp2, 2);
					process_bec2(bec_hd->next, tmp, tmp2, 2);
					bec_hd->next = NULL;
					bec = bec_hd;
					if (!process_iam2(iam_hd->next, tmp, tmp2))
					{
						PRINTRECCALL2(vis_vars, invis_vars, tmp, tmp2, 2);
					}
					iam_hd->next = NULL;
					iam = iam_hd;
					fprintf(tmp, "\n\t).\n\n\n");
					fprintf(tmp2, "\n\t).\n\n\n");
				}
			}
			else
			{
				/* Else print out normal guard*/
				if (OP1(tptr))
				{
					print_tree_format2(OP1(tptr), tmp, tmp2, 1);
				}
				print_tree_format2(OP2(tptr), tmp, tmp2, 1);
				process_bec2(bec_hd->next, tmp, tmp2, 1);
				bec_hd->next = NULL;
				bec = bec_hd;
				if (!process_iam2(iam_hd->next, tmp, tmp2))
				{
					PRINTRECCALL2(vis_vars, invis_vars, tmp, tmp2, 1);
				}
				iam_hd->next = NULL;
				iam = iam_hd;
				fprintf(tmp, ".\n\n\n");
				fprintf(tmp2, ".\n\n\n");
			}
		}
	}
	else
	{
		/* No initial section just print recursive call */
		PRINTRECCALL2(vis_vars, invis_vars, tmp, tmp2, 1);
		fprintf(tmp, ".\n\n");
		fprintf(tmp2, ".\n\n");
	}

	/* Close and force flush of buffers */
	fclose(tmp);
	fclose(tmp2);
	forget = FALSE;

	/* Now that we have finished the entry points to the class
	   merge them into one file, tmp
	*/
	if (!(tmp = fopen(tempname, "a")))
	{
		perror("\nCompiler error : unable to open temporary file");
		return FALSE;
	}
	if (!(tmp2 = fopen(temp2, "r")))
	{
		perror("\nCompiler error : unable to open temporary file");
		return FALSE;
	}

	/* Copy tmp2 to tmp, just use one char at time no rush */
	while ((i = getc(tmp2)) != EOF)
		putc(i, tmp);

	/* Remove tmp2, no need for it any more */
	fclose(tmp2);
	unlink(temp2);

	/* From here on only tmp (major tmp file) is used */

	/* Print out declaration for c_classname */
	fprintf(tmp, "%s %s c_%s(i, i", tok_lookup(TOK_PROLOG), tok_lookup(TOK_LAZYDET), classname);
	PRINTVARSMODES(vis_vars, tmp);
	PRINTVARSMODES(invis_vars, tmp);
	PRINTVARSMODES(dbs, tmp);
	fprintf(tmp, ", i).\n");

	bec = bec_hd;
	iam = iam_hd;
	tptr = OP2(root);
	lptr = LIST(OP2(tptr));

	/* Now process all the clauses */
	for(;lptr; lptr = lptr->next)
	{
		forget = FALSE;
		tptr2 = lptr->tree;
		fprintf(tmp, "c_%s(", classname);
		if (OP(OP1(tptr2)) == TOK_ANY)
		{
			fprintf(tmp, "%s, %s", ANY, ID);
		}
		else
		{
			fprintf(tmp, "[");
			print_tree(OP1(tptr2), tmp);
			fprintf(tmp, " | %s], %s", INPUTSTR, ID);
		}
		PRINTVARS(vis_vars, tmp);
		PRINTVARS(invis_vars, tmp);
		PRINTVARS(dbs, tmp);
		fprintf(tmp, ", %s) %s\n", MESG, tok_lookup(TOK_RULE));
		tptr2 = OP2(tptr2);

		if (tptr2)
		{
			if (OP(tptr2) == TOK_THEN)
			{
				if (!OP1(OP1(tptr2)))
				{
					fprintf(tmp, "\t(");
				}
				else
				{
					fprintf(tmp, "\t");
				}
				print_tree(OP1(tptr2), tmp);
				fprintf(tmp, " then\n");
				tptr2 = OP2(tptr2);
				if (tptr2 && OP(tptr2) == TOK_ELSE)
				{
					print_tree_format(OP1(tptr2), tmp, 2);
					process_bec(bec_hd->next, tmp, 2);
					bec_hd->next = NULL;
					bec = bec_hd;
					if (!process_iam(iam_hd->next, tmp))
					{
						PRINTRECCALL(vis_vars, invis_vars, tmp, 2);
					}
					iam_hd->next = NULL;
					iam = iam_hd;
					fprintf(tmp, "\n\telse\n");
					if (OP(OP2(tptr2)) == TOK_CPAR)
					{
						print_tree_format(OP1(OP2(tptr2)), tmp, 2);
					}
					else
					{
						print_tree_format(OP2(tptr2), tmp, 2);
					}
					process_bec(bec_hd->next, tmp, 2);
					bec_hd->next = NULL;
					bec = bec_hd;
					if (!process_iam(iam_hd->next, tmp))
					{
						PRINTRECCALL(vis_vars, invis_vars, tmp, 2);
					}
					iam_hd->next = NULL;
					iam = iam_hd;
					fprintf(tmp, "\n\t).\n\n\n");
				}
				else
				{
					if (OP(tptr2) == TOK_CPAR)
					{
						print_tree_format(OP1(tptr2), tmp, 2);
					}
					else
					{
						print_tree_format(tptr2, tmp, 2);
					}
					process_bec(bec_hd->next, tmp, 2);
					bec_hd->next = NULL;
					bec = bec_hd;
					if (!process_iam(iam_hd->next, tmp))
					{
						PRINTRECCALL(vis_vars, invis_vars, tmp, 2);
					}
					iam_hd->next = NULL;
					iam = iam_hd;
					fprintf(tmp, "\n\t).\n\n\n");
				}
			}
			else
			{
				if (OP(tptr2) == TOK_NIF)
				{
					fprintf(tmp, "\t");
					print_tree(OP1(tptr2), tmp);
					fprintf(tmp, " ->\n");
					tptr2 = OP2(tptr2);
					if (tptr2 && OP1(tptr2) && OP(OP1(tptr2)) == TOK_SEMICOLON)
					{
						tptr2 = OP1(tptr2);
						print_tree_format(OP1(tptr2), tmp, 2);
						process_bec(bec_hd->next, tmp, 2);
						bec_hd->next = NULL;
						bec = bec_hd;
						if (!process_iam(iam_hd->next, tmp))
						{
							PRINTRECCALL(vis_vars, invis_vars, tmp, 2);
						}
						iam_hd->next = NULL;
						iam = iam_hd;
						fprintf(tmp, "\n\t;\n");
						print_tree_format(OP2(tptr2), tmp, 2);
						process_bec(bec_hd->next, tmp, 2);
						bec_hd->next = NULL;
						bec = bec_hd;
						if (!process_iam(iam_hd->next, tmp))
						{
							PRINTRECCALL(vis_vars, invis_vars, tmp, 2);
						}
						iam_hd->next = NULL;
						iam = iam_hd;
						fprintf(tmp, "\n\t).\n\n\n");
					}
					else
					{
						print_tree_format(OP1(tptr2), tmp, 2);
						process_bec(bec_hd->next, tmp, 2);
						bec_hd->next = NULL;
						bec = bec_hd;
						if (!process_iam(iam_hd->next, tmp))
						{
							PRINTRECCALL(vis_vars, invis_vars, tmp, 2);
						}
						iam_hd->next = NULL;
						iam = iam_hd;
						fprintf(tmp, "\n\t).\n\n\n");
					}
				}
				else
				{
					if (OP1(tptr2))
					{
						print_tree_format(OP1(tptr2), tmp, 1);
					}
					fprintf(tmp, "\t%.1s,\n", tok_lookup(TOK_CUT));
					print_tree_format(OP2(tptr2), tmp, 1);
					process_bec(bec_hd->next, tmp, 1);
					bec_hd->next = NULL;
					bec = bec_hd;
					if (!process_iam(iam_hd->next, tmp))
					{
						PRINTRECCALL(vis_vars, invis_vars, tmp, 1);
					}
					iam_hd->next = NULL;
					iam = iam_hd;
					fprintf(tmp, ".\n\n\n");
				}
			}
		}
		else
		{
			PRINTRECCALL(vis_vars, invis_vars, tmp, 1);
			fprintf(tmp, ".\n\n\n");
		}
	}

	/* Now print out the pnu code*/
	tptr = OP1(tptr);
	tptr2 = OP1(tptr);
	print_tree(tptr2, tmp);

	/* Now what we have is a file, tmp with all the pnp
	   code in it so we run pnp on the tmp file and append
	   that to "file.nl" 
	   pnp_comm will be "PNPtmp file nameAPP"file.nl"
	   where PNP = "~/lee/bin/pnp < "
		 APP = " >> "
	*/
	i = strlen(tempname);
	m = strlen(file_nl);
	j = LEN_APP + m;
	pnp_comm = copy_string(PNP, (LEN_PNP + i + j));
	for(j = 0, k = LEN_PNP; j < i; k++, j++)
	{
		pnp_comm[k] = tempname[j];
	}
	for(j = 0; j < LEN_APP; k++, j++)
	{
		pnp_comm[k] = str[j];
	}
	for(j = 0; j < m; k++, j++)
	{
		pnp_comm[k] = file_nl[j];
	}
	pnp_comm[k] = EOS;

	/* Call system to carry out command
	   NOTE : From what i can gather if system returns 1 it failed,
	   otherwise, it succeeded
	*/

	/* But before we do that close tmp */
	fclose(tmp);

	fprintf(stderr,"\n******** Calling pnp processor (hold on a minute) ********\n");
	if (system(pnp_comm))
	{
		fprintf(stderr,"\n******** Pnp processing failed ********\n");
		m = FALSE;
	}
	else
	{
		fprintf(stderr,"\n******** Pnp processing succeeded ********\n");
		/* Now print out NU prolog code and finished*/
		if (!(filenl = fopen(file_nl, "a")))
		{
			fprintf(stderr,"\nCompiler error : unable to open '%s", file_nl);
			perror("'");
			return FALSE;
		}
		print_tree(OP2(tptr), filenl);
		m = TRUE;
	}
	free(pnp_comm);

	/* If no pnp file requested remove it */
	if (!pnpf)
	{
		unlink(tempname);
	}
	else
	{
		/* Append to pnp file the nu-prolog code and a note */
		if (!(tmp = fopen(tempname, "a")))
		{
			fprintf(stderr,"\nCompiler error : unable to open '%s", tempname);
			perror("'");
			return FALSE;
		}
		fprintf(tmp,"\n%c The following NU-Prolog code will disappear after being run through pnp.\n", '%');
		fprintf(tmp,"%c So BEWARE. It is prefered to re-compile the class.\n\n", '%');
		print_tree(OP2(tptr), tmp);
		fclose(tmp);
	}

	/* Do some cleaning up */
	FREE(vis_vars);
	FREE(invis_vars);
	FREE(dbs);
	bec_hd->next = iam_hd->next = NULL;
	free(bec_hd);
	free(iam_hd);

	/* close the final file which is are product */
	fclose(filenl);
	/* If pnp produced error file.nl is inconsistent
	   so remove it
	*/
	if (!m)
	{
		unlink(file_nl);
	}
	free(tempname);
	tempname = NULL;
	return m;

}


/*  Interrupt : is the function that is called if the compiler is interrupted
    during compilation, so as to remove temp files that are used, basically
    a clean up job
*/
void interrupt(int sig)
{
	/* If signal comes in again do normal */
	signal(sig, SIG_DFL);

	if (tempname)
	{
		unlink(tempname);
	}
	unlink(temp2);
	if (file_nl)
	{
		unlink(file_nl);
	}
	fprintf(stderr,"\n\nInterrupted -- %s removed.\n", file_nl);
	exit(1);
}

/* Use_eq_or_is : given the parse tree this function works out that if a
   variable was to be assigned the value the tree represents should an
   equal sign (=) or the arithmetic is be used
   Returns either EQ or IS to indicate answer.
*/
static char use_eq_or_is(TREEPTR root)
{
	int op;
	TREEPTR tptr = root;

	while (tptr)
	{
		op = OP(tptr);
		if (op >= TOK_VAR && op <= TOK_CONSTANT)
		{
			return EQ;
		}
	
		if (op >= TOK_MOD && op < TOK_END_TABLE)
		{
			return IS;
		}
	
		if (op == TOK_OBLK)
		{
			return EQ;
		}
	
		if (op == TOK_OPAR )
		{
			tptr = OP2(tptr);
		}
		else
		{
			if (op == TOK_CPAR)
			{
				tptr = OP1(tptr);
			}
			else
			{
				return EQ;
			}
		}
	}
	return EQ;
}

/* Process_bec : does what the name suggests, processing a list of
   becomes and produces the correct code for them
*/
static void process_bec(TLISTPTR beclist, FILE *fp, int n)
{
	int len, len2;
	VARPTR var;
	TREEPTR tree;
	char *str;
	TLISTPTR lptr, lptr2;

	lptr = beclist;

	while (lptr)
	{
		var = VART(OP1(lptr->tree));
		if (var->value)
		{
			fprintf(stderr,"Warning : becomes has been used on the same variable %s more than once using the later call\n", var->varname);
		}
		if (!(OP(OP2(lptr->tree)) == TOK_VAR))
		{
			var_counter++;
			for(len = 0, len2 = var_counter; len2; len++)
			{
				len2 /= 10;
			}
			str = copy_string(VAR_TMP, (LEN_VAR_TMP + len));
			len = LEN_VAR_TMP;
			itoa(str, len, var_counter);
			tree = build_var(var_insert(str, len));
			var->value = tree;
			if (forget)
			{
				fprintf(fp, "\n");
				print_tabs(fp, n);
				fprintf(fp, "%s", str);
				forget = FALSE;
			}
			else
			{
				fprintf(fp, ",\n");
				print_tabs(fp, n);
				fprintf(fp, "%s", str);
			}
			if(use_eq_or_is(OP2(lptr->tree)) == EQ)
			{
				fprintf(fp, " = ");
			}
			else
			{
				fprintf(fp, " is ");
			}
			print_tree(OP2(lptr->tree), fp);
			free(str);
		}
		else
		{
			str = VART(OP2(lptr->tree))->varname;
			len = strlen(str);
			tree = build_var(var_insert(str, len));
			var->value = tree;
		}
		lptr2 = lptr;
		lptr = lptr->next;
		lptr2->next = NULL;
		lptr2->tree = NULL;
		free(lptr2);
	}
}


/* Process_iam : does what the name suggests, processing a list of
   i_am's and produces the correct code for them
*/
static char process_iam(TLISTPTR iamlist, FILE *fp)
{
	char retvalue = FALSE;
	TREEPTR tree;
	TLISTPTR lptr, lptr2;

	lptr = iamlist;

	while(lptr)
	{
		tree = lptr->tree;
		if (!OP1(tree))
		{
			if (forget)
			{
				forget = FALSE;
				print_tree(OP2(tree), fp);
			}
			else
			{
				fprintf(fp, ",\n");
				print_tree(OP2(tree), fp);
			}
		}
		else
		{
			tree = OP2(tree);
			if (!forget)
			{
				fprintf(fp, ",\n");
			}
			else
			{
				forget = FALSE;
			}
			fprintf(fp, "l_%s(%s, %s, %s", PREDT(tree).functor->faname, MESG, INPUTSTR, ID);
			if (PREDT(tree).args)
			{
				fprintf(fp, ", ");
				print_tree(OP2(PREDT(tree).args), fp);
			}
			else
			{
				fprintf(fp, ")");
			}
		}
		lptr2 = lptr;
		lptr = lptr->next;
		free(lptr2);
		retvalue = TRUE;
	}

	return retvalue;
}

/* Process_bec2 : does what the name suggests, processing a list of
   becomes and produces the correct code for them
   But the code produced is written to two files.
*/
static void process_bec2(TLISTPTR beclist, FILE *fp1, FILE *fp2, int n)
{
	int len, len2;
	VARPTR var;
	TREEPTR tree;
	char *str;
	TLISTPTR lptr, lptr2;

	lptr = beclist;

	while (lptr)
	{
		var = VART(OP1(lptr->tree));
		if (var->value)
		{
			fprintf(stderr,"Warning : becomes has been used on the same variable %s more than once using the later call\n", var->varname);
		}
		if (!(OP(OP2(lptr->tree)) == TOK_VAR))
		{
			var_counter++;
			for(len = 0, len2 = var_counter; len2; len++)
			{
				len2 /= 10;
			}
			str = copy_string(VAR_TMP, (LEN_VAR_TMP + len));
			len = LEN_VAR_TMP;
			itoa(str, len, var_counter);
			tree = build_var(var_insert(str, len));
			var->value = tree;
			if (forget)
			{
				fprintf(fp1, "\n");
				fprintf(fp2, "\n");
				print_tabs2(fp1, fp2, n);
				fprintf(fp1, "%s", str);
				fprintf(fp2, "%s", str);
				forget = FALSE;
			}
			else
			{
				fprintf(fp1, ",\n");
				fprintf(fp2, ",\n");
				print_tabs2(fp1, fp2, n);
				fprintf(fp1, "%s", str);
				fprintf(fp2, "%s", str);
			}
			if(use_eq_or_is(OP2(lptr->tree)) == EQ)
			{
				fprintf(fp1, " = ");
				fprintf(fp2, " = ");
			}
			else
			{
				fprintf(fp1, " is ");
				fprintf(fp2, " is ");
			}
			print_tree2(OP2(lptr->tree), fp1, fp2);
			free(str);
		}
		else
		{
			str = VART(OP2(lptr->tree))->varname;
			len = strlen(str);
			tree = build_var(var_insert(str, len));
			var->value = tree;
		}
		lptr2 = lptr;
		lptr = lptr->next;
		lptr2->next = NULL;
		lptr2->tree = NULL;
		free(lptr2);
	}
}

/* Process_iam2 : does what the name suggests, processing a list of
   i_am's and produces the correct code for them
   But the code produced is written to two files.
*/
static char process_iam2(TLISTPTR iamlist, FILE *fp1, FILE *fp2)
{
	char retvalue = FALSE;
	TREEPTR tree;
	TLISTPTR lptr, lptr2;

	lptr = iamlist;

	while(lptr)
	{
		tree = lptr->tree;
		if (!OP1(tree))
		{
			if (forget)
			{
				forget = FALSE;
				print_tree2(OP2(tree), fp1, fp2);
			}
			else
			{
				fprintf(fp1, ",\n");
				fprintf(fp2, ",\n");
				print_tree2(OP2(tree), fp1, fp2);
			}
		}
		else
		{
			tree = OP2(tree);
			if (!forget)
			{
				fprintf(fp1, ",\n");
				fprintf(fp2, ",\n");
			}
			else
			{
				forget = FALSE;
			}
			fprintf(fp1, "l_%s(%s, %s, %s", PREDT(tree).functor->faname, MESG, INPUTSTR, ID);
			fprintf(fp2, "l_%s(%s, %s, %s", PREDT(tree).functor->faname, MESG, INPUTSTR, ID);
			if (PREDT(tree).args)
			{
				fprintf(fp1, ", ");
				fprintf(fp2, ", ");
				fprintf(stderr," ***** Here with forget = %d\n", forget);
				print_tree2(OP2(PREDT(tree).args), fp1, fp2);
			}
			else
			{
				fprintf(fp1, ")");
				fprintf(fp2, ")");
			}
		}
		lptr2 = lptr;
		lptr = lptr->next;
		free(lptr2);
		retvalue = TRUE;
	}

	return retvalue;
}

/* Print_tree : prints out the tree passed to it to the file also
   passed to it, no formatting what so all occurs while printing
*/
static void print_tree(TREEPTR root, FILE *fp)
{
	int op;
	FAPTR pred;
	TLISTPTR list;

	if (root)
	{
		op = OP(root);
		if (op < 0)
		{
			print_tree(OP1(root), fp);
			print_tree(OP2(root), fp);
			return;
		}

		if (op >= TOK_CUT && op <= TOK_PIPE)
		{
			print_tree(OP1(root), fp);
			if (!forget)
			{
				fprintf(fp, "%.1s ", tok_lookup(op));
				if (op == TOK_FULLSTOP)
				{
					fprintf(fp, "\n\n");
				}
			}
			forget = FALSE;
			print_tree(OP2(root), fp);
			return;
		}

		if (op < TOK_START_TABLE )
		{
			if (op == TOK_VAR)
			{
				fprintf(fp, "%s ", VART(root)->varname);
				return;
			}

			if (op == TOK_ATOM)
			{
				pred = PREDT(root).functor;
				if (pred->defined && !pred->sys)
				{
					fprintf(fp, "%s_%s(", classname, pred->faname);
					if (PREDT(root).args)
					{
						print_tree(OP1(OP2(PREDT(root).args)), fp);
						fprintf(fp, ", %s, %s", INPUTSTR, ID);
					}
					else
					{
						fprintf(fp, "%s, %s", INPUTSTR, ID);
					}
					PRINTVARS(vis_vars, fp);
					PRINTVARS(invis_vars, fp);
					PRINTVARS(dbs, fp);
					fprintf(fp, ", %s) ", MESG);
				}
				else
				{
					fprintf(fp, "%s", pred->faname);
					if (strncmp(pred->faname, CALLCLASS, LEN_CCLASS))
					{
						print_tree(PREDT(root).args, fp);
					}
					else
					{
						fprintf(fp, "(%s, ", MESG);
						print_tree(OP2(PREDT(root).args), fp);
					}
					fprintf(fp, " ");
				}
				return;
			}

			fprintf(fp, "%s ", CONSTT(root)->constant);
			return;
		}

		if (op > TOK_START_TABLE && op < TOK_END_TABLE)
		{
			print_tree(OP1(root), fp);
			fprintf(fp, "%s", tok_lookup(op));
			if (op != TOK_GROUND)
			{
				fprintf(fp, " ");
			}
			print_tree(OP2(root), fp);
			return;
		}

                if (op == TOK_DBS)
                {
                        list = LIST(root);

                        while(list)
                        {
                                print_tree(list->tree, fp);
                                list = list->next;
                        }
                        return;
                }

                if (op == TOK_PNU)
                {
                        list = LIST(root);
			if (list)
			{
				if (list->tree)
				{
					print_tree(OP1(list->tree), fp);
					fprintf(fp, "%s ", tok_lookup(OP(list->tree)));
					fprintf(fp, "%s_%s", classname,  PREDT(OP2(list->tree)).functor->faname);
					print_tree(PREDT(OP2(list->tree)).args, fp);
					fprintf(fp, ", i, i");
					PRINTVARSMODES(vis_vars, fp);
					PRINTVARSMODES(invis_vars, fp);
					PRINTVARSMODES(dbs, fp);
					fprintf(fp, ", i).\n");
					list = list->next;
				}
                        	while(list)
                        	{
					if (list->tree && OP(list->tree) == TOK_RULE)
					{
						print_tree(OP1(list->tree), fp);
						fprintf(fp, " %s\n", tok_lookup(TOK_RULE));
						print_tree_format(OP2(list->tree), fp, 1);
						fprintf(fp, ".\n\n\n");
					}
					else
					{
                                		print_tree(list->tree, fp);
						fprintf(fp, ".\n\n\n");
					}
                                	list = list->next;
                        	}
			}
                        return;
                }

                if (op == TOK_NU)
                {
                        list = LIST(root);
			/* Print out when declaration */
			if (list && list->tree)
			{
				print_tree(list->tree, fp);
			}
			list = list->next;

                        while(list)
                        {
				if (list->tree && OP(list->tree) == TOK_RULE)
				{
					print_tree(OP1(list->tree), fp);
					fprintf(fp, " %s\n", tok_lookup(TOK_RULE));
					print_tree_format(OP2(list->tree), fp, 1);
					fprintf(fp, ".\n\n\n");
				}
				else
				{
                               		print_tree(list->tree, fp);
					fprintf(fp, ".\n\n\n");
				}
                               	list = list->next;
                        }
                        return;
                }

		if (op == TOK_MESSAGE)
		{
			fprintf(fp, "%s", SEND);
			print_tree(OP1(root), fp);
			fprintf(fp, ",");
			print_tree(OP2(root), fp);
			fprintf(fp, ")");
			return;
		}

		if (op == TOK_BECOMES)
		{
			bec->next = NEW(TLIST);
			bec = bec->next;
			bec->next = NULL;
			bec->tree = root;
			forget = TRUE;
			return;
		}

		if (op == TOK_IAM)
		{
			iam->next = NEW(TLIST);
			iam = iam->next;
			iam->next = NULL;
			iam->tree = root;
			forget = TRUE;
			return;
		}

		print_tree(OP1(root), fp);
		print_tree(OP2(root), fp);
		return;
	}

	return;
}


/* Print_tree2 : prints out the tree passed to it to the files also
   passed to it, no formatting what so all occurs while printing
*/
static void print_tree2(TREEPTR root, FILE *fp1, FILE *fp2)
{
	int op;
	FAPTR pred;
	TLISTPTR list;

	if (root)
	{
		op = OP(root);
		if (op < 0)
		{
			print_tree2(OP1(root), fp1, fp2);
			print_tree2(OP2(root), fp1, fp2);
			return;
		}

		if (op >= TOK_CUT && op <= TOK_PIPE)
		{
			print_tree2(OP1(root), fp1, fp2);
			if (!forget)
			{
				fprintf(fp1, "%.1s ", tok_lookup(op));
				fprintf(fp2, "%.1s ", tok_lookup(op));
				if (op == TOK_FULLSTOP)
				{
					fprintf(fp1, "\n\n");
					fprintf(fp2, "\n\n");
				}
			}
			forget = FALSE;
			print_tree2(OP2(root), fp1, fp2);
			return;
		}

		if (op < TOK_START_TABLE )
		{
			if (op == TOK_VAR)
			{
				fprintf(fp1, "%s ", VART(root)->varname);
				fprintf(fp2, "%s ", VART(root)->varname);
				return;
			}

			if (op == TOK_ATOM)
			{
				pred = PREDT(root).functor;
				if (pred->defined && !pred->sys)
				{
					fprintf(fp1, "%s_%s(", classname, pred->faname);
					fprintf(fp2, "%s_%s(", classname, pred->faname);
					if (PREDT(root).args)
					{
						print_tree2(OP1(OP2(PREDT(root).args)), fp1, fp2);
						fprintf(fp1, ", %s, %s", INPUTSTR, ID);
						fprintf(fp2, ", %s, %s", INPUTSTR, ID);
					}
					else
					{
						fprintf(fp1, "(%s, %s", INPUTSTR, ID);
						fprintf(fp2, "(%s, %s", INPUTSTR, ID);
					}
					PRINTVARS2(vis_vars, fp1, fp2);
					PRINTVARS2(invis_vars, fp1, fp2);
					PRINTVARS2(dbs, fp1, fp2);
					fprintf(fp1, ", %s) ", MESG);
					fprintf(fp2, ", %s) ", MESG);
				}
				else
				{
					fprintf(fp1, "%s", pred->faname);
					fprintf(fp2, "%s", pred->faname);
					if (strncmp(pred->faname, CALLCLASS, LEN_CCLASS))
					{
						print_tree2(PREDT(root).args, fp1, fp2);
					}
					else
					{
						fprintf(fp1, "(%s, ", MESG);
						fprintf(fp2, "(%s, ", MESG);
						print_tree2(OP2(PREDT(root).args), fp1, fp2);
					}
					fprintf(fp1, " ");
					fprintf(fp2, " ");
				}
				return;
			}

			fprintf(fp1, "%s ", CONSTT(root)->constant);
			fprintf(fp2, "%s ", CONSTT(root)->constant);
			return;
		}

		if (op > TOK_START_TABLE && op < TOK_END_TABLE)
		{
			print_tree2(OP1(root), fp1, fp2);
			fprintf(fp1, "%s", tok_lookup(op));
			fprintf(fp2, "%s", tok_lookup(op));
			if (op != TOK_GROUND)
			{
				fprintf(fp1, " ");
				fprintf(fp2, " ");
			}
			print_tree2(OP2(root), fp1, fp2);
			return;
		}

                if (op == TOK_DBS)
                {
                        list = LIST(root);

                        while(list)
                        {
                                print_tree2(list->tree, fp1, fp2);
                                list = list->next;
                        }
                        return;
                }

		if (op == TOK_MESSAGE)
		{
			fprintf(fp1, "%s", SEND);
			fprintf(fp2, "%s", SEND);
			print_tree2(OP1(root), fp1, fp2);
			fprintf(fp1, ",");
			fprintf(fp2, ",");
			print_tree2(OP2(root), fp1, fp2);
			fprintf(fp1, ")");
			fprintf(fp2, ")");
			return;
		}

		if (op == TOK_BECOMES)
		{
			bec->next = NEW(TLIST);
			bec = bec->next;
			bec->next = NULL;
			bec->tree = root;
			forget = TRUE;
			return;
		}

		if (op == TOK_IAM)
		{
			iam->next = NEW(TLIST);
			iam = iam->next;
			iam->next = NULL;
			iam->tree = root;
			forget = TRUE;
			return;
		}

		print_tree2(OP1(root), fp1, fp2);
		print_tree2(OP2(root), fp1, fp2);
		return;
	}

	return;
}

/* Used in the following functions to indicate no printing of tabs */
static char notabs;

/* Print_tree_format : prints the tree passed to it to the file passed
   to it indenting by some number (tabs) tab stops
   Thus, some formatting occurs, indenting of if's, -> etc.
*/
static void print_tree_format(TREEPTR root, FILE *fp, int tabs)
{
	TREEPTR tptr;
	int newtabs = tabs;

	while( root && OP(root) == TOK_COMMA)
	{
		print_tree_format(OP1(root), fp, tabs);
		if (!forget)
		{
			fprintf(fp, ",\n"); 
		}
		notabs = forget;
		forget = FALSE;
		print_tree_format(OP2(root), fp, tabs);
		return;
	}
	if (!root)
		return;
	
	if (!notabs || !forget)
	{
		print_tabs(fp, tabs);
	}
	if (OP(root) == TOK_THEN)
	{
		print_tree(OP1(root), fp);
		fprintf(fp, " then\n");
		tptr = OP2(root);
		if (tptr && OP(tptr) == TOK_ELSE)
		{
			print_tree_format(OP1(tptr), fp, ++newtabs);
			fprintf(fp, "\n");
			print_tabs(fp, tabs);
			fprintf(fp, "else\n");
			if (OP2(tptr) && OP(OP2(tptr)) == TOK_CPAR)
			{
				print_tree_format(OP1(OP2(tptr)), fp, newtabs);
				fprintf(fp, "\n");
				print_tabs(fp, tabs);
				fprintf(fp, ")");
			}
			else
			{
				print_tree_format(OP2(tptr), fp, newtabs);
			}
		}
		else
		{
			print_tree_format(tptr, fp, ++newtabs);
		}
		return;

	}
	if (OP(root) == TOK_NIF)
	{
		print_tree(OP1(root), fp);
		fprintf(fp, " ->\n");
		tptr = OP2(root);
		if (tptr && OP1(tptr) && OP(OP1(tptr)) == TOK_SEMICOLON)
		{
			tptr = OP1(tptr);
			print_tree_format(OP1(tptr), fp, ++newtabs);
			fprintf(fp, "\n");
			print_tabs(fp, tabs);
			fprintf(fp, ";\n");
			print_tree_format(OP2(tptr), fp, newtabs);

		}
		else
		{
			print_tree_format(OP1(tptr), fp, ++newtabs);
		}
		fprintf(fp, "\n");
		print_tabs(fp, tabs);
		fprintf(fp, ")");
		return;
	}
	print_tree(root, fp);
	return;
}

/* Print_tree_format2 : prints the tree passed to it to the files passed
   to it indenting by some number (tabs) tab stops
   Thus, some formatting occurs, indenting of if's, -> etc.
*/
static void print_tree_format2(TREEPTR root, FILE *fp1, FILE *fp2, int tabs)
{
	TREEPTR tptr;
	int newtabs = tabs;

	while( root && OP(root) == TOK_COMMA)
	{
		print_tree_format2(OP1(root), fp1, fp2, tabs);
		if (!forget)
		{
			fprintf(fp1, ",\n"); 
			fprintf(fp2, ",\n"); 
		}
		notabs = forget;
		forget = FALSE;
		print_tree_format2(OP2(root), fp1, fp2, tabs);
		return;
	}
	if (!root)
		return;
	
	if (!notabs || !forget)
	{
		print_tabs2(fp1, fp2, tabs);
	}
	if (OP(root) == TOK_THEN)
	{
		print_tree2(OP1(root), fp1, fp2);
		fprintf(fp1, " then\n");
		fprintf(fp2, " then\n");
		tptr = OP2(root);
		if (tptr && OP(tptr) == TOK_ELSE)
		{
			print_tree_format2(OP1(tptr), fp1, fp2, ++newtabs);
			fprintf(fp1, "\n");
			fprintf(fp2, "\n");
			print_tabs2(fp1, fp2, tabs);
			fprintf(fp1, "else\n");
			fprintf(fp2, "else\n");
			print_tree_format2(OP2(tptr), fp1, fp2, newtabs);
		}
		else
		{
			print_tree_format2(tptr, fp1, fp2, ++newtabs);
		}
		return;

	}
	if (OP(root) == TOK_NIF)
	{
		print_tree2(OP1(root), fp1, fp2);
		fprintf(fp1, " ->\n");
		fprintf(fp2, " ->\n");
		tptr = OP2(root);
		if (tptr && OP1(tptr) && OP(OP1(tptr)) == TOK_SEMICOLON)
		{
			tptr = OP1(tptr);
			print_tree_format2(OP1(tptr), fp1, fp2, ++newtabs);
			fprintf(fp1, "\n");
			fprintf(fp2, "\n");
			print_tabs2(fp1, fp2, tabs);
			fprintf(fp1, ";\n");
			fprintf(fp2, ";\n");
			print_tree_format2(OP2(tptr), fp1, fp2, newtabs);

		}
		else
		{
			print_tree_format2(OP1(tptr), fp1, fp2, ++newtabs);
		}
		fprintf(fp1, "\n");
		fprintf(fp2, "\n");
		print_tabs2(fp1, fp2, tabs);
		fprintf(fp1, ")");
		fprintf(fp2, ")");
		return;
	}
	print_tree2(root, fp1, fp2);
	return;
}

/* Print_tabs : print n tabs to file fp */
static void print_tabs(FILE *fp, int n)
{
	while (n > 0)
	{
		fprintf(fp, "\t");
		n--;
	}
}

/* Print_tabs2 : print n tabs to files fp1, fp2 */
static void print_tabs2(FILE *fp1, FILE *fp2, int n)
{
	while (n > 0)
	{
		fprintf(fp1, "\t");
		fprintf(fp2, "\t");
		n--;
	}
}

