
/*   This program was originally written in the GLUE language
     by Ashish Gupta and Sanjai Tiwari, with modifications by Marcia Derr.
     It was (rather loosely) translated into CORAL by R. Ramakrishnan.   */


%  Copyright (c)        1991 by Ashish Gupta and Sanjai Tiwari
%               1992 Modifications by Marcia A. Derr

% * High level overview.  In a building, there are constructible
% objects: beams, columns, ewalls, iwalls and slabs. Each of these
% objects needs different kinds of resources in order for it to be
% built. Given the following restrictions, we need to schedule the
% construction of a given building in terms of the objects that comprise
% the building:
%
%  ** Some objects support others (eg: beam on 2nd floor depends on
%     some beams in the first floor). So the order of constructing
%     them has to be respected.
%  ** Within each object, given amounts of some resource R1 needs to be
%     used before resource R2 can be used (eg: All units of concrete
%     better be used before any finishing work starts).
%  ** the tasks scheduled for any day should not use more than the
%     amount of the resource available per day.
%
% Keeping the above restrictions in mind, we do the following:
%
%  ** treat the construction of every object as a high level task.
%  ** Using the dependencies, find the high level tasks that are
%     feasible at the current time. (high level task corresponds
%     to each object)
%  ** Decompose each feasible tasks into subtasks on the basis of the
%     resources it uses (subtasks correspond to <object,resource>
%     pairs). 
%  ** From these subtasks select those that can be done keeping in mind
%     the constraints specified by resource ordering and resource
%     availability. 


% Initialization

module init.
export  indep(f), subtask(bfff).

@make_index requires(bfff).
@make_index supp_by_facts(bf).
@make_index same_resource(fb).


@pipelining.

	
	dep(X) :- supp_by_facts(X,Y).

	indep(T) :- constructable1(T), not dep(T).

subtask(Task,Resource,Units,Subtask_Id) :- 
        same_resource(Class,Task) ,
	requires(Class, Resource, Units, Days), 
	gen(Days,Subtask_Id).


end_module. %init

module misc.
export gen(bf).
@index_deltas -.

gen(D,1).
gen(D,N) :- gen(D,M), N=M+1, N<=D.

end_module. %misc


module pconstruct.
export feasible_subtask(bffff),  result(fffff), 
feasible_task(bf), complete_subtask(bffff), 
violates_res_prec(bffff), 
complete_subtask_subset(bfff), complete_task(bf),
choose_first_subtask(bffff).

@index_deltas -.
%@non_ground_facts -.

@make_index choose_first_subtask_bffff (A,B,C,D,E) (A,B,C,D). 

@allowed_adornments feasible_subtask[bffff].
@allowed_adornments feasible_task[bf].
@allowed_adornments complete_subtask[bffff].
@allowed_adornments complete_task[bf].
@allowed_adornments violates_res_prec[bffff].
@allowed_adornments complete_subtask_subset[bfff].
@allowed_adornments choose_first_subtask[bffff].



result(L1,T,R,U,SId) :- layer(L), L1=L+1, complete_subtask(L1,T,R,U,SId).


% Definition of feasible tasks at layer L

	feasible_task(1,T) :- 
		indep(T).

	feasible_task(L,T) :-  LL=L-1, LL>0,
		complete_task(LL,Y), supp_by_facts(T,Y), constructable1(T).


% Definition of feasible subtasks at layer L

	feasible_subtask_cand(L,T,R,U,SId) :- 
		feasible_task(L,T), subtask(T,R,U,SId).

	feasible_subtask_cand(L,T,R,U,SId) :- LL=L-1, LL>0,
		feasible_subtask(LL,T,R,U,SId).

	feasible_subtask(L,T,R,U,SId) :- 
		feasible_subtask_cand(L,T,R,U,SId),
		not complete_subtask(L,T,R,U,SId).


	complete_task(L1,T) :-  L=L1-1, L>0,
		feasible_subtask(L,T,_,_,_), not feasible_subtask(L1,T,_,_,_). 



% Identify tasks that are complete at beginning of layer L1

	complete_subtask(L1,T,R,U,SId) :-  L=L1-1, L>0,
		choose_first_subtask(L,T,R,U,SId), 
		not violates_res_prec(L,T,R,U,SId).


	choose_first_subtask(L,T,R,U,min(<SId>)) :-
		feasible_subtask(L,T,R,U,SId).


	violates_res_prec(L,T,R,U,SId) :-
		complete_subtask_subset(L,T,R1,UnitsDone) ,
		same_resource(Class,T) ,
		res1_before_res2(Class,R1,Units,R) ,
		Units > UnitsDone.

% Update book-keeping on resources consumed by completed subtasks

	complete_subtask_subset(1,T,R,0).

	complete_subtask_subset(L1,T,R,Units) :- 
		complete_subtask(L1,T,R,U,SId), L=L1-1,L>0,
		complete_subtask_subset(L,T,R,Uold), 
		Units = U + Uold.



	
end_module. % pconstruct

module psupported.
export constructable1(f).

constructable1(X) :- columns(_,X,_,_).
constructable1(X) :- beams(_,X,_,_).
constructable1(X) :- ewalls(_,X,_,_,_,_).
constructable1(X) :- iwalls(_,X,_,_,_,_).
constructable1(X) :- slabs(X,_,_,_,_).

end_module.  %psupported


