/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************
	CORAL Software :: U.W.Madison

	interp.C

	Contains routines to 'interpret' coral modules into an
	intermediate representation that can be executed when
	there is a query.

 ***********************************************************************/

#include "externs.h"
#include "interp.h"
#include "pipelined.h"
#include "derived-rel.h"
#include "profile.h"
#include "apply.h"
#include "annotations.h"
#include "globals.h"
#include <stdlib.h>
#include <string.h>
#include "ordsearch.h"

static long display_rules = 0 ;

extern void sprint_pred_name(char *buf, char *name, char *adorn);
extern void sprint_pred_name(char *buf, char *name, BitVector *adorn);
extern void sprint_magic_name(char *buf, char *name, BitVector *adorn);
extern void sprint_magic_name(char *buf, char *predname, char *adorn);
extern int print_new_line_after_rule;
extern int print_adorn_for_pred;

extern int getSuccessBacktrackPoint(SemiNaive *sn, Clause *clause, int);
extern int getfirstFailureBacktrackPoint(SemiNaive *sn, int rhs_index, int);
extern int getnextFailureBacktrackPoint(SemiNaive *sn, int rhs_index, int);
extern int has_grouping_args(ArgList &args);
// Defined in annotations.C
extern void add_make_index_annotation(PredAnnotations* &, Symbol *pred,
				      BitVector *bv,
				      ArgList *al1, ArgList *al2);

extern void AddProfileInfo(ProfileInfo *A, ProfileInfo *B) ;

// Defined in util.C
extern int is_magic(Name name);
extern int is_supp(Name name);

extern DatabaseStruct BuiltinDB ;

ArgList *MagicArgs = NULL;
BindEnv *MagicBindEnv = NULL;
Relation *MagicRelation = NULL;


// points to current moduleInfo defaults
ModuleInfo *CurrModuleInfo = NULL;

int get_offset(Table *F, Name name, int arity, int print_error)
{
	for (int i = 0; i< F->numRels; i++)
	    if ((F->rels[i].relArity == arity) &&
		(F->rels[i].relName == name))
		break ;

// The offset returned should be within the array bounds.
	if (i >= F->numRels) {
	    if (print_error) {
	      fprintf(exEnv.error_file,"Error: Couldn't find relation %s with arity %d\n",
			name->string(), arity);
	    }
	    return(-1); // NOTE: divesh added this
	}
	ASSERT((i < F->numRels)) ;
	return(i) ;
}


// NOTE : Predicate field in Literal is often not set !! So use pred field.
int get_offset(Table *F, Literal *pred, int print_error)
{
	return get_offset(F, pred->pred, pred->arity(), print_error) ;
}

void Table::AddValue(int i, Name relName, int arity)
{
	rels[i].AssignVal(relName, arity) ;
}

Table::Table(Table *F)
{
	numRels = F->numRels ;
	rels = new RelationStruct[numRels];
	for (int i = 0; i< numRels; i++)
	 rels[i].AssignVal(F->rels[i].relName, F->rels[i].relArity) ;
	next = NULL;
	context = NULL;
	moduledata = NULL;
}

void RelationStruct::AssignVal(Name name, int arity)
{
	relName = name ;
	relArity = arity ;
}

RuleInfoArray::RuleInfoArray()
{
	numRules = 0 ;
	rInfoPtrs = NULL;
}

void RuleInfoArray::AddRule(RuleInfo *rInfo)
{
	RuleInfo **temp = rInfoPtrs ;
	rInfoPtrs = new RuleInfo*[++numRules] ;

	for (int i=0; i< numRules-1; i++)
		rInfoPtrs[i] = temp[i] ;

	rInfoPtrs[i] = rInfo ;
	if (numRules > 1) delete [] temp ;

}

void RuleInfoArray::printon(FILE *outf)
{
	for (int i=0; i< numRules; i++)
	    fprintf(outf, "%s\n", rInfoPtrs[i]->pInfo.rule) ;

}

// Constructor for RMarkArray. All RMarks * are initially NULL.
// Even RMarks are 'old' and odd RMarks are 'new'.
// Size should be even !!
RMarkArray::RMarkArray(int size)
{
	markList = new RMark*[size] ;
	for (int i = 0; i< size; i++) {
		markList[i] = NULL ;
	}
	array_size = size ;
}

// Copies a new RMark at position i+1 to the old RMark at position i.
// Obviously, i should be even.
void RMarkArray::New_To_Old(int i)
{
    if (i < array_size - 1) {
	markList[i] = markList[i+1] ;
	markList[i+1] = NULL;
    }
}

RMarkArray::~RMarkArray()
{
	delete [] markList ;
}

SemiNaive::SemiNaive()
{
  scc = NULL;
  rInfo = NULL ;
  first = 0 ;
  markOffsets = NULL ;
  predOrders = NULL ;
  firstFailureBacktrack = NULL;
  nextFailureBacktrack = NULL;  
  successBacktrack = 0;
  recursive = 0 ;
  indices = NULL;
  ignore_this_rule = 0;
}


/** WARNING: The following routine damages the varlists that are
	passed to it.
**/
VarLink *intersect_var_lists(VarLink *varlist1, VarLink *varlist2)
{
    VarLink *ptr1, *ptr2;
    VarLink *head=NULL, *prev=NULL;
    for(ptr1=varlist1; ptr1;) {
	for(ptr2=varlist2; ptr2; ptr2=ptr2->next) {
	    if (ptr1->arg->var == ptr2->arg->var)
		break;
	}
	if (ptr2) {
	    if (! head) head = ptr1;
	    if (! prev) prev = ptr1;
	    else {
		prev->next = ptr1;
		prev = ptr1;
	    }
	    ptr1 = ptr1->next;
	    prev->next = NULL;
	}
	else ptr1 = ptr1->next;
    }
    return head;
}

VarLink * build_var_list(Literal * pred, VarLink *list)
/*
 * Construct a list of all the variables in the bound arguments of the given
 * predicate.  Format the list as a Arg * for a predicate.
 */
{
    FOR_EACH_ARG(pred_arg, pred->args) {
	list = pred_arg->var_list(list);
    } END_EACH_ARG
    return list;
}


/** WARNING: The following routine damages the varlists that are
	passed to it.
**/
void create_an_annotation(PredAnnotations *&annos, 
			  Literal *lit, VarLink *var_list)
{
    VarLink *link;
    int var_count = 0;
    for (link = var_list ; link != NULL; link = link->next)
	var_count++;

    // no point generating an index with all arguments free !!
    if (!var_count) return ;

    ArgList *arglist = ArgList::New(var_count);

    for (link = var_list; var_count > 0; var_count -- ) {
	VarLink *next = link->next;
	(*arglist)[var_count-1]= link->arg;
	delete link;
	link = next;
    }
   
    Name name;
    if (lit->kind == COR_MAGIC) {
	char *buf = new char [lit->pred->length()+10];
	sprint_magic_name(buf,lit->pred->string(),"");
	name = EnterSymbol(buf);
    }
    else if (lit->kind == COR_DONE_MAGIC) {
	char *buf = new char [lit->pred->length()+15];
	sprintf(buf, "done_");
	char *buf1 = buf + strlen(buf);
	sprint_magic_name(buf1,lit->pred->string(),"");
	name = EnterSymbol(buf);
    }
    else {
	name = lit->pred;
    }


    PredAnnotations *tmp_anno = annos ;

    add_make_index_annotation(annos,name, &(lit->adorn),&(lit->args),arglist);

}

VarLink *vars_between(struct rule *adrule, int start, int end,
					VarLink *rest, int *predOrders = NULL)
{
    int i;
    VarLink *result=rest;
    for (i=start; i <= end; i++)
      if (predOrders)
	result = build_var_list(adrule->preds[predOrders[i]], result);
      else 
	result = build_var_list(adrule->preds[i], result);
    return result;
}

int is_a_builtin(Name name)
{
  int ret_val = 1;
  Association *ptr1 ;
  ptr1 = SymbolLookup(BuiltinDB.RelationTable, name) ;

  if (HashNone(ptr1->arg)) ret_val = 0;
  //else ret_val = !((BuiltinRelation *)(ptr1->val))->is_printable_builtin() ;

  return ret_val;
}

static void generateIndexes(SemiNaive *snrule)
{
  int i;
  VarLink *leading_vars, *indexing_vars ;

#define RULEPTR snrule->rInfo->rule  
#define PREDINDEX(i) snrule->predOrders[i]

  /* The following code indexes predicates in the rule */
  for(i=1; i < RULEPTR->num_preds; i++)
    // no point building an index on an arithmetic predicate
    if (!is_a_builtin(RULEPTR->preds[PREDINDEX(i)]->pred)) {
      leading_vars = vars_between(RULEPTR,0,i-1,NULL, snrule->predOrders);
      indexing_vars = intersect_var_lists(
	build_var_list(RULEPTR->preds[PREDINDEX(i)],NULL), leading_vars);
      create_an_annotation(snrule->scc->module->index_annotations,
			   RULEPTR->preds[PREDINDEX(i)], indexing_vars);
    }

  // if there is a grouping argument in the head, create an index on the
  // the head predicate that is bound on all arguments except the grouping
  // argument

  if (has_grouping_args(RULEPTR->head->args)) {
    char *buf = new char[RULEPTR->head->args.count() + 1];
    for(i=0;i< RULEPTR->head->args.count();i++)
      if(RULEPTR->head->args[i]->kindof()==COR_GROUPING)
	buf[i] = 'f';
      else buf[i] = 'b';
    buf[i] = '\0';
    ArgList *new_arglist = ArgList::New(1);
    (*new_arglist)[0] = EnterSymbol(buf);


    PredAnnotations *tmp_anno = snrule->scc->module->index_annotations;

    add_make_index_annotation(snrule->scc->module->index_annotations,
			      RULEPTR->head->pred, &(RULEPTR->head->adorn),
			      new_arglist, NULL);
  }

#undef RULEPTR
#undef PREDINDEX
}

static void initPredOrders(SemiNaive *sn)
{
  sn->predOrders[0] = sn->first;
  for (int i = 1; i < sn->rInfo->num_literals; i++) {
    if ( i <= sn->first)
      sn->predOrders[i] = i-1;
    else sn->predOrders[i] = i ;
  }
}

static void initFailureBacktrack(SemiNaive *sn, int ng_facts)
{
  for (int i = sn->rInfo->num_literals-1; i >0; i--) {
    sn->firstFailureBacktrack[i] = getfirstFailureBacktrackPoint(sn, i, ng_facts);

    sn->nextFailureBacktrack[i] =  getnextFailureBacktrackPoint(sn, i, ng_facts) ;
  }

  sn->firstFailureBacktrack[0] = -1;
  sn->nextFailureBacktrack[0] = -1;
}

void SemiNaive::Add_Rule_Info(SccInfo *in_scc, RuleInfo *rule_info,
			      Clause *clause, int non_ground_facts)
{
  scc = in_scc;
  rInfo = rule_info;
  markOffsets = new int[(rInfo->num_literals)*2];
  indices = new int [rInfo->num_literals];

  predOrders = new int[rInfo->num_literals];
  initPredOrders(this);

  // NOTE : predOrders MUST be assigned BEFORE the backtrack arrays
  firstFailureBacktrack = new int [rInfo->num_literals];
  nextFailureBacktrack = new int [rInfo->num_literals];
  initFailureBacktrack(this, non_ground_facts);
  
  successBacktrack = getSuccessBacktrackPoint(this, clause, non_ground_facts);
  
  generateIndexes(this);

  clean_cache();

  // this is an optimization for supplementary magic with indexing
  // the magic rules are ignored, and the magic facts are generated
  // along withthe firing of the goalid() literal of the supp rule
  //            --- Praveen

  if ((in_scc->module->module_info.SupMagicIndexing) && 
      is_magic(clause->head->pred)) ignore_this_rule = 1;

}

void SemiNaive::clean_cache() 
{
	for (int i=0; i < rInfo->num_literals; i++)
	    indices[i] = -1;
}

// This function finds the offset of an RMark in the RMarkArray for an scc
// It depends on the order in which the RMarks are actually allocated, which
// happens in SccInfo::Solve(). The function returns the offset of the first
// RMark for that predicate (i.e.) the old RMark. The new RMark is in the
// adjacent position.
static int Find_RMark_Offset(SccInfo *scc_info, Predicate *rpred)
{
    Predicate **pred_ptrs = scc_info->pred_ptrs;
    int total_numpreds = scc_info->numPreds + scc_info->num_lowerPreds;
    for (int i=0; i < total_numpreds; i++) {
	if (pred_ptrs[i] == rpred) return (2*i);
    }

    // Default --- this should never be reached !!
    fprintf(exEnv.error_file,
	    "CORAL:: Error - Find_RMark_Offset - can't find %s(arity %d)\n",
	    rpred->name->string(), rpred->arity);
    return(-1) ;
}

/********************
static int Find_RMark_Offset(Scc *scc, Predicate *rpred)
{
	Predicate *pred ;
	int i = 0 ;
	for (pred = scc->preds; pred != NULL; pred = pred->next)
		if (pred == rpred) return(i) ;
		else i +=2 ;

	// Default --- this should never be reached !!
	fprintf(exEnv.error_file, 
		"CORAL:: Error - Find_RMark_Offset - can't find %s(arity %d)\n",
		rpred->name->string(), rpred->arity);
	return(-1) ;
}
********************/

static int has_fail_in_body(struct rule *rule)
{ 
    int i;

    for(i=0; i < rule->num_preds; i++)
	if(rule->preds[i]->pred == FailSymbol)
	   return i;
    return -1;

}

Predicate *find_ext_magicquerypred(ParserStruct& parserStruct)
{
    for (Predicate *pred = parserStruct.AllPredicates; pred != NULL; 
				pred=pred->allprednext) {
	if ((!pred->clauses) && (pred->is_magic())) { // found it
	    return pred;
	}
    }
    return NULL;
}

void SccInfo::Create_SN_Structs(ParserStruct& parserStruct)
{

  // First, allocate enough memory to hold all seminaive rule structures
  numSNRules = 0 ;
  Predicate *pred; Clause *clause;
  Predicate *kpred;
  int k;

  for (pred = scc->preds; pred != NULL; pred = pred->next)
    for (clause = pred->clauses; clause != *pred->clause_tail;
	 clause = clause->pnext) {
      if (!clause->seen) continue ;
      if (!clause->recursive) {
	if (! module->module_info.SaveModule) {
	  numSNRules++ ; continue ;
	}
	else {
	  // with SaveModules have to count the number of literals
	  // whose predicates are defined in lower SCCs.
	  // NOTE: something special needs to be done for non-magic modules
	  int thisclausesn = 0;
	  int ipred = 0;
	  for (; ipred < clause->num_preds; ipred++) {
	    // check to see if it is in a lower SCC
	    // versus being an external predicate
	    if ((clause->preds[ipred]->predicate->clauses) ||
		(clause->preds[ipred]->predicate->is_magic())) { 
	      numSNRules++; thisclausesn++;
	    }
	  }
	  if (thisclausesn == 0) { 
	    // all predicates in body are external to module
	    numSNRules++ ; 
	  }
	  continue;
	}
      }
      for (int ipred = 0; ipred <  clause->num_preds; ipred++) {

	/*
	 * Treat negated literals as base !!
	 */
	if (clause->preds[ipred]->negated) continue;

	/** The following is to special case the creation of the seminaive structures
	 * for the case of Ordered Search.  Three special things need to be done
	 * (1) `not p' is always treated as base, even if p is derived.
	 * (2) done_m_p is treated as derived even though there are no rules for it.
	 * (3) any literal following a done_m_p is treated as base, even if p
	 *     is derived. (this also covers case 1)
	 **/

	if (module->module_info.UseOrdSearch) {
	  if ((ipred > 0) && 
	      (clause->preds[ipred-1]->predicate->is_mp_done()))
	    continue;

	  /* Treat mp_done literals as derived */
	  if (clause->preds[ipred]->predicate->is_mp_done())
	    clause->preds[ipred]->predicate->scc = scc ;
	}
	if (clause->preds[ipred]->predicate->scc != scc) {
	  if (module->module_info.SaveModule) {
	    // check to see if it is in a lower SCC
	    // versus being an external predicate
	    if ((clause->preds[ipred]->predicate->clauses) ||
	        	(clause->preds[ipred]->predicate->is_magic())) 
	      numSNRules++;
	  }
	  continue;
	}
	numSNRules++ ;
      }
    }
  sn_rules = new SemiNaive[numSNRules]; 
  sn_offset_arr = new int[numSNRules];
  sn_first_offsets = new int[numSNRules];

  // Now fill in values in seminaive rule structures

  int cur_snr = -1, i ,j ;

  // first find the number of predicates in this Scc
  for (numPreds = 0, pred = scc->preds; pred != NULL; 
			++numPreds, pred = pred->next) ;

  // now fill in the information about lower Sccs, which is useful in
  // the case of SaveModules
  Predicate  *ext_magicquerypred = NULL;
  if (module->numSccs == 1) { // this is the first Scc
    if (module->module_info.SaveModule) {
      ext_magicquerypred = find_ext_magicquerypred(parserStruct);
    }
    if (ext_magicquerypred) num_lowerPreds = 1;
    else num_lowerPreds = 0;
  }
  else { // add the numPreds and num_lowerPreds of the previous scc
    ASSERT((module->numSccs > 1)) ;
    num_lowerPreds = module->sccArray[module->numSccs-2]->num_lowerPreds +
			module->sccArray[module->numSccs-2]->numPreds ;
  }

  // assign the predicate pointers and the offset array
  if (!module->module_info.SaveModule) {
    pred_ptrs = new Predicate *[numPreds] ;
    offset_arr = new int[numPreds];
  }
  else {
    pred_ptrs = new Predicate * [numPreds + num_lowerPreds];
    offset_arr = new int [numPreds + num_lowerPreds];
  }
  for (i = 0, pred = scc->preds; pred != NULL; i++,pred = pred->next){
    pred_ptrs[i] = pred ;
    // allow at least one iteration
  }
  if (module->module_info.SaveModule) {
    // get the predicate pointers from the lower scc
    if (module->numSccs > 1) {
      Predicate **lower_pred_ptrs = 
			module->sccArray[module->numSccs-2]->pred_ptrs;
      ASSERT((i == numPreds));
      for (j=0; j < num_lowerPreds; j++, i++) {
	pred_ptrs[i] = lower_pred_ptrs[j];
      }
    }
    else { // module->numSccs == 1
      ASSERT((num_lowerPreds <= 1));
      if (num_lowerPreds == 1) {
	pred_ptrs[numPreds] = ext_magicquerypred;
      }
    }
  }

  /******************
   * There is a correspondence between the predicate positions in 
   * this->pred_ptrs, and the offsets in the frame given by 
   * this->offset_arr of those predicates in rm_Arr.
   ******************/
  
  for (i=0; i<numPreds; i++)
    offset_arr[i] = get_offset(module->F, pred_ptrs[i]) ;
  if (module->module_info.SaveModule) {
    for (i=numPreds; i < numPreds+num_lowerPreds; i++) 
      offset_arr[i] = get_offset(module->F, pred_ptrs[i]) ;
  }


  // Consider non-recursive clauses in this scc first.

  for (j=0, pred = scc->preds; pred != NULL; ++j, pred = pred->next) {
    for (clause = pred->clauses;
	 clause != *pred->clause_tail;
	 clause = clause->pnext) {
      if (!clause->seen) continue; 	
      if (clause->recursive) continue;

      // we know it is a non-recursive clause by now
      int thisclausesn = 0; 
      if (module->module_info.SaveModule) { 
	// have to generate SN rules corresponding to lower predicates
	int ipred, jpred;
	for (ipred=0; ipred < clause->num_preds; ipred++) {
	  // all body literals are in lower sccs or external predicates
	  ASSERT((clause->preds[ipred]->predicate->scc != scc));
	  if ((clause->preds[ipred]->predicate->clauses) ||
		(clause->preds[ipred]->predicate->is_magic())) { 
	    // lower scc or containing MagicQueryPredKind without any rules

	    thisclausesn++; // counting number of sn rules generated 
	    cur_snr++ ; 
	    ASSERT((cur_snr < numSNRules));
	    sn_rules[cur_snr].first = ipred;
	    sn_rules[cur_snr].recursive = 0 ;
	    sn_offset_arr[cur_snr] = j ;
	    for (k=0, kpred = scc->preds; kpred != NULL; ++k, kpred = kpred->next)
	      if (kpred == clause->preds[ipred]->predicate) {
		sn_first_offsets[cur_snr] = k;
		break;
	      }
	    sn_rules[cur_snr].Add_Rule_Info(this,
		       module->ruleArray->rInfoPtrs[clause->rule_number],
		        clause, module->module_info.NonGroundFacts);
	    for (jpred = 0; jpred < clause->num_preds; jpred++) {
	      Literal *bodypred = clause->preds[jpred];

	      if (ipred == jpred) {
	        // NOTE: check whether external predicates have 
		// such rmarks defined on it or not!
		// they may be needed 
		sn_rules[cur_snr].markOffsets[2*jpred] =
		  	Find_RMark_Offset(this, bodypred->predicate);
	      }
	      else 
		sn_rules[cur_snr].markOffsets[2*jpred] = -1;
	      
	      if (bodypred->predicate->clauses || 
			bodypred->predicate->is_magic()) {
		int tmp = Find_RMark_Offset(this, bodypred->predicate);
		if (ipred  > jpred)
		  sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp;
		else
		  sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp+1;
	      }
	      else 
		sn_rules[cur_snr].markOffsets[2*jpred+1] = -1;
	    }
	  }
	}
      }

      if (thisclausesn ==0) { 
	// Either no SaveModule or a rule with only external predicates
	// in a SaveModule
	// Same actions need to be performed
	cur_snr++ ;
	ASSERT((cur_snr < numSNRules));
	sn_rules[cur_snr].first = 0 ;
	sn_rules[cur_snr].recursive = 0 ;
	sn_offset_arr[cur_snr] = j ;
	if (clause->num_preds)
	  for (k=0,kpred = scc->preds; kpred != NULL; ++k, kpred = kpred->next)
	    if (kpred == clause->preds[0]->predicate) {
	      sn_first_offsets[cur_snr] = k;
	      break;
	    }
	sn_rules[cur_snr].Add_Rule_Info(this,
			module->ruleArray->rInfoPtrs[clause->rule_number],
			clause, module->module_info.NonGroundFacts);

	for (int jpred = 0; jpred < clause->num_preds; jpred++)
	  sn_rules[cur_snr].markOffsets[2*jpred] = 
	    sn_rules[cur_snr].markOffsets[2*jpred+1] = -1 ;
      }
    }
  }
  
  // Now handle recursive rules

  if (scc->recursive) { // If there are recursive rules:
    
    if (module->module_info.SaveModule) {
      // have to generate non-recursive SemiNaive rules from recursive rules
      for (j=0, pred = scc->preds; pred != NULL; ++j, pred = pred->next) {
	for (clause = pred->clauses;
	     clause != *pred->clause_tail;
	     clause = clause->pnext) {
	  if (!clause->seen) continue;
	  if (!clause->recursive) continue;

          int ipred, jpred, i;
	  for (ipred=0; ipred < clause->num_preds; ipred++) {
	    if (module->module_info.UseOrdSearch) {
	      if ((ipred > 0) &&
			(clause->preds[ipred-1]->predicate->is_mp_done()))
		continue;
	    }

	    // disregard literals defined in the same scc
	    if (clause->preds[ipred]->predicate->scc == scc) continue;
	    
	    if ((clause->preds[ipred]->predicate->clauses) ||
	    		(clause->preds[ipred]->predicate->is_magic())) {
	      // lower scc or containing MagicQueryPredKind without any rules

	      cur_snr++ ;
	      ASSERT((cur_snr < numSNRules));
	      i = has_fail_in_body(clause);
	      if (i >= 0) {
		sn_rules[cur_snr].first = i;
		for (k=0, kpred = scc->preds; kpred != NULL; ++k, kpred = kpred->next)
		  if (kpred == clause->preds[i]->predicate) {
		    sn_first_offsets[cur_snr] = k;
		    break;
		  }
	      }
	      else {
		sn_rules[cur_snr].first = ipred;
		for (k=0, kpred = scc->preds; kpred != NULL; ++k, kpred = kpred->next)
		  if (kpred == clause->preds[ipred]->predicate) {
		    sn_first_offsets[cur_snr] = k;
		    break;
		  }
	      }

	      sn_rules[cur_snr].recursive = 0 ;
	      sn_offset_arr[cur_snr] = j ;
	      sn_rules[cur_snr].Add_Rule_Info(this,
			 module->ruleArray->rInfoPtrs[clause->rule_number],
			 clause, module->module_info.NonGroundFacts);

	      for (jpred = 0; jpred < clause->num_preds; jpred++) {
		Literal *bodypred = clause->preds[jpred];

		if (ipred == jpred) {
		  sn_rules[cur_snr].markOffsets[2*jpred] =
			  Find_RMark_Offset(this, bodypred->predicate);
		}
		else
		  sn_rules[cur_snr].markOffsets[2*jpred] = -1;

		if (bodypred->predicate->scc == scc) {
		  // only until the global old mark; not the whole relation
		  sn_rules[cur_snr].markOffsets[2*jpred+1] = 
		  		Find_RMark_Offset(this, bodypred->predicate);
		  // sn_rules[cur_snr].markOffsets[2*jpred+1] = -1;
		}
		else if (bodypred->predicate->clauses ||
			  bodypred->predicate->is_magic()) {

		  int tmp = Find_RMark_Offset(this, bodypred->predicate);
		  if (ipred  > jpred)
		    sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp;
		  else
		    sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp+1;
		}
		else { // external non querymagic predicate 
		  sn_rules[cur_snr].markOffsets[2*jpred+1] = -1;
		}
	      } 
	    }
	  }
	}
      }
    }

    // now generate recursive SemiNaive rules from recursive rules
    for (j=0, pred = scc->preds; pred != NULL; j++, pred = pred->next)
      for (clause = pred->clauses;
	   clause != *pred->clause_tail;
	   clause = clause->pnext) {
	if (!clause->seen) continue;
	if (!clause->recursive) continue;

	int ipred, jpred, i;
	for (ipred = 0; ipred < clause->num_preds; ipred++) {

	  //
	  //  Treat negated literals as base !!
	  //
	  if (clause->preds[ipred]->negated) continue;

	  // with Ordered Search, have to special case the 
	  // negated literals as well as the grouped literal.
	  // the check below assumes that both follow a 
	  // done_m_p literal.

	  if (module->module_info.UseOrdSearch) {
	    if ((ipred > 0) && 
			(clause->preds[ipred-1]->predicate->is_mp_done()))
	      continue;
	  }

	  /* don't need to special case for mp_done since 
	     it is covered by the next if condition */

	  if (clause->preds[ipred]->predicate->scc != scc) continue;

	  cur_snr++ ; 
	  ASSERT((cur_snr < numSNRules));
	  i = has_fail_in_body(clause);
	  if (i >= 0) {
	    sn_rules[cur_snr].first = i;
	    for (k=0, kpred = scc->preds; kpred != NULL; ++k, kpred = kpred->next)
	      if (kpred == clause->preds[i]->predicate) {
		sn_first_offsets[cur_snr] = k;
		break;
	      }
	  }
	  else {
	    sn_rules[cur_snr].first = ipred;
	    for (k=0, kpred = scc->preds; kpred != NULL; ++k, kpred = kpred->next)
	      if (kpred == clause->preds[ipred]->predicate) {
		sn_first_offsets[cur_snr] = k;
		break;
	      }
	  }	    
	  sn_rules[cur_snr].recursive = 1 ;
	  sn_offset_arr[cur_snr] = j ;
	  sn_rules[cur_snr].Add_Rule_Info(this,
		    module->ruleArray->rInfoPtrs[clause->rule_number],
		    clause, module->module_info.NonGroundFacts);
	  
	  for (jpred = 0; jpred < clause->num_preds; jpred++) {
	    
	    Literal *bodypred = clause->preds[jpred];
	    
	    if (ipred == jpred)
	      sn_rules[cur_snr].markOffsets[2*jpred] = 
			Find_RMark_Offset(this, bodypred->predicate) ;
			// Find_RMark_Offset(scc, bodypred->predicate) ;
	    else 
	      sn_rules[cur_snr].markOffsets[2*jpred] = -1 ;
	    
	    if (bodypred->predicate->scc != scc)
	      sn_rules[cur_snr].markOffsets[2*jpred+1] = -1;
	    else {
	      int tmp = Find_RMark_Offset(this, bodypred->predicate) ;
			// Find_RMark_Offset(scc, bodypred->predicate) ;
	      if (ipred > jpred)
		sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp ;
	      else
		sn_rules[cur_snr].markOffsets[2*jpred+1] = tmp+1;
	    }
	  }
	}
      }
  }
  
}

static void PrintSccRecursiveRules(SccInfo* scc, int start, FILE *outf)
{
 int j;
  fprintf(outf,"\n[Iterating over recursive rules ...]") ;
  fprintf(outf,"\n---------------------------------------\n");
  
  for (int i = start; i < scc->numSNRules ; i++)
    if (!scc->sn_rules[i].ignore_this_rule) {
      fprintf(outf,"%s\n",scc->sn_rules[i].rInfo->pInfo.rule);
      fprintf(outf, "\nPredord :: ");
      for (j = 0; j < scc->sn_rules[i].rInfo->num_literals; j++)
	fprintf(outf, "%d ",scc->sn_rules[i].predOrders[j]);      
      fprintf(outf, "\n1st bt :: ");
      for (j = 0; j < scc->sn_rules[i].rInfo->num_literals; j++)
	fprintf(outf, "%d ",scc->sn_rules[i].firstFailureBacktrack[j]);
      fprintf(outf, "\nnext bt :: ");
      for (j = 0; j < scc->sn_rules[i].rInfo->num_literals; j++)
	fprintf(outf, "%d ",scc->sn_rules[i].nextFailureBacktrack[j]);
      fprintf(outf, "\n");
    }
  
  fprintf(outf,"---------------------------------------\n\n");
  fflush(outf);

}

static void PrintSccNonRecursiveRules(SccInfo* scc, FILE *outf)
{
  fprintf(outf,"\n[Applying non-recursive rules ...]") ;
  fprintf(outf,"\n---------------------------------------\n");
  
  for (int i = 0; i < scc->numSNRules ; i++) {
    if (scc->sn_rules[i].recursive) break ;
    if (!scc->sn_rules[i].ignore_this_rule) {
      fprintf(outf,"%s\n",scc->sn_rules[i].rInfo->pInfo.rule);
      fprintf(outf, "\n");
    }
  }
  
  fprintf(outf,"---------------------------------------\n\n");

}

void SccInfo::display_annotation_info(Table *F) 
{
  FILE *file = exEnv.trace_file;
  PredAnnotations *cur;
  QFModuleData *md = F->moduledata;
  
  for(int i=0; i < numPreds; i++) {
    Name predname = F->rels[offset_arr[i]].relPtr->name;
    
    for(cur= md->agg_sel_annotations; cur; cur=cur->next()) {
      if (cur->pred==predname) {
	fprintf(file,"%s%s  ", "@",SymbolString(AggSelectionSymbol));
	cur->print(file);
	fprintf(file, ".\n");
      }
    }
    for(cur= md->prioritize_annotations; cur; cur=cur->next()) {
      if (cur->pred==predname) {
	fprintf(file,"%s%s  ", "@",SymbolString(PrioritizeSymbol));
	cur->print(file);
	fprintf(file, ".\n");
      }
    }
    fprintf(exEnv.trace_file, "%s: subsumption checking - %s \n\n",
	    F->rels[offset_arr[i]].relPtr->name->string(),
	    F->rels[offset_arr[i]].relPtr->check_subsum? "on" : "off");
    /* PRAVEEN:: Add the index information dumping routines here. */
  }
  fprintf(file,"---------------------------------------\n\n");
}



static void EndOfScc(SccInfo* scc, FILE *outf)
{
  int i, j ;

  if (exEnv.profile_scc) {
    for (i=0; i< scc->numSNRules; i++)
      if (!scc->sn_rules[i].ignore_this_rule) {
	scc->sn_rules[i].rInfo->pInfo.print(outf, "SCC",
					    scc->sn_rules[i].predOrders,
					    scc->sn_rules[i].rInfo->num_literals);
      }
  }
}

static void clearStatistics(QFMaterializedModuleData *md, FILE *outf)
{
  int i, j;
  long profile_rules = ((exEnv.GlobalRelationOptions | md->rel_options) &
			REL_PROFILE_STATS) ;
  
  if (profile_rules && (exEnv.profile_module)) {
    md->module_info.GlobalProfileInfo.clear();
    md->module_info.GlobalProfileInfo.set_rule_name("");
    for (i = 0; i < md->numSccs; i++)
/***
      for (j = 0; j < ((md->sccArray)[i])->numSNRules; j++) {
	AddProfileInfo(&(md->sccArray[i]->sn_rules[j].rInfo->pInfo),
		       &(md->module_info.GlobalProfileInfo));
	md->sccArray[i]->sn_rules[j].rInfo->pInfo.clear();
***/
      for (j=0; j < md->ruleArray->numRules; j++) {
	AddProfileInfo(&(md->ruleArray->rInfoPtrs[j]->pInfo),
		       &(md->module_info.GlobalProfileInfo));
	md->ruleArray->rInfoPtrs[j]->pInfo.clear();
      }

    md->module_info.GlobalProfileInfo.print(outf, "MODULE SUMMARY");
    md->module_info.GlobalProfileInfo.clear();
  }
  else {
    for (i = 0; i < md->numSccs; i++)
      for (j = 0; j < md->sccArray[i]->numSNRules; j++)
	md->sccArray[i]->sn_rules[j].rInfo->pInfo.clear();
  }

}

SccInfo::SccInfo(ParserStruct& parserStruct, 
			QFMaterializedModuleData *md, Scc *scc)
{
  PredAnnotations *cur;
  int i;

  module = md ;
  this->scc = scc ;
  this->Create_SN_Structs(parserStruct);
  for (i=0; i < numPreds; i++) {
    for(cur = md->prioritize_annotations; cur; cur=cur->next()) {
      if (cur->pred==pred_ptrs[i]->name)
	has_hidden_relations = 1;
    }
  }
/*
  fprintf(stderr, "numpreds = %d; " , this->numPreds);
  for (i = 0; i< numSNRules; i++)
    fprintf(stderr, "%d ", sn_first_offsets[i]);
  fprintf(stderr, "\n");
*/

  rec_rule_start = 0;
}	


RMarkArray * SccInfo::setupSolve(Table *F)
{
  int sn_count;
  int i;
  
  // Create array of RMarks
  // By default, all RMark *s are NULL 
  RMarkArray *rm_Arr = NULL;
  RMarkArray *hidden_rel_rmarks = NULL;

  if (!module->module_info.SaveModule) 
    rm_Arr = new RMarkArray(2*numPreds) ;
  else {
    rm_Arr = new RMarkArray(2*(numPreds+num_lowerPreds)) ;
    // set up RMarks for the case of non-recursive SN rules for SaveModules
    HashSimpleRelation *hrel = NULL;
    for (i=0; i < numPreds+num_lowerPreds; i++) {
      hrel = (HashSimpleRelation *) F->rels[offset_arr[i]].relPtr;
      if (hrel) {
	ASSERT((hrel->there_are_marks()));
	// set up the old marks to the status of the relations as of
	// the end of the previous call to the SaveModule
	rm_Arr->markList[2*i] = hrel->headMark.prev;
	
	// we don't need to get new marks for any of the relations 
	// for SaveModule
      }
    }
  }

  if (has_hidden_relations) {
    hidden_rel_rmarks = new RMarkArray(2*numPreds) ;
    for (i=0; i < numPreds; i++) {
      if (F->rels[offset_arr[i]].relPtr->priority_info)
        hidden_rel_rmarks->markList[2*i+1] =
                               F->rels[offset_arr[i]].relPtr->getMark() ;
    }
  }

  if ((display_rules) && (!sn_rules[0].recursive) && 
      (!exEnv.C_quiet_mode_default))
    PrintSccNonRecursiveRules(this, exEnv.trace_file) ;

  /* For Basic SemiNaive, evaluate non-recursive rules
   * For Predicate SemNaive, this is incorporated into the
   * first iteration of the recursive rules.
   */

#define CUR_POS sn_offset_arr[sn_count]
  for (sn_count = 0; sn_count < numSNRules ; sn_count++)
    if (sn_rules[sn_count].recursive) break ;
    else if (!module->use_psn)
      if (!sn_rules[sn_count].ignore_this_rule)
         sn_rules[sn_count].evaluate(F, rm_Arr) ;
#undef CUR_POS


  // I dont know if it is possible that the lowest scc has a rule for
  // the m_query relation , but the rule is non-recursive. But just in
  // case this is possible, i catch it here, and insert the m_query
  // seed tuple at this point
  if (MagicRelation && module->use_psn && (sn_count == numSNRules)) {
      MagicRelation->insert_new(*MagicArgs, MagicBindEnv);
      MagicRelation = NULL; MagicArgs = NULL; MagicBindEnv  = NULL;
  }

  /*
   * NOTE : if the scc has only non-recursive rules, evaluate them all
   *        even for PSN.
   */

  if (module->use_psn && (sn_count == numSNRules)) {
    for (sn_count = 0; sn_count < numSNRules ; sn_count++)
      if (!sn_rules[sn_count].ignore_this_rule)
        sn_rules[sn_count].evaluate(F, rm_Arr) ;
  }

  rec_rule_start = sn_count ;

  if (has_hidden_relations) {
    for (i=0; i < numPreds; i++) {
      if (F->rels[offset_arr[i]].relPtr->priority_info)
      F->rels[offset_arr[i]].relPtr->freeMark(
                      hidden_rel_rmarks->markList[2*i+1], /*hide_em=*/ 1);
    }
  }


  if (sn_count == numSNRules) return rm_Arr;


  // Note on RMarks for recursive rules. For each predicate, an 'old' RMark
  // and a 'new' RMark are created at consecutive positions in the RMarkArray.
  // The order in which these RMarks are created defines their positions in
  // the array, and this fact is used by Find_RMark_Offset() .

  if (display_rules && !exEnv.C_quiet_mode_default)
    PrintSccRecursiveRules(this, rec_rule_start, exEnv.trace_file) ;

  return rm_Arr;
}

static int evaluateNonRecursive(SccInfo *scc, Table *F,
                                 RMarkArray *rm_Arr, int curpos)
{
  int sn_count;
  int new_tuples = 0;

  if (MagicRelation && 
      (F->rels[scc->offset_arr[curpos]].relPtr == MagicRelation)) {
    MagicRelation->insert_new(*MagicArgs,MagicBindEnv);
    MagicRelation = NULL; MagicArgs = NULL; MagicBindEnv  = NULL;
    new_tuples = 1;
  }

      
  for (sn_count = 0; sn_count < scc->rec_rule_start ; sn_count++)
    if (scc->sn_offset_arr[sn_count] == curpos)
      if (!scc->sn_rules[sn_count].ignore_this_rule)
            /*** ignore for now 
	      new_tuples += 
	      (scc->pred_success_arr[curpos] = 
	      scc->sn_rules[sn_count].evaluate(F, rm_Arr));
	     ***/
        new_tuples += scc->sn_rules[sn_count].evaluate(F, rm_Arr);

  return new_tuples;

}

/*
 * performs one iteration over the recursive rules and returns 1 if the iteration
 * should continue, and 0 if the iterative loop should be exited
 */
int SccInfo::iterateRecursiveRules(Table *F, RMarkArray *rm_Arr, 
					int loop_iter_num, int iteration_number)
{
  int new_tuples = 0;
  int sn_count;
  
  if (module->use_psn) {
    // Predicate SemiNaive

    int curpos = sn_offset_arr[rec_rule_start];
    rm_Arr->markList[2*curpos+1] = F->rels[offset_arr[curpos]].
      relPtr->getMark() ;

    for (sn_count = rec_rule_start; sn_count < numSNRules ; sn_count++){
      if (!sn_rules[sn_count].ignore_this_rule)
        new_tuples += sn_rules[sn_count].evaluate(F, rm_Arr) ;

      if ((sn_count == numSNRules -1) || (sn_offset_arr[sn_count+1] != curpos)) {
	int hide_em = 0;
	if (module->use_psn && (loop_iter_num == 1) && (iteration_number == 1))
	  new_tuples += evaluateNonRecursive(this, F, rm_Arr, curpos);

	if ((! module->module_info.SaveModule) || (loop_iter_num > 1)) {
	    F->rels[offset_arr[curpos]].relPtr->freeMark(
			  rm_Arr->markList[2*curpos]) ;
	}

        if (has_hidden_relations  &&
	    F->rels[offset_arr[curpos]].relPtr->priority_info) {
          hide_em = 1;
          F->rels[offset_arr[curpos]].relPtr->freeMark(
			       rm_Arr->markList[2*curpos+1], hide_em) ;
          rm_Arr->markList[2*curpos+1] = F->rels[offset_arr[curpos]].
	    relPtr->getMark() ;
	}

	rm_Arr->New_To_Old(2*curpos);

	if (sn_count < numSNRules -1) {
	  curpos = sn_offset_arr[sn_count+1] ;
	  rm_Arr->markList[2*curpos+1] = F->rels[offset_arr[curpos]].
	    relPtr->getMark() ;
	}
      }
    }
  }

  else {
    // Basic SemiNaive
    int i;
    for (i=0; i < numPreds; i++) {
      rm_Arr->markList[2*i+1] = F->rels[offset_arr[i]].relPtr->getMark() ;
    }

    for (sn_count = rec_rule_start; sn_count < numSNRules ; 
	 sn_count++) {

#define CUR_SN_RULE sn_rules[sn_count]
// position in the pred_success_arr of the head predicate of this sn rule
#define CUR_POS sn_offset_arr[sn_count]
// position in the pred_success_arr of the first rhs predicate of this sn rule
#define FIRST_POS sn_first_offsets[sn_count]

       if (!CUR_SN_RULE.ignore_this_rule)
	 new_tuples += CUR_SN_RULE.evaluate(F, rm_Arr);

#undef CUR_SN_RULE
#undef CUR_POS
#undef FIRST_POS
     }
    
    for (i=0; i < numPreds; i++){
      int hide_em = 0;
      // the old marks should not be freed, if we are in the
      // first iteration of SaveModules since these correspond to 
      // the global old pointers obtained at the end of evaluation of
      // the previous call to the module.
      if ((! module->module_info.SaveModule) || (loop_iter_num > 1)) {
        F->rels[offset_arr[i]].relPtr->freeMark(rm_Arr->markList[2*i]) ;
      }
      if (has_hidden_relations  &&
	 F->rels[offset_arr[i]].relPtr->priority_info) {
	hide_em = 1;
	F->rels[offset_arr[i]].relPtr->freeMark(rm_Arr->markList[2*i+1],
						hide_em);
	rm_Arr->markList[2*i+1] = F->rels[offset_arr[i]].relPtr->getMark();
      }
      rm_Arr->New_To_Old(2*i) ;
    }
  }
	
  GoalNode *g_node; 

  if (new_tuples) return 1;

  else {
    /* There are a bunch of special cases in which the number of new_tuples is == 0,
       but there is still some computation to go. However, if the value of new_tuples
       is non_zero, then we can return right away
     */
    if (has_hidden_relations)
      for (int i =0; i < numPreds; i++)
        if (F->rels[offset_arr[i]].relPtr->priority_info)
           if (F->rels[offset_arr[i]].relPtr->pop_from_hidden_relation() == 0)
                  continue;
           else {
	     return 1;
	     // return, so newly unhidden tuple
	     // can be displayed to user if required.
	     // Tuple will get used in a later call on this routine
	   }

    if (!(module->module_info.UseOrdSearch))
      return 0;

    // Ordered Search
    // have reached a fixpoint; need to get unmarked tuple 
    // from context, and put it in the appropriate relation

    int any_new_fact_in_done = 0;
    g_node = F->context->get_unmarked_goal(any_new_fact_in_done);
    
    // exit condition from Ordered Search
    if (g_node == NULL) {
      if (!any_new_fact_in_done) {
	return 0;
      }
      else return 2 ;
    }

    if ((!module->module_info.SaveModule) || 
	(! g_node->owner_rel->check_subsum) ||
	(! g_node->owner_rel->is_subsumed(g_node))) {
      g_node->owner_rel->insert_tuple(g_node);
      // NOTE: should remove g_node from 
      // g_node->owner_rel->for_ordsearch's index
      return 2;
    }

  return 0;
  }
}



// Iterator will be non-null ONLY if this is a top-level scc
// and staggered evaluation is desired on it.
void SccInfo::Solve(Table *F, TupleIterator *iterator)
{
  RMarkArray *rm_Arr;

  // if staggered evaluation is not being performed, or this is the
  // not the top level scc, or this is the first time the top level 
  // scc is being executed

  if (!(iterator) || !(iterator->state->rm_Arr)) {
    rm_Arr = setupSolve(F);

    if (iterator) {
      // first time this top-level scc is being executed
      iterator->state->rm_Arr = rm_Arr;
    }
  }
  else {
    rm_Arr = iterator->state->rm_Arr;
  }

  /******
  // Sudarshan:  Handling prioritize at the level of SN updates is critical.
  //   I tried to handle hiding facts at the level of fact insertion, but
  //   that caused major problems if a relation has aggregation
  //   (even if non-recursive) as well as prioritize.  The trouble is that
  //   if facts are hidden during aggregation, the result of aggregation is
  //   garbage.  So now facts are hidden only when they are moved out of the
  //   delta relations.
  //
  // The priority code works as follows.
  // a)  get rmarks on prioritized relations even before exit rules,
  // b)  after each iteration or exit rule firing, move tuples from
  //     deltas to the hidden relation, instead of to the previous
  //   delta relation, if the relation uses priority.
  ******/

  if (rec_rule_start == numSNRules) {
     if (rm_Arr) delete rm_Arr ;    
     if (iterator) iterator->state->rm_Arr = NULL;
     if (has_hidden_relations) {  // Shouldn't happen normally, since the
              // annotation is meaningless in this case.
              // To be safe, unhide all hidden facts.
        clean_hidden_relations(F);
     }
#ifdef DO_PROFILE
     if ((exEnv.GlobalRelationOptions | module->rel_options) &
			REL_PROFILE_STATS)
        EndOfScc(this, exEnv.profile_file) ;
#endif
     return;
  }

  // loop_iter_num is useful for SaveModules since the first time
  // around, the old RMarks correspond to global old pointers and 
  // should not be deleted

  int loop_iter_num = 0;
  int ret_val;

  // Evaluate recursive rules
  for(; ;) {

    loop_iter_num++ ;

    // if staggered evaluation is being used, then the iter number is
    // actually stored in the iterator's state, and not in the variable
    // loop_iter_num.
    if (iterator)
      ret_val = iterateRecursiveRules(F, rm_Arr, loop_iter_num,
                                      iterator->state->iter_number);
    else
      ret_val = iterateRecursiveRules(F, rm_Arr, loop_iter_num, loop_iter_num);

    if (!ret_val) break;
    if ((iterator) && (ret_val != 2)) return;
  }
  
  // since the loop is going through at least once, these marks 
  // will not be the global old marks 

  for (int i=0; i < numPreds; i++)
    F->rels[offset_arr[i]].relPtr->freeMark(rm_Arr->markList[2*i]) ;

  delete rm_Arr ;

  if (iterator) iterator->state->rm_Arr = NULL;

#ifdef DO_PROFILE
     if ((exEnv.GlobalRelationOptions | module->rel_options) &
			REL_PROFILE_STATS)
        EndOfScc(this, exEnv.profile_file) ;
#endif
}


void SccInfo::clean_hidden_relations(Table *F) 
{
  for(int i=0; i < numPreds; i++) {
    if(has_hidden_relations  &&
       F->rels[offset_arr[i]].relPtr->priority_info)
      F->rels[offset_arr[i]].relPtr->pop_all_from_hidden_relation();
  }
}


QFMaterializedModuleData::QFMaterializedModuleData(Table *table,
						   RuleInfoArray *rArray,
						   ParserStruct& parserStruct)
{
  F = table ;
  ruleArray = rArray ;
  numSccs = 0 ;
  sccArray = NULL ;
  multiset_annotations = parserStruct.MultisetAnnotations;
  agg_sel_annotations  = parserStruct.agg_sel_annotations;
  prioritize_annotations  = parserStruct.prioritize_annotations;
  index_annotations    = parserStruct.make_index_annotations;
  free_table_list = NULL;
  free_query_relations = NULL;
  free_answer_relations = NULL;
  saved_state = 0;
  no_magic = 0;
  rel_options = 0;
}

void QFMaterializedModuleData::AddScc(ParserStruct& parserStruct, Scc *scc)
{
	int i;

	SccInfo **temp = sccArray ;
	sccArray = new SccInfo*[++numSccs] ;
	for (i=0; i< numSccs-1; i++)
		sccArray[i] = temp[i] ;

	sccArray[i] = new SccInfo(parserStruct, this, scc) ;
	if (numSccs > 1) delete [] temp ;

}

void QFMaterializedModuleData::clean_cache()
{
	for (int i = 0; i < numSccs; sccArray[i++]->clean_cache()) ;
}

/*
 * iterator is guaranteed to be either NULL, or have a state field allocated.
 * If iterator->state->rm_Arr is NULL, it implies that this is the first
 * time this module is being invoked using this iterator
 */
void QFMaterializedModuleData::Solve(Table *table, TupleIterator *iterator)
{
  if (!(iterator) || !(iterator->state->rm_Arr)) { 
    clean_cache();	
    // This is conservative.  In most cases it is 
    // not needed

    for (int i = 0; i < numSccs-1; i++) {
      sccArray[i]->Solve(table);
    }
  }
  sccArray[numSccs-1]->Solve(table, iterator);
  if (iterator) iterator->state->iter_number++;
}



/* Each Relation used in the module is cached in a global variable */


// NOTE : Rewrite this !!
void add_pred_extern(Predicate *pred, enum PredKind kind,
			     struct collection *predicates)
{
    Name name = pred->name;
    int arity = pred->arity;
    struct rule *new_rule;
    /* WARNING: Rule lists are misused here for storing predicate information */
    FOR_EACH_PRED_FOR(rule, name, predicates) {
	if (rule->num_preds == arity) {
	    if (kind==LocalPredKind && RulePredKind(rule)==ExternalPredKind)
		SetRulePredKind(rule, kind);
	    return;
	}
    } END_EACH_PRED
    new_rule = new rule;
    new_rule->head = new Literal();
    new_rule->head->pred = name;
    new_rule->num_preds = arity;
    SetRulePredKind(new_rule, kind);
    add_rule(new_rule, predicates);
}

RuleInfo *create_rule_info(Table *F, struct rule *rule)
{
    int num_preds = rule->num_preds;
    int num_deletes = rule->head_deletes_count;
    int ipred;

    /* Variable declarations */
    
    RuleInfo *rInfo = new RuleInfo(rule);

//  Adding info for rInfo->arg_list, and rInfo->relations.

//  ArgList **arglist = new ArgList*[num_preds+1] ;

// NOTE : Should change Arg** to ArgList *
    for (ipred = 0; ipred < num_preds; ipred++)
	rInfo->arg_list[ipred] = (Arg **)(&rule->preds[ipred]->args) ;

    rInfo->arg_list[num_preds] = (Arg **)(&rule->head->args) ;
    for (ipred=0; ipred < num_deletes; ipred++) {
	rInfo->arg_list[num_preds+1+ipred] =
			(Arg **)(&rule->head_deletes[ipred]->args) ;
    }

// If the predicate is negated, a negative value is placed in the
// rInfo->relations[] field. This info is used by SemiNaive::evaluate() 
    for (ipred = 0; ipred < num_preds; ipred++)
	if (rule->preds[ipred]->negated)
	    rInfo->relations[ipred]= -2 
	      - (rInfo->offsets[ipred] = get_offset(F, rule->preds[ipred]));
	else
	    rInfo->relations[ipred]= 
	      (rInfo->offsets[ipred] = get_offset(F, rule->preds[ipred]));

    if (rule->head->negated)
	rInfo->relations[num_preds] = -2 - 
	  (rInfo->offsets[num_preds] = get_offset(F, rule->head)) ;
    else
	rInfo->relations[num_preds] = 
	  (rInfo->offsets[num_preds] = get_offset(F, rule->head)) ;

    rInfo->headrel_ptr = &(F->rels[rInfo->offsets[num_preds]].relPtr);

    for (ipred = 0; ipred < num_deletes; ipred++) {
	rInfo->relations[num_preds+1+ipred]= -2 -
	  (rInfo->offsets[num_preds+1+ipred] = 
	      get_offset(F, rule->head_deletes[ipred]));
    }

    rInfo->relations[num_preds+num_deletes+1] = -1 ;
    rInfo->offsets[num_preds+num_deletes+1] = -1 ;


    return(rInfo) ;
}

int count_var_list(VarLink *rest)
{
    VarLink *cur;
    int max_var_num = -1;
    for( cur=rest;  cur; cur=cur->next) {
	if (cur->arg->var > max_var_num)
	    max_var_num = cur->arg->var;
    }
    return max_var_num+1;
}



/*** WARNING  - with recursion deeper than 
	exEnv.C_max_recursion_depth_default this code will complain and abort.
*******/

Relation *find_local_relation(Name name, int arity)
{
    if (TableStackCount <= 0 || TableStack[TableStackCount-1] == NULL)
	return NULL;
    int offset = get_offset(TableStack[TableStackCount-1], name, arity,
			/* print_error */ 0);
    if (offset < 0)
	return NULL;

    return (TableStack[TableStackCount-1]->rels[offset].relPtr);
}

extern Relation *Find_External_Relation(Name name, int arity);

/* Initialize the cached Relation pointers. */


// Used by psn to determine whether to insert the query tuple before
// evaluating the scc, or during scc evaluation. the function in_scc
// returns non-zero if the Relation rel corresponds to one of the
// recursive predicates of the scc.
static int in_scc(SccInfo *scc, Relation *rel)
{
  for (int i = 0; i < scc->numPreds; i++)
    {
      if ((scc->pred_ptrs[i]->name == rel->name) &&
	  (scc->pred_ptrs[i]->arity == rel->arity())) return 1;
    }
  return 0;
}

static int determine_subsum_status(QFModuleData *md,
				   Name name,
				   int default_val)
{
  PredAnnotations *cur;
  
  for(cur= md->multiset_annotations; cur; cur=cur->next()) {
    char buf[1000];
    sprint_pred_name(buf, SymbolString(cur->pred), cur->adorn);
    /*
     * Warning: multiset and checksubsumption annotations have
     * the name stored separate from the adornment 
     */
    if (EnterSymbol(buf) == name) {
      default_val = cur->check_subsum;
    }
  } 
  return default_val;
}

static void init_rule_pointers(Table *F, QFModuleData *md,
			Relation *magic, int reused_table, 
			int old_magic_relation)
{
    struct collection *predicates = md->rules;
    F->moduledata = md;

    /* WARNING: Rule lists are misused here for storing predicate info */

    FOR_EACH_PRED(rule, predicates) {
	Name name = rule->head->pred;
	int arity = rule->num_preds;
	int offset = get_offset(F, name, arity);
	switch (RulePredKind(rule)) {
	    case QueryPredKind:
	        // check whether it is a non SaveModule or the first 
	        // invocation of a SaveModule
	        if ((! md->module_info.SaveModule) || 
			(! ((QFMaterializedModuleData *) md)->saved_state)) {
		/***********
		* the condition tested is not (! reused_table) as for 
		* LocalPredKind below since answer relations are used 
		* at the calling point even after the table is returned 
		* to the free table list.
		***********/
		    int subsum_stat = determine_subsum_status(md, name,
						       md->module_info.CheckSubsum);
	            if (md->free_answer_relations && 
			md->free_answer_relations->reln->check_subsum == subsum_stat){
			ASSERT((! md->module_info.SaveModule)) ;
			F->rels[offset].AssignRel(
					md->free_answer_relations->reln);
		        RelationLink *tmp = md->free_answer_relations;
		        md->free_answer_relations = tmp->next;
		        delete tmp;

	            } 
	            else {
		        F->rels[offset].AssignRel(make_local_relation(name, 
				   arity, md->module_info.IndexDeltas, subsum_stat));
			/*
	                F->rels[offset].relPtr->check_subsum =
						md->module_info.CheckSubsum;
			*/
		        F->rels[offset].relPtr->ordsearch =
						md->module_info.UseOrdSearch;
		        F->rels[offset].relPtr->r_kind = COR_R_ANSWER;
			if ((md->module_info.SaveModule) &&
			  (! ((QFMaterializedModuleData *) md)->saved_state)) {
			    F->rels[offset].relPtr->getMark();
			}
	            }
		    F->rels[offset].relPtr->local_options = md->rel_options |
						exEnv.GlobalRelationOptions;
		      
		  }
	        break;
	    case LocalPredKind:
	        if (! reused_table) { // Relation not already set up
		    // with SaveModules, the relation should also have all
		    // the tuples computed until the last time.

		    ASSERT(((! md->module_info.SaveModule) || ((md->module_info.SaveModule) && (! ((QFMaterializedModuleData *) md)->saved_state))));

		    int subsum_stat = determine_subsum_status(md, name,
						       md->module_info.CheckSubsum);

		    F->rels[offset].AssignRel(make_local_relation(name, arity, 
				  md->module_info.IndexDeltas, subsum_stat));
		    /*
	            F->rels[offset].relPtr->check_subsum =
		      			md->module_info.CheckSubsum;
		     */
		    F->rels[offset].relPtr->ordsearch =
					md->module_info.UseOrdSearch;
		    F->rels[offset].relPtr->r_kind =
		    			F->rels[offset].relPtr->find_r_kind();

		    if ((F->rels[offset].relPtr->r_kind ==
		    	    COR_R_MAGIC) && (!md->module_info.CheckSubsum))
		        F->rels[offset].relPtr->check_subsum =
		    			md->module_info.MultiSet;
		    if ((md->module_info.SaveModule) &&
			  (! ((QFMaterializedModuleData *) md)->saved_state)) {
		        F->rels[offset].relPtr->getMark();
		    }
	     	}
	     	else {
		  F->rels[offset].relPtr->local_options = md->rel_options |
						exEnv.GlobalRelationOptions;
		  continue;
		}

		F->rels[offset].relPtr->local_options = md->rel_options |
						exEnv.GlobalRelationOptions;
	        break;

	    case MagicQueryPredKind:
	        if ((md->module_info.UseOrdSearch) && (! F->context)) {
		    F->context = new Context();
	        }

	        // create a suitable magic relation
	        // F->rels[offset].AssignRel(magic) ;
	        if ((!md->module_info.SaveModule) || 
			(! ((QFMaterializedModuleData *) md)->saved_state)) {
					// (!F->rels[offset].relPtr)) {
	            // create a new relation 

		    int subsum_stat = determine_subsum_status(md, name,
			  (md->module_info.CheckSubsum | md->module_info.MultiSet));
	            if (md->free_query_relations && 
			md->free_query_relations->reln->check_subsum == subsum_stat){
		        RelationLink *tmp = md->free_query_relations;
		        F->rels[offset].AssignRel(tmp->reln);
		        md->free_query_relations = tmp->next;
		        delete tmp;
		        old_magic_relation = 1;
		      }
		    else {
		      F->rels[offset].AssignRel(make_local_relation(name, arity,
				       md->module_info.IndexDeltas, subsum_stat));
		      old_magic_relation = 0;
		    }
		    if ((md->module_info.SaveModule) &&
			  (! ((QFMaterializedModuleData *) md)->saved_state)) {
			F->rels[offset].relPtr->getMark();
		    }
	        }
	        // else retain the old relation 
	     
	        // NOTE: the tuples in "magic" have not been added as yet

	        F->rels[offset].relPtr->ordsearch = 
						md->module_info.UseOrdSearch;
	        F->rels[offset].relPtr->r_kind = COR_R_MAGIC;

	        /*
		  F->rels[offset].relPtr->check_subsum = 
				(md->module_info.MultiSet |
				md->module_info.CheckSubsum);
	         */

	        if (md->module_info.UseOrdSearch) { 
		    if (! F->rels[offset].relPtr->for_ordsearch) {
			F->rels[offset].relPtr->for_ordsearch = 
					new HashSimpleRelation(arity);
		        // what other indices need to be created?
	            }

	            magic->add_all_goals(F->rels[offset].relPtr->for_ordsearch,
					F->context, F->rels[offset].relPtr);

	        }
		F->rels[offset].relPtr->local_options = md->rel_options |
						exEnv.GlobalRelationOptions;

	        break;

	    case ExternalPredKind: 
	        /********************** 
		* If reused_table is set, we should be able to avoid the 
		* overhead of finding external relations all over again.  
		* However if anything has changed externally (say, because 
		* of a rel_close), we need to do this.
		* Perhaps we should maintain a flag and find relations again
		* only if required
	        **********************/ 
	        F->rels[offset].AssignRel(Find_External_Relation(name, arity));
		/*****
	        F->rels[offset].relPtr->r_kind =
					F->rels[offset].relPtr->find_r_kind();
		*****/

		F->rels[offset].relPtr->local_options = md->rel_options |
					exEnv.GlobalRelationOptions;
	        break;
	      }

	PredAnnotations *cur;

	for(cur= md->index_annotations; cur; cur=cur->next())
	    if (cur->pred==rule->head->pred) { 
		VarLink *rest;
		rest = cur->arglist1->var_list(NULL);
		if ( cur->arglist2 )
		    rest = cur->arglist2->var_list(rest);

		int num_vars = count_var_list(rest);

		if (cur->arglist2)
		    F->rels[offset].relPtr->add_index(cur->arglist1, 
						num_vars, cur->arglist2) ;
		else F->rels[offset].relPtr->add_index(cur->arglist1) ;

	    }

	/***** moved before make_local_relation --- P
	for(cur= md->multiset_annotations; cur; cur=cur->next()) {
	  char buf[1000];
	  sprint_pred_name(buf, SymbolString(cur->pred), cur->adorn);
	  if (EnterSymbol(buf) == rule->head->pred) {
	    F->rels[offset].relPtr->check_subsum = cur->check_subsum;
	  }
	}
        ****/

        for(cur= md->agg_sel_annotations; cur; cur=cur->next()) {
	  if (cur->pred==rule->head->pred)
	    F->rels[offset].relPtr->add_agg_sel_anno(cur);
	}

	for(cur = md->prioritize_annotations; cur; cur=cur->next()) {
          if (cur->pred==rule->head->pred)
	    F->rels[offset].relPtr->add_prioritize_info(cur);
	}

	if (RulePredKind(rule) == MagicQueryPredKind) {
	    // this has to be done here instead of in the switch 
	    // statement above because the check_subsum bit could have
	    // altered because of the annotations.

	    // first add the tuples in "magic" to the relation
	    if (!md->module_info.UseOrdSearch) {
		magic->add_all_tuples(F->rels[offset].relPtr, 
				/* take marks into account */ 1,
				/* can check for duplicates */ 1) ;
		// there could be marks because of save modules

		if (((QFMaterializedModuleData *)md)->use_psn) {
		  MagicRelation = F->rels[offset].relPtr;
		  if (!in_scc(((QFMaterializedModuleData *)md)->sccArray[0],
			      MagicRelation)) {
		    MagicRelation->insert_new(*MagicArgs, MagicBindEnv);
		    MagicRelation = NULL;
		    MagicArgs  = NULL; MagicBindEnv = NULL;
		  }
		}
	      }
	    else { // get the fact from the Context

		int any_new_fact_in_done = 0;
		GoalNode *g_node =
		        F->context->get_unmarked_goal(any_new_fact_in_done);

		ASSERT((g_node != NULL));
		ASSERT((g_node->owner_rel == F->rels[offset].relPtr)) ;
                if ((!md->module_info.SaveModule) || 
			(! F->rels[offset].relPtr->check_subsum) ||
			(! F->rels[offset].relPtr->is_subsumed(g_node)))
		    F->rels[offset].relPtr->insert_tuple(g_node);

	    }


	    /***** 
	    * From this point on, the passed in relation "magic"
	    * is useless.  However, it gets reclaimed only in 
	    * clear_rule_pointers.
	    *****/
	}

    } END_EACH_PRED

    display_rules = ((exEnv.GlobalRelationOptions | md->rel_options) &
			REL_DISPLAY_INSERTIONS) ;
}

static void clear_rule_pointers(Table *F, QFModuleData *md,
			Relation *magic)
{
    struct collection *predicates = md->rules;

    /* WARNING: Rule lists are misused here to store predicate information */
    FOR_EACH_PRED(rule, predicates) {
	Name name = rule->head->pred;
	int arity = rule->num_preds;
	int offset = get_offset(F, name, arity);
	switch (RulePredKind(rule)) { 
	    case ExternalPredKind: 
		break;
	    case QueryPredKind:
		if ((md->module_info.SaveModule) &&
			  (! ((QFMaterializedModuleData *) md)->saved_state)) {
		    ((HashSimpleRelation *) F->rels[offset].relPtr)->
						free_allMarks();
		}
		if (md->module_info.SaveModule) {
		    F->rels[offset].relPtr->getMark();
		}
		break;
	    case LocalPredKind: {
		Relation *temp = F->rels[offset].relPtr;
		if (! md->module_info.SaveModule) {
		    temp->empty_relation( /*deleteTuples =*/ 1 );
		    if (temp->mp_done_reln) { // for OrdSearch
		        temp->mp_done_reln->empty_relation(1);
		    }
		}
		else { // get appropriate mark after releasing previous marks
		    HashSimpleRelation *h_temp = (HashSimpleRelation *) temp;
		    h_temp->free_allMarks();
		    h_temp->getMark();
		    if (h_temp->mp_done_reln) {
			h_temp->mp_done_reln->free_allMarks();
			h_temp->mp_done_reln->getMark();
		    }
		}
		if (temp->for_ordsearch) {
		    // this can be done even for SaveModules
		    temp->for_ordsearch->empty_relation(0);
		}
	    	break;
	    }
	    case MagicQueryPredKind: {
		Relation *temp = F->rels[offset].relPtr;
		if (! md->module_info.SaveModule) {
		    temp->empty_relation( /*deleteTuples =*/ 1 );
		    temp->ordsearch = 0; 
		        // else Ordered Search will crash the next time around
		    if (temp->mp_done_reln) { // for OrdSearch
		        temp->mp_done_reln->empty_relation(1);
		    }
		}
		else { // get appropriate mark after releasing previous marks
		    HashSimpleRelation *h_temp = (HashSimpleRelation *) temp;
		    h_temp->free_allMarks();
		    h_temp->getMark();
		    if (h_temp->mp_done_reln) {
			h_temp->mp_done_reln->free_allMarks();
			h_temp->mp_done_reln->getMark();
		    }
		}

		if (temp->for_ordsearch) {
		    // this can be done even for SaveModules
		    temp->for_ordsearch->empty_relation(0);
		}

		if (! md->module_info.SaveModule) {
		    md->free_query_relations = new RelationLink(
				temp, md->free_query_relations);
		    F->rels[offset].relPtr = NULL;
		}
	    	break;
	    }
	}

    } END_EACH_PRED

    // reclaim "magic" 
    magic->empty_relation(0);
    magic->ordsearch=0;
	// else Ordered Search will crash the next time around
    md->free_query_relations = new RelationLink(magic, 
					md->free_query_relations);

    // reclaim table for non SaveModules
    if (! md->module_info.SaveModule) {
        F->next = md->free_table_list;
        md->free_table_list = F;
    }
    else { // have saved the state of saved module
	((QFMaterializedModuleData *) md)->saved_state = 1;
    }
}

Scc *SccList = NULL;
Scc **SccListTail = &SccList;

struct SccWork {
    int lowlink;
    int number;
    struct SccWork *next;
    Predicate *pred;
    int recursive;
};

struct SccWork *SccWorkList = NULL;
int SccCount = 0;
struct SccWork SccDone[1]; // dummy

static struct Scc *NewScc()
{
    struct Scc *scc = new Scc;
    scc->preds = NULL;
    scc->next = NULL;
    scc->recursive = 0;
    *SccListTail = scc;
    SccListTail = &scc->next;
    scc->emitted = -1;
    return scc;
}

void initialize_scc_structs()
{
    SccList = NULL ;
    SccListTail = &SccList;
    SccWorkList = NULL;
    SccCount = 0;
}

inline int min(int i, int j) { return i < j ? i : j; }

int find_predicates_invoked_from(Predicate *pred)
{
    // Based on code by Thomas Ball from an algorithm by Robert Tarjan

    struct SccWork *work = new SccWork();
    Clause *clause;
    int ipred;
    // number is a sequential node number
    // lowlink is lowest number of a node reachable from current node.
    work->lowlink = work->number = ++SccCount;

    work->next = SccWorkList;
    work->pred = pred;
    work->recursive = 0;
    pred->work = work;
    SccWorkList = work;
    // Iterate over all Clauses that define 'pred'.
    for (clause = pred->clauses; clause; clause = clause->pnext) {

	// This should take care of 'bogus' magic dependencies
	if (clause->head->predicate->is_magic() && clause->num_preds) {
	    Predicate *first_pred = clause->preds[0]->predicate;
	    if (first_pred->is_magic() && first_pred->mark == 0)
		continue ;
	}

        clause->seen = 1;
	// Iterate over all Predicates invoked by 'clause'.
	for (ipred = 0; ipred < clause->num_preds; ipred++) {
	    Literal *child_literal = clause->preds[ipred];
	    Predicate *child = child_literal->predicate;
	    if (child->work == NULL) { /*  not visited yet */
		int child_lowlink = find_predicates_invoked_from(child);
		work->lowlink = min(work->lowlink, child_lowlink);
	    }
	    else if (child->work != SccDone) {
		if (child->work->number < work->number)
		    work->lowlink = min(work->lowlink, child->work->lowlink);
		child->work->recursive = 1;
	    }
	}
    }
    int lowlink = work->lowlink;
    if (work->lowlink == work->number) {
	struct Scc *scc = NewScc();
	Predicate **pred_tail = &scc->preds;
	while (SccWorkList && SccWorkList->number >= work->number) {
	    register SccWork *sccWork = SccWorkList;
	    Predicate *work_pred = sccWork->pred;
	    work_pred->next = scc->preds;
	    work_pred->scc = scc;
#ifdef DEBUG
	    if (work_pred->work != sccWork) abort();
#endif
	    work_pred->work = SccDone;
	    *pred_tail = work_pred;
	    pred_tail = &work_pred->next;
	    SccWorkList = sccWork->next;
	    delete sccWork;
	    scc->recursive |= work->recursive;
	}
	*pred_tail = NULL;

    }
    return lowlink;
}

static void mark_predicates_invoked_from(Predicate *pred)
{
    Clause *clause;
    int ipred;

    pred->mark = 1 ;
    // Iterate over all Clauses that define 'pred'.
    for (clause = pred->clauses; clause; clause = clause->pnext) {

	if (clause->head->predicate->is_magic()) continue ;

	// Iterate over all Predicates invoked by 'clause'.
	for (ipred = 0; ipred < clause->num_preds; ipred++) {
	    Literal *child_literal = clause->preds[ipred];
	    Predicate *child = child_literal->predicate;
	    if (child->mark == 0) { /*  not visited yet */
		mark_predicates_invoked_from(child);
	    }
	}
    }

}

static void clean_up_after_one_scc()
{
    // if there are no base predicates, need to clean up SccList
    struct Scc *base_scc = SccList;
    if (!(base_scc->preds)) {
	// there are no base predicates
	ASSERT((base_scc->next != NULL));
	ASSERT((base_scc->next->preds != NULL));
		// must at least have the query predicate

	SccList = base_scc->next;
	delete base_scc;
    }

    // NOTE: should the SccWorkList also be deleted?
}

int group_predicates_in_one_scc(Predicate *pred)
{
    // If there are base predicates, there have to be at least two
    // SCCs; the first one must be the base SCC

    struct Scc *base_scc;
    struct Scc *scc;
    if (!SccList) {
	base_scc = NewScc();
	scc = NewScc(); 	// this should link it to be the 2nd SCC
        scc->recursive = 1;		// NOTE: is this a right assumption?
    }
    else {
	base_scc = SccList;
	scc = SccList->next;
    }

    struct SccWork *work;
    if (!SccWorkList) {
	 work = new SccWork();
	 work->next = SccWorkList;
	 SccWorkList = work;
    }
    else work = SccWorkList;

    pred->work = work;  // visited

    Clause *clause;
    int ipred;
    
    pred->scc = scc;
    pred->next = scc->preds;
    scc->preds = pred;

    for (clause = pred->clauses; clause; clause = clause->pnext) {

	// This should take care of 'bogus' magic dependencies
	if (clause->head->predicate->is_magic() && clause->num_preds) {
	    Predicate *first_pred = clause->preds[0]->predicate;
	    if (first_pred->is_magic() && first_pred->mark == 0)
		continue ;
	}

	clause->seen = 1;

	// Iterate over all Predicates invoked by 'clause'.
	for (ipred = 0; ipred < clause->num_preds; ipred++) {
	    Literal *child_literal = clause->preds[ipred];
	    Predicate *child = child_literal->predicate;
	    if ((child->work == NULL) && (child->clauses)) {
		// predicates not yet visited and those
		// which have rules defined in this module
		group_predicates_in_one_scc(child);
	    }
	    else if (child->work == NULL) {
		// predicates not yet visited
		// and which are external predicates
		child->work = work;
		if (child->is_mp_done()) { 
		    // NOTE: for Ordered Search
		    child->scc = scc;
		    child->next = scc->preds;
		    scc->preds = child;
		}
		else {
		    child->scc = base_scc;
		    child->next = base_scc->preds;
		    base_scc->preds = child;
		}
	    }
	}
    }

    return 1;
}


static void interpret_scc(ParserStruct &parserStruct,
		QFMaterializedModuleData *md, Scc *scc,
		ExportInfo *export_info)
{
    Predicate *pred; Clause *clause;
    int numRules = 0 ;

    // First emit code to solve Scc's needed by this Scc.
    // (This is, after all, bottom-up evaluation.)

    for (pred = scc->preds; pred != NULL; pred = pred->next) {
	for (clause = pred->clauses;
	     clause != *pred->clause_tail;
	     clause = clause->pnext) {
	    if (!clause->seen) continue;
	    int ipred;
	    numRules++ ;
	    for (ipred = 1; ipred <= clause->num_preds; ipred++) {
		Predicate *child = clause->preds[ipred-1]->predicate;
		if (child->scc == scc) continue;
		if (child->scc->emitted == export_info->sequence)
		    continue; // Already dumped.
		child->scc->emitted = export_info->sequence;
		interpret_scc(parserStruct, md, child->scc, export_info);
	    }
	}
    }

// Check that the scc has some rules to evaluate. Otherwise, it is useless
    if (numRules)  md->AddScc(parserStruct, scc) ;

}

static Table *initialize_do_query(QFModuleData *md, Relation *magic,
                                  int old_magic_relation)
{
    // Make a copy of the table. This is to handle recursive calls to do_query.
    // Each call should see an unmodified Table.
    // Except in the case of SaveModule, where each call should see
    // the result of the last call to the module, and recursive calls
    // through a saved module are disallowed.
  
    Table *table_copy;
    int reused_table = 0;

    if (md->module_info.SaveModule) {
	table_copy = md->F;
	ASSERT((md->F != NULL));
	ASSERT((md->F->rels != NULL));
	// check whether this module was called for the first time 
	if (((QFMaterializedModuleData *) md)->saved_state) {
	    reused_table = 1;
	    // each call should see the result of the last call to the 
	    // saved module.
	}
    }
    else {
    	if (md->free_table_list) {
      	    table_copy = md->free_table_list;
      	    md->free_table_list = md->free_table_list->next;
      	    reused_table = 1;
    	}
    	else table_copy = new Table(md->F) ;
    }

    init_rule_pointers(table_copy, md, magic, reused_table, 
			old_magic_relation) ;
  
    return table_copy;
}

void cleanup_do_query(QFModuleData *md, Relation *magic, Table *table_copy)
{
  if (md->module_info.UseOrdSearch) {
    if (table_copy->context) {
      table_copy->context->free_all();
    }
  }
  clear_rule_pointers(table_copy, md, magic) ;

  // probably unnecessary here -- Praveen
  MagicRelation = NULL;
  MagicBindEnv = NULL;
  MagicArgs = NULL;

  clearStatistics((QFMaterializedModuleData *)md, exEnv.profile_file);
}


/*
 * This function is called at run time to execute a query on the module.
 * The magic fact for the query is in the magic relation magic. old_magic_relation
 * specifies if the magic relation is new or old. The iterator is passed in
 * to allow for a staggered bottom-up evaluation mode, where answers at the top
 * level of the module are returned at the end of each iteration of the
 * seminaive loop. This requires that the state of the seminaive computation
 * has to be stored across calls to do_query, and the tuple_iterator is the
 * only structure that can store this state information
 */
Relation *do_query(QFModuleData *md, Relation *magic,
		   int old_magic_relation, TupleIterator *iterator)
{
  Table *table_copy;
  ModuleInfo *temp_module_info;

#ifdef DO_PROFILE
  md->module_info.GlobalProfileInfo.clear();
#endif

  /*
   * If staggered evaluation is not being performed, or this is the
   * first time do_query is being called on this module for this query
   * initialize the table of relations
   */
  if (!(iterator) || !(iterator->state)) {
    /* 
     * Make a copy of the table. This is to handle recursive calls to do_query.
     * Each call should see an unmodified Table.
     */
    table_copy = initialize_do_query(md, magic, old_magic_relation);
  }
  else {
    table_copy = iterator->state->table;
  }

  // this is NEW CODE : even in the case of lazy evaluation, the table stack
  //                    needs to be pushed.

  if (TableStackCount >= exEnv.C_max_recursion_depth_default) {
    fprintf(exEnv.error_file,
	    "CORAL Error::Out of recursion stack space !!\n");
    fprintf(exEnv.error_file, " Current max stack size = %d\n", 
	    exEnv.C_max_recursion_depth_default);
    fprintf(exEnv.error_file,
	    "Use assign(recursion_depth, <new_value>) to increase it !\n");
    fprintf(exEnv.error_file,
	    " Aborting query !\n");
    exEnv.C_interrupt_raised = 1;
  }
  TableStack[TableStackCount++] = table_copy;
  temp_module_info = CurrModuleInfo;
  CurrModuleInfo = &(table_copy->moduledata->module_info);


  Relation *answer_rel =
    table_copy->rels[get_offset(table_copy, md->export_pred)].relPtr;

  /*
   * for the two cases, save_module + and eager_eval -, staggered evaluation
   * is not executed. Since md->Solve() is called with NULL as the iterator
   * argument, the module is completely evaluated, and the iterator->state
   * field remains NULL
   */
  if (!iterator || md->module_info.SaveModule || 
				!(md->module_info.EagerEval)) {
    // no need to evaluate a no_magic module a second time around
    if ((! md->module_info.SaveModule) || 
      		(! ((QFMaterializedModuleData *) md)->no_magic) ||
		(!((QFMaterializedModuleData *) md)->saved_state)) {
        md->Solve(table_copy, NULL);
    }
    cleanup_do_query(md, magic, table_copy);
    TableStack[--TableStackCount] = NULL;
    CurrModuleInfo = temp_module_info;
    return answer_rel;
  }

  /*
   * if this is the first time the module is being called with the iterator,
   * the iterator->state field will be created.
   * otherwise, the existing iterator->state field will be used.
   * when there are no further answers, the state field is deleted and
   * set to NULL by SccInfo::Solve(), and this signifies that the
   * Table and temporary relations can be cleaned up.
   */
  else {
    // this is the first invocation of the module using this iterator
    if (!(iterator->state)) {
      iterator->state = new ModuleExecInfo(md, table_copy, magic, do_query);
    }

    // this sets the rm_Arr field
    md->Solve(table_copy, iterator);

    // if no more answers
    if (!(iterator->state->rm_Arr)) {
      cleanup_do_query(iterator->state->md, iterator->state->magic,
		       iterator->state->table);
      // NOTE that the state->scan_mark field is redundant !!
      delete iterator->state;
      iterator->state = NULL;
      TableStack[--TableStackCount] = NULL;
      CurrModuleInfo = temp_module_info;
      return answer_rel;
    }

    // even for lazy evaluation, pop the table stack, or else,
    // the table stack abstraction is lost !
    TableStack[--TableStackCount] = NULL;    
    CurrModuleInfo = temp_module_info;
    return answer_rel;
  }
}

/*------------------------------------------------------------------

  interpret_materialized_qf()

  This routine creates the structures within the QFMaterializedModuleData,
  and inserts it into the IterativeMethod table. It assumes that SCC
  analysis has already been performed w.r.t. the particular exported
  predicate.

 -------------------------------------------------------------------*/

static void interpret_materialized_qf(QFMaterializedModuleData *md,
				      ExportInfo *export_info,
				      ParserStruct& parserStruct)
{
    Table *F = md->F ;
    RuleInfoArray *rArray = md->ruleArray ;

    Scc *scc;
    struct collection *predicates = new struct collection;
    init_collection(predicates, 1);
    Name exported_name = export_info->pred_name;

    Predicate *query_predicate = export_info->predicate;
    int query_arity = query_predicate->arity;

    // Make sure that all Relations used by the query are initialized
    // by init_rule_pointers. Use add_pred_extern to enter them into
    // the the predicates collection to keep track of them.
    // First enter the the query and magic predicates:
    add_pred_extern(query_predicate, QueryPredKind, predicates);
    int magic_arity = export_info->adornment.count();
    char *magic_name = new char[exported_name->length()+4+query_arity];
    sprint_magic_name(magic_name, exported_name->string(), 
				&export_info->adornment);


    Predicate *magic_predicate =
	LookupPredicate(parserStruct, EnterSymbol(magic_name), magic_arity);
    if (magic_predicate) 
        add_pred_extern(magic_predicate, MagicQueryPredKind,
			 predicates);
    else {
	fprintf(exEnv.error_file,
		"Warning: No magic predicate for exported predicate %s\n",
		exported_name->string() );
	if (parserStruct.CurModule.SaveModule) {
	    md->no_magic = 1;
        }
    }

    // Then enter the remaining predicates.
    for (scc = SccList; scc != NULL; scc = scc->next) {
	Predicate *pred; Clause *clause; int ipred; Literal** literals;

	// Add predicates for non-recursive clauses in this scc.
	for (pred = scc->preds; pred != NULL; pred = pred->next) {
	    clause = pred->clauses;
	    // If pred->clauses == *pred->clause_tail, then
	    // the pred has no clauses, and hence is an ExternalPredKind.
	    if (clause != *pred->clause_tail) {
	        add_pred_extern(pred, LocalPredKind, predicates);
	    }
	    else if (pred->is_mp_done()) {
		// NOTE: this is for ordered search
		add_pred_extern(pred, LocalPredKind, predicates);
	    }
	    else add_pred_extern(pred, ExternalPredKind, predicates);

	    // This also determines if clause is recursive.
	    for (; clause != *pred->clause_tail; clause = clause->pnext) {
		if (!clause->seen) continue;
		clause->recursive = 0;
		for (literals = clause->preds, ipred = clause->num_preds;
		     --ipred>= 0; literals++) {
		    add_pred_extern(literals[0]->predicate,
				     ExternalPredKind,
				     predicates);

		    // with Ordered Search, have to special case the 
		    // negated literals
		    if (parserStruct.CurModule.UseOrdSearch) {
			if (literals[0]->negated) continue;
		    }
		    // don't need to special case for done_m_p since 
		    // it is covered by the next if condition

		    if (literals[0]->predicate->scc == scc)
			clause->recursive = 1;
		}
	    }
	}
    }

    md->rules = predicates;
    md->export_pred = export_info->predicate;
    md->module_info = parserStruct.CurModule;
    if (md->module_info.UsePredicateSN) {
      if (md->module_info.UseOrdSearch) {
        fprintf(exEnv.error_file, 
                "CORAL error: Cannot use PSN with Ordered Search ... yet :-)");
        fprintf(exEnv.error_file, " using basic SN instead\n");
        md->use_psn = 0;
        md->module_info.UsePredicateSN = 0;
      }
      else if (md->module_info.SaveModule) {
        fprintf(exEnv.error_file,
                "CORAL error: Cannot use PSN with SaveModule ... yet :-)");
        fprintf(exEnv.error_file, " using basic SN instead\n");
        md->use_psn = 0;
        md->module_info.UsePredicateSN = 0;
      }
      else if (md->module_info.SupMagicIndexing) {
        fprintf(exEnv.error_file,
                "CORAL error: Cannot use PSN with SupMagicIndexing ... yet :-)");
        fprintf(exEnv.error_file, " using basic SN instead\n");
        md->use_psn = 0;
        md->module_info.UsePredicateSN = 0;
      }
    }
    else md->use_psn = md->module_info.UsePredicateSN;

    interpret_scc(parserStruct, md, export_info->predicate->scc, export_info);


    // Insert the module structure along with its associated exported
    // query form into the IterativeMethod table.
    if (exported_name) {
	DerivedMethod* query_method = new DerivedMethod(exported_name,
					export_info->adornment, do_query,
					(QFModuleData *)md); 
    }
}

/*------------------------------------------------------------------
 interpret_module()

 Top-level routine to interpret a CORAL module. For each exported query form,
 a QFModuleData structure is created that encapsulates the information needed
 to evaluate that query. The exported query form and the associated 
 QFModuleData structure are stored in a table of IterativeMethods.

 -------------------------------------------------------------------*/
void interpret_module(ParserStruct &parserStruct)
{

    ExportInfo *export_info;

    if (parserStruct.CurModule.ExportedPreds == NULL) {
	fprintf(exEnv.error_file, "CORAL: Error - No exported query form.\n");
	return;
    }

    // Create the Table structure (containing a pointer to each Relation used).
    int pred_count = 0 ;
    for (Predicate *pred = parserStruct.AllPredicates; pred != NULL;
					pred = pred->allprednext) 
	pred_count++ ;

    Table *module_table = new Table(pred_count);
    RuleInfoArray *module_rules = new RuleInfoArray;

    // This enters the name and arity of each relation in the Table into the
    // Table array 'module_table'. The relations are also initialized at the
    // same time.

    int i = 0 ;
    for (pred = parserStruct.AllPredicates; pred != NULL; 
				pred=pred->allprednext, i++){
	module_table->AddValue(i, pred->name, pred->arity) ;
	pred->frame_offset = i;
    }


// For each clause, create a rule-info structure that contains the 
// information necessary to evaluate it.

    int rule_number = 0;
    FOR_ALL_RULES(clause) {
	clause->rule_number = rule_number++;
	module_rules->AddRule(create_rule_info(module_table, clause));
    } END_EACH_RULE

    QFModuleData *module_data ;

    // Process each exported query form independently.
    for (export_info = parserStruct.CurModule.ExportedPreds;
	 export_info != NULL; export_info = export_info->next) {
	Name query_symbol;
	Name exported_name = export_info->pred_name;
	int query_arity;

	if (parserStruct.CurModule.UsePipelining) 
	  module_data = new 
	    QFPipelinedModuleData(module_table, module_rules, parserStruct);
	else
	  module_data = new 
	    QFMaterializedModuleData(module_table, module_rules, parserStruct);

	if (export_info->adorn_symbol == 0) {
	    query_arity = 0;
	    query_symbol = exported_name;
	}
	else 
	{
	    Name exported_adorn_name = export_info->adorn_symbol;
	    query_arity = (int) exported_adorn_name->length();
		
		if (parserStruct.CurModule.UsePipelining)
		{
			int len = (int) exported_name->length() + 1;
			char *query_name = new char[len+1];
			sprintf(query_name,"%s",exported_name->string());
			query_symbol = EnterSymbol(query_name);
		}
		else
		{
	    	int len = (int) exported_name->length() + 1 + query_arity;
	    	char *query_name = new char[len+1];
	   		sprint_pred_name(query_name, exported_name->string(), 
				exported_adorn_name->string());
	    	query_symbol = EnterSymbol(query_name);
		}

	}
	// Ensure that the exported query form is a valid one.
	export_info->predicate = 
	  LookupPredicate(parserStruct, query_symbol, query_arity);
	if (!export_info->predicate) {
	    fprintf(exEnv.error_file,
		"Error: Exported predicate %s not defined in module\n",
		query_symbol->string() );
	    continue;
	}

	if (parserStruct.CurModule.UsePipelining) {
	  interpretPipelined ((QFPipelinedModuleData *) module_data,
			      export_info, parserStruct);
	}

	else { // For MATERIALIZED modules
	  // Do Scc (Strongly Connected Components) analysis.
	  FOR_ALL_RULES(clause) {
	    clause->seen = 0;
	      
	    clause->recursive = 0;
	    for (int ipred = 0; ipred < clause->num_preds; ipred++) {
	      clause->preds[ipred]->predicate->work = NULL;
	      clause->preds[ipred]->predicate->mark = 0;
	    }
	  } END_EACH_RULE
	  initialize_scc_structs() ;

	  if (parserStruct.CurModule.SingleScc ||
	      parserStruct.CurModule.UseOrdSearch ||
	      parserStruct.CurModule.HasPrioritize) {
	    mark_predicates_invoked_from(export_info->predicate);
	    group_predicates_in_one_scc(export_info->predicate);
	    clean_up_after_one_scc();

	  }
	  else {
	    mark_predicates_invoked_from(export_info->predicate);
	    find_predicates_invoked_from(export_info->predicate);
	  }
	  
	  interpret_materialized_qf((QFMaterializedModuleData *)module_data,
				    export_info, parserStruct);
	}

    }

}
