/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************
	CORAL Software :: U.W.Madison

	database.C

	Contains declarations for handling multiple databases.
 ***********************************************************************/

#include "externs.h"
#include "globals.h"
#include "cor_error.h"
#include "interface.h"
#include <strings.h>

#define COR_DB_RELS "db_rels"
#define COR_BUILTINS "builtin_rels"

DatabaseStruct BuiltinDB(EnterSymbol(COR_BUILTINS));
DatabaseStruct *CurDB = &BuiltinDB;
struct SymbolTable *DatabaseTable = NULL ;

extern Association * DeleteInTab(struct SymbolTable *tab, ConstPointer arg);

#ifdef WITH_PERSISTENCE
extern void listFiles(FILE *);
#endif

extern Relation *makePersistentRel(FuncArg *) ;
extern void deletePersistentRelation(Name, int);
extern int SM_initialized;
typedef Association (*Func)(Pointer, Association*);
extern void ForEachTabElement(struct SymbolTable *, Func, Pointer) ;

DatabaseStruct *lookupDB(Name name)
{
  Association *ptr;
  if (DatabaseTable == NULL)
    DatabaseTable = AllocTab(COR_DATABASE_TABLE_INCR, NULL);
  
  ptr = SymbolLookup(DatabaseTable, name);
  
  if (HashNone(ptr->arg)) return 0 ;
  else return (DatabaseStruct *)(ptr->val) ;
}

DatabaseStruct *createDB(Name name)
{
  char err_msg[35];
  DatabaseStruct *db ;

  if (db = lookupDB(name)) {
    sprintf(err_msg, "%10s.. exists already", SymbolString(name));
    err_msg[34] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "createDB");
    return NULL ;
  }

  Association *ptr;
  ptr = SymbolLookup(DatabaseTable, name);

  db = new DatabaseStruct(name);
  ptr->arg = (Name)name;
  ptr->val = db ;
  TabInserted(DatabaseTable, ptr);
  db->RelationTable = AllocTab(COR_RELATION_TABLE_INCR, NULL);
  return db ;
}

// Used by init_coral to add the default DB to the DatabaseTable
int addDBStruct(DatabaseStruct *db)
{
  if (db) {
    if (lookupDB(db->name)) {
      return 0 ;
    }

    Association *ptr;
    ptr = SymbolLookup(DatabaseTable, db->name);

    ptr->arg = (Name)(db->name);
    ptr->val = db ;
    TabInserted(DatabaseTable, ptr);
    return 1;
  }
  else
    return 0 ;
}

static Association clear_relation(Pointer , Association* ass)
{
  if (! --(((Relation *)(ass->val))->ref_count)) {
    ((Relation *)(ass->val))->empty_relation(0) ;
    delete (Relation *)(ass->val) ;
  }
  return *ass ;
}

int destroyDB(Name name)
{
  char err_msg[35];

  if (name == CurDB->name) {
    CORAL_error(COR_MULTIPLE_DB_ERR, "cannot destroy current DB", "destroyDB");
    return 0 ;
  }

  if (!strcmp(SymbolString(name), COR_BUILTINS)) {
    fprintf(stderr, "Cannot destroy the builtin database \n");
    return 0 ;
  }

#ifdef WITH_PERSISTENCE
  if (!strcmp(SymbolString(name), COR_DB_RELS)) {
    fprintf(stderr,
	    "Cannot destroy the persistent database \n");
    return 0 ;
  }
#endif

  DatabaseStruct *db ;
  if (!(db = lookupDB(name))) {
    sprintf(err_msg, "%10s .. does not exist", SymbolString(name));
    err_msg[34] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "destroyDB");
    return 0;
  }

  if (!DeleteInTab(DatabaseTable, name)) {
    sprintf(err_msg, "%10s.. does not exist", SymbolString(name));
    err_msg[34] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "destroyDB");
    return 0 ;
  }

  ForEachTabElement(db->RelationTable, (Func)clear_relation, 0) ;
  delete db->RelationTable ;
  delete db ;
  return 1;
}

int currentDB(Name name)
{
  char err_msg[35];
  DatabaseStruct *db;

  if (!strcmp(SymbolString(name), COR_BUILTINS)) {
    fprintf(stderr, "Cannot make the builtin database the current database\n");
    return 0 ;
  }

#ifdef WITH_PERSISTENCE
  if (!strcmp(SymbolString(name), COR_DB_RELS)) {
    fprintf(stderr,
	    "Cannot make the persistent database the current database\n");
    return 0 ;
  }
#endif

  if (!(db = lookupDB(name))) {
    sprintf(err_msg, "%10s.. does not exist", SymbolString(name));
    err_msg[34] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "currentDB");
    return 0 ;
  }
  CurDB = db ;
  return 1;
}

int renameDB(Name dbname, Name newName)
{
  char err_msg[35];
  DatabaseStruct *db;

  if (!strcmp(SymbolString(dbname), COR_BUILTINS)) {
    fprintf(stderr, "Cannot rename the builtin database\n");
    return 0 ;
  }

#ifdef WITH_PERSISTENCE
  if (!strcmp(SymbolString(dbname), COR_DB_RELS)) {
    fprintf(stderr, "Cannot rename the persistent database\n");
    return 0 ;
  }
#endif

  if (!(db = lookupDB(dbname))) {
    sprintf(err_msg, "%10s.. does not exist", SymbolString(dbname));
    err_msg[34] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "renameDB");
    return 0 ;
  }

  if (lookupDB(newName)) {
    sprintf(err_msg, "%10s.. already exists", SymbolString(newName));
    err_msg[34] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "renameDB");
    return 0 ;
  }

  DeleteInTab(DatabaseTable, dbname);
  Association *ptr ;
  ptr = SymbolLookup(DatabaseTable, newName);
  ptr->arg = (Name)newName;
  ptr->val = db ;
  TabInserted(DatabaseTable, ptr);
  db->name = newName ;
  return 1;
}

static Association dump_relation(Pointer outf, Association* ass)
{
  ((Relation *)(ass->val))->print_facts((FILE *)outf) ;
  return *ass ;
}

int saveDB(Name dbname, FILE *outf)
{
  char err_msg[35];
  DatabaseStruct *db;

  if (!dbname)  db = CurDB;

  else if (!strcmp(SymbolString(dbname), COR_BUILTINS)) {
    fprintf(stderr, "Cannot save the builtin database\n");
    return 0;
  }

  else if (!(db = lookupDB(dbname))) {
    sprintf(err_msg, "%10s.. does not exist", SymbolString(dbname));
    err_msg[34] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "saveDB");
    return 0;
  }

  fprintf(outf, "db_create(%s).\n", SymbolString(db->name));
  fprintf(outf, "db_current(%s).\n", SymbolString(db->name));
  ForEachTabElement(db->RelationTable, (Func)dump_relation, (Pointer)outf) ;

  return 1;
}

int copyToDB(FuncArg *srcrel, Name dbname, FuncArg *dstrel)
{
  DatabaseStruct *db ;
  char err_msg[100] ;

  if (!(db = lookupDB(dbname))) {
    sprintf(err_msg, "%10s .. DB does not exist", SymbolString(dbname));
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "copyToDB");
    return 0;
  }

  Association *ptr = SymbolLookup(db->RelationTable, srcrel->functor());
  if (HashNone(ptr->arg)) {
    sprintf(err_msg,"rel %10s ..does not exist in DB %s",
	 SymbolString(srcrel->functor()), SymbolString(dbname));
    CORAL_error(COR_BAD_REL_NAME, err_msg, "copyToDB");
    return 0 ;
  }
  Relation *rel1 = (Relation *)(ptr->val);

  ptr = SymbolLookup(CurDB->RelationTable, dstrel->functor());
  if (!HashNone(ptr->arg)) {
    sprintf(err_msg,"rel %10s ..already exists in current DB",
	 SymbolString(dstrel->functor()));
    CORAL_error(COR_BAD_REL_NAME, err_msg, "copyToDB");
    return 0 ;
  }

  Relation *rel2 = make_relation(dstrel->functor(), dstrel->arity());

  Tuple *tuple ;
  FOR_EACH_TUPLE(tuple, rel1) {
    rel2->insert_new(tuple) ;
  }END_EACH_TUPLE(tuple)

  return 1 ;
}

int addToDB(FuncArg *srcrel, Name dbname, FuncArg *dstrel)
{
  DatabaseStruct *db ;
  char err_msg[100] ;

  if (!(db = lookupDB(dbname))) {
    sprintf(err_msg, "%10s .. DB does not exist", SymbolString(dbname));
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "addToDB");
    return 0;
  }

  Association *ptr = SymbolLookup(db->RelationTable, srcrel->functor());
  if (HashNone(ptr->arg)) {
    sprintf(err_msg,"rel %10s ..does not exist in DB %s",
	 SymbolString(srcrel->functor()), SymbolString(dbname));
    CORAL_error(COR_BAD_REL_NAME, err_msg, "addToDB");
    return 0 ;
  }
  Relation *rel1 = (Relation *)(ptr->val);

  ptr = SymbolLookup(CurDB->RelationTable, dstrel->functor());
  if (HashNone(ptr->arg)) {
    sprintf(err_msg,"rel %10s ..does not exist in current DB",
	 SymbolString(dstrel->functor()));
    CORAL_error(COR_BAD_REL_NAME, err_msg, "addToDB");
    return 0 ;
  }
  Relation *rel2 = (Relation *)(ptr->val);

  Tuple *tuple ;
  FOR_EACH_TUPLE(tuple, rel1) {
    rel2->insert_new(tuple) ;
  }END_EACH_TUPLE(tuple)

  return 1 ;
}

int removeRelFromDB(FuncArg *rel, Name dbname)
{
  char err_msg[45] ;
  DatabaseStruct *db ;

#ifdef WITH_PERSISTENCE
  if (!strcmp(SymbolString(dbname), COR_DB_RELS)) {
    // the relation is to be removed from the persistent database
    deletePersistentRelation(rel->functor(), rel->arity());
    return 1;
  }
#endif

  if (dbname == CurDB->name) db = CurDB ;
  else if (!(db = lookupDB(dbname))) {
    sprintf(err_msg, "db %10s ..does not exist", SymbolString(dbname));
    err_msg[44] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "removeRelInDB");
    return 0 ;
  }

  Association *ptr = SymbolLookup(db->RelationTable, rel->functor());
  if (HashNone(ptr->arg)) {
    sprintf(err_msg,"rel %10s ..does not exist", SymbolString(rel->functor()));
    err_msg[44] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "removeRelInDB");
    return 0 ;
  }

  else {
   if (! --(((Relation *)(ptr->val))->ref_count)) {
     ((Relation *)(ptr->val))->empty_relation(0) ;
     delete (Relation *)(ptr->val) ;
   }
   DeleteInTab(db->RelationTable, rel->functor());
  }

}

int clearRelInDB(FuncArg *rel, Name dbname)
{
  char err_msg[45] ;

  DatabaseStruct *db ;
  if (dbname == CurDB->name) db = CurDB ;
  else if (!(db = lookupDB(dbname))) {
    sprintf(err_msg, "db %10s ..does not exist", SymbolString(dbname));
    err_msg[44] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "clearRelInDB");
    return 0 ;
  }

  Association *ptr = SymbolLookup(db->RelationTable, rel->functor());
  if (HashNone(ptr->arg)) {
    sprintf(err_msg,"rel %10s ..does not exist", SymbolString(rel->functor()));
    err_msg[44] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "clearRelInDB");
    return 0 ;
  }

//  else ((Relation *)(ptr->val))->empty_relation(0) ;
  else ((Relation *)(ptr->val))->empty_relation(1) ;
  return 1 ;
}

static void InsertRel(DatabaseStruct *db, Name n, int ,  Relation *r)
{
  if (!db->RelationTable)
    db->RelationTable = AllocTab(COR_RELATION_TABLE_INCR, NULL);
  Insert(db->RelationTable, n, r);
}


int openRelInCurDB(FuncArg *rel, Name dbname)
{
  char err_msg[35];
  DatabaseStruct *db;

  if (!strcmp(SymbolString(dbname), COR_BUILTINS)) {
    fprintf(stderr, "Access already exists to builtin relations\n");
    return 0 ;
  }

#ifdef WITH_PERSISTENCE
  if (!strcmp(SymbolString(dbname), COR_DB_RELS)) {
    // the persistent database is being accessed
    Relation *p_rel = makePersistentRel(rel) ;
    InsertRel(CurDB, rel->functor(), rel->arity(), p_rel);  
    return 1;
  }
#endif

  else if (!(db = lookupDB(dbname))) {
    sprintf(err_msg, "%10s.. does not exist", SymbolString(dbname));
    err_msg[34] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "openRelInDB");
    return 0 ;
  }

  Association *ptr = SymbolLookup(db->RelationTable, rel->functor());
  if (!HashNone(ptr->arg)) {
    InsertRel(CurDB, rel->functor(), rel->arity(), (Relation *)ptr->val);
    (((Relation *)(ptr->val))->ref_count)++ ;
    return 1;
  }
  
  CORAL_error(COR_REL_NOT_FOUND, rel->functor()->string(),"OpenRelInCurDB");
  return 0;

}

static Association displayDBName(Pointer , Association *ass)
{
  fprintf(stderr, "%s\n" , SymbolString((Symbol *)(ass->arg)));
  return *ass ;
}

int listEveryDB()
{
#ifdef WITH_PERSISTENCE
  if (SM_initialized)
    fprintf(stderr, "%s\n", COR_DB_RELS);
#endif
  ForEachTabElement(DatabaseTable, (Func)displayDBName, 0) ;
  return 1 ;
}


Association display_relation_name(Pointer outf, Association *ass)
{
  ((Relation *)(ass->val))->print_name((FILE *)outf);

  return *ass ;
}
//
// Will print out a builtin relation only if it is printable.
//
Association display_builtin_relation_name(Pointer outf, Association *ass)
{
  Relation *r = ((Relation *)(ass->val)); // save so we dont have to deref 
  
  if(((BuiltinRelation*)r)->is_printable_builtin()) // if builtin % printable, do so.
    r->print_name((FILE *)outf) ;

  return *ass ;
}

int listRelsInDB(Name dbname, FILE *outfile = stderr)
{
  char err_msg[35];
  DatabaseStruct *db;

#ifdef WITH_PERSISTENCE
  if (!strcmp(SymbolString(dbname), COR_DB_RELS)) {
    // the persistent database is being accessed
    listFiles(outfile) ;
    return 1;
  }
#endif

  if (!(db = lookupDB(dbname))) {
    sprintf(err_msg, "%10s.. does not exist", SymbolString(dbname));
    err_msg[34] = '\0';
    CORAL_error(COR_MULTIPLE_DB_ERR, err_msg, "currentDB");
    return 0 ;
  }
  if (!strcmp(SymbolString(dbname), COR_BUILTINS)) // id the relation is a builtin
    ForEachTabElement(db->RelationTable, (Func)display_builtin_relation_name,
		      outfile) ;
  else
    ForEachTabElement(db->RelationTable, (Func)display_relation_name,
		      outfile) ;

  return 1 ;
}
