/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

#ifdef WITH_PERSISTENCE

#include "persistent-rel.h"
#include <strings.h>
#include "interface.h"
#include "cor_error.h"

extern int unify_literal(TupleIterator&, Tuple *, int) ;
extern int SM_initialized;

#define DELETE_INDEX_ENTRY 1
#define DELETE_OBJ_AND_INDEX_ENTRY 2


/*------------------------------------------------------------------
  PersistentRel() : Constructor that initializes the persistent-rel
  structure from the catalog if it exists in the catalog. Otherwise,
  an error is raised.

 Oddities/Quirks :: Note that the file names are only guarenteed to be
 unique upto COR_MAX_FILE_NAME_LEN

 -------------------------------------------------------------------*/
PersistentRel::PersistentRel(Name name, int arity, int& error)
  : Relation(arity, -1)
{
  if (!SM_initialized) return ;

  OID oid ;

  SM.startTrans();

  if (!SM.lookupFile(name, arity, &oid)) {
    fprintf(stderr, "SORRY ! Unknown persistent relation referenced !\n");
    error = 1 ;
    return ;
  }

  USERDESC *u_desc ;
  if (!SM.retrieveObject(&oid, &u_desc)) {
    fprintf(stderr, "SORRY ! Unknown persistent relation referenced !\n");
    fprintf(stderr, "Error in catalogs !!\n");
    error = 1 ;
    return ;
  }

  fd.copyDesc((FileDescStruct *)(u_desc->basePtr));

  this->name = name ;
  this->oid  = oid ;
  ref_count = 2 ;            // This is to prevent it from being emptied
                             // when it is closed in a database other than
			     // db_rels.
  count = fd.count;
  SM.releaseObject(u_desc) ;
}

void PersistentRel::print_name(FILE *outf)
{
  fprintf(outf, "%15s/%d  : (exodus)  ", SymbolString(name), arity()) ;
  fprintf(outf, ": indexes ");
  fd.print_index_adornments(outf);
//  fprintf(outf, ": count %d\n", count);
  fprintf(outf, "\n");
}

void PersistentRel::print_facts(FILE *outf, ArgList *) const
{
  fprintf(outf, "rel_open(%s(", SymbolString(name));
  for (int i = 0; i < arity() -1 ; i++) {
    fprintf(outf, "x,");
  }

  if (arity()) fprintf(outf, "x), db_rels).\n") ;
  else fprintf(outf, "), db_rels).\n");

}

/*------------------------------------------------------------------
  insertObject() : Inserts the tuple object specified by the arglist
  into the relation.

 Oddities/Quirks :: The schema describes the structure of the data.
 Each field of the tuple is separately added to the object. In the case
 of strings, the length of the string(an int) is first written to the
 file, followed by the actual string. The object is also added to the
 appropriate existing indexes.

 -------------------------------------------------------------------*/
void PersistentRel::insertObject(ArgList &arglist)
{
  if (!SM_initialized) return  ;

  // Check that arity of relation and arglist match - defensive programming
  if (arglist.arity() != arity()) {
    CORAL_error(COR_BAD_ARGLIST, "this error shouldn't happen",
		"PersistentRel::insert");
    return  ;
  }

  SM.startTrans();

  // Using the buffer group of the Storage Manager ! This is probably
  // not optimal, but its better than creating a separate buffer pool
  // just for this
  OID oid ;
  int e = sm_CreateObject(SM.bufGroup(), &(fd.fid), NEAR_LAST_PHYSICAL,
			  NULL, NULL, 0,
			  NULL, &oid) ;
  ErrorCheck(e, "sm_CreateObject");

  int str_len = 0 ;
  int data_len = 0 ;
  char *data_ptr = NULL;
  int tmp_int ;
  double tmp_double ;

  // Fill in each field of the object
  for (int i = 0; i < arity(); i++) {
    switch(fd.schema[i]) {

    case COR_STRING_DBTYPE :
      str_len = strlen(data_ptr = make_string(arglist[i])) + 1 ;
      e = sm_AppendToObject(SM.bufGroup(), &oid, (FOUR)sizeof(int),
			    (char *)(&str_len)) ;
      ErrorCheck(e, "sm_AppendToObject");
      data_len = str_len * sizeof(char) ;
      break ;

    case COR_INT_DBTYPE :
      tmp_int = make_int(arglist[i]);
      data_ptr = (char *)(&tmp_int) ;
      data_len = sizeof(int) ;
      break ;

    case COR_DOUBLE_DBTYPE :
      tmp_double = make_double(arglist[i]);
      data_ptr = (char *)(&tmp_double) ;
      data_len = sizeof(double) ;
      break ;

    default :
      CORAL_error(COR_BAD_SCHEMA, "INVALID field reached", "insertObject");
      return  ;
      // This should not happen !!
    }

    e = sm_AppendToObject(SM.bufGroup(), &oid, (FOUR)data_len,
			  (char *)(data_ptr)) ;
    ErrorCheck(e, "sm_AppendToObject");
    delete data_ptr;
  }

  USERDESC *u_desc ;
  int index_offset = -1 ;

  // Add object to all valid indexes

  if (fd.num_indexes) {
    SM.retrieveObject(&oid, &u_desc) ; 
    for (i = 0; i < fd.num_indexes; i++) {
      if ((index_offset = fd.nextIndex(index_offset)) < 0) {
	CORAL_error(COR_INDEX_ERR, NULL, "PersistentRel::insertObject");
	SM.releaseObject(u_desc) ;
	return ;
      }
      fd.addObjectToIndex(index_offset, u_desc) ;
    }
  }
  (fd.count)++;
  
  if (fd.num_indexes)
    SM.releaseObject(u_desc) ;
}

/*------------------------------------------------------------------
  PersistentRel::checkTuple(USERDESC *, TupleIterator &)

   Determine if the retrieved tuple satisfies the bindings so far.
   If so, add to the bindings(change the bindenv) and return
   To do this, create an arglist that reflects the tuple
   and call unify_arglists(iterator.arglist, new_arglist,
  			     iterator.bindenv, NULL_BINDENV);
   This automatically checks for validity of the tuple and if valid
   also pushes the new environment on to the stack.

 Oddities/Quirks :: 
  In general, the tuple may contain variable length strings as fields.
  So the length of a string field is stored as an integer in front of it.

 -------------------------------------------------------------------*/
Tuple * PersistentRel::checkTuple(USERDESC *u_desc, TupleIterator &iterator)
{ 
  if (!SM_initialized) return NULL ;

  ArgList *new_arglist ;
  StackMark  stackmark0;
  int success = 0 ;
  int offset = 0 ;
  int str_len = 0 ;
  int data_len = 0 ;

  // Create an arglist corresponding to the retrieved tuple.
    new_arglist = ArgList::New(this->arity()) ;

    for (int i=0; i < arity(); i++) {
      switch(fd.schema[i]) {

      case COR_STRING_DBTYPE :
	memcpy((char *)&str_len ,(char *)(u_desc->basePtr+offset),sizeof(int));
	offset = offset + sizeof(int) ;
	data_len = str_len * sizeof(char) ;
	if ((((iterator.arg_list)[i])->kindof() != COR_VARIABLE) ||
	    (((VarArg *)((iterator.arg_list)[i]))->var_name() !=DontCareSymbol))
	  (*new_arglist)[i] = make_arg((char *)(u_desc->basePtr + offset)) ;
	else
	  (*new_arglist)[i] = iterator.arg_list[i];
	  
	break ;
	
      case COR_INT_DBTYPE :
	data_len = sizeof(int) ;
	int itmp ;
	if ((((iterator.arg_list)[i])->kindof() != COR_VARIABLE) ||
	   (((VarArg *)((iterator.arg_list)[i]))->var_name() !=DontCareSymbol)){
	  memcpy((char *)&itmp, (char *)(u_desc->basePtr+offset), sizeof(int));
	  (*new_arglist)[i] = make_arg(itmp) ;
	}
	else
	  (*new_arglist)[i] = iterator.arg_list[i];
	break ;
	
      case COR_DOUBLE_DBTYPE :
	data_len = sizeof(double) ;
	double dtmp ;
	if ((((iterator.arg_list)[i])->kindof() != COR_VARIABLE) ||
	   (((VarArg *)((iterator.arg_list)[i]))->var_name() !=DontCareSymbol)){
	  memcpy((char *)&dtmp, (char *)(u_desc->basePtr+offset), sizeof(double));
	  (*new_arglist)[i] = make_arg(dtmp) ;
	}
	else
	  (*new_arglist)[i] = iterator.arg_list[i];
	break ;
	
      default :
	CORAL_error(COR_BAD_SCHEMA, "INVALID field reached", "checkTuple");
	return 0 ;
      }

      offset += data_len ;
    }

  // Create a Tuple corresponding to the retrieved tuple.
    Tuple *tuple = new Tuple(new_arglist) ;

  // Call unify_literal
  if (unify_literal(iterator, tuple, exEnv.dbg_indexing))
    return tuple ;

  delete tuple;
  delete new_arglist;
  stackmark0.pop_to();
  return NULL ;
}

#undef FAIL_RETURN
#define FAIL_RETURN      {	iterator.scanDesc = NULL ;\
                                iterator.ipos = Scanning_Done ;\
                                iterator.set_no_match() ; \
	                        return NULL; }

/*------------------------------------------------------------------
  fileScanGetNext(iterator) : Gets the next tuple from the persistent
  relation using a file scan. The tuple should satisfy the selection
  implicit in the arglist of the iterator. The new bindings due to
  the selected tuple are (automatically) pshed onto the global
  trail stack.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
Tuple * PersistentRel::fileScanGetNext(TupleIterator &iterator, int delete_flag)
{
  if (!SM_initialized) FAIL_RETURN ;

  int e ;
  USERDESC *u_desc ;
  BOOL     eof ;

  SM.startTrans();

  if (exEnv.C_interrupt_raised) FAIL_RETURN;

  // If this is the first time get_next is being called, the scan
  // may have to be initialized. However, subsequent get_next() calls
  // should continue from where they left off.

    if (!iterator.scanDesc) {
      e = sm_OpenScanWithGroup(&(fd.fid), PHYSICAL_SCAN, SM.bufGroup(),
			       (SCANDESC **)&(iterator.scanDesc), NULL);
      ErrorCheck(e, "sm_OpenScanWithGroup") ;
      if (e < 0) FAIL_RETURN ;
    }

    for (;;) {
      e = sm_ScanNextObject((SCANDESC *)iterator.scanDesc, 0,
			    READ_ALL, &u_desc, &eof) ;
      ErrorCheck(e, "sm_ScanNextObject") ;


      // If there are no more tuples, return.
      if ((e <0) || eof) {
	e = sm_CloseScan((SCANDESC *)iterator.scanDesc) ;
	ErrorCheck(e, "sm_CloseScan") ;
	FAIL_RETURN ;
      }

      // Determine if the retrieved tuple satisfies the bindings so far.
      // If so, add to the bindings(change the bindenv) and return
      Tuple *tuple ;
      if (tuple = checkTuple(u_desc, iterator)) {
	if (delete_flag) {
	  SM.destroyObject(&(u_desc->oid), SM.bufGroup());
	}
	return tuple;
      }
      // Else, get next tuple (return to loop )

    }

}

#undef FAIL_RETURN
#define FAIL_RETURN   { INVALIDATE_CURSOR(*((SMCURSOR *)iterator.cursor)) ;\
			iterator.ipos = Scanning_Done ;\
			iterator.set_no_match() ;\
			return NULL ;	}

/*------------------------------------------------------------------
  indexScanGetNext(TupleIterator &iterator, IID iid) :
  Gets the next tuple from the persistent relation using an index scan
  using the given index id. The tuple should satisfy the selection
  implicit in the arglist of the iterator. The new bindings due to
  the selected tuple are (automatically) pushed onto the global
  trail stack.

 Oddities/Quirks ::

 -------------------------------------------------------------------*/
Tuple * PersistentRel::indexScanGetNext(TupleIterator &iterator, int index,
					int delete_flag)
{
  if (!SM_initialized)  FAIL_RETURN ;

  USERDESC *u_desc ;
  BOOL     eof ;
  KEY key ;
  OID oid ;
  int tmp_int ;
  double tmp_double ;
  char value_buff[COR_MAX_KEY_LEN] ;
  int buff_offset = 0 ;
  char *data_ptr = NULL;
  int data_len ;
  int str_len ;
  int e ;

  SM.startTrans();

  if (exEnv.C_interrupt_raised) FAIL_RETURN;

  // If this is the first time get_next is being called, the scan
  // may have to be initialized. However, subsequent get_next() calls
  // should continue from where they left off.

  if (CURSOR_IS_INVALID(*((SMCURSOR *)iterator.cursor))) {

  // Fill value_buff with the key value.
  // This is done by copying the appropriate attributes into value_buff

    ASSERT(iterator.arg_list.count() == arity()) ;

    for (int i = 0; i < arity(); i++) {
      if (!(iterator.arg_list[i]->isConstant())) continue ;

      switch(fd.schema[i]) {
	
      case COR_INT_DBTYPE :
	if (iterator.arg_list[i]->kindof() != COR_NUM_CONST) {
	  CORAL_error(COR_BAD_ARGLIST, "int expected", "indexScanGetNext");
	  FAIL_RETURN
	}
	tmp_int = make_int(iterator.arg_list[i]) ;
	data_ptr = (char *)&tmp_int ;
	data_len = sizeof(int) ;
	break ;

      case COR_STRING_DBTYPE  :
	if (iterator.arg_list[i]->kindof() != COR_SYMBOL_CONST) {
	  CORAL_error(COR_BAD_ARGLIST, "string expected", "indexScanGetNext");
	  FAIL_RETURN
	}
	data_ptr = make_string(iterator.arg_list[i]) ;
	str_len = strlen(data_ptr)+1 ;
	data_len = str_len * sizeof(char) ;
	break ;

      case COR_DOUBLE_DBTYPE :
	if (iterator.arg_list[i]->kindof() != COR_NUM_CONST) {
	  CORAL_error(COR_BAD_ARGLIST, "double expected", "indexScanGetNext");
	  FAIL_RETURN
	}
	tmp_double = make_double(iterator.arg_list[i]) ;
	data_ptr = (char *)&tmp_double ;
	data_len = sizeof(double) ;
	break ;

      default :
	CORAL_error(COR_BAD_SCHEMA, "INVALID field reached",
		      "indexScanGetNext");
	FAIL_RETURN;
	// This should not happen !!
      }

      if ((buff_offset+data_len < COR_MAX_KEY_LEN) && 
	  fd.indexes[index].attrs[i]){
	memcpy((char *)(value_buff + buff_offset),(char *)(data_ptr),
	       data_len ) ;
	buff_offset += data_len ;
      }
    }

 
    key.length = (TWO)(buff_offset) ;
    key.valuePtr = (void *)(value_buff) ;

    // Initialize the index scan
    e = sm_FetchInit(&(fd.indexes[index].iid), SM.bufGroup(),
		     &key, SM_EQ, &key, SM_EQ, (SMCURSOR *)iterator.cursor) ;
    
    ErrorCheck(e, "sm_FetchInit") ;
    if (e < 0) FAIL_RETURN ;
  }


  // At this stage, the scan has already been initialized either during
  // this function invocation, or in some previous invocation.
  for (;;) {
    // Get the next OID from the index
    e = sm_FetchNext((SMCURSOR *)iterator.cursor, NULL, &oid, &eof) ;
    ErrorCheck(e, "sm_FetchNext") ;

    // If there are no more index entries, return.
    if ((e <0) || (eof)) FAIL_RETURN ;
/*
    
    if (!SM.retrieveObject(&oid, &u_desc)) {
      fprintf(stderr, "CORAL::Error in index !!\n");
      continue ;
    }
*/
    if (!SM.retrieveObject(&oid, &u_desc)) FAIL_RETURN ;
    
    Tuple *tuple ;
    // Determine if the retrieved tuple satisfies the bindings so far.
    // If so, if delete_flag is set, delete the object, else
    // add to the bindings(change the bindenv) and return
    if (tuple = checkTuple(u_desc, iterator)) {
      if (delete_flag)  {
	// remove the index entry for the object
	(fd.indexes[index]).delete_entry(u_desc, fd.schema);
	if (delete_flag == DELETE_OBJ_AND_INDEX_ENTRY)
	  SM.destroyObject(&(u_desc->oid), SM.bufGroup());
      }
      SM.releaseObject(u_desc) ;
      return tuple;
    }

    // Else, get next tuple (return to loop )
    SM.releaseObject(u_desc) ;
  }

#undef FAIL_RETURN
}

int PersistentRel::findIndex(TupleIterator& iterator)
{
  if (!SM_initialized) return -1 ;

  char index_buff[COR_MAX_ARGLIST_SIZE] ;

  // If the desired index has already been computed, it would have
  // been stored in indexnump. In such a case, use the stored index.
/*
  if ( (iterator.indexnump) && ((*iterator.indexnump) >= 0)
      &&  ((*iterator.indexnump) < arity()) )
    return (*(iterator.indexnump)) ;
*/

  // propositional case.
  if (iterator.arg_list.count() == 0) { 
    if (arity() != 0)
      fprintf(stderr,"Error: Persistent::findIndex: arity mismatch\n");
//      if (iterator.indexnump) *iterator.indexnump = 0;
    return  -1;
  }

  for (int i = 0; i < arity(); i++) {
    index_buff[i] = (iterator.arg_list[i]->isConstant()) ? 1 : 0 ;
  }
  int chosen_index = fd.findBestMatch(index_buff) ;
/*

  if ((chosen_index >= 0) && (iterator.indexnump))
    *(iterator.indexnump) = chosen_index;
  else if ((chosen_index < 0) && (iterator.indexnump))
    *(iterator.indexnump) = 0;
*/

#ifdef DEBUG
    if (exEnv.dbg_indexing) {
      fprintf(stderr, "Looking for best index for the pattern");
      for (i = 0; i < arity(); i++) {
	fprintf(stderr,"%d", index_buff[i]);
      }
      fprintf(stderr, "\n Offset of best index found = %d\n", chosen_index);
    }
#endif

  return chosen_index ;
}

void PersistentRel::initialize_iterator(TupleIterator& iterator)
{
  if (!SM_initialized) return ;

#ifdef DEBUG
  if (exEnv.dbg_get_next) {
    fprintf(stderr, "\nPersistentRelation::initialize_iterator: ");
    iterator.relation->name->print(NULL, stderr);
    fprintf(stderr, " ( ");
    iterator.arg_list.print(iterator.bindenv, stderr);
    fprintf(stderr, " )\n");
  }
#endif

  iterator.stack_mark.get_mark();
  iterator.ipos = Scanning_Persistent;
  iterator.index = (GenericIndex *)(findIndex(iterator));
  if (((int)(iterator.index)) >= 0) {
    // Index scan cursor needs to be created
    iterator.cursor = (void *) new SMCURSOR ;
    INVALIDATE_CURSOR(*((SMCURSOR *)iterator.cursor));
  }
}

BindEnv *PersistentRel::get_next(TupleIterator& iterator)
{
  if (!SM_initialized) return NULL ;

  this->get_next_tuple(iterator) ;
  if (iterator.no_match()) return NULL;
  return(iterator.bindenv) ;
}

Tuple *PersistentRel::get_next_tuple(TupleIterator& iterator)
{
  if (!SM_initialized) return NULL ;

  Tuple *tuple ;

  SM.startTrans();

  if ((iterator.ipos == Scanning_Init)) 
    initialize_iterator( iterator );
  else iterator.stack_mark.pop_to();

// If no index is available, use a full file scan.
  if (((int)(iterator.index)) < 0)
    // This sets iterator.scanDesc
    tuple = fileScanGetNext(iterator) ;
  else {
    // iterator.index is set
#ifdef DEBUG
    if (exEnv.dbg_indexing) {
      fprintf(stderr, "Index scan with index %d\n",(int)(iterator.index));
    }
#endif
    tuple = indexScanGetNext(iterator, (int)(iterator.index)) ;
  }

  return tuple ;
}

static void start_iterator(TupleIterator& iterator)
{
  if (!SM_initialized) return ;

#ifdef DEBUG
  if (exEnv.dbg_get_next) {
    fprintf(stderr, "\nPersistentRelation::initialize_iterator: ");
    iterator.relation->name->print(NULL, stderr);
    fprintf(stderr, " ( ");
    iterator.arg_list.print(iterator.bindenv, stderr);
    fprintf(stderr, " )\n");
  }
#endif

  iterator.stack_mark.get_mark();
  iterator.ipos = Scanning_Persistent;
}

int PersistentRel::tuple_delete(Tuple *tuple)
{
  if (!SM_initialized) return 0;
  
  SM.startTrans();

  Tuple *return_tuple;
  int object_deleted = 0;

  // set up index scan cursor on all available indexes
  for (int i = 0; i < COR_MAX_INDEXES; i++)
    if (fd.indexes[i].valid()) {
      TupleIterator iterator(this, tuple);

      start_iterator(iterator);
      iterator.index = (GenericIndex *)i;
      iterator.cursor = (void *) new SMCURSOR ;
      INVALIDATE_CURSOR(*((SMCURSOR *)iterator.cursor));

      if (object_deleted)
	return_tuple = indexScanGetNext(iterator, (int)(iterator.index), DELETE_INDEX_ENTRY);
      else {
	return_tuple = indexScanGetNext(iterator, (int)(iterator.index),
					DELETE_OBJ_AND_INDEX_ENTRY);
	object_deleted = 1;
      }
      iterator.stack_mark.pop_to();
      INVALIDATE_CURSOR(*((SMCURSOR *)iterator.cursor)) ;
    }

  // if the object has not been deleted yet (because no index is present),
  // it is deleted using a full file scan

  if (object_deleted) {
    (fd.count)--;
    return 1;
  }

  TupleIterator iterator(this, tuple);
  start_iterator(iterator);
  iterator.index = (GenericIndex *)(-1);

  return_tuple = fileScanGetNext(iterator, DELETE_OBJ_AND_INDEX_ENTRY) ;
  iterator.stack_mark.pop_to();
  int e = sm_CloseScan((SCANDESC *)iterator.scanDesc) ;
  ErrorCheck(e, "sm_CloseScan") ;
  iterator.scanDesc = NULL ;
  (fd.count)--;
  return 1;
}

int PersistentRel::is_subsumed(Tuple *tuple, RMark *, RMark *)
{
  if (!SM_initialized) return 0 ;
  StackMark stackmark ;

  TupleIterator tuple_its(this, tuple);
  if (tuple_its.get_next_tuple()) {
    stackmark.pop_to();
    return 1;
  }
  stackmark.pop_to();
  return 0;
}

int PersistentRel::insert_tuple(Tuple *tuple)
{
  if (!SM_initialized) return 0 ;

  // Only flat tuples can be persistent !
  if (tuple->env_size) {
    CORAL_error(COR_BAD_TUPLE, "Only flat tuples can be persistent",
		"PersistentRel::insert");
    return 0 ;
  }

/*
  if (check_subsum==1) {
    if (is_subsumed(tuple))
      return 0;
  }
*/

  // Materialize a tuple in the desired format as an object to be inserted
  // insert it into the file
  insertObject(tuple->args()) ;
  delete &(tuple->args());
  delete tuple;
  return 1 ;
}

int PersistentRel::insert_new(ArgList& args, BindEnv *env,
			      BindEnv *,
			      Tuple *,
			      int)
{
  if (!SM_initialized) return 0 ;

  // Only flat tuples can be persistent !
  if (env) {
    CORAL_error(COR_BAD_TUPLE, "Only flat tuples can be persistent",
		"PersistentRel::insert");
    return 0 ;
  }

  if (check_subsum==1) {
    Tuple newtuple(&args, env);
    if (is_subsumed(&newtuple)) {

#ifdef DO_TRACE
      if (DISPLAY_INSERTIONS)
	fprintf(exEnv.trace_file, " - subsumed)]\n");
#endif

      return 0;
    }
  }

  insertObject(args) ;
  delete &args;
  return 1 ;
}




/*------------------------------------------------------------------
  PersistentRel::add_index(char *, int)

  Adds an index to the persistent relation.

 Oddities/Quirks ::  The second argument is a dummy and is present
 only for compatibility with non-persistent relations.

 -------------------------------------------------------------------*/
int PersistentRel::add_index(char *adorn, int )
{
  if (!SM_initialized) return 0 ;

    ASSERT((adorn != NULL) && (strlen(adorn) <= COR_MAX_ARGLIST_SIZE));

    int e ;

    SM.startTrans();

    // Check that an index can be added
    int index_offset ;
    if ((index_offset = fd.canAddIndex()) < 0 ) {
      CORAL_error(COR_INDEX_ERR, "Exceeded maximum number of indexes",
		  "add_index") ;
      return COR_I_FAIL ;
    }
    

    // Before creating a new index, find a BestMatch to see if
    // the index already exists !! --- PRAVEEN

    // Create a new index on the file on the specified attributes

    char index_buf[COR_MAX_ARGLIST_SIZE] ;

    for(int i = 0; i < arity(); i++, adorn++) {
	if(*adorn == 0) {
	    fprintf(stderr,
	    "PersistentRelation::add_index adorn too short: \n");
	    fprintf(stderr, "    --- arity of %s=%d, adorn len=%d)\n", 
			name->string(), arity(), i); 
	    return(COR_I_FAIL);
	}
	index_buf[i] = (*adorn == 'b' || *adorn == '1') ? 1 : 0 ;
    }

    if (*adorn != 0) {
	fprintf(stderr,"PersistentRelation::add_index adorn too long\n");
	return(COR_I_FAIL);
    }

    fd.addIndex(index_offset, index_buf) ;

    // If there are any existing tuples in the file, filter them
    // through the index

    SCANDESC *s_desc ;
    USERDESC *u_desc ;
    BOOL eof ;
    e = sm_OpenScanWithGroup(&(fd.fid), PHYSICAL_SCAN, SM.bufGroup(), &s_desc, NULL);
    ErrorCheck(e, "sm_OpenScanWithGroup") ;

    for (;;) {
      e = sm_ScanNextObject(s_desc, 0, READ_ALL, &u_desc, &eof) ;
      ErrorCheck(e, "sm_ScanNextObject") ;

      // If there are no more tuples, break
      if (eof) break ;

      fd.addObjectToIndex(index_offset, u_desc) ;
    }

    e = sm_CloseScan(s_desc) ;
    ErrorCheck(e, "sm_CloseScan") ;

    // Change the catalog entry to reflect the new index
    if (!SM.retrieveObject(&oid, &u_desc)) {
      CORAL_error(COR_ESM_ERR, NULL, "PersistentRel::add_index");
      return COR_I_FAIL;
    }

    // Safer to write the entire object, instead of calculating the
    // offset. You never know with C++ class structures what their
    // true sizes are !!
    e = sm_WriteObject(SM.bufGroup(), 0, sizeof(FileDescStruct),
		       (char *)(&fd), u_desc, TRUE /* release */) ;
    ErrorCheck(e, "sm_WriteObject") ;
    
    return COR_I_SUCCESS;
}

int PersistentRel::add_index(ArgList *args, int dummy)
{
  if (!SM_initialized) return 0 ;

    ASSERT(args->count() != 0);
    if (args->count() > 1) {
        fprintf(stderr, "More than one argument to add_index ");
	fprintf(stderr, "-- using first one to build index.\n");
    }
    ASSERT((*args)[0]->kindof() == COR_SYMBOL_CONST);
    char *arg_str = ((Symbol *) ((*args)[0]))->string();

    return add_index(arg_str, dummy);
}


Relation * makePersistentRel(FuncArg *rel)
{
  if (!SM_initialized) return NULL ;
  int error = 0 ;

  PersistentRel *p_rel = new PersistentRel(rel->functor(), rel->arity(),error);
  if (error){
    delete p_rel ;
    return NULL ;
  }
  else return p_rel ;
}

void deletePersistentRelation(Name name, int arity)
{
  if (!SM_initialized) return  ;
  fprintf(stderr, "\nAre you sure you want to delete persistent relation ");
  fprintf(stderr, "%s arity %d ? (Y/N)\n", SymbolString(name), arity);
  int input = getchar() ;
  if ((input == 'Y')||(input == 'y'))
    SM.deleteFile(name, arity);
  else
    fprintf(stderr, "Delete of persistent relation aborted by user !\n");
}

void PersistentRel::empty_relation(int delete_tuples)
{
  if (!SM_initialized) return ;

  int e ;
  OID oid, nextoid ;
  OBJHDR objhdr ;
  BOOL     eof ;

  if (!delete_tuples) return ;

  SM.startTrans();

  e = sm_GetFirstOid(SM.bufGroup(), &(fd.fid), &oid, &objhdr, &eof);
  ErrorCheck(e, "sm_GetFirstOid") ;

  if (e < 0) return ;

  for (;;) {
    e = sm_GetNextOid(SM.bufGroup(), &oid, &nextoid, &objhdr, &eof) ;
    ErrorCheck(e, "sm_GetNextOid");

    sm_DestroyObject(SM.bufGroup(), &oid) ;
    oid = nextoid ;

    // If there are no more tuples, return.
    if (eof || (e < 0)) {
      return ;
    }
  }

}

void PersistentRel::release(TupleIterator& iterator)
{
  int e ;

  if (((int)(iterator.index)) < 0) {
    // Full file scan 
    if (iterator.scanDesc) {
      e = sm_CloseScan((SCANDESC *)iterator.scanDesc) ;
      ErrorCheck(e, "sm_CloseScan") ;
      iterator.scanDesc = NULL ;
    }
  }
  else {
    // index scan
    if (iterator.cursor) {
      delete (SMCURSOR *)(iterator.cursor);
      iterator.cursor = NULL;
    }
  }

}

#else

/* dummy declarations for compilation without persistence */

#include "generic-rel.h"

Relation * makePersistentRel(FuncArg *)
{
  return NULL ;
}

void deletePersistentRelation(Name, int)
{
}

#endif
