/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************
	CORAL Software :: U.W.Madison

	generic-sm.h : Contains routines for manipulating the EXODUS
	Storage Manager, for the persistent relations of CORAL.
 ***********************************************************************/

#ifdef WITH_PERSISTENCE

#include "generic-sm.h"
#include "builtin-rel.h"
#include "cor_error.h"
#include <strings.h>

StorageManager SM ;
extern int SM_initialized ;
extern BindEnv *SchemaSolver(BuiltinRelation& rel, TupleIterator& iterator);
extern BindEnv *StartBulkLoadSolver(BuiltinRelation& rel, TupleIterator& iterator);
extern BindEnv *EndBulkLoadSolver(BuiltinRelation& rel, TupleIterator& iterator);

/*
 * ErrorCheck( ) is a simple routine for checking
 * Storage Manager return codes.
 */
int ErrorCheck (
    int    e,              /* error return code */
    char   *routine        /* the routine that caused an error */
)
{
    if (e < 0) {
	if (!SM_initialized) {
	    fprintf(stderr, "Warning: Cannot connect to ");
	    fprintf(stderr, "EXODUS storage manager\n");
	}
        fprintf(stderr, "Got a Storage Manager error from %s\n", routine);
        fprintf(stderr, "Error = %s\n", sm_Error(sm_errno));
	fflush(stderr);
	if (SM_initialized)
	  SM.shutdown();
	return 1 ;
    }
    return(0);
}

void StorageManager::init()
{
  int e ;

  SM_initialized = 0 ;

  if ((exEnv.C_exec_mode == COR_DOING_FACTOR_MAGIC) ||
      (exEnv.C_exec_mode == COR_DOING_MAGIC) ||
      (exEnv.C_exec_mode == COR_DOING_FACTOR) ||
      (exEnv.C_exec_mode == COR_DOING_EXIST)) return ;
  /*
   * Initialize the storage manager.
   */

  char *error_msg ;
  char *config_file = getenv("ESMCONFIG");
  int bufsize = -1;

  if (config_file) {
    e = sm_ReadConfigFile(getenv("ESMCONFIG"), NULL, &error_msg) ;
    if (e >= 0) {
      e = sm_GetClientOption("bufpages", (void *)(&bufsize));
      if (ErrorCheck(e, "sm_GetClientOption")) return ;
    }
  }

  if ((!config_file) || (bufsize < COR_CLIENT_BUF_SIZE )) {
    // Use default config values
    bufsize = COR_CLIENT_BUF_SIZE;
    e = sm_SetClientOption("bufpages", (void *)(&bufsize), SM_int);
    if (ErrorCheck(e, "sm_SetClientOption")) return ;
  }

  e = sm_Initialize(getenv("ESERVERHOST"), getenv("ESERVERNAME") );
  if (ErrorCheck(e, "sm_Initialize")) return;

  /*
   *	get the disk name
   */
  char* diskName = getenv("EVOLUME");
  // Default disk name is 'coraldisk'
  if (!diskName) diskName = "coraldisk" ;
    
  /*
   * Mount the volume.
   */
  e = sm_MountVolume(diskName, &_volid);
  if (ErrorCheck(e, "sm_MountVolume")) {
    fprintf(stderr, "Name of volume : %s\n", diskName);
    return;
  }

  /*
   * Open a buffer group.
   */
  if (buf.open(COR_SM_BUF_SIZE, BF_MRU)) return;
  
  if (catalog.init("CORAL_CATALOG_FILE", "CORAL_CATALOG_INDEX")) return ; ;

  SM_initialized = 1 ;

  /*
   * Install schema, start_bulk_load and end_bulk_load builtin relations
   */
  Relation *temp = new BuiltinRelation(1, EnterSymbol("schema"), SchemaSolver);
  temp = new BuiltinRelation(1, EnterSymbol("start_bulk_load"),
				       StartBulkLoadSolver);
  temp = new BuiltinRelation(1, EnterSymbol("end_bulk_load"),
				       EndBulkLoadSolver);


}

void StorageManager::shutdown()
{
  if (SM_initialized) {
    sm_AbortTransaction(trans.tid);
    sm_CloseBufferGroup(buf.bufGroup);
    sm_DismountVolume(_volid) ;
    sm_ShutDown();
    SM_initialized = 0 ;
  }
}
/*------------------------------------------------------------------
 retrieveObject : Retrieves an object from the storage manager.
 The var parameter is a pointer which, when the function returns,
 points to the retrieved object in the buffer pool. 

 Oddities/Quirks ::  The retrieved object is pinned in the buffer pool
 and will have to be explicitly unpinned using an sm_release() call.
 The object should be accessed only INDIRECTLY via the 'basePtr'
 field of the USERDESC structure, because EXODUS may relocate the
 contents of the buffer pool. Also note that the object is now
 PINNED in the buffer pool, and has to be unpinned using the function
 releaseObject().

 -------------------------------------------------------------------*/
int StorageManager::retrieveObject(OID *oid, USERDESC ** u_desc)
{

  if (!SM_initialized) return 0;

  SM.startTrans();

  int e = sm_ReadObject(bufGroup(), oid, 0, READ_ALL,
			u_desc) ;
  if (ErrorCheck(e, "retrieveObject:sm_ReadObject")) return 0;

  return 1 ;
}

/*------------------------------------------------------------------
 releaseFileStruct : Releases an object pinned in the buffer pool.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
int StorageManager::releaseObject(USERDESC *u_desc)
{
  if (!SM_initialized) return 0;

  SM.startTrans();

  int e = sm_ReleaseObject(u_desc) ;
  if (ErrorCheck(e, "releaseObject:sm_ReleaseObject")) return 0;

  return 1 ;
}

/*------------------------------------------------------------------
 destroyObject : Destroys an object. If 
 bufGroup is 0, the object is in the catalog file.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
int StorageManager::destroyObject(OID *oid, int bufGroup)
{
  if (!SM_initialized) return 0;

  int e ;

  SM.startTrans();

  // If bufGroup is 0, the object belongs to the catalog
  if (!bufGroup) bufGroup = this->bufGroup() ;

  e = sm_DestroyObject(bufGroup, oid) ;
  if (ErrorCheck(e, "destroyObject:sm_DestroyObject")) return 0;

  return 1 ;
}


int StorageManager::lookupFile(Name name, int arity, OID* oid)
{
  if (!SM_initialized) return 0;

  char buf[COR_MAX_REL_NAME_LEN + 6] ;

  SM.startTrans();

  if (arity > COR_MAX_ARGLIST_SIZE) {
    CORAL_error(COR_TOO_MANY_ARGS, NULL, "lookupFile") ;
    return 0;
  }

  ASSERT(name->length() <= COR_MAX_REL_NAME_LEN) ;
  sprintf(buf, "%s%3d", name->string(), arity);

  return catalog.lookupFile(buf, oid) ;

}

int StorageManager::addFile(Name name, int arity, FileDescStruct *obj)
{
  if (!SM_initialized) return 0;

  char buf[COR_MAX_REL_NAME_LEN + 6] ;

  SM.startTrans();

  if (arity > COR_MAX_ARGLIST_SIZE) {
    CORAL_error(COR_TOO_MANY_ARGS, NULL, "addFile") ;
    return 0;
  }

  ASSERT(name->length() <= COR_MAX_REL_NAME_LEN) ;
  sprintf(buf, "%s%3d", name->string(), arity);

  return catalog.addFile(buf, obj) ;
}

int StorageManager::deleteFile(Name name, int arity)
{
  if (!SM_initialized) return 0;

  OID oid ;
  USERDESC *u_desc ;
  SM.startTrans();

  // Look up the file in the catalog
  if (! this->lookupFile(name, arity, &oid)) return 0;

  // Retrieve the file descriptor object
  if (! this->retrieveObject(&oid, &u_desc)) return 0;
  FileDescStruct fd ;
  fd.copyDesc((FileDescStruct *)(u_desc->basePtr));

  // Destroy the file and its contents
  int e = sm_DestroyFile(SM.bufGroup(), &(fd.fid));
  if (ErrorCheck(e, "sm_DestroyFile")) return 0;

  // Destroy all indexes on the file
  for (int i = 0; i < fd.num_indexes; i++)
    if (fd.indexes[i].valid()) {
      e = sm_DestroyIndex(&(fd.indexes[i].iid), SM.bufGroup());
      if (ErrorCheck(e, "sm_DestroyIndex")) return 0;
    }


  // Delete the entry for the file in the catalogs
  char buf[COR_MAX_REL_NAME_LEN + 6] ;

  if (arity > COR_MAX_ARGLIST_SIZE) {
    CORAL_error(COR_TOO_MANY_ARGS, NULL, "deleteFile") ;
    return 0;
  }

  ASSERT(name->length() <= COR_MAX_REL_NAME_LEN) ;
  sprintf(buf, "%s%3d", name->string(), arity);

  if (!catalog.deleteFile(buf, &oid)) return 0;
    
  this->releaseObject(u_desc) ;
  return 1;
}

void StorageManager::listFiles(FILE *outf)
{
  if (!SM_initialized) return ;

  catalog.listFiles(outf) ;
}

/************** BufferGroup methods **********************/

void BufferGroup::close() 
{
  if (!SM_initialized) return;

  SM.startTrans();

  if (BUFGROUP_IS_INVALID(bufGroup)) return ;
  int e = sm_CloseBufferGroup(bufGroup);
  ErrorCheck(e, "sm_CloseBufferGroup");
  INVALIDATE_BUFGROUP(bufGroup);
}

/************** Transaction methods **********************/
int Transaction::commit() 
{
  if (TID_IS_INVALID(tid)) return 0 ;

  int e = sm_CommitTransaction(tid) ;
  ErrorCheck(e, "sm_CommitTransaction");

  INVALIDATE_TID(tid) ;
  return 1 ;
}


/************** BPlusTree methods **********************/

/*------------------------------------------------------------------
   BPlusTreeIndex::create

   Creates a B+ tree index.

   Oddities/Quirks :: 

 -------------------------------------------------------------------*/
void BPlusTreeIndex::create(char *attr_str, int arity, int unique)
{
  if (!SM_initialized) return;

  SM.startTrans();

  for (int i = 0; i < arity; i++)
    this->arity += (attrs[i] = attr_str[i]) ;  

  int e = sm_CreateIndex(SM.volid(), SM.bufGroup(), SM_BTREENDX,
			 SM_string, COR_MAX_KEY_LEN, sizeof(OID), 
			 unique, &iid);
  ErrorCheck(e, "BPlusTreeIndex::create");

}

void BPlusTreeIndex::insert(USERDESC *u_desc, Schema &schema)
{
  if (!SM_initialized) return ;

  char value_buff[COR_MAX_KEY_LEN] ;
  int buff_offset = 0 ;

  SM.startTrans();

  int offset = 0 ;
  int str_len = 0 ;
  int data_len = 0 ;
  int e ;

  // For safety, check that the index is valid
  ASSERT(valid()) ;

  // Fill value_buff with the key value.
  // This is done by copying the appropriate attributes into value_buff
  for (int i = 0; i < schema.arity; i++) {
    switch(schema[i]) {

    case COR_STRING_DBTYPE :
      memcpy((char *)&str_len, (char *)(u_desc->basePtr+offset), sizeof(int));
      offset = offset + sizeof(int) ;
      data_len = str_len * sizeof(char) ;
      break ;

    case COR_INT_DBTYPE :
      data_len = sizeof(int) ;
      break ;

    case COR_DOUBLE_DBTYPE :
      data_len = sizeof(double) ;
      break ;

    default :
      CORAL_error(COR_BAD_SCHEMA, "INVALID field reached","BPlusTree::insert");
      return  ;
      // This should not happen !!
    
    }

    if ((buff_offset + data_len < COR_MAX_KEY_LEN) && attrs[i]){
      memcpy((char *)(value_buff + buff_offset),
	     (char *)(u_desc->basePtr + offset), data_len ) ;
      buff_offset += data_len ;
    }
    offset += data_len ;
  }

  KEY key ;
  key.length = (TWO)(buff_offset) ;
  key.valuePtr = (void *)(value_buff) ;
  e = sm_InsertEntry(&iid, SM.bufGroup(), &key, &(u_desc->oid));
  ErrorCheck(e, "sm_InsertEntry") ;

}

void BPlusTreeIndex::delete_entry(USERDESC *u_desc, Schema &schema)
{
  if (!SM_initialized) return ;

  char value_buff[COR_MAX_KEY_LEN] ;
  int buff_offset = 0 ;

  SM.startTrans();

  int offset = 0 ;
  int str_len = 0 ;
  int data_len = 0 ;
  int e ;

  // For safety, check that the index is valid
  ASSERT(valid()) ;

  // Fill value_buff with the key value.
  // This is done by copying the appropriate attributes into value_buff
  for (int i = 0; i < schema.arity; i++) {
    switch(schema[i]) {

    case COR_STRING_DBTYPE :
      memcpy((char *)&str_len, (char *)(u_desc->basePtr+offset), sizeof(int));
      offset = offset + sizeof(int) ;
      data_len = str_len * sizeof(char) ;
      break ;

    case COR_INT_DBTYPE :
      data_len = sizeof(int) ;
      break ;

    case COR_DOUBLE_DBTYPE :
      data_len = sizeof(double) ;
      break ;

    default :
      CORAL_error(COR_BAD_SCHEMA, "INVALID field reached","BPlusTree::insert");
      return  ;
      // This should not happen !!
    
    }

    if ((buff_offset + data_len < COR_MAX_KEY_LEN) && attrs[i]){
      memcpy((char *)(value_buff + buff_offset),
	     (char *)(u_desc->basePtr + offset), data_len ) ;
      buff_offset += data_len ;
    }
    offset += data_len ;
  }

  KEY key ;
  key.length = (TWO)(buff_offset) ;
  key.valuePtr = (void *)(value_buff) ;
  e = sm_RemoveEntry(&iid, SM.bufGroup(), &key, &(u_desc->oid));
  ErrorCheck(e, "sm_RemoveEntry") ;

}

BPlusTreeIndex::BPlusTreeIndex()
{

  INVALIDATE_IID(iid) ; 
  for (int i = 0; i < COR_MAX_ARGLIST_SIZE; i++)
    attrs[i] = 0 ;
  arity = 0 ;
}

int BPlusTreeIndex::matches(char *attr_str, int arity) {
     return !(memcmp(attrs, attr_str, arity)) ;
};

void BPlusTreeIndex::destroy() 
{
  if (!SM_initialized) return ;

  SM.startTrans();

  int e = sm_DestroyIndex(&iid, SM.bufGroup()) ;
  ErrorCheck(e, "sm_DestroyIndex") ;
  INVALIDATE_IID(iid) ; 
  for (int i = 0; i < COR_MAX_ARGLIST_SIZE; i++)
    attrs[i] = 0 ;
};    

void BPlusTreeIndex::print_adornment(FILE *outf)
{
  for (int i = 0; i < arity; i++)
    fprintf(outf, "%c", ((attrs[i]) ? 'b' : 'f'));
}

/******* FileDescStruct methods *******************/

//
// Constructor takes a schema definition of a relation as parameter.
// The schema definition is in the form of a FuncArg
//
FileDescStruct::FileDescStruct(FuncArg *arg, int& error)
     : schema(arg->args, error)
{
  OID oid ;

  SM.startTrans();

  // If there was an error in the schema, return.
  if (error != 0) {
    fprintf(stderr, "WARNING :: error in schema\n");
    INVALIDATE_FID(fid) ;
    num_indexes = 0 ;
    return ;
  }

  // Check if the relation is already defined. If so, return.
  if (SM.lookupFile(arg->functor(), arg->arity(), &oid)) {
    error = COR_DUP_SCHEMA ;
    INVALIDATE_FID(fid) ;
    num_indexes = 0 ;
    return ;
  }

  // If not, create a new file to hold the relation
  int e = sm_CreateFile(SM.bufGroup(), SM.volid(), &fid) ;
  ErrorCheck(e, "sm_CreateFile");

  num_indexes = 0 ;
  count  = 0;
}


void FileDescStruct::copyDesc(FileDescStruct *fdstruct)
{
  if (!SM_initialized) return ;

  fid = fdstruct->fid ;
  for (int i = 0; i < COR_MAX_INDEXES; i++)
    indexes[i] = fdstruct->indexes[i];
  num_indexes = fdstruct->num_indexes ;
  schema = fdstruct->schema ;
}

// Returns offset where an index entry can be made
int FileDescStruct::canAddIndex() 
{
  if (!SM_initialized) return -1;

  if (num_indexes == COR_MAX_INDEXES) return -1 ;

  for (int i=0; i < COR_MAX_INDEXES; i++)
    if (!(indexes[i].valid())) return i ;

  // This should not be reached
  return (-1) ;
}

// Returns offset of the next valid index after offset i
// The first time, this is called with i = -1
int FileDescStruct::nextIndex(int i)
{
  if (!SM_initialized) return -1;

  for (i += 1; i < COR_MAX_INDEXES; i++)
    if (indexes[i].valid()) return i ;
  return (-1) ;
}

// Adds an index at position 'offset' in the index array. The attributes
// of the index are specified by buf.
void FileDescStruct::addIndex(int offset, char *buf) 
{
  if (!SM_initialized) return;

  if (!(indexes[offset].valid())) {
    indexes[offset].create(buf, schema.arity, 0) ;
    num_indexes++ ;
  }
}

// Delete an index whose offset is specified 
void FileDescStruct::destroyIndex(int index_offset) 
{
  if (!SM_initialized) return ;

  if (indexes[index_offset].valid()) {
    indexes[index_offset].destroy() ;
    num_indexes-- ;
  }
}

// Adds an object to a specified index
void FileDescStruct::addObjectToIndex(int index_offset, USERDESC *u_desc) 
{
  if (!SM_initialized) return ;

  if (indexes[index_offset].valid())
    indexes[index_offset].insert(u_desc, schema) ;
}

// Find an index that best matches the given attr string
// IMPROVE THIS FUNCTION LATER !!
int FileDescStruct::findBestMatch(char *attr_str) 
{
  if (!SM_initialized) return -1;

  if (num_indexes == 0) return -1 ;

  for (int i = 0; i < COR_MAX_INDEXES; i++)
    if (indexes[i].valid() && (indexes[i].matches(attr_str, schema.arity)))
      return i ;
  
  return -1 ;
}

void FileDescStruct::print_index_adornments(FILE *outf)
{
  if (!SM_initialized || !num_indexes) return;
  for (int i = 0; i < COR_MAX_INDEXES; i++)
    if (indexes[i].valid()) {
      indexes[i].print_adornment(outf);
      fprintf(outf, " ");
    }
}

Schema::Schema(ArgList &arglist, int& error)
{
  int i = 0 ;
  FOR_EACH_ARG(arg, arglist){
    if (strcmp(SymbolString(arg), "int"))
      if (strcmp(SymbolString(arg), "double"))
	if (strcmp(SymbolString(arg), "string")) {
	  error = COR_BAD_SCHEMA ;
	  return ;
	}
	else types[i] = COR_STRING_DBTYPE ;
      else types[i] = COR_DOUBLE_DBTYPE ;
    else types[i] = COR_INT_DBTYPE ;
    i++ ;
  }END_EACH_ARG

  arity = i ;

}

void Schema::operator=(Schema& schema2)
{
  for (int i = 0; i < COR_MAX_ARGLIST_SIZE; i++ )
    types[i] = schema2.types[i] ;
  arity = schema2.arity ;
}

void Schema::printon(FILE *outf)
{
  fprintf(outf, "(");
  for (int i = 0; i < arity; i++) {
    switch (types[i]) {
    case COR_INT_DBTYPE : fprintf(outf, "int") ; break ;
    case COR_DOUBLE_DBTYPE : fprintf(outf, "double") ; break ;
    case COR_STRING_DBTYPE : fprintf(outf, "string") ; break ;
      default : break;

    }

    if (i < arity - 1) fprintf(outf, ", ");
  }
  fprintf(outf, ")");
}


// This function is called by gram.y . Those applications that do not
// require linking with persistence code, are instead linked with
// the routine of the same name in dummy.C, that does nothing

void commitTrans()
{
  if (SM_initialized) SM.commitTrans() ;
}

void abortTrans()
{
  if (SM_initialized) SM.abortTrans() ;
}

void shutdown_server()
{
  if (SM_initialized) SM.shutdown() ;
}

void listFiles(FILE *outf)
{
  if (SM_initialized) SM.listFiles(outf) ;
  else
    CORAL_error(COR_ESM_ERR, "dbserver not initialized", "listFiles");
}

void sm_init()
{
  SM.init() ;
}

#else

/* dummy declarations for compilation without exodus */

#include <stdio.h>

void commitTrans()
{
}

void abortTrans()
{
}

void shutdown_server()
{
}

void sm_init()
{
}

void listFiles(FILE *)
{
}

#endif
