/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************
	CORAL Software :: U.W.Madison

	catalog.C : Contains catalog manipulation routines for the
	portion of CORAL that handles persistent relations.
 ***********************************************************************/

#ifdef WITH_PERSISTENCE

#include "generic-sm.h"
#include <strings.h>

extern int SM_initialized;

/*------------------------------------------------------------------
  init() :  initializes catalog by creating catalog file and catalog 
	    index if they do not exist already.  

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
int Catalog::init(char *cat_file_name, char *cat_index_name)
{
  int data_length = -1;
  int catalog_exists = 1 ;
  int e ;

   // Start a transaction
   SM.startTrans();

   // Check if catalog file already exists
   e = sm_GetRootEntry(SM.volid(), cat_file_name,
			   (char *)&cat_fid, &data_length);

   if (sm_errno == esmBADROOTNAME) {
     // Entry does not exist
     catalog_exists = 0 ;
     e = sm_CreateFile(SM.bufGroup(), SM.volid(), &cat_fid) ;
     if (ErrorCheck(e, "sm_CreateFile")) {
       SM.abortTrans();
       return -1;
     }

     e = sm_SetRootEntry(SM.volid(), cat_file_name,
			 (char *)&cat_fid, sizeof(FID));
     if (ErrorCheck(e, "sm_SetRootEntry")) {
       SM.abortTrans();
       return -1;
     }
   }

   else if (data_length != sizeof(FID)) {
     SM.abortTrans();
     fprintf(exEnv.error_file, "Error creating catalog file\n") ;
     return -1;
   }


   data_length = -1 ;
   // Check if catalog index already exists
   e = sm_GetRootEntry(SM.volid(), cat_index_name,
			   (char *)&cat_iid, &data_length);

   if (sm_errno == esmBADROOTNAME) {
     // Entry does not exist
     if (catalog_exists) {
       fprintf(exEnv.error_file, "Catalog file exists, but index does not exist !\n");
       fprintf(exEnv.error_file, "Disk has to be reformatted !\n");
       SM.abortTrans();
       return -1;
     }

     e = sm_CreateIndex( SM.volid(), SM.bufGroup(),
			SM_BTREENDX, SM_string,
			COR_MAX_REL_NAME_LEN, sizeof(OID),
			TRUE, &cat_iid) ;
     if (ErrorCheck(e, "sm_CreateIndex")) {
       SM.abortTrans();
       return -1; 
      }

     e = sm_SetRootEntry(SM.volid(), cat_index_name,
			 (char *)&cat_iid, sizeof(IID));
     if (ErrorCheck(e, "sm_SetRootEntry")) {
       SM.abortTrans();
       return -1;
     }
   }

   else if (!catalog_exists) {
     fprintf(exEnv.error_file, "Catalog index exists, but file does not exist !\n");
     fprintf(exEnv.error_file, "Disk has to be reformatted !\n");
     SM.abortTrans();
     return -1;
   }

   else if (data_length != sizeof(IID)) {
     SM.abortTrans() ;
     fprintf(exEnv.error_file, "Error creating catalog file\n") ;
     return -1 ;
   }

   if (!SM.commitTrans()) {
     fprintf(exEnv.error_file, "Commit error\n");
     // CORAL_error
     return -1 ;
   }
  return 0;

}

/*------------------------------------------------------------------
 lookupFile : Checks catalog index to see if a file of the specified name
 exists. If so, the OID of the entry for that file in the catalog file
 is returned via the var parameter. A return value of 0 implies that
 the entry was not found.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
int Catalog::lookupFile(char *name, OID *oid)
{
  if (!SM_initialized) return 0;

  SMCURSOR cursor ;
  KEY key ;
  key.length = (TWO)(strlen(name)+1) ;
  key.valuePtr = (void *)name ;
  char eof ;

  SM.startTrans();
  
  int e = sm_FetchInit(&cat_iid, SM.bufGroup(), &key, SM_EQ,
		        &key, SM_EQ, &cursor);
  if (ErrorCheck(e, "sm_FetchInit")) return 0;

  e = sm_FetchNext(&cursor, NULL, oid, &eof) ;
  if (ErrorCheck(e, "sm_FetchNext")) return 0;

  if (eof) return 0 ; // Entry not found

  return 1 ;
}

/*------------------------------------------------------------------
 addFile : Adds a file to the catalog file, and adds
 the appropriate entry in the catalog index. The data
 to be entered is pointed to by obj.

 -------------------------------------------------------------------*/
int Catalog::addFile(char *file_name, FileDescStruct *obj)
{
  if (!SM_initialized) return 0;

  // Add the entry to the catalog file
  OID oid ;

  SM.startTrans();

  int e = sm_CreateObject(SM.bufGroup(), &cat_fid, NEAR_LAST_PHYSICAL,
			  NULL, NULL, (FOUR)sizeof(FileDescStruct),
			  (char *)obj, &oid) ;
  if (ErrorCheck(e, "sm_CreateObject")) return 0;


  // Add the oid to the catalog index
  KEY key ;
  key.length = (TWO)(strlen(file_name)+1) ;
  key.valuePtr = (void *)file_name ;

  e = sm_InsertEntry(&cat_iid, SM.bufGroup(), &key, &oid) ;
  if (ErrorCheck(e, "sm_InsertEntry")) return 0;

  return 1 ;
}

/*------------------------------------------------------------------
 deleteFileStruct : Deletes a file from the catalogs.

 Oddities/Quirks :: 

 -------------------------------------------------------------------*/
int Catalog::deleteFile(char *file_name, OID *oid)
{
  if (!SM_initialized) return 0;

  SM.startTrans();

  KEY key ;
  key.valuePtr = (void *)file_name ;
  key.length = (TWO)(strlen(file_name) + 1) ;

  int e = sm_RemoveEntry(&cat_iid, SM.bufGroup(), &key, oid);
  if (ErrorCheck(e, "sm_RemoveEntry")) return 0;

  // Delete the file descriptor object itself
  e = sm_DestroyObject(SM.bufGroup(), oid);
  if (ErrorCheck(e, "sm_DestroyObject")) return 0 ;

}


void Catalog::listFiles(FILE *outf)
{

  SM.startTrans();

  SMCURSOR cursor ;
  KEY key1, key2 ;

  char value_buff1[COR_MAX_KEY_LEN+1], value_buff2[COR_MAX_KEY_LEN+1] ;

  for (int i = 0; i < COR_MAX_KEY_LEN ; i++) {
    value_buff1[i] = '\0' ;
    value_buff2[i] = '~' ;
  }
  value_buff2[COR_MAX_KEY_LEN -1] = '\0' ;

 
  key1.length = (TWO)(COR_MAX_KEY_LEN) ;
  key1.valuePtr = (void *)(value_buff1) ;

  key2.length = (TWO)(COR_MAX_KEY_LEN) ;
  key2.valuePtr = (void *)(value_buff2) ;

  // Initialize the index scan
  int e = sm_FetchInit(&cat_iid, SM.bufGroup(),
		   SM_BOF, SM_EOF, &cursor) ;
    
  ErrorCheck(e, "sm_FetchInit") ;
  for (;;) {
    OID oid ;
    char eof ;

    // Get the next entry from the index
    key1.length = (TWO)(COR_MAX_KEY_LEN);
    e = sm_FetchNext(&cursor, &key1, &oid, &eof) ;

    // If there are no more index entries, return.
    if (eof) {
      return ;
    }

    if (e < 0) {
#ifdef DEBUG
      fprintf(exEnv.error_file, "CORAL::WARNING Entry with key length too large\n");
#endif
      ErrorCheck(e,"sm_FetchNext");
      return ;
    }

    USERDESC * u_desc ;
    if (! SM.retrieveObject(&oid, &u_desc)) {
      CORAL_error(COR_ESM_ERR, NULL, "Catalog::listFiles");
      return ;
    }
    
    value_buff1[key1.length] = '\0' ;
    fprintf(outf, "%s : ", value_buff1) ;
    ((FileDescStruct *)(u_desc->basePtr))->schema.printon(outf);
    fprintf(outf, " : (exodus)\n");
//    fprintf(outf, ": count %d\n", ((FileDescStruct *)(u_desc->basePtr))->count);
  }

}
#endif
