/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/
/*****************************************************************************/
/*	 cleanup.c
	 Delete useless unit rules, rules which don't have exit rules for the
	 head predicate, rules that have undefined head predicate, and then
	 rules that are not reachable from the query.
*/
/*****************************************************************************/
#include "exist.h"
#include "externs.h"
#include "rules.h"
#include "reduce.h"
#include "cleanup.h"

static struct ruleNode *makeGraph(struct rule *, struct predNode *);
static void traverse(struct predNode *, struct ruleNode *);
static struct rule *delUnsolvable(struct rule *, struct ruleNode *);
static struct rule *delUnreachable(struct rule *, struct predNode *, int *);
static int unitRuleUsed(struct rule *, struct projection **, int) ;
static void printGraph(predNode *pNode, ruleNode *rNode);
static int rule_count = 0, pred_count = 0;

static void printPreds(predNode *pNode)
{
	fprintf(stderr, "Predicate Table --- pred_count = %d\n", pred_count);
	for (pNode = pNode->next; pNode; pNode = pNode->next)
		fprintf(stderr, "%s\n", SymbolString(pNode->pred->name));
}
/*****************************************************************************/
struct rule *cleanup(struct rule *rules)
{
	ruleNode *rNode;
	int i;

	do {
		pred_count = rule_count = 0;
		predNode *pNode = new predNode;
		rNode = makeGraph(rules, pNode);
		traverse(pNode, rNode);
		rules = delUnsolvable(rules, rNode);
		rules = delUnreachable(rules, pNode, &i);
	} while (i);

	return(rules);
}

/*****************************************************************************/
/* delete unit rules of the form p[a] :- p[a1].
   The first rule will not be deleted.
*/
/*****************************************************************************/
static struct rule *delUnsolvable(struct rule *rules, ruleNode *rNode)
{
	ruleNode *rn;
	struct rule *r, *b;

	if (rNode==NULL)
		return(NULL);
	if (rNode->solved==COR_UNSOLVED)
		return(NULL);
	for (rn=rNode->next, r=(struct rule *)rules->link.next, b=rules; 
		(rn && r); rn=rn->next, r=(struct rule *)r->link.next) {

		if (rn->solved==COR_UNSOLVED)
			b->link.next = r->link.next;
		else
			b = r;
	}
	return(rules);
}

/****************************************************************************
 returns true if the predicate is already in the predNode structure
 ****************************************************************************/
static predNode * LookupPred(predNode *pNode, Predicate *pred)
{
	for (pNode = pNode->next; pNode; pNode = pNode->next)
	   if (pNode->pred == pred) return pNode ;

	return NULL ;
}

/****************************************************************************
 returns true if the predicate is already in the unreachTable structure
 ****************************************************************************/
static unreachTable * LookupPred(unreachTable *table, Predicate *pred)
{
	for (int i = 0; i < pred_count; i++)
	   if (table[i].pred == pred) return &(table[i]) ;

	return NULL ;
}

/****************************************************************************
 adds a predicate to the predNode list
 ****************************************************************************/
static void AddPred(predNode *pNode, Literal *p)
{
	predNode *tmp = new predNode(p, rule_count);
	tmp->next = pNode->next ;
	pNode->next = tmp ;
	pred_count++ ;
}
/*****************************************************************************/
/* construct the rule/goal graph
   return the pointer to the first rule node, predicate nodes are organised
   in an array format and is returned as a parameter.
   As a side-effect, compute rule_count and pred_count.
*/
/*****************************************************************************/
static ruleNode *makeGraph(struct rule *rules,
			predNode *pNode) /* dimension = COR_MAX_PREDS */
{
	if (rules==NULL)
		return(NULL);

	struct rule *r;
	ruleNode *tmp;
	int i;

	for (r = rules; r; r= (struct rule *)r->link.next, rule_count++) ;

	for (r = rules; r; r= (struct rule *)r->link.next) {
	    if (!(LookupPred(pNode, r->head->predicate)))
		AddPred(pNode, r->head);
	    for (i = 0; i < r->num_preds; i++) {
		if (!(LookupPred(pNode, r->preds[i]->predicate))){
		  AddPred(pNode, r->preds[i]);
		}
	    }
	}

	ruleNode *rNode = new ruleNode(rules) ;

	for (i=0; i<rules->num_preds; i++)
	    LookupPred(pNode, rules->preds[i]->predicate)->addRNode(rNode);

	tmp = rNode;
	for (r=(struct rule *)rules->link.next; r != NULL; 
			r=(struct rule *)r->link.next) {
		tmp->next = new ruleNode(r);
		tmp = tmp->next ;
		for (i=0; i<rules->num_preds; i++) 
		  LookupPred(pNode,rules->preds[i]->predicate)->addRNode(rNode);
	}
	return(rNode);
}

/*****************************************************************************/
/* find a better algorithm later
*/
/*****************************************************************************/
static void traverse(predNode *pNode, ruleNode *rNode)
{
	ruleNode *rn;
	int i;
	int num;
	Predicate *j ;

	for (rn=rNode; rn!=NULL; ) {
	    if (rn->r==NULL)
		continue;
	    if (rn->solved==COR_SOLVED) {
		rn = rn->next;
	    } else {
		num = rn->r->num_preds;
		for (i=0; i<num; i++) {
		    j = rn->r->preds[i]->predicate;
		    if (j == NULL) 
			continue;
		    if (LookupPred(pNode, j)->solved == COR_UNSOLVED){
			rn = rn->next;
			break;		/* if any of the pred are unsolvable */
		    }			/* the rule is still unsolvable */
		}
		if (i==num) {
		    rn->solved = COR_SOLVED;
		    if (rn->head != NULL)
		      LookupPred(pNode, rn->head)->solved = COR_SOLVED;

		    rn = rNode;	  /* THIS IS INEFFICIENT! IMPROVE IT LATER */
		}
	    } /* else */
	} /* for */
}

/*****************************************************************************/
static void init_table(unreachTable *table, predNode *pNode)
{
    int i;
	for (i=0, pNode=pNode->next; pNode; pNode=pNode->next, i++) {
	    table[i].pred = pNode->pred ;
	    table[i].numMarked = 0;
	}
}
/*****************************************************************************/
static struct rule *delUnreachable(struct rule *rules, predNode * pNode, int *del)
{
	*del = 0;		/* no rules is deleted so far */

	if (rules==NULL)
		return(NULL);

	struct rule *r, *b;
	struct unreachTable * table = new struct unreachTable[pred_count];
	int i, j;


	init_table(table, pNode) ;

	for (r=rules; r ; r=(struct rule *)r->link.next)
	    for (i=0; i<r->num_preds; i++)
		LookupPred(table, r->preds[i]->predicate)->add_bv(
							r->preds[i]->adorn);

	for (r=(struct rule *)rules->link.next, b=rules; r; 
				r=(struct rule *)r->link.next) {
		Predicate *ipred = r->head->predicate;
		if (ipred == NULL) continue;
		unreachTable *uTable = LookupPred(table, ipred);
		for (j=0; j<uTable->numMarked; j++)
		    if (uTable->vector[j] == r->head->adorn)
			break;
		if (j==uTable->numMarked) {
			b->link.next = r->link.next;
			*del = 1; 
			/* set the flag to show that some rule is deleted */
		} else
			b = r;
	}
	return(rules);
}

/*****************************************************************************/
static void printGraph(predNode *pNode, ruleNode *rNode)
{
	ruleNode *r;
	int i, j;

	fprintf(stderr, "\nThe rule/goal graph:\n");
	if (rNode==NULL)
		return;
	fprintf(stderr, "rule nodes:\n");
	for (r=rNode, i=0; r!=NULL; r=r->next, i++) {
		fprintf(stderr, "rule #%d (%s): ",i+1,
			((r->solved==COR_SOLVED)? "solved":"unsolved"));	
		if (r->r->head != NULL)
			fprintf(stderr, "%s\n", SymbolString(r->head->name));
		else 
			fprintf(stderr, "\n");
	}
	fprintf(stderr, "predicate nodes:\n");

	for (predNode *pN = pNode->next;pN;pN = pN->next) {
	    fprintf(stderr, "%s (%s):", SymbolString(pN->pred->name),
			((pN->solved==COR_SOLVED)? "solved" : "unsolved"));
	    for (j=0; j<pN->numRule; j++)
		if (pN->r[j]->head != NULL)
		    fprintf(stderr, " %s", SymbolString(pN->r[j]->head->name));
		fprintf(stderr, ".\n");
	}
}

/*****************************************************************************/

