/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

#include "rules.h"
#include "magic.h"
#include "magic-externs.h"
#include "globals.h"

extern int adorn_type;
static void deepen_adorn(struct collection *, 
		struct collection * , Name name, BitVector& adorn);
static int isbound(Arg * a, BitVector *u);
extern void setup_vars(Clause*,struct ClauseInfo[]) ;
extern void gen_sip_rule(struct ClauseInfo[], BitVector&);
extern void cleanup_vars(struct ClauseInfo[]);
extern void set_best_allowed_adorn(Literal *lit);
extern int listing ;
static int adorn_rule(struct collection *, 
		struct rule *, struct collection *, Name, BitVector&);
static void adorn_rule_preds(struct collection *, struct rule *, int, 
				Literal *, struct collection *) ;

static int equals_pred(Literal *lit1, Literal *lit2);

extern int has_grouping_args(ArgList &args);

extern Name build_ad_grouped_pred_name(Literal *head);
extern int grouped_pred_arity(Literal *head);

extern char *init_any_adorn(int num);
extern void add_disallowed_adornment(ParserStruct&, Name, char*, int);
extern void add_allowed_adornment(ParserStruct&, Name, char*, int);

extern void create_grouping_rule_1(struct rule *rule, Name new_pred_name,
		int new_pred_arity, struct collection *pre_ad_rules);
extern void create_grouping_rule_2(struct rule *rule, Name new_pred_name,
		int new_pred_arity, struct collection *pre_ad_rules);
extern void update_annotations(Name name, BitVector &adorn);

static int group_rule_num = 0;


void adorn(struct collection *pre_ad_rules, struct collection * ad_rules)
/*
 * Construct the adorned system.  adorn_type==1: construct as described in
 * "On the Power of Magic", adorn_type==2: as in "Magic Templates:
 * A Spellbinding Approach to Logic Programs".
 *
 * Type 2 is different from type 1 as follows:
 *
 * - bound query args are any arg position that contains a function symbol,
 *   constant, or repeated variable.
 * - to compute the adornment of a derived predicate, an argument is considered
 *   bound if *some* variable appearing in it is in the union of sip labels.
 */
{
    BitVector	adorn;

    
    if (parserStruct.CurModule.ExportedPreds == NULL)
	fprintf(stderr, "'export' missing or bad format.\n");
    else {
	for (ExportInfo *export_info = parserStruct.CurModule.ExportedPreds;
	     export_info != NULL;
	     export_info = export_info->next) {
	     deepen_adorn(pre_ad_rules, ad_rules, 
			export_info->pred_name, export_info->adornment);
	}
    }

    if ( listing ) {
    	printf("Adorned rule set:\n");
    	FOR_EACH_RULE(adrule, ad_rules){
		print_rule(adrule, stdout) ;
		} END_EACH_RULE
    	printf("------- End adorned rule set -------\n") ;
	}
}

static void deepen_adorn(struct collection *pre_ad_rules, 
			struct collection * ad_rules, 
			Name name, BitVector& adorn)
{

    FOR_EACH_RULE_FOR(rule, name, ad_rules) {
	if (rule->head->adorn == adorn)
	    return;
    } END_EACH_RULE;

    // Find rules that match name and adorn.
	
	FOR_EACH_RULE_FOR(rule, name, pre_ad_rules) {
	    if (!adorn_rule(pre_ad_rules, rule, ad_rules, name, adorn))
		break ;
	} END_EACH_RULE;

     // Update the annotations

     update_annotations(name, adorn);


}

static int adorn_rule(struct collection *pre_ad_rules, 	
			struct rule *rule, struct collection *ad_rules,
			Name name, BitVector& adorn)
{
    struct sips *	s;

    /*
     * Build the skeleton of the new adorned rule
     */
    Clause * adrule = copy_rule(rule);
    adrule->head = copy_pred(rule->head);
    adrule->head->base_pred = 0;
    adrule->head->adorn = adorn;
    alloc_preds(adrule, rule->num_preds);
    /* attach only the appropriate sip */
    adrule->sips = new sips;
    adrule->sips->adorn = adorn;
    add_rule(adrule, ad_rules);
    adrule->sips->next = NULL;
    /*
     * For this rule, find its sip with the same adornment
     * as the worklist predicate
     */
    int sips_complete = 1;
    struct sip *asip;
    for (s=rule->sips; s; s=s->next)
	if (s->adorn == adorn) {
	    for(int pred_num=0; pred_num < rule->num_preds; 
						pred_num++) {
		for(asip=s->sip_list;asip;asip=asip->next)
		     if (asip->head==pred_num+1)
			break;
		if (!asip)
		    sips_complete = 0;
		}
	    break ;
	}
    if (s!=NULL && sips_complete) 
	adrule->sips->sip_list = s->sip_list;
    else {
	if (rule->head->args.count() > COR_MAX_ARGLIST_SIZE) {
    	    fprintf(stderr,
	    "Error: Rule (line %d) has too many args (%d) - ignored\n",
	    	rule->line_number, rule->head->args.count());
            fflush(stderr);
	    return 1 ;
	}
	struct ClauseInfo rule_info[1];
    	setup_vars(rule, rule_info);
    	gen_sip_rule(rule_info, adorn);
	cleanup_vars(rule_info);
	for (s=rule->sips; s; s=s->next)
	    if (s->adorn == adorn)
		 break;
	if (! s) {
	    fprintf(stderr,
		    "Internal Error: no sip matching adornment ") ;
	    print_adornment(adorn, stderr);
	    fprintf(stderr, " for %s (line %d)\n",
		SymbolString(name), rule->line_number);
	    return 0  ;
	    }
	adrule->sips->sip_list = s->sip_list;
    }

    /*
     * Now compute the adornment for each derived predicate
     * in the rule's right-hand-side
     */
    for (int p=0; p<rule->num_preds; ++p)
	adorn_rule_preds(pre_ad_rules, adrule, p, rule->preds[p], ad_rules) ;

     /***** Now handle any head deletes *******/
    
    adrule->head_deletes_count = rule->head_deletes_count;
    adrule->head_deletes = new Literal* [adrule->head_deletes_count];
    for (int hdnum=0; hdnum < rule->head_deletes_count; hdnum++) {
      adrule->head_deletes[hdnum] = copy_pred(rule->head_deletes[hdnum]);
      adrule->head_deletes[hdnum]->base_pred = 0; 
      // So adorn will get printed

      int rulepos;
      for(rulepos=0; rulepos < rule->num_preds; rulepos++) {
	if (equals_pred(rule->preds[rulepos], rule->head_deletes[hdnum]))
	  break;
      }
      if (rulepos < rule->num_preds) {
	adrule->head_deletes[hdnum]->adorn = adrule->preds[rulepos]->adorn
	  ;
      }
    }
    
   return 1;

}

static void adorn_rule_preds(struct collection *pre_ad_rules,
			struct rule *adrule, int p,
			Literal *pred, struct collection *ad_rules)
{
    struct sip *	arc;

  /*
   * Add the predicate to the adorned rule
   */
  adrule->preds[p] = copy_pred(pred);
  if (pred->base_pred)
	adrule->preds[p]->adorn.setlen(0);
  else {
	/*
	* Compute the union of labels of the sip arcs
	* entering the node labelled by pred
	*/
	BitVector u;
	for (arc=adrule->sips->sip_list; arc; arc=arc->next) {
	    /* sub 1 since p indexes rhs only (not head) */
	    if (arc->head-1 == p) {
		u.or_bits(&arc->bound_vars);
	    }
	}
	/*
	 * For each arg in pred, if all vars in that arg
	 * are in the union, mark the arg as bound
	 */
	int i = 0;
	FOR_EACH_ARG(a, pred->args) {
	    if (isbound(a, &u))
		adrule->preds[p]->adorn.set(i);
		i++;
	} END_EACH_ARG
	/* Find best allowed adornment */
	set_best_allowed_adorn(adrule->preds[p]);
	deepen_adorn(pre_ad_rules, ad_rules, 
			pred->pred, adrule->preds[p]->adorn);
  }
}

static int isconst2(Arg * a)
/*
 * Recursively traverse the given argument and return 1 if at least one
 * sub-arg is a constant or a function symbol.
 */
{
    switch (a->kindof()) {
      case COR_VARIABLE:
      case COR_DONTCARE:
	return (0);
      default:
	return (1);
    }
}


static int isbound(Arg * a, BitVector *u)
/*
 * Type 1:
 * Recursively traverse the given argument and return 1 if all
 * sub-args are in the set of variables represented by the BitVector u
 *
 * Type 2:
 * Recursively traverse the given argument and return 1 if there is at least
 * one sub-arg that is in the set of variables represented by the BitVector u
 */
{
    if ((adorn_type == 2) || (parserStruct.CurModule.UseReturnUnify)) {
	switch (a->kindof()) {
	    case COR_VARIABLE:
		return u->test(((VarArg*)a)->var);
	    case COR_SYMBOL_CONST:
	    case COR_NUM_CONST:
		return (1);
	    case COR_DONTCARE:
		return (0);
	    case COR_FUNCTOR:
		FOR_EACH_ARG(arg, ((FuncArg*)a)->args) {
		    if (isbound(arg, u))
			return (1);
		} END_EACH_ARG
		return (0);
	}
    }
    else {
	switch (a->kindof()) {
	    case COR_VARIABLE:
		return u->test(((VarArg*)a)->var);
	    case COR_SYMBOL_CONST:
	    case COR_NUM_CONST:
		return (1);
	    case COR_DONTCARE:
		return (0);
	    case COR_FUNCTOR:
		FOR_EACH_ARG(arg, ((FuncArg*)a)->args) {
		    if ( ! isbound(arg, u))
			return (0);
		} END_EACH_ARG
		return (1);
	}
    }
}

static int equals_pred(Literal *lit1, Literal *lit2)
{
  if(lit1->name() != lit2->name() || lit1->arity() != lit2->arity())
    return 0;
  for(int i=0; i < lit1->arity(); i++) {
    if (! lit1->args[i]->equals(lit2->args[i]) )
      return 0;
  }
  return 1;
}



extern int is_disallowed_adorn(Name name, BitVector &adorn);

void set_best_allowed_adorn(Literal *lit)
{
    PredAnnotations *cur = parserStruct.AllowedAdornList;

    /* This method is a modified version of the method used in 
		iterative-rel.C */

    int best_shortfall = 1000; /* Infinity */
    BitVector *best_adorn= NULL;
    int restrictions = 0; /* Adornments restricted for this predicate */
    for(;cur != NULL; cur = cur->next()) {
	if (cur->pred == lit->predicate->name) {
	    int i = lit->adorn.len();
	    if (cur->adorn->len() != i) 
		continue; 
	    restrictions = 1; /* An adornment restriction has been specified 
				for this predicate */
	    int shortfall= 0;
	    while (--i >= 0) {
		if ( ! lit->adorn.test(i) && cur->adorn->test(i) )
			break; /* Allowed adorn overbound */
		if ( lit->adorn.test(i) && !cur->adorn->test(i) )
		    shortfall++;
	    }
	    if ( i >= 0 ) 
		continue ; /* Allowed adorn is overbound  - ignore it */
	    if (( shortfall < best_shortfall) && 
			(!is_disallowed_adorn(cur->pred, *(cur->adorn))))
		best_adorn = cur->adorn;
	}
    }
    if ( !restrictions)
	return;
    if (!best_adorn) {
	fprintf(stderr, 
	"Warning: adornment for %s does not match any allowed adornment.\n%s\n",
			lit->predicate->name->string(),
	"		-- adornment restrictions ignored." 
			);
	
	return;
    }
    lit->adorn.init(*best_adorn);
    return;

}

int is_disallowed_adorn(Name name, BitVector &adorn)
{
    PredAnnotations *cur = parserStruct.DisallowedAdornList;
    int adorn_length = adorn.len();

    for(; cur != NULL; cur = cur->next()) {
	if (cur->pred == name) {
	    if (cur->adorn->len() != adorn_length) 
		continue;
	    else if (*(cur->adorn) == adorn)
		return 1;
	}
    }
    return 0;
}

/**********************
 * the following code is to ensure that a rule with grouping in 
 * the head and more than one body predicate is split into two rules 
 * as follows:
 * 	p(tbar, <Y>) :- l1, l2, ... ln.  is split up as 
 *		(1): p(Xbar, <Y>) :- pprime(Xbar, Y).  and
 *		(2): pprime(tbar, Y) :- l1, l2, ... ln.
 * this is to ensure correctness of the evaluation of the magic
 * rewritten program.
 *
 * it also ensures that grouping args are not bound in the adorned program.
 * consequently, magic works for programs with grouping
**********************/

void preprocess_for_grouping(struct collection *orig_rules,
		struct collection *pre_ad_rules)
{
    int j;
    char *disallowed_head_adorn, *possible_allowed_head_adorn;
    struct rule *preadrule;
    Name new_pred_name;
    int new_pred_arity;

    group_rule_num = 0;
    FOR_EACH_RULE(rule, orig_rules) {
	Literal *head = rule->head;
	if (has_grouping_args(head->args)) {
	    if (rule->num_preds == 0) { 
		fprintf(stderr, "ERROR: Cannot have grouping rule with ");
		fprintf(stderr, "empty body.\n");
		return;
	    }
	    else { // valid grouping rule in the program
		// FIRST emit grouping argument restrictions
		disallowed_head_adorn = init_any_adorn(head->args.count());
		possible_allowed_head_adorn=init_any_adorn(head->args.count());

		int i = 0;
		FOR_EACH_ARG(a, head->args) {
		    if (a->kindof()==COR_GROUPING) {
			disallowed_head_adorn[i] = 'b';
			possible_allowed_head_adorn[i] = 'f';
		    }
		    i++;
		} END_EACH_ARG

		add_disallowed_adornment(parserStruct, head->pred, 
					 disallowed_head_adorn,
					 head->args.count());
		add_allowed_adornment(parserStruct, head->pred, 
				      possible_allowed_head_adorn,
				      head->args.count());
		delete disallowed_head_adorn;
		delete possible_allowed_head_adorn;
	         
		// NEXT rewrite grouping rules into two rules
		if (rule->num_preds > 1) {
		    new_pred_name = build_ad_grouped_pred_name(head);
		    new_pred_arity = grouped_pred_arity(head);
	            create_grouping_rule_1(rule, new_pred_name, new_pred_arity,
				    pre_ad_rules);
	            create_grouping_rule_2(rule, new_pred_name, new_pred_arity,
				    pre_ad_rules);
		    continue;
		}
	    }
	}

	// either there are no grouping args in the head or the body
	// had exactly one predicate -- the same thing needs to be done
	// just copy out the rule

	preadrule = copy_rule(rule);
	preadrule->head = rule->head;
	alloc_preds(preadrule, rule->num_preds);
	for (j=0; j <rule->num_preds; j++) {
	    preadrule->preds[j] = rule->preds[j];
	}
	preadrule->head_deletes_count = rule->head_deletes_count;
	preadrule->head_deletes = rule->head_deletes;
	add_rule(preadrule, pre_ad_rules);
    } END_EACH_RULE
}

Name build_ad_grouped_pred_name(Literal *head)
{
    char *head_name = head->pred->string();
    int head_name_length = strlen(head_name);

    char *buf = new char [head_name_length+10];
    group_rule_num++; // this gives it the unique numbering

    sprintf(buf, "grp_%d_",group_rule_num);
    char *buf1 = buf + strlen(buf);
    strcpy(buf1, head_name);
    buf1[head_name_length] = '\0';
    
    return EnterSymbol(buf);
}

int grouped_pred_arity(Literal *head)
{
    int ret_val = 0;
    FOR_EACH_ARG(a, head->args) {
	if (a->kindof() != COR_GROUPING) {
	    ret_val++;
	}
	else { // it is a grouping argument
	    ret_val += ((Grouping *) a)->arity();
	}
    } END_EACH_ARG
    return ret_val;
}

char *init_any_adorn(int num)
{
    char *ret_val = new char[num+1];
    for (int i = 0; i < num; i++) 
	ret_val[i] = 'a';
    ret_val[num] = '\0';
    return ret_val;
}

void create_grouping_rule_1(struct rule *rule, Name new_pred_name,
		int new_pred_arity, struct collection *pre_ad_rules)
{
/**************
 * this adds the rule of the form
 *		p(Xbar, <Y>) :- pprime(Xbar, Y)
 * obtained from 
 *		p(tbar, <Y>) :- l1, l2, ... ln
**************/

    struct rule *grouping_rule_1 = copy_rule(rule);
    Literal *head = rule->head;
    int j, k;
    Arg **arg_ptr, **grp_arg_ptr;
    Grouping *new_groupingarg;

    // create the new head literal
    Literal *new_head = AllocLiteral(parserStruct, 
				     head->pred, head->args.count());
    j = 0;
    arg_ptr = new_head->args.first();
    FOR_EACH_ARG(a, head->args) {
	if (a->kindof() != COR_GROUPING) {
	    *arg_ptr++ = make_var(j++);
	}
	else { // it was a grouping argument earlier and it has to be now
	    new_groupingarg = Grouping::New(((Grouping *)a)->arity());
	    new_groupingarg->kind = ((Grouping *) a)->kind;
	    grp_arg_ptr = new_groupingarg->first();
	    for (k=0; k < new_groupingarg->arity(); k++) {
		*grp_arg_ptr++ = make_var(j++);	
	    }
	    *arg_ptr++ = new_groupingarg;
	}
    } END_EACH_ARG

    // allocate the head
    grouping_rule_1->head = new_head;

    // create the new body literal
    Literal *new_body = AllocLiteral(parserStruct, 
				     new_pred_name, new_pred_arity);
    arg_ptr = new_body->args.first();
    for (j=0; j < new_pred_arity; j++) {
	*arg_ptr++ = make_var(j);
    }

    // allocate the body
    alloc_preds(grouping_rule_1, 1);
    grouping_rule_1->preds[0] = new_body; 

    add_rule(grouping_rule_1, pre_ad_rules);
}

void create_grouping_rule_2(struct rule *rule, Name new_pred_name,
		int new_pred_arity, struct collection *pre_ad_rules)
{
/**************
 * this implicitly adds the rule of the form
 *		pprime(tbar, Y) :- l1, l2, ... ln.
 * obtained from 
 *		p(tbar, <Y>) :- l1, l2, ... ln
**************/

    int j;
    struct rule *grouping_rule_2;
    Arg **arg_ptr, **grp_arg_ptr;
    Literal *head = rule->head;

    grouping_rule_2 = copy_rule(rule);

    // create the new head 
    Literal *new_head = AllocLiteral(parserStruct, 
				     new_pred_name, new_pred_arity);
    arg_ptr = new_head->args.first();
    FOR_EACH_ARG(a, head->args) {
	if (a->kindof() != COR_GROUPING) {
	    *arg_ptr++ = a;
	}
	else { // it is a grouping argument 
	    grp_arg_ptr = ((Grouping *) a)->first();
	    for (j=0; j < ((Grouping *) a)->arity(); j++) {
	        *arg_ptr++ = *grp_arg_ptr++;
	    }
	}
    } END_EACH_ARG

    // allocate the head
    grouping_rule_2->head = new_head;

    // copy out the body
    alloc_preds(grouping_rule_2, rule->num_preds);
    for (j=0; j <rule->num_preds; j++) {
	grouping_rule_2->preds[j] = rule->preds[j];
    }
    add_rule(grouping_rule_2, pre_ad_rules);
}

