
module simulation_model.
export model(bbbf), attribute(bbbf), rate_of_change(bbbf).

%  This is a simple ecological simulation model.  A Prolog version is 
%  presented in the book Ecologic: Logic-Based Approaches to 
%  Ecological Modelling, by Robertson et al.

% Typical queries are of the form ?model(mass, deer_pop, 10, N).
% Intuitively, the mass of the deer population at time 10 is computed.
% Robertson et al. note that a standard Prolog system takes exponential time,
% taking over an hour of real time for values of T around 15.

model(mass, deer_pop, T, N) :-
	sumdeer({young_deer,old_deer}, T,  N).

attribute(mass, young_deer, 0, 10).
attribute(mass, old_deer, 0, 20).

attribute(mass, X, T, N) :-
	previous_time(T,Tp),
	attribute(mass, X, Tp, N1),
	sumchange({growth_rate,mortality_rate}, Tp, X, Sum),
	N = N1+Sum.

rate_of_change(growth_rate, X, T, N) :- 
	attribute(mass, X, T, N1), N = 0.3 * N1.

rate_of_change(mortality_rate, X, T, N) :- 
	attribute(mass, X, T, N1), N = -0.2 * N1.
	
sumdeer(S, T, sum(<N>)) :- member(S, X), attribute(mass, X, T, N).

sumchange(S, Tp, X, sum(<N>)) :- member(S, R), rate_of_change(R, X, Tp,N).

end_module.

module time.
export previous_time(bf).
@pipelining.

previous_time(T, Tp) :- T>0, Tp = T-1.

end_module.
