/************************************************************************
 ========================================================================
 CORAL 
 (c)  Copyright R. Ramakrishnan and The CORAL Group, 
 University of Wisconsin at Madison.
 (1992) All Rights Reserved.
 Version 0.1
 ========================================================================



 ------------------------------------------------------------------------
 CORAL Version 0.1
 RESEARCH SOFTWARE DISCLAIMER -------------------------------------------
 ------------------------------------------------------------------------

    As unestablished, research software, this program is provided free of 
    charge on an "as is" basis without warranty of any kind, either 
    express or implied.  Acceptance and use of this program constitutes 
    the user's understanding that (s)he will have no recourse for any 
    actual or consequential damages, including, but not limited to, 
    lost profits or savings, arising out of the use of or inability to 
    use this program.  

 ------------------------------------------------------------------------
 USER AGREEMENT ---------------------------------------------------------
 ------------------------------------------------------------------------

     BY ACCEPTANCE AND USE OF THIS EXPERIMENTAL PROGRAM
     THE USER AGREES TO THE FOLLOWING:

     a.  This program is provided free of charge for the user's personal, 
	 non-commercial, experimental use.

     b.  All title, ownership and rights to this program and any copies 
         remain with the copyright holder, irrespective of the ownership 
	 of the media on which the program resides.

     c.  The user is permitted to create derivative works to this program.  
         However, all copies of the program and its derivative works must
         contain the CORAL copyright notice, the UNESTABLISHED SOFTWARE 
         DISCLAIMER and this USER AGREEMENT.

     d.  The user understands and agrees that this program and any 
         derivative works are to be used solely for experimental purposes 
	 and are not to be sold or commercially exploited in any manner 
	 WITHOUT EXPRESS WRITTEN PERMISSION.

     e.  We request that the user supply us with a copy of any changes, 
         enhancements, or derivative works which the user may create,
	 with the user's permission to redistribute it.
	 Copies of such material should be sent to:  CORAL@CS.WISC.EDU

-------------------------------------------------------------------------
*************************************************************************/

/***********************************************************************
	CORAL Software :: U.W.Madison

	coral-includes.h : Header file

	Specifications for interfacing CORAL with C++

 ***********************************************************************/
#ifndef CORAL_INCLUDES_H
#define CORAL_INCLUDES_H

#include "interface.h"
#include "globals.h"
#include "parser.h"
#include "generic-index.h"
#include "cor_error.h"

/* *** Using C++ from declarative CORAL *** *

 * In order to declare a 'built-in' relation defined using a C++
 * function, the user needs to make a _CORAL_export declaration,
 * of the form (for example)
 *
 *         _coral_export int newrel(int, double);
 *
 * and provide the function int newrel(int, double). This creates a
 * built-in relation with the name 'newrel', and with arity 3, where
 * the third argument is the return value of the function.
 *
 *
 * The types allowed are int, short, long, float, double, char * and 
 * Arg *. User defined types are not allowed.
 *
 * However, the user should note that internally, CORAL stores all
 * ints, shorts and longs as longs. Similarly, all floats are stored
 * as doubles. 
 *


 * *** Accessing CORAL structures from C++ *** *

 ** the class Arg is available, defined as specified below :
   class Arg {
    public :
      void printon(FILE *file);    // print Arg to the file
      int equals(Arg *arg2);       // equality method 
   };
 **

 * To create an argument *
extern Arg *make_arg (int i);
extern Arg *make_arg (long l);
extern Arg *make_arg (short s);
extern Arg *make_arg (float f);
extern Arg *make_arg (double n);
extern Arg *make_arg (char *name);

 *
 * To create a variable argument. Distinct var_nums represent distinct 
 * variables, and identcial var_nums represent identical variables
 *
 * NOTE: All variable arguments within a particular tuple should
 *       have different var_nums if they are meant to be distinct.
 *       Further, these variables should be numbered starting from
 *       0, and increasing in steps of 1.
 *
 *

extern Arg *make_var (int var_num);
extern Arg *make_var (char *print_name, int var_num);

 * To create a functor argument *
extern Arg *make_arg (char *func_name, ArgList *args);

 * To create a cons cell (used for list arguments) *
extern Arg *make_cons (Arg *a, Arg *b);

 * To create an empty list [] *
extern Arg *make_nil ();

 *
 * Functions to determine the nature of an argument .
 * All these functions return 0 if false, and non-zero otherwise
 *

extern int is_int (Arg *a);
extern int is_long(Arg *a);
extern int is_short(Arg *a);
extern int is_float (Arg *a);
extern int is_double (Arg *a);
extern int is_string (Arg *a);
extern int is_num (Arg *a);
extern int is_constant (Arg *a);
extern int is_var (Arg *a);
extern int is_functor (Arg *a);
extern int is_list (Arg *a);

 *
 * Functions to extract values from an Arg*
 *

extern int make_int (Arg *a);            // integer represented by 'a'
extern short make_short(Arg *a);         // short represented by 'a'
extern long make_long (Arg *a);          // long represented by 'a'
extern float make_float (Arg *a);        // float represented by 'a'
extern double make_double (Arg *a);      // double represented by 'a'
extern char *make_string (Arg *a);       // string represented by 'a'

 * Functions to extract values from functor Args of the form f(a1, .., an) *
extern char *functor_name (Arg *a);      // functor name of argument (i.e) 'f'
extern ArgList *functor_args (Arg *a);   // functor arguments (i.e) a1, .., an

 * Functions to extract values from list Args of the form [a1, .., an] *
extern Arg *make_car (Arg *a);           // car of the list
extern Arg *make_cdr (Arg *a);           // cdr of the list




 ** the class ArgList is available, defined as specified below :
   class ArgList {
    public :
      int arity();               // arity of the ArgList
      Arg*& operator [] (int i); // indexing operator 
      void printon(FILE *file);  // print ArgList to the file
   };

 NOTE :: Do not use the C++ new() function to create an ArgList. Instead
         use the functions below
 **


 *
 * To create a list of arguments : the first parameter is the length of the
 * argument list, and the remaining parameters are of type Arg *
 *
extern ArgList *make_arglist (int len ...);


 *
 * In case the variable argument passing is not properly supported by
 * the C++ compiler (as in ATT C++ 1.9). (is obsolete with ATT C++ 2.0 and later)
 *
extern ArgList *make_arglist1 (Arg* a1);
extern ArgList *make_arglist2 (Arg* a1, Arg *a2);
extern ArgList *make_arglist3 (Arg* a1, Arg *a2, Arg *a3);
extern ArgList *make_arglist4 (Arg* a1, Arg *a2, Arg *a3, Arg *a4);
 * To concatenate two lists of arguments *
extern ArgList *concat_arglists (ArgList *al1, ArgList *al2);






 *
 * To create a list of distinct variable arguments of length n
 *
extern ArgList *make_vararglist (int n);

 ** the class Tuple is available, defined as specified below :

   class Tuple {
   public :
      void do_delete();         // mark the tuple as deleted
      int arity();              // arity of the tuple 
      Arg*& operator [] (int i);// indexing operator 
      void printon(FILE *file); // print the contents of the tuple
   } ;

   VERSION 2 :: methods to manipulate environments and handle non-ground
                tuples
 **

 *
 * To create a Tuple
 *
 * A typical way of creating a Tuple is
 *
 *        Tuple *tuple = make_tuple(make_vararglist(3));
 *
 *    which creates a Tuple whose argument list has three variables.
 *
extern Tuple *make_tuple(int arity);
extern Tuple *make_tuple(ArgList *arglist);


 ** the class Relation is available, defined as specified below :
   class Relation {
   public :
      int tuple_count();                  // number of tuples in the relation
      int arity();                        // arity of the relation
      void empty_relation();              // empty the relation of all tuples
      int check_subsumption();            // returns non-zero if checking
                                          // for subsumption is turned on.
      void set_subsumption(int flag);     // sets the status of subsumption
                                          // checking to 'flag'
      int tuple_insert(Tuple *tuple);     // insert a tuple to the relation

      int tuple_delete(Tuple *tuple);     // value based deletion: delete 
                                          // all tuples that are subsumed
                                          // by the argument tuple

      int tuple_update(Tuple *old_tuple, Tuple *new_tuple));
                                          // logical update of a tuple, 
                                          // implemented by first doing a 
                                          // delete_tuple(rel, old_tuple) 
					  // followed by a 
					  // rel->insert_new(new_tuple)

      int add_index(char *adorn);         // create an index as specified by
                                          // the adorn string.(eg."bbf" means
					  // index on the first two fields)

      int add_index(ArgList *pattern, int num_vars, ArgList *index_pattern)
                                          // create a pattern-from index as
					  // specified by the pattern and the
					  // index_pattern. For instance,
					  // add_index((X,Y), 2, (X)) adds
					  // and index on the first field.
					  // The notation (X,Y) here means
					  // the ArgList with X and Y as
					  // its two Args.


   } ;
**

 *
 * Function for finding an existing database relation.
 * Returns null if the relation is not found
 *
extern Relation *find_relation (char *db_rel_name, int arity);

 * Function to create a new relation
extern Relation *make_relation(char *rel_name, int arity);

 *
 * Interface function for querying and calling the declarative module.
 * To be used when the answer is to be materialized ans stored. If the
 * answer need not be materialized, it is more efficient to set up
 * a C_ScanDesc on the relation, using the query tuple as an argument 
 * to it, so that only the desired tuples are returned.
 * 
 * The query is a tuple of the relation, and any facts in the relation
 * that unify with the query tuple are answers to the query.
 * 
 * For example, (X,Y) is a query asking for all tuples in a binary relation.
 *	
 * Similarly, (1,Y) returns all tuples that have 1 in the first column.
 *	
 * If the result parameter is NULL, a new relation is allocated and result
 * points to it. Otherwise, answer tuples are added to the relation 'result'.
 * 
 * Returns:  Number of answer tuples.  Result < 0 implies error.
 *
extern int call_coral(char *exp_pred_name, Tuple *query,
		      Relation *&result);


 ** the class C_ScanDesc is available, defined as specified below :

class C_ScanDesc {
  C_ScanDesc(Relation *rel);                     // constructor 
  C_ScanDesc(Relation *rel, Tuple *match_tuple); // constructor
  ~C_ScanDesc();                                 // destructor
  Tuple *next_tuple();                           // returns the next tuple
  int no_match();                                // is non-zero when there
                                                 // are no more tuples left
						 // to scan
};

 *
 * The C_ScanDesc abstraction allows for scans over a relation.
 * The scans can be either over an entire relation :
 *      C_ScanDesc *scan = new C_ScanDesc(rel);
 *
 *  or over those tuples in a relation that match a given tuple :
 *      C_ScanDesc *scan = new C_ScanDesc(rel, match_tuple);
 *
 *
 * A typical use of such a C_ScanDesc is to loop over all the scanned
 * tuples, performing some action
 *      for (tuple = scan->next_tuple(); !(scan->no_match());
 *                                   tuple = scan->next_tuple()) {
 *            .
 *            .
 *            .
 *      }
 * 
 *
 * A scan on a relation is 'closed' only when the C_ScanDesc is destroyed
 * using :
 *     delete scan;
 * 
 *



 *
 * any declarative CORAL calls (anything that can be entered at the
 * CORAL prompt, including module declarations, builtin calls, fact 
 * insertions, and imperative rules) can be embedded within C++
 * bracketed by 
 *
 *    \[
 *        <CORAL code >
 *    \]
 *
 * Note that the bracketing symbols /[ and ]/ must occur alone
 * on a separate line of the program.
 *
*/

#endif  /* CORAL_INCLUDES_H */
