/*
 *	Sowam Bytecode Interpreter
 *
 *	File:
 *		unify.c
 *		Unifikation, Occur-Check, Reject
 *
 *	Author:
 *		V. Siebert
 *
 *	History:
 *
 */

#include "sowam.h"


/*
 * Einige Register der SOWAM
 */
#define R_E		(s_regs.e)
#define R_B		(s_regs.b)
#define R_R		(s_regs.r)
#define R_HR		(s_regs.hr)
#define R_T		(s_regs.t)
#define R_HP		(s_regs.h)

/*
 * Prueft, ob t durch Narrowing bzw. Rewriting ersetzt werden kann.
 * e gibt an, ob wir uns schon im Environment befinden.
 * Aus Geschwindigkeitsgruenden benutzen wir zunaechst ein Makro.
 */
#define IS_CONSTRUCTOR(t,e)	((e) || is_constructor (t,LOC(t) == ENV))

static int
is_constructor (t,e)
    term t;
    int e;
{
    while (!e && (TAG(t) == T_STRUCT || TAG(t) == T_VAR)) {
	e = LOC(t) == ENV;
	t = REF(t);
    }
    return (e || TAG(t) == T_LIST || (TAG(t) == T_AF && !IS_FUNCTION(t)));
}


/*
 * Makros fuer die Emulation der PDL (Push Down List) der orig. WAM
 * Wir benutzen dafuer den freien Speicher auf dem Local-Stack.
 */
#define PUSH_PDL(t)             (*lsptr++ = (t))
#define POP_PDL()               (*--lsptr)
#define PUSH_PAIR(t1,t2)        (PUSH_PDL(t1), PUSH_PDL(t2), ++upl)
#define PUSH_IPAIR(e1,e2)	(* (int *) (lsptr++) = (e1), \
				 * (int *) (lsptr++) = (e2), ++upl)
#define POP_PAIR(t1,t2)         (upl-- ? ((t2) = POP_PDL(), (t1) = POP_PDL(), 1) : 0)
#define POP_IPAIR(e1,e2)	(upl-- ? ((e2) = * ((int *) --lsptr), \
					  (e1) = * ((int *) --lsptr), 1): 0)
#define PUSH_SINGLE(t)          (PUSH_PDL(t), ++upl)
#define POP_SINGLE(t)           (upl-- ? ((t) = POP_PDL(), 1) : 0)
#define REPLACE(t1,lc,tg,ad) \
    {termptr = &REF(t1); \
     if (!trail_undef(termptr)) return; \
     SET_LOC(*termptr, lc); \
     SET_TAG(*termptr, tg); \
     SET_VAL(*termptr, ad);}
#define PROCEED_UNIFY(t1,t2)	{if(!POP_PAIR(t1,t2)) return;}
#define PROCEED_OCCUR(t)        {if(!POP_SINGLE(t)) return 0;}
#define CHECK_SPACE(n)		{if(lsptr+2*(n) >= HP_MAX) hp_overflow();}

/*
 * Occur check
 */
#define OCCUR(addr,t,lsptr)   (s_glflags.occur_check ? occur(addr,t,lsptr) : 0)

static int
occur (addr,t,lsptr)
    int addr;
    term t;
    term *lsptr;
{
    /*
     * Arbeits (Lauf-) variablen --> Register
     */
    register int upl = 0;		/* unify (occur) pairs left */
    register term *termptr;

    forever {
        switch (TAG(t)) {
            case T_AF:
                PROCEED_OCCUR(t);
                break;
            case T_UNDEF:
                if (VAL(t) == addr)
                    return 1;
                PROCEED_OCCUR(t);
                break;
            case T_VAR:
                if (VAL(t) == addr)
                    return 1;
		t = REF(t);
                break;
            case T_LIST:
                PUSH_SINGLE(TAIL(t));
                t = REF(t);
                break;
            case T_STRUCT:
                if (TAG(REF(t)) == T_AF) {
                    int i,st = ARITY(VAL(REF(t)));

		    if (st == 0) {
			PROCEED_OCCUR(t);
		    } else {
			for (i=2; i<=st; i++)
			    PUSH_SINGLE(AF_ARG(t,i));
			t = AF_ARG(t,1);
		    }
                } else
		    t = REF(t);
                break;
	    default:
		internal_error("occur");
        }
    }
}


/*
 * Allg. Unifikationsalgorithmus
 */
void
unify (t1,t2)
    term t1,t2;
{
  /*
   * Arbeits (Lauf-) variablen --> Register
   */
  register int upl = 0;		/* unify pairs left */
  register term *lsptr = R_HP;
  register term *termptr;

  forever {
    while (TAG(t1) == T_VAR ||
	   TAG(t1) == T_STRUCT && TAG(REF(t1)) != T_AF)
      t1 = REF(t1);
    while (TAG(t2) == T_VAR ||
	   TAG(t2) == T_STRUCT && TAG(REF(t2)) != T_AF)
      t2 = REF(t2);

    if (TAG(t1) == T_UNDEF) {
      if (TAG(t2) == T_UNDEF) {
	/*
	 * t1,t2 ungebundene Variablen
	 */
	if (VAL(t1) > VAL(t2)) {
	  REPLACE(t1,ENV,T_VAR,VAL(t2));
	} else if (VAL(t2) > VAL(t1)) {
	  REPLACE(t2,ENV,T_VAR,VAL(t1));
	}
      } else {
	/*
	 * t1 ungebundene Variable
	 * t2 beliebig
	 */
	if (OCCUR(VAL(t1),t2,lsptr))
	  break;

	REPLACE(t1,ENV,TAG(t2),VAL(t2));
      }
      PROCEED_UNIFY(t1,t2);
    } else if (TAG(t2) == T_UNDEF) {
      /*
       * t2 ungebundene Variable
       * t1 beliebig
       */
      if (OCCUR(VAL(t2),t1,lsptr))
	break;

      REPLACE(t2,ENV,TAG(t1),VAL(t1));
      PROCEED_UNIFY(t1,t2);
    } else if (TAG(t1) == T_LIST) {
      if (TAG(t2) != T_LIST)
	break;
      /*
       * t1,t2 sind Listen
       */
      CHECK_SPACE(1);

      PUSH_PAIR(TAIL(t1), TAIL(t2));
      t1 = REF(t1);
      t2 = REF(t2);
    } else if (TAG(t1) == T_AF) {
      if (TAG(t2) == T_STRUCT)
	t2 = REF(t2);
      if (VAL(t1) != VAL(t2)) break;
      PROCEED_UNIFY(t1,t2);
    } else if (TAG(t2) == T_AF) {
      if (TAG(t1) == T_STRUCT)
	t1 = REF(t1);
      if (VAL(t1) != VAL(t2)) break;
      PROCEED_UNIFY(t1,t2);
    } else if (TAG(t1) == T_STRUCT) {
      int i,st;

      if (TAG(t2) != T_STRUCT || VAL(REF(t1)) != VAL(REF(t2)))
	break;
      /*
       * t1,t2 sind Strukturen
       */
      st = ARITY(VAL(REF(t1)));

      if (st != 0) {
	CHECK_SPACE(st-1);

	for (i=2; i<=st; i++)
	  PUSH_PAIR(AF_ARG(t1,i), AF_ARG(t2,i));
	t1 = AF_ARG(t1,1);
	t2 = AF_ARG(t2,1);
      } else
	PROCEED_UNIFY(t1,t2);
    } else
      break;
  }
  fail();
}

/*
 * Allg. Matchalgorithmus
 */
void
match (t,pattern)
     term t,pattern;
{
  /*
   * Arbeits (Lauf-) variablen --> Register
   */
  register int upl = 0;		/* unify pairs left */
  register term *lsptr = R_HP;
  register term *termptr;

  forever {
    while (TAG(t) == T_VAR ||
	   TAG(t) == T_STRUCT && TAG(REF(t)) != T_AF)
      t = REF(t);
    while (TAG(pattern) == T_VAR ||
	   TAG(pattern) == T_STRUCT && TAG(REF(pattern)) != T_AF)
      pattern = REF(pattern);

    if (TAG(pattern) == T_UNDEF) {
      /*
       * pattern ist eine ungebundene Variable
       * t beliebig
       */
      if (OCCUR(VAL(pattern),t,lsptr))
	break;

      if (TAG(t) != T_UNDEF || VAL(pattern) != VAL(t))
	REPLACE(pattern,ENV,TAG(t),VAL(t));
      PROCEED_UNIFY(t,pattern);
    } else if (TAG(t) == T_UNDEF)
      break;
    else if (TAG(t) == T_LIST) {
      if (TAG(pattern) != T_LIST)
	break;
      /*
       * t,pattern sind Listen
       */
      CHECK_SPACE(1);

      PUSH_PAIR(TAIL(t), TAIL(pattern));
      t = REF(t);
      pattern = REF(pattern);
    } else if (TAG(t) == T_AF) {
      if (TAG(pattern) == T_STRUCT)
	pattern = REF(pattern);
      if (VAL(t) != VAL(pattern)) break;
      PROCEED_UNIFY(t,pattern);
    } else if (TAG(pattern) == T_AF) {
      if (TAG(t) == T_STRUCT)
	t = REF(t);
      if (VAL(t) != VAL(pattern)) break;
      PROCEED_UNIFY(t,pattern);
    } else if (TAG(t) == T_STRUCT) {
      int i,st;

      if (TAG(pattern) != T_STRUCT || VAL(REF(t)) != VAL(REF(pattern)))
	break;
      /*
       * t,pattern sind Strukturen
       */
      st = ARITY(VAL(REF(t)));

      if (st != 0) {
	CHECK_SPACE(st-1);

	for (i=2; i<=st; i++)
	  PUSH_PAIR(AF_ARG(t,i), AF_ARG(pattern,i));
	t = AF_ARG(t,1);
	pattern = AF_ARG(pattern,1);
      } else
	PROCEED_UNIFY(t,pattern);
    } else
      break;
  }
  fail();
}


/*
 * NICHT-Rekursiver reject-Algorithmus
 * Prueft die komplette Tiefe des Baums ausgehend von
 * den zwei Termen t1,t2.
 */
void
reject_terms (t1,t2)
    term t1,t2;
{
  /*
   * Arbeits (Lauf-) variablen --> Register
   */
  register int upl = 0;		/* unify pairs left (reject pairs left) */
  register term *lsptr = R_HP;
  register int e1 = 0;
  register int e2 = 0;

  forever {
    if (!e1)
      e1 = (LOC(t1) == ENV);
    if (!e2)
      e2 = (LOC(t2) == ENV);
    if (TAG(t1) == TAG(t2) && VAL(t1) == VAL(t2)) {
      if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
	return;
    } else if (TAG(t1) == T_VAR ||
	       TAG(t1) == T_STRUCT && TAG(REF(t1)) != T_AF)
      t1 = REF(t1);
    else if (TAG(t2) == T_VAR ||
	     TAG(t2) == T_STRUCT && TAG(REF(t2)) != T_AF)
      t2 = REF(t2);
    else if (TAG(t1) == T_UNDEF) {
      if (TAG(t2) == T_UNDEF) {
	if (UNBINDABLE(t1) && UNBINDABLE(t2))
	  break;
	if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
	  return;
      } else {
	if (UNBINDABLE(t1))
	  break;
	if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
	  return;
      }
    } else if (TAG(t2) == T_UNDEF) {
      if (UNBINDABLE(t2))
	break;
      if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
	return;
    } else if (!(IS_CONSTRUCTOR(t1,e1) && IS_CONSTRUCTOR(t2,e2))) {
      if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
	return;
    } else if (TAG(t1) == T_AF && TAG(t2) != T_LIST) {
      if (TAG(t2) == T_STRUCT)
	t2 = REF(t2);
      if (VAL(t1) != VAL(t2))
	break;
      if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
	return;
    } else if (TAG(t2) == T_AF && TAG(t1) != T_LIST) {
      if (TAG(t1) == T_STRUCT)
	t1 = REF(t1);
      if (VAL(t1) != VAL(t2))
	break;
      if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
	return;
    } else if (TAG(t1) != TAG(t2))
      break;
    else if (TAG(t1) == T_STRUCT) {
      if (VAL(REF(t1)) != VAL(REF(t2))) {
	if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
	  return;
      } else {
	int i,st = ARITY(VAL(REF(t1)));

	if (st == 0) {
	  if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
	    return;
	} else {
	  CHECK_SPACE(2*st-2);
	  for (i=2; i<=st; i++) {
	    PUSH_PAIR(AF_ARG(t1,i), AF_ARG(t2,i));
	    PUSH_IPAIR(e1,e2);
	  }
	  t1 = AF_ARG(t1,1);
	  t2 = AF_ARG(t2,1);
	}
      }
    } else if (TAG(t1) == T_LIST) {
      CHECK_SPACE(2);
      PUSH_PAIR(TAIL(t1), TAIL(t2));
      PUSH_IPAIR(e1,e2);
      t1 = REF(t1);
      t2 = REF(t2);
    } else if (!(POP_IPAIR(e1,e2) && POP_PAIR(t1,t2)))
      return;
  }
  fail();
}
