/*
 *
 *	sowam_c
 *		Bytecode-Interpreter fuer die SOWAM
 *
 *
 *	FILE
 *		sowam.c
 *
 *	PURPOSE
 *		Kommandozeilenparamter parsen, Speicherbloecke
 *		reservieren, Programmcode einlesen und starten ...
 *
 *
 *	AUTHORS
 *		Mike Wilhelm, Frank Lindert, Volker Siebert, Andreas Schwab
 *
 *
 */


#include <errno.h>
#include <sys/types.h>
#include <sys/times.h>
#include <string.h>
#include "getopt.h"
#include "sowam.h"

#define BANNER		"A-WAM -- Bytecode Interpreter, Version %s\n"

struct registers	s_regs;
struct flags		s_flags;
struct global_regs	s_glregs;
struct global_flags	s_glflags;
term 			s_xregs[256];

code_addr	b_code;
term 		*b_hp_ls;
term 		**b_occ_stk;
struct trail	*b_trail;
struct af_entry	*b_af_area;

	/* Groessen der Speicherbereiche */
int     hp_max = DEFAULT_HP_MAX;
int     ls_max = DEFAULT_LS_MAX;
int     os_max = DEFAULT_OS_MAX;
int     trl_max = DEFAULT_TRL_MAX;
int	af_max = DEFAULT_AF_MAX;
int	ta_max = DEFAULT_TA_MAX;

#ifdef STAT
term 		*heap_used;
term 		*ls_used;
term 		**os_used;
struct trail	*trail_used;

int		num_bos;

#define s_exit(n)	(statistics(), exit(n))
#else
#define s_exit(n)	exit(n)
#endif

term 		*hp_end;
term 		*ls_end;
term 		**os_end;
struct trail	*trail_end;

struct tms		start_time;

char *program_name;

#ifdef PILS
void label(str1, str2, str3)
     char *str1, *str2, *str3;
{
  extern char *getenv();
  extern char *tgetstr();
  char *tname;
  char bp[1024];
  char *so = NULL, *se = NULL;
  char buf[1024], *pbuf = buf;

  if ((tname = getenv("TERM")) && tgetent(bp, tname) == 1)
    {
      so = tgetstr("so", &pbuf);
      se = tgetstr("se", &pbuf);
    }
  if (so == NULL || se == NULL) so = se = "";

  fprintf(stderr, "\n\n\
       SSSSSS\n\
       PPPPPP        %s                              %s\n\
        IIII         %s  A-WAM Bytecode Interpreter  %s\n\
        LLLL         %s                              %s\n\
        SSSS         %s   Copyright (C) 1990 PILS    %s\n\
        PPPP         %s                              %s\n\
       IIIIII\n\
       LLLLLL\n\
      SS    SS\n\
     PP      PP\n\
    II        II\n\
    LL        LL\n\
    SS        SS     %s\n\
    PP        PP     %s\n\
    II        II     %s\n\
     LL      LL\n\
     SSSSSSSSSS\n\n\n",
	  so, se, so, se, so, se, so, se, so, se,
	  str1, str2, str3);
}
#endif

void usage()
{
#ifdef PILS
  label("=======================",
	"don't worry - try again",
	"=======================");
#endif

  fprintf(stderr, "\
Usage: %s [-odt] [-H bytes] [-L bytes] [-T bytes] [-O bytes]\n\
       [--occur-check] [--debug] [--trace] [--heap-size=bytes]\n\
       [--ls-size=bytes] [--trail-size=bytes] [--occ-size=bytes]\n\
       filename[.byt]\n", program_name);

  fprintf(stderr, "\n\
Default sizes:	heap		%8d\n\
		local stack	%8d\n\
		trail		%8d\n\
		occurence stack	%8d\n",
	  DEFAULT_HP_MAX * sizeof(term),
	  DEFAULT_LS_MAX * sizeof(term),
	  DEFAULT_TRL_MAX * sizeof(struct trail),
	  DEFAULT_OS_MAX * sizeof(term *));
  exit(2);
}


void internal_error(str)
     char *str;
{
#ifdef PILS
  label("Congratulations ...",
	"You found an internal BUG !@#%$&*",
	str);
#else
  fprintf(stderr, "\n%s: internal error: %s\n", program_name, str);
#endif

  exit(2);
}

void notimp (str)
     char *str;
{
#ifdef PILS
  label("Function", str, "isn't implemented yet");
#else
  fprintf(stderr, "%s: function %s not implemented yet\n", program_name, str);
#endif

  exit(2);
}

#ifdef STRERROR_MISSING
static char *
private_strerror (errnum)
     int errnum;
{
  extern char *sys_errlist[];
  extern int sys_nerr;

  if (errnum > 0 && errnum < sys_nerr)
    return sys_errlist[errnum];
  return "Unknown system error";
}
#define strerror private_strerror
#endif /* STRERROR_MISSING */

void error(message)
     char *message;
{
#ifdef PILS
  label(message ? message : "", strerror(errno), "");
#else
  fprintf(stderr, "%s: ", program_name);
  if (message && *message)
    fprintf(stderr, "%s: ", message);
  fprintf(stderr, "%s\n", strerror(errno));
#endif

  exit(2);
}

void fatal_error(str)
     char *str;
{
#ifdef PILS
  label("FATAL ERROR:", str, "");
#else
  fprintf(stderr, "%s: FATAL ERROR: %s\n", program_name, str);
#endif
  exit (2);
}

void runtime_error(str)
     char *str;
{
#ifdef PILS
  label("RUNTIME ERROR:", str, "");
#else
  fprintf(stderr, "%s: RUNTIME ERROR: %s\n", program_name, str);
#endif
  exit (2);
}

#ifdef STAT
void statistics()
{
  printf("\nMemory usage (bytes):\n");
  printf("                    max     used\n");
  printf("Heap:            %8d %8d\n", hp_max * sizeof(term),
	 (heap_used - HP_MIN) * sizeof(term));
  printf("Local Stack:     %8d %8d\n", ls_max * sizeof(term),
	 (ls_used - LS_MIN) * sizeof(term));
  printf("Trail:           %8d %8d\n", trl_max * sizeof(struct trail),
	 (TRL_MAX - trail_used) * sizeof(struct trail));
  printf("Occurence Stack: %8d %8d\n", os_max * sizeof(term *),
	 (os_used - OS_MIN) * sizeof(term *));
  printf("\nNumber of rebuild_occ_stack: %d\n", num_bos);
}
#endif

void os_overflow()
{
#ifdef PILS
  label ("=========================",
	 "Occurrence-Stack overflow",
	 "=========================");
#else
  fprintf(stderr, "%s: occurence stack overflow\n", program_name);
#endif
  s_exit (2);
}

void trl_overflow()
{
#ifdef PILS
  label("==============",
	"Trail overflow",
	"==============");
#else
  fprintf(stderr, "%s: trail overflow\n", program_name);
#endif
  s_exit (2);
}

void ls_overflow()
{
#ifdef PILS
  label("====================",
	"Local-Stack overflow",
	"====================");
#else
  fprintf(stderr, "%s: local stack overflow\n", program_name);
#endif
  s_exit (2);
}

void hp_overflow()
{
#ifdef PILS
  label ("=============",
	 "Heap overflow",
	 "=============");
#else
  fprintf(stderr, "%s: heap overflow\n", program_name);
#endif
  s_exit (2);
}


void af_overflow()
{
#ifdef PILS
  label ("================",
	 "AF-Area overflow",
	 "================");
#else
  fprintf(stderr, "%s: af area overflow\n", program_name);
#endif
  s_exit (2);
}

void stop(success)
     int success;
{
#ifdef PILS

  switch (success)
    {
    case YES:
      label("=============",
	    "     YES     ",
	    "=============");
      break;
    case NO:
      label("=============",
	    "     NO      ",
	    "=============");
      break;
    case ABORT:
      label("=================",
	    "Execution aborted",
	    "=================");
      s_exit(1);
      break;
    }

#else

  switch (success)
    {
    case YES:
      printf("\nyes\n");
      break;

    case NO:
      printf("\nno\n");
      break;

    case ABORT:
      fprintf(stderr, "%s: execution aborted\n", program_name);
      s_exit(1);
    }

#endif

  s_exit(0);
}


void *xmalloc(size)
     unsigned size;
{
  void *tmp = (void *) malloc(size);
  if (tmp == NULL)
    error("");
  return tmp;
}

void *xrealloc(old, size)
     void *old;
     unsigned size;
{
  void *tmp = (void *) realloc(old, size);
  if (tmp == NULL)
    error ("");
  return tmp;
}

char *savestring(s)
     char *s;
{
  int len = strlen(s);
  char *saved_s = xmalloc(len + 1);
  strcpy(saved_s, s);
  return saved_s;
}

void allocate_areas()
{
  b_hp_ls = (term *) xmalloc ((hp_max + ls_max) * sizeof(term));
  b_occ_stk = (term **) xmalloc (os_max * sizeof(term *));
  b_trail = (struct trail *) xmalloc (trl_max * sizeof(struct trail));

  hp_end	= b_hp_ls + hp_max;
  ls_end	= hp_end + ls_max;
  os_end	= b_occ_stk + os_max;
  trail_end	= b_trail + trl_max;
}

static struct option long_opts[] =
{
  { "occur-check",  0, NULL, 'o' },
  { "debug",        0, NULL, 'd' },
  { "trace",        0, NULL, 't' },
  { "heap-size",    1, NULL, 'H' },
  { "ls-size",      1, NULL, 'L' },
  { "trail-size",   1, NULL, 'T' },
  { "occ-size",	    1, NULL, 'O' },
  { NULL,	    0, NULL, 0 }
};

#define OPTIONS		"doctH:L:T:O:"

static int option_index;

void main(argc, argv)
     int argc;
     char *argv[];
{
  int c;

  program_name = argv[0];

  printf(BANNER, sowam_version);

  while ((c = getopt_long(argc, argv, OPTIONS, long_opts, NULL))
	 != EOF)
    {
      switch (c)
	{
	case 'o':
	case 'c':
	  s_glflags.occur_check = TRUE;
	  break;
	case 't':
	  s_glflags.trace = TRUE;
	  break;
	case 'd':
	  s_glflags.debug = TRUE;
	  break;
	case 'H':
	  hp_max = atoi(optarg) / sizeof(term);
	  break;
	case 'L':
	  ls_max = atoi(optarg) / sizeof(term);
	  break;
	case 'T':
	  trl_max = atoi(optarg) / sizeof(struct trail);
	  break;
	case 'O':
	  os_max = atoi(optarg) / sizeof(term *);
	  break;
	case '?':
	  usage(argv[0]);
	  break;
	default:
	  internal_error ("getopt");
	  break;
	}
    }
  if (argc - optind != 1)
    usage(argv[0]);

  /*
   * Alles klar, ab geht's
   */

  allocate_areas();
  read_bytecode(argv[optind]);
  sowam_setup();
  times(&start_time);
  interpret();
  s_exit(0);
}
