/*
 *
 *	sowam_c
 *		Bytecode-Interpreter fuer die SOWAM
 *
 *
 *	FILE
 *		output.c
 *
 *	PURPOSE
 *		Terme ausgeben
 *
 *
 *	AUTHORS
 *		Berthold Josephs, Andreas Schwab
 *
 *	HISTORY
 *		[001]	16.05.1990
 *			Erstimplementation
 *
 */

#include <string.h>
#include <sys/types.h>
#include <sys/times.h>
#include "prologdef.h"
#include "plread.h"
#include "streams.h"

enum char_class { cc_white, cc_alnum, cc_symbol, cc_single };

static enum char_class last_char_class = cc_white;

static void write_struct PROTO((FILE *, term *, int));
static void write_number PROTO((FILE *, term *, int));
static void write_name PROTO((FILE *, char *));
static void invalid_alf_term PROTO((term *));

#define write_open_paren(f) \
  ((last_char_class != cc_white && \
    last_char_class != cc_single ? fputc(' ',(f)) : 0), \
   fputc('(', (f)), \
   last_char_class = cc_single)

#define write_close_paren(f) \
  (fputc(')', (f)), last_char_class = cc_single)

#define write_space(f, class) \
  (((class) != cc_single && \
    last_char_class == (class) ? fputc(' ', (f)) : 0), \
   last_char_class = (class))

static void
write_struct(f, t, max_depth)
     FILE *f;
     term *t;
     int max_depth;
{
  int arity;
  int i;

  arity = AF_ARITY(VAL(*t));
  write_name(f, AF_NAME(VAL(*t)));
  if (arity != 0)
    {
      fputc('(', f);
      if (--max_depth <= 0)
	fprintf(f, "...");
      else
	for (i = 0; i < arity; i++)
	  {
	    if (i > 0) fputc(',', f);
	    write_internal(f, ++t, 1000, FALSE, max_depth);
	  }
      fputc(')', f);
    }
}

static void
write_number(f, t, max_depth)
     FILE *f;
     term *t;
     int max_depth;
{
  long i;

  for (i = 0; TAG(*t) == T_AF && VAL(*t) == SUCC_SYM; i++)
    {
      ++t;
      DEREF_TERMP(t);
    }

  if (TAG(*t) == T_AF && VAL(*t) == ZERO_SYM)
    fprintf(f, "%ld", i);
  else
    {
      fprintf(f, "(%ld+", i);
      write_internal(f, t, 1000, FALSE, max_depth-1);
      fputc(')', f);
      last_char_class = cc_single;
    }
}

static void
write_name(f, name)
     FILE *f;
     char *name;
{
  char *p;

#ifdef STRIP_QUALIFIER
  if ((p = strchr(name, QUALIFIER_SEPERATOR)) && p != name)
    fprintf(f, "%.*s", p - name, name);
  else
#endif
    fprintf(f, "%s", name);
}


#define SYMCHARS	"#$&*+-./:<=>?@\\^`~"

static void
write_alf_name(f, name)
     FILE *f;
     char *name;
{
  char *p;
  int do_quote = FALSE;
  enum char_class this_char_class;

  if (name[0] == '\0')
    do_quote = TRUE;
  else if (index(SYMCHARS, name[0]) != NULL)
    {
      this_char_class = cc_symbol;
      if (name[0] == '.' && name[1] == '\0')
	do_quote = TRUE;
      else
	for (p = &name[1]; *p; p++)
	  if (index(SYMCHARS, *p) == NULL)
	    {
	      do_quote = TRUE;
	      break;
	    }
    }
  else if (name[0] >= 'a' && name[0] <= 'z')
    {
      this_char_class = cc_alnum;
      for (p = &name[1]; *p; p++)
	if (!(*p >= 'a' && *p <= 'z' ||
	      *p >= 'A' && *p <= 'Z' || *p == '_' ||
	      *p >= '0' && *p <= '9')) {
	  do_quote = TRUE;
	  break;
	}
    }
  else if (strcmp(name, "[]") == 0 || strcmp(name, "!") == 0)
    this_char_class = cc_single;
  else
    do_quote = TRUE;

  if (do_quote)
    {
      write_space(f, cc_alnum);
      fputc('\'', f);
      while (*name)
	{
	  if (*name == '\'')
	    fputc('\'', f);
	  fputc(*name, f);
	  ++name;
	}
      fputc('\'', f);
    }
  else
    {
      write_space(f, this_char_class);
      fputs(name, f);
    }
}

static void
write_alf_string(f, str)
     FILE *f;
     char *str;
{
  write_space(f, cc_alnum);
  fputc('"', f);
  while (*str)
    {
      if (*str == '"')
	fputc('"', f);
      fputc(*str, f);
      ++str;
    }
  fputc('"', f);
}

void
write_internal(f, t, prec, same_prec, max_depth)
     FILE *f;
     term *t;
     int prec, same_prec, max_depth;
{
  int arity;
  int do_parens;

  if (max_depth <= 0)
    {
      fprintf(f, "...");
      return;
    }

  DEREF_TERMP(t);

  switch (TAG(*t))
    {
    case T_AF:
      if (A_F(VAL(*t)) >= MIN_STREAM)
	{
	  fprintf(f, "{stream %d}", A_F(VAL(*t)) - MIN_STREAM);
	  break;
	}

      arity = AF_ARITY(VAL(*t));
      do_parens = (AF_PRIORITY(VAL(*t)) > prec ||
		   AF_PRIORITY(VAL(*t)) == prec && !same_prec);
      switch (AF_OPTYPE(VAL(*t)))
	{
	case nonfix:
	  if (VAL(*t) == SUCC_SYM)
	    write_number(f, t, max_depth);
	  else
	    write_struct(f, t, max_depth);
	  break;

	case prefix:
	  if (arity == 1)
	    {
	      if (do_parens)
		fputc('(', f);
	      write_name(f, AF_NAME(VAL(*t)));
	      fputc(' ', f);
	      write_internal(f, t+1, AF_PRIORITY(VAL(*t)), FALSE, max_depth-1);
	      if (do_parens) fputc(')', f);
	    }
	  else
	    write_struct(f, t, max_depth);
	  break;

	case postfix:
	  if (arity == 1)
	    {
	      if (do_parens)
		fputc('(', f);
	      write_internal(f, t+1, AF_PRIORITY(VAL(*t)), FALSE, max_depth-1);
	      fputc(' ', f);
	      write_name(f, AF_NAME(VAL(*t)));
	      if (do_parens) fputc(')', f);
	    }
	  else
	    write_struct(f, t, max_depth);
	  break;

	case infixnot:
	case infixleft:
	case infixright:
	  if (arity == 2)
	    {
	      if (do_parens) fputc('(', f);
	      write_internal(f, t+1, AF_PRIORITY(VAL(*t)),
			     AF_OPTYPE(VAL(*t)) == infixleft,
			     max_depth-1);
	      fputc(' ', f);
	      write_name(f, AF_NAME(VAL(*t)));
	      fputc(' ', f);
	      write_internal(f, t+2, AF_PRIORITY(VAL(*t)),
			     AF_OPTYPE(VAL(*t)) == infixright,
			     max_depth-1);
	      if (do_parens) fputc(')', f);
	    }
	  else
	    write_struct(f, t, max_depth);
	  break;

	default:
	  internal_error("illegal op_type in write_internal");
	  break;
	}
      break;

    case T_UNDEF:
      {
	long offset = t - b_hp_ls;
	if (t != &REF(*t))
	  internal_error("illegal undef var");
	fputc('_', f);
	if (offset >= hp_max)
	  {
	    offset -= hp_max;
	    fputc('L', f);
	  }
	else
	  fputc('H', f);
	fprintf(f, "%d", offset);
      }
      break;

    case T_LIST:
      fputc('[', f);
      forever
	{
	  if (max_depth <= 0)
	    {
	      fprintf(f, "...]");
	      break;
	    }
	  t = &REF(*t);
	  write_internal(f, t++, 1000, FALSE, max_depth--);
	  DEREF_TERMP(t);
	  if (TAG(*t) == T_LIST)
	    fputc(',', f);
	  else if (ISNIL(*t))
	    {
	      fputc(']', f);
	      break;
	    }
	  else
	    {
	      fputc('|', f);
	      write_internal(f, t, 1000, FALSE, max_depth-1);
	      fputc(']', f);
	      break;
	    }
	}
      break;

    default:
      internal_error("illegal TAG in write_internal");
      break;
    }
}

static void
invalid_alf_term(t)
     term *t;
{
  fprintf(stderr, "\n* Invalid ALF-Term: ");
  write_internal(stderr, t, 1200, TRUE, 20);
  fputc('\n', stderr);
}

static void
write_alf_struct(f, name, args)
     FILE *f;
     char *name;
     term *args;
{
  term *t = args;

  write_alf_name(f, name);
  fputc('(', f);
  forever
    {
      last_char_class = cc_single;
      t = &REF(*t);
      write_alf_internal(f, t++, 1000, FALSE);
      DEREF_TERMP(t);
      if (TAG(*t) == T_LIST)
	fputc(',', f);
      else if (ISNIL(*t))
	break;
      else
	{
	  invalid_alf_term(args);
	  return;
	}
    }
  fputc(')', f);
  last_char_class = cc_single;
}

void
write_alf_internal(f, alf_term, outer_prec, same_prec)
     FILE *f;
     term *alf_term;
     int outer_prec, same_prec;
{
  int arity;
  int do_parens;
  term *t = alf_term, *arg, *arg1, *arg2;
  char *name;
  int symbol, prec, type;

  DEREF_TERMP(t);

  if (TAG(*t) != T_AF)
    {
      invalid_alf_term(alf_term);
      return;
    }

  switch (ARITY(VAL(*t)))
    {
    case 0:
      if (VAL(*t) != EOF_SYM && VAL(*t) != READERROR_SYM)
	invalid_alf_term(alf_term);
      break;

    case 1:
      switch (A_F(VAL(*t)))
	{
	case A_F(CONST_SYM):
	  t++;
	  DEREF_TERMP(t);
	  if (TAG(*t) != T_AF || ARITY(VAL(*t)) != 0)
	    {
	      invalid_alf_term(alf_term);
	      break;
	    }
	  lookup_operator(VAL(*t), &prec, &type);
	  do_parens = type && (prec > outer_prec ||
			       prec == outer_prec && !same_prec);
	  if (do_parens) write_open_paren(f);
	  write_alf_name(f, AF_NAME(VAL(*t)));
	  if (do_parens) write_close_paren(f);
	  break;

	case A_F(VAR_SYM):
	  t++;
	  DEREF_TERMP(t);
	  if (TAG(*t) != T_AF || ARITY(VAL(*t)) != 0)
	    invalid_alf_term(alf_term);
	  else
	    {
	      write_space(f, cc_alnum);
	      fputs(AF_NAME(VAL(*t)), f);
	    }
	  break;

	case A_F(STR_SYM):
	  t++;
	  DEREF_TERMP(t);
	  if (TAG(*t) != T_AF || ARITY(VAL(*t)) != 0)
	    invalid_alf_term(alf_term);
	  else
	    write_alf_string(f, AF_NAME(VAL(*t)));
	  break;

	case A_F(INT_SYM):
	  t++;
	  DEREF_TERMP(t);
	  write_space(f, cc_alnum);
	  write_number(f, t, 0x7fffffff);
	  break;

	default:
	  invalid_alf_term(alf_term);
	  break;
	}
      break;

    case 2:
      switch (A_F(VAL(*t)))
	{
	case A_F(COMP_SYM):
	  arg = t + 1;
	  DEREF_TERMP(arg);
	  if (TAG(*arg) != T_AF || ARITY(VAL(*arg)) != 0)
	    {
	      invalid_alf_term(alf_term);
	      break;
	    }
	  symbol = VAL(*arg);
	  t += 2;
	  DEREF_TERMP(t);
	  if (TAG(*t) == T_LIST)
	    {
	      arg1 = &REF(*t);
	      arg = &TAIL(*t);
	      DEREF_TERMP(arg);
	      if (TAG(*arg) == T_LIST)
		{
		  arg2 = &REF(*arg);
		  arg = &TAIL(*arg);
		  DEREF_TERMP(arg);
		  if (TAG(*arg) == T_LIST)
		    arity = 3;	/* 3 == viele */
		  else if (ISNIL(*arg))
		    arity = 2;
		  else
		    {
		      invalid_alf_term(alf_term);
		      break;
		    }
		}
	      else if (ISNIL(*arg))
		arity = 1;
	      else
		{
		  invalid_alf_term(alf_term);
		  break;
		}

	      if (arity > 2)
		/* kann kein Operator sein */
		write_alf_struct(f, AF_NAME(symbol), t);
	      else
		{
		  /* Operatortyp und Praezedenz holen */
		  lookup_operator(symbol, &prec, &type);
		  do_parens = (prec > outer_prec ||
			       prec == outer_prec && !same_prec);
		  if (arity == 1 && type & (OP_PREFIX|OP_POSTFIX))
		    {
		      if (do_parens) write_open_paren(f);
		      if (type & OP_PREFIX)
			{
			  write_alf_name(f, AF_NAME(symbol));
			  write_alf_internal(f, arg1, prec, FALSE);
			}
		      else if (type & OP_POSTFIX)
			{
			  write_alf_internal(f, arg1, prec, FALSE);
			  write_alf_name(f, AF_NAME(symbol));
			}
		      if (do_parens) write_close_paren(f);
		    }
		  else if (arity == 2 && type & OP_INFIX)
		    {
		      if (do_parens) write_open_paren(f);
		      write_alf_internal(f, arg1, prec, (type & OP_YFX) != 0);
		      write_alf_name(f, AF_NAME(symbol));
		      write_alf_internal(f, arg2, prec, (type & OP_XFY) != 0);
		      if (do_parens) write_close_paren(f);
		    }
		  else
		    write_alf_struct(f, AF_NAME(symbol), t);
		}
	    }
	  else if (ISNIL(*t))
	    {
	      lookup_operator(symbol, &prec, &type);
	      do_parens = type && (prec > outer_prec ||
				   prec == outer_prec && !same_prec);
	      if (do_parens) write_open_paren(f);
	      write_alf_name(f, AF_NAME(symbol));
	      if (do_parens) write_close_paren(f);
	    }
	  else
	    invalid_alf_term(alf_term);
	  break;

	case A_F(LIST_SYM):
	  fputc('[', f);
	  forever
	    {
	      last_char_class = cc_single;
	      write_alf_internal(f, ++t, 1000, FALSE);
	      ++t;
	      DEREF_TERMP(t);
	      if (TAG(*t) == T_AF)
		{
		  if (VAL(*t) == LIST_SYM)
		    fputc(',', f);
		  else if (VAL(*t) == CONST_SYM)
		    {
		      arg = t + 1;
		      DEREF_TERMP(arg);
		      if (!ISNIL(*arg))
			{
			  fputc('|', f);
			  last_char_class = cc_single;
			  write_alf_internal(f, t, 1000, FALSE);
			}
		      fputc(']', f);
		      break;
		    }
		  else
		    {
		      fputc('|', f);
		      last_char_class = cc_single;
		      write_alf_internal(f, t, 1000, FALSE);
		      fputc(']', f);
		      break;
		    }
		}
	      else
		{
		  invalid_alf_term(alf_term);
		  break;
		}
	    }
	  last_char_class = cc_single;
	  break;

	default:
	  invalid_alf_term(alf_term);
	  break;
	}
      break;

    default:
      invalid_alf_term(alf_term);
      break;
    }
}

void
write_term(f, t)
     FILE *f;
     term *t;
{
  write_internal(f, t, 1200, TRUE, 0x7fffffff);
}

void
debug_write_term(t, max_depth)
     term *t;
     int max_depth;
{
  write_internal(stdout, t, 1200, TRUE, max_depth);
}

void
write_alf_term(f, t)
     FILE *f;
     term *t;
{
  last_char_class = cc_single;
  write_alf_internal(f, t, 1200, TRUE);
}

void
write_and_ask(arg)
     int arg;
{
  int c, c1;
  term *t;

  t = &X_REG(arg);
  if (TAG(*t) == T_LIST)
    forever
      {
	t = &REF(*t);
	write_term(stdout, t++);
	if (TAG(*t) == T_LIST)
	  printf(", ");
	else
	  break;
      }
  else
    internal_error("write_and_ask");

  putchar(' ');
  fflush(stdout);
  forever
    {
      while ((c = getchar()) == ' ' || c == '\t');
      if (c == '\n' || c == EOF) stop(TERMINATE);
      while ((c1 = getchar()) != '\n' && c1 != EOF);
      if (c == ';')
	{
	  times(&start_time);
	  fail();
	  return;
	}
      fputs("Action (\";\" for more choices, otherwise <return>): ", stdout);
    }
}
